//>>> _using
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SharpDX;
using SharpDX.Direct3D11;
using SharpDX.Windows;
//<<< _using
using System.IO;

using Newtonsoft.Json;

namespace Framefield.Core.ID06d920b1_01ae_413c_82f7_98aabc5241eb
{
    public class Class_SoundLevels : OperatorPart.Function
    {
        //>>> _inputids
        private enum InputId
        {
            FilePath = 0,
            Time = 1,
            Threshold = 2,
            Smooth = 3,
            BeatTimeMode = 4,
            FlashDecay = 5,
            MinTimeBetweenPeaks = 6
        }
        //<<< _inputids
        
        //>>> _outputids
        private enum OutputId
        {
            Level = 0,
            BeatIndex = 1,
            BeatTime = 2,
            SmoothLevel = 3
        }
        //<<< _outputids


        private void SmoothBuffer(ref float[] inBuffer, ref float[] outBuffer, int sampleCount, int stepWidth =1) 
        {
            for(var i = 0; i < inBuffer.Length; i++ ) 
            {
                var average =0f;
                var count = 0f;
                
                for(var ds = 0 ; ds < sampleCount; ds++ ) 
                {
                    var smoothI = i + (-sampleCount / 2 + ds) * stepWidth  ;
                    
                    if(smoothI < 0 || smoothI >= inBuffer.Length)
                        continue;
                        
                    average+= inBuffer[smoothI];
                    count ++;
                }
                outBuffer[i] = average/count;                    
            }
        }

        public override OperatorPartContext Eval(OperatorPartContext context, List<OperatorPart> inputs, int outputIdx) {
            //>>> _params
            var FilePath = inputs[(int)InputId.FilePath].Eval(context).Text;
            var Time = inputs[(int)InputId.Time].Eval(context).Value;
            var Threshold = inputs[(int)InputId.Threshold].Eval(context).Value;
            var Smooth = inputs[(int)InputId.Smooth].Eval(context).Value;
            var BeatTimeMode = (int) inputs[(int)InputId.BeatTimeMode].Eval(context).Value;
            var FlashDecay = inputs[(int)InputId.FlashDecay].Eval(context).Value;
            var MinTimeBetweenPeaks = inputs[(int)InputId.MinTimeBetweenPeaks].Eval(context).Value;
            //<<< _params
                        
            
            var needsRecalcAverage = false;




            if( FilePath != _filepath) {
                //Logger.Info("here" + FilePath);
                _filepath = FilePath;
                
                if (File.Exists(_filepath))
                {
                    using (var reader = new StreamReader(FilePath))
                    {
                        var jsonString = reader.ReadToEnd();
                        _uplevels = JsonConvert.DeserializeObject<float[]>(jsonString);
                        if (_uplevels == null || _uplevels.Length == 0)
                        {
                            Logger.Warn("Loading sound levels palletes failed");
                            return context;
                        }        
                        //Logger.Info(this, "Updated volumes. Samples" + _uplevels.Length);
                    }                    
                }
                else {
                    Logger.Warn(this, "File doesn't exist:  " + _filepath);
                }
                needsRecalcAverage = true;
            }
            
            var smoothWindow = (int)Utilities.Clamp(Smooth, 1, MAX_SMOOTH_WINDOW);
            
            needsRecalcAverage |= smoothWindow != _smoothWindow;
            
            
            if(needsRecalcAverage) 
            {
                _minTimeBetweenPeaks = MinTimeBetweenPeaks;
                _smoothWindow = smoothWindow;                                
                _maxLevel = 0f;
                _averageLevels = new float[_uplevels.Length];
                _averageLevels2 = new float[_uplevels.Length];
                
                foreach(var l in _uplevels) 
                {
                    _maxLevel = (float)Math.Max(l, _maxLevel);
                }
                
                SmoothBuffer( ref _uplevels, ref _averageLevels2, _smoothWindow, 1);
                SmoothBuffer( ref _averageLevels2, ref _averageLevels, _smoothWindow , 2);
                SmoothBuffer( ref _averageLevels, ref _averageLevels2, _smoothWindow , 4);
                SmoothBuffer( ref _averageLevels2, ref _averageLevels, _smoothWindow , 8);
            }
            
            
            bool needsRescanBeats = needsRecalcAverage;
            
            needsRescanBeats |= Threshold != _threshold;            
            needsRescanBeats |= MinTimeBetweenPeaks != _minTimeBetweenPeaks;
            
            if(needsRescanBeats) {
                _minTimeBetweenPeaks = MinTimeBetweenPeaks;
                _threshold = Threshold;                
                UpdateAllBeatNumbers();
            }



            var index = (int)(Time * SAMPLE_RESOLUTION_PER_SECOND); 
            bool needToFindNewBoundarys = (index <= _beatStartIndex || index >= _beatEndIndex);
            needToFindNewBoundarys |= needsRescanBeats;
            
            if(needToFindNewBoundarys) {            
               FindBoundarysFromBeatNumbers(index, out _beatStartIndex, out _beatEndIndex);
            }
            
            //FindBeatBoundaries(index, ref _beatStartIndex, ref _beatEndIndex);
            
            context.Value = 0;            
            
            if(_uplevels != null && index < _uplevels.Length && index > 0 && _maxLevel > 0.001f) 
            {
                switch((OutputId)outputIdx) {
                    case OutputId.Level:
                        context.Value = _uplevels[index];
                        break;
                        
                    case OutputId.BeatTime:
                        var mode = (OutputType)(BeatTimeMode);
                        switch((OutputType)(BeatTimeMode)) 
                        {                    
                            case OutputType.Peaks:
                                context.Value = (float)Math.Max(0, _uplevels[index] - _averageLevels[index]);                    
                            	break;
                            	
                            case OutputType.Flashes:
                                var t = ((float)index - _beatStartIndex) / (float)SAMPLE_RESOLUTION_PER_SECOND;
                                context.Value =  (float)Math.Pow(2.71f, -FlashDecay* t);
                            	break;
                            	
                            case OutputType.FlashesWithIntensity:
                                var t2 = ((float)index - _beatStartIndex) / (float)SAMPLE_RESOLUTION_PER_SECOND;
                                context.Value =  (float)Math.Pow(2.71f, -FlashDecay* t2) * _uplevels[_beatStartIndex];
                            	break;
                            	
                            case OutputType.TimeSincePeak:
                                context.Value = ((float)index - _beatStartIndex) / (float)SAMPLE_RESOLUTION_PER_SECOND;
                            	break;
                            	
                            case OutputType.TimeToPeak:
                                context.Value = (_beatEndIndex  - (float)index) / (float)SAMPLE_RESOLUTION_PER_SECOND;                    
                            	break;
                        }
                        break;
                    
                    case OutputId.BeatIndex:
                        //context.Value = (float)_beatStartIndex;
                        context.Value = (float)_beatNumbers[index];
                    	break;
                    	
                    case OutputId.SmoothLevel:
                        context.Value = _averageLevels[index];
                        break;
                                            	
                }
            }
            
            return context;
        }
        
        private enum OutputType  {
            Peaks = 1,
            Flashes,
            TimeSincePeak,
            TimeToPeak,
            FlashesWithIntensity,
        }
        
        private void UpdateAllBeatNumbers() 
        {
        
            _beatNumbers = new int[_uplevels.Length];
            var lastUpperIndex = 0;
            var lastLowerIndex = 0;
            var lastBeatStartIndex =0;
            int beatsFound = 0;
            
            for(int i=0; i < _uplevels.Length; i++) 
            {
                if( i>= lastUpperIndex) 
                {
                    
                    FindBeatBoundaries(i, ref lastLowerIndex, ref lastUpperIndex);
                    if(lastUpperIndex > lastBeatStartIndex + _minTimeBetweenPeaks * SAMPLE_RESOLUTION_PER_SECOND) {                    
                        beatsFound++;
                        lastBeatStartIndex = lastUpperIndex;
                    }
                }
                _beatNumbers[i] = beatsFound;
            }
            
            
        }
        

        private bool IsIndexValid( int timeIndex) {            
            if(timeIndex < 0 || timeIndex >= _uplevels.Length)
                return false;
            return true;
        }
        
        
        /** Use the beat index array to find boundaries.
            We can't use the original method, because MinTimeBetween peaks would be too complicated to implement
        */
        private void FindBoundarysFromBeatNumbers(int timeIndex, out int leftBoundaryIndex, out int rightBoundaryIndex) 
        {
            leftBoundaryIndex = timeIndex;
            rightBoundaryIndex = timeIndex;        
            if( !IsIndexValid(timeIndex)) return;
            
            
            var currentBeatNumber = _beatNumbers[timeIndex];
            
            while(leftBoundaryIndex > 0 && _beatNumbers[leftBoundaryIndex - 1] == currentBeatNumber) {
                leftBoundaryIndex--;
            }

            while(rightBoundaryIndex < _beatNumbers.Length - 1 && _beatNumbers[rightBoundaryIndex + 1] == currentBeatNumber) {
                rightBoundaryIndex++;
            }            
        }


        private void FindBeatBoundaries(int timeIndex, ref int leftBoundaryIndex, ref int rightBoundaryIndex) 
        {
            if( !IsIndexValid(timeIndex)) return;

            const int MAX_STEPS = 12 * SAMPLE_RESOLUTION_PER_SECOND;
            
            // Find left boundary
            bool wasAboveThreshold = false;
            var maxLevel = 0f;
            var steps = 0;
                            
            for(steps = 0; steps < MAX_STEPS && (timeIndex - steps) >=0; steps++) 
            {                                
                leftBoundaryIndex = timeIndex - steps;
                var level = _uplevels[leftBoundaryIndex] - _averageLevels[leftBoundaryIndex];                
                maxLevel = (float)Math.Max(maxLevel, level);
                var isAboveThreshold = level > _threshold;
                var isFlank= (steps > 0 && wasAboveThreshold && !isAboveThreshold);
                
                if(isFlank) {
                    break;
                }                
                wasAboveThreshold = isAboveThreshold;                
            }

            // Find right boundary            
            for(steps = 0; steps < MAX_STEPS && (timeIndex + steps) < _averageLevels.Length; steps++) 
            {                                
                rightBoundaryIndex = timeIndex + steps;
                var level = _uplevels[rightBoundaryIndex] - _averageLevels[rightBoundaryIndex];                
                var isAboveThreshold = level > _threshold;
                var isFlank= (steps > 0 && !wasAboveThreshold && isAboveThreshold);
                
                if(isFlank) {
                    break;
                }                
                wasAboveThreshold = isAboveThreshold;                
            }
        }
        
        private const int MAX_SMOOTH_WINDOW = 500;
        
        private const int SAMPLE_RESOLUTION_PER_SECOND = 100;
        string _filepath = ""; 
        string _content ="";        
        private float[] _uplevels = new float[0];
        private int _smoothWindow=-1;
        
        private float[] _averageLevels= new float[0];
        private float[] _averageLevels2 = new float[0];
        private int[] _beatNumbers = new int[0];
        
        private float _maxLevel;
        private int _beatStartIndex;
        private int _beatEndIndex;
        private float _threshold=0;
        private float _minTimeBetweenPeaks = 0.05f;
        //private const float MIN_BEAT_LENGTH = 0.05f;
        
    }
}

