#version 430 core

layout (location=0) out vec4 color;
layout (location=1) out vec4 bright;

layout (location=2) uniform int t;
in vec2 coord;

const float height = 1080.0;
const float width = 1920.0;
const float pi = 3.1415926536;

float zaw(float freq, float time)
{
	float f = 1.0 / freq;
	return 1.0 - freq * mod(time, f);
}

float alternate(float a, float b, float freq, float time)
{
	return (zaw(freq, time) < 0.5) ? a : b;
}

float beat(float time)
{
	return zaw(2.0, time) * alternate(0.25, 1.0, 0.5, time);
}

vec2 fixCoords(vec2 uv)
{
	return vec2(uv.x * width / height, uv.y);
}

vec3 rotate(in vec3 pos, in vec3 angles)
{
	float xrot = angles.x;
	float yrot = angles.y;
	float zrot = angles.z;

	mat3 rmx = mat3(vec3(1., 0., 0.), vec3(0., cos(xrot), sin(xrot)), vec3(0., -sin(xrot), cos(xrot)));
	mat3 rmy = mat3(vec3(cos(yrot), 0., -sin(yrot)), vec3(0., 1., 0.), vec3(sin(yrot), 0., cos(yrot)));
	mat3 rmz = mat3(vec3(cos(zrot), sin(zrot), 0.), vec3(-sin(zrot), cos(zrot), 0.), vec3(0., 0., 1.));
	return pos * rmx * rmy * rmz;
}

float sphere(vec3 pos, float s)
{
	return length(pos) - s;
}

float torus(vec3 p, vec2 t)
{
	vec2 q = vec2(length(p.xz)-t.x,p.y);
	return length(q)-t.y;
}

float fixsin(in float val)
{
	return 0.5 + 0.5 * sin(val);
}

float sdf1(in vec3 pos, in float time)
{
	return sphere(pos, 1.0);
}

float sdf(in vec3 pos, in float time)
{
	return sphere(pos, 0.5+beat(time));
}

float sdf2(in vec3 pos, in float time)
{
	float d = sphere(pos, 0.5+beat(time));
	vec3 torusPos = rotate(pos, vec3(0, cos(time), sin(time)));
	float scale = 1.5 + sin(time);
	d = min(d, scale * torus(torusPos/scale, vec2(1.0 + beat(time), 0.2 + beat(time) * 0.2)));
	return d;
}

vec3 rayCast(vec3 end)
{
	vec3 start = vec3(0.0, 0.0, end.z - 1.0);
	return normalize(end - start);
}

vec3 scene1(in float time)
{
	vec3 rgb = vec3(0);
	vec2 uv =  fixCoords(coord);

	uv.x = uv.x + cos(uv.y + time) * time * 0.2;
	uv.y = uv.y + cos(uv.x + time) * time * 0.2;

	rgb.b = 1.5 * step(0.95, cos(uv.x*10.0));
	rgb.g = 1.5 * step(0.95, cos(uv.y*10.0));
	return rgb;
}

vec3 sdfNormal(vec3 pos, float time)
{
	const float ep = 0.0001;
	vec2 e = vec2(1.0,-1.0)*0.5773;
	return normalize(vec3(
				e.xyy * sdf(pos + ep * e.xyy, time) +
				e.yyx * sdf(pos + ep * e.yyx, time) +
				e.yxy * sdf(pos + ep * e.yxy, time) +
				e.xxx * sdf(pos + ep * e.xxx, time)));
}

vec3 sdfNormal2(vec3 pos, float time)
{
	const float ep = 0.0001;
	vec2 e = vec2(1.0,-1.0)*0.5773;
	return normalize(vec3(
				e.xyy * sdf2(pos + ep * e.xyy, time) +
				e.yyx * sdf2(pos + ep * e.yyx, time) +
				e.yxy * sdf2(pos + ep * e.yxy, time) +
				e.xxx * sdf2(pos + ep * e.xxx, time)));
}

vec3 scene2(in float time)
{
	vec2 fix = fixCoords(coord);
	vec3 start = vec3(0.0, 3.0, 6.0);
	vec3 dir = normalize(vec3(fix-vec2(0.0, 1.0), -2.0));

	vec3 pos = start;
	float dist = sdf(pos, time);
	float len = 0.0;
	int steps = 0;
	while (abs(dist) > 0.001 && len < 100.0 && steps < 100)
	{
		steps++;
		len += dist;
		pos = start + len * dir;
		dist = sdf1(pos, time);
	}
	vec3 rgb = vec3(fract(len + beat(time)));
	return rgb;
}

vec3 scene3(in float time)
{
	vec2 uv = 5.0 * fixCoords(coord);

	vec2 xy = vec2(0);
	xy.x = uv.x + time + beat(time);
	xy.y = sin(xy.x) * uv.y + time + beat(time);

	vec3 rgb = vec3(0);
	rgb.r = step(0.0, sin(xy.x+xy.y));
	rgb.b = step(0.0, sin(xy.x-xy.y));
	return rgb;
}

float raySphere(in vec3 start, in vec3 dir, in vec3 cen, in float r, out vec3 point)
{
	vec3 m = start - cen;
	float b = dot(m, dir);
	float c = dot(m, m) - r*r;
	if(c > 0.0 && b > 0.0)
		return -1.0;
	float discr = b*b - c;
	if(discr < 0.0)
		return -1.0;
	float t = -b - sqrt(discr);
	point = start + t * dir;
	return t;
}

float backgroundGrid(in float time, in vec2 uv)
{
	vec2 p = uv * 3.0;
	vec2 clp = clamp(uv * 3.0, vec2(-4.0, -2.0), vec2(4.0, 2.0));

	vec2 fp = floor(p);
	vec2 cp = ceil(p);

	vec2 d = (clp == p)
		? min(abs(p-fp), abs(p-cp))
		: vec2(max(abs(clp.x - p.x), abs(clp.y - p.y)), 1.0);
	return min(d.x, d.y);
}

vec2 ballPosition(in float time)
{
	return vec2(1.5 * cos(time * 0.2), 1.5*abs(sin(time)*sin(time))-0.7);
}

float ballGrid(in float time, in vec2 uv, in float rad, in vec2 ball)
{
	vec3 start = vec3(uv, -3.0);
	vec3 dir = vec3(0.0, 0.0, 1.0);
	vec3 point;
	float t = raySphere(start, dir, vec3(ball, 0.0), rad, point);
	if(t < 0.0)
		return 1.0;
	vec3 d = point - vec3(ball, 0.0);

	float xrot = 0.2 * cos(time*0.2);
	float yrot = 0.3 * sin(time*0.3);
	float zrot = 0.12 * time;

	mat3 rmx = mat3(vec3(1., 0., 0.), vec3(0., cos(xrot), sin(xrot)), vec3(0., -sin(xrot), cos(xrot)));
	mat3 rmy = mat3(vec3(cos(yrot), 0., -sin(yrot)), vec3(0., 1., 0.), vec3(sin(yrot), 0., cos(yrot)));
	mat3 rmz = mat3(vec3(cos(zrot), sin(zrot), 0.), vec3(-sin(zrot), cos(zrot), 0.), vec3(0., 0., 1.));

	d.xyz = d.xzy;
	d = rmx * rmy * rmz * d;
	float r = length(d);
	float theta = acos(d.z / r);
	float gamma = atan(d.y / d.x);

	float grid = pi;
	float deltaTheta = min(abs(floor(theta * grid) - theta * grid), abs(ceil(theta * grid) - theta * grid));
	float deltaGamma = min(abs(floor(gamma * grid) - gamma * grid), abs(ceil(gamma * grid) - gamma * grid));
	return min(deltaTheta, deltaGamma);
}

vec3 scene4(in float time)
{
	vec2 uv = fixCoords(coord);
	vec2 ball = ballPosition(time);
	float val = backgroundGrid(time, uv);
	float r = 0.4 + 0.2 * beat(time);
	if (distance(ball, uv) < r)
		val = ballGrid(time, uv, r, ball);
	vec3 rgb = vec3(0);
	rgb.r = smoothstep(0.96, 0.98, 1.0 - val);
	rgb.g = smoothstep(0.96, 0.98, 1.0 - val);
	rgb.b = smoothstep(0.85, 0.98, 1.0 - val);
	return rgb;
}

vec3 scene5(in float time)
{
	vec2 fix = fixCoords(coord);
	vec3 start = vec3(0.0, 3.0, 6.0);
	vec3 dir = normalize(vec3(fix-vec2(0.0, 1.0), -2.0));

	vec3 pos = start;
	float dist = sdf(pos, time);
	float len = 0.0;
	int steps = 0;
	while (abs(dist) > 0.001 && len < 100.0 && steps < 100)
	{
		steps++;
		len += dist;
		pos = start + len * dir;
		dist = sdf(pos, time);
	}
	vec3 normal = sdfNormal(pos, time);
//	vec3 sun = vec3(0.57703);
//	float dif = clamp(dot(normal, sun), 0.0, 1.0);
//	vec3 rgb = (dist < 0.01) ? vec3(0.02, 0.05, 0.08) + dif * vec3(1.0, 0.9, 0.8) : vec3(0);
	vec3 rgb = (len > 100.0) ? vec3(fract(normal)) : vec3(0.0);
	return rgb;
}

vec3 scene6(in float time)
{
	vec2 fix = fixCoords(coord);
	vec3 start = vec3(0.0, 3.0, 6.0);
	vec3 dir = normalize(vec3(fix-vec2(0.0, 1.0), -2.0));

	vec3 pos = start;
	float dist = sdf2(pos, time);
	float len = 0.0;
	int steps = 0;
	while (abs(dist) > 0.001 && len < 100.0 && steps < 100)
	{
		steps++;
		len += dist;
		pos = start + len * dir;
		dist = sdf2(pos, time);
	}
	vec3 normal = sdfNormal2(pos, time);
	vec3 sun = vec3(0.57703);
	float dif = clamp(dot(normal, sun), 0.0, 1.0);
	vec3 rgb = (len < 100.0) ? vec3(0.02, 0.05, 0.08) + dif * vec3(1.0, 0.9, 0.8) : fract(normal * 0.5 + time);
	return rgb;
}

vec3 scene7(in float time)
{
	vec2 fix = fixCoords(coord);
	vec3 start = vec3(0.0, 3.0, 6.0);
	vec3 dir = normalize(vec3(fix-vec2(0.0, 1.0), -2.0));

	vec3 pos = start;
	float dist = sdf2(pos, time);
	float len = 0.0;
	int steps = 0;
	while (abs(dist) > 0.001 && len < 100.0 && steps < 100)
	{
		steps++;
		len += dist;
		pos = start + len * dir;
		dist = sdf2(pos, time);
	}
	vec3 normal = sdfNormal2(pos, time);
	vec3 sun = vec3(0.57703);
	float dif = clamp(dot(normal, sun), 0.0, 1.0);
	vec3 rgb = (len < 100.0) ? vec3(0.02, 0.05, 0.08) + dif * normal : fract(normal * 0.5 + time);
	return rgb;
}

vec3 scene(in float time)
{
	float times[6] = float[](10.0, 15.0, 20.0, 25.0, 25.0, 30.0);
	vec3 rgb = vec3(0.0, 0.0, 0.0);
	//rgb += scene7(time);
	if (time < times[0])
		rgb += scene1(time);
	if (time > times[0] && time < times[1])
		rgb += scene2(time);
	if (time > times[1] && time < times[2])
		rgb += scene3(time);
	if (time > times[2] && time < times[3])
		rgb += scene4(time);
	if (time > times[3] && time < times[4])
		rgb += scene5(time);
	if (time > times[4] && time < times[5])
		rgb += scene6(time);
	if (time > times[5])
		rgb += scene7(time);
	return rgb;
}

void main(){
	vec3 rgb = scene(t * 0.001);

	color = vec4(clamp(rgb, 0.0, 1.0), 1.0f);
	float brightness = dot(rgb, vec3(0.2126, 0.7152, 0.0722));
	bright = (brightness > 1.0) ? vec4(rgb, 1.0f) : vec4(0.0, 0.0, 0.0, 1.0);
}

