import XCTest
import libc
@testable import Core

private enum PortalTestError: Error {
    case someError
    case anotherError
}

class PortalTests: XCTestCase {
    static let allTests = [
        ("testPortalResult", testPortalResult),
        ("testPortalFailure", testPortalFailure),
        ("testPortalNotCalled", testPortalNotCalled),
        ("testPortalTimedOut", testPortalTimedOut),
        ("testTimeout", testTimeout),
        ("testDuplicateResults", testDuplicateResults),
        ("testDuplicateErrors", testDuplicateErrors),
        ("testPortalThrowHandler", testPortalThrowHandler),
        ("testPortalErrorDebugging", testPortalErrorDebugging),
    ]

    func testPortalResult() throws {
        var array: [Int] = []

        array.append(1)
        let result: Int = try Portal.open { portal in
            array.append(2)
            background {
                sleep(1)
                array.append(4)
                portal.close(with: 42)
            }
            array.append(3)
        }
        array.append(5)

        XCTAssert(array == [1,2,3,4,5])
        XCTAssert(result == 42)
    }

    func testPortalFailure() {
        var array: [Int] = []

        do {
            array.append(1)
            let _ = try Portal<Int>.open { portal in
                array.append(2)
                background {
                    sleep(1)
                    array.append(4)
                    portal.close(with: PortalTestError.someError)
                }
                array.append(3)
            }
            XCTFail("Portal should throw")
        } catch PortalTestError.someError {
            // success
        } catch {
            XCTFail("Unexpected error thrown: \(error)")
        }

        XCTAssert(array == [1,2,3,4])
    }

    func testPortalNotCalled() {
        do {
            let _ = try Portal<Int>.open { portal in
                portal.destroy()
            }
            XCTFail("Should not have passed")
        } catch PortalError.notClosed {
            // pass
        } catch {
            XCTFail("Wrong error: \(error)")
        }
    }

    func testPortalTimedOut() {
        do {
            let _ = try Portal<Int>.open(timeout: 0) { portal in
                //
            }
            XCTFail("Should not have passed")
        } catch PortalError.timedOut {
            // pass
        } catch {
            XCTFail("Wrong error: \(error)")
        }
    }

    func testTimeout() throws {
        let result = try Portal<Int>.timeout(1) {
            return 1
        }

        XCTAssertEqual(result, 1)
    }

    func testDuplicateResults() throws {
        let response = try Portal<Int>.open { portal in
            portal.close(with: 10)
            // subsequent resolutions should be ignored
            portal.close(with: 400)
        }

        XCTAssert(response == 10)
    }

    func testDuplicateErrors() {
        do {
            let _ = try Portal<Int>.open { portal in
                portal.close(with: PortalTestError.someError)
                // subsequent rejections should be ignored
                portal.close(with: PortalTestError.anotherError)
            }
            XCTFail("Test should not pass")
        } catch PortalTestError.someError {
            // success
        } catch {
            XCTFail("Unexpected error thrown")
        }
    }

    func testPortalThrowHandler() throws {
        do {
            _ = try Portal<Int>.open { _ in throw PortalTestError.someError }
        } catch PortalTestError.someError {}
    }

    func testPortalErrorDebugging() {
        XCTAssertTrue(PortalError.notClosed.printable.contains("not properly closed"))
        XCTAssertTrue(PortalError.timedOut.printable.contains("timed out"))
    }
}
