#include "../Level/PlayField.as"
#include "../PathFinder.as"


funcdef void EnemyStepped(Enemy@);

class Enemy : iRefObj
{
	private Scene@ mScene;
	private Object@ mObject;
	private PlayField@ mPlayField; //this is fine, because the playfield doesn't ref Enemies.
	private Location mFleeStart;
	private Location mLocation;
	Location get_location() { return mLocation; }
	private Direction mCurrentDir = Dir_Undefined;

	private bool mFleeing = false;
	private pUInt mFleeDistance = 6;
	private Animation@ mModelAnim;
	private SubAnimation@ mIdleAnim;
	private SubAnimation@ mMoveAnim;
	private OpenALSource@ mFleeSound;
	
	private bool mPlayingMoveInvert;
		
	//Directions
	private Quaternion mWest = Quaternion(Vector3(0,0,1), 0.5*PI);
	private Quaternion mEast= Quaternion(Vector3(0,0,1), -0.5*PI);
	private Quaternion mNorth= Quaternion();
	private Quaternion mSouth= Quaternion(Vector3(0,0,1), 1*PI);

	EnemyStepped@ stepped;

	void animEnded(Entity@ entity, Animation@ anim)
	{
		if (@anim == @mMoveAnim) {
			
			if (!mPlayingMoveInvert) {
				mObject.animator.play(mMoveAnim, false, -1.0);
				mPlayingMoveInvert = true;
				return;
			}
			
			setIdle();
			return;
		}
	
		if (anim.name != "Turn" && anim.name != "Move" && anim.name != "Flee") 
			return;
	
		if (anim.name == "Flee") {
			return;
		}

		if (stepped !is null)
			stepped(this);
		//Check for game over here...
	}
		
	Enemy(PlayField@ playField, Scene@ scene, EnemyPosition ep)
	{
			@mScene = scene;
			@mPlayField = playField;

			switch (ep.type)
			{
				case ET_Blight:
					{

						Object@ object = GAME.objectFactory.load(GAME.assetRoot + "3D/actors/blight.fbx");
						//Object@ object = GAME.objectFactory.createCube();
						object.xForm = mPlayField.getWorldLocation(ep.loc);
						object.setScale(mPlayField.tileSize * 0.5);
						object.translate(0,0,0.5);

						//The blight has glowing eyes!
						auto blight = object.model.getSurface("blight");
						auto eyes = object.model.getSurface("eyes");
						auto glow = object.model.getSurface("eye_glow");

						if (eyes !is null)
							eyes.material.mode = DM_EMISSIVE;
						

						@object.animator = Animator();
						object.animator.animationEnded += AnimEndedCallback(this.animEnded);
						mScene.add(object);
						@mObject = object;
						
						@mModelAnim = GAME.resourcePool.getAnimation(GAME.assetRoot + "3D/actors/blight.fbx:Take 001");
						@mIdleAnim = SubAnimation(mModelAnim, 11, 40);
						@mMoveAnim = SubAnimation(mModelAnim, 1, 10);
						
						@mFleeSound = GAME.audio.createFXSource();

						mObject.link.add(mFleeSound, LM_OVERRIDE);
						mPlayingMoveInvert = false;
							
						setIdle();

						mLocation = ep.loc;
					}
					break;
				default:
						echo ("Unfamiliar enemy type found!");
			}
	}

	void setIdle()
	{
		mObject.animator.stop(mIdleAnim);
		mObject.animator.stop(mMoveAnim);
		mObject.animator.play(mIdleAnim, true);
		mPlayingMoveInvert = false;
	}

	void setMoving()
	{
		mPlayingMoveInvert = false;
		mObject.animator.stop(mIdleAnim);
		mObject.animator.stop(mMoveAnim);
		mObject.animator.play(mMoveAnim, false);
	}	
	
	void tick() 
	{
			if (mFleeing) { mFleeing = false; return; }

			PathFinder pf(mPlayField);
			auto path = pf.findPath(mLocation, mPlayField.playerLocation);
			if (path.length() == 0) return;

			//Path is in reverse... this way, if we ever need to walk more than one step, you can pop them from the back.. cheaper.
			Location newLoc = path[path.length() -1];

			Direction newDir;
			if (newLoc.x > mLocation.x) newDir = Dir_E;
			else if (newLoc.x < mLocation.x) newDir = Dir_W;
			else if (newLoc.y > mLocation.y) newDir = Dir_S;
			else newDir = Dir_N;

			if (newDir != mCurrentDir) { //Turning motion
				auto object = mObject;
				//HACK .. need to make sure the enemy animation ends before the player tick ends... -_-
				ObjectAnimation@ oa = ObjectAnimation("Turn", GAME.animationSpeed + 0.1f);
					switch (newDir)	{
						case Dir_N:
							oa.addFrames(object.xForm.quaternion, mNorth);
							break;
						case Dir_E:
							oa.addFrames(object.xForm.quaternion, mEast);
							break;
						case Dir_S:
							oa.addFrames(object.xForm.quaternion, mSouth);
							break;
						case Dir_W:
							oa.addFrames(object.xForm.quaternion, mWest);
							break;
					}

				oa.build(IM_LINEAR);
				object.animator.play(oa, false);
				mCurrentDir = newDir;

			} else { //Linear motion else { //Lineair animation
				auto xForm = mPlayField.getWorldLocation(newLoc);
				xForm.translate(0,0,0.5);
				
				//HACK .. need to make sure the enemy animation ends before the player tick ends... -_-
				ObjectAnimation@ oa = ObjectAnimation("Move", GAME.animationSpeed + 0.2f); 
				oa.addFrames(Vector3(0,0,0), xForm.position - mObject.xForm.position);
				oa.build(IM_LINEAR);

				mObject.animator.play(oa, false);

				setMoving();
				
				mPlayField.updateEnemyPos(mLocation, newLoc);
				mLocation = newLoc;
			}
	}

	void flee(Location playerLoc, Direction playerDir) {

		mFleeing = true;
		mFleeStart = mLocation;

		ObjectAnimation@ oa = ObjectAnimation("Flee", 12);
		oa.addFrame(Vector3(0,0,0));

		switch (playerDir)
		{
			//turning the enemy away from the player.
			case Dir_N:
				mObject.quaternion = mNorth;
				break;
			case Dir_E:
				mObject.quaternion = mEast;
				break;
			case Dir_S:
				mObject.quaternion = mSouth;
				break;
			case Dir_W: 
				mObject.quaternion = mWest;
		}

		mCurrentDir = playerDir;

		PathFinder pf(mPlayField);
		auto path = pf.planEscape(mPlayField.playerLocation, mLocation , mFleeDistance);

		pUInt count = 0;
		for (pUInt i = path.length()-1; i !=~0; --i) {
			auto xForm = mPlayField.getWorldLocation(path[i]);
			xForm.position = xForm.position - mObject.xForm.position;
			xForm.translate(0,0,0.5);
			oa.addFrame(xForm.position);
		}

		if (path.length() != 0)
			mLocation = path[0];

		oa.build(IM_LINEAR);

		mObject.animator.play(oa, false);
		
		mFleeSound.play(GAME.resourcePool.getSound(GAME.assetRoot + "Sounds/ghost_flee.ogg"));
		
		setMoving();
		
		mPlayField.updateEnemyPos(mFleeStart, mLocation);
	}

	void removeFromScene() {
		mScene.remove(mObject);
	}
	
	~Enemy()
	{
		echo ("Destroying Enemy...");
	}
				
}
 