#include "Level/PlayField.as"


class SearchTile
{
    SearchTile() {}
    Location prev;
    pInt gscore = MAX_PINT; //nr of steps away from start
    pInt fscore = MAX_PINT; //gscore + heuristic score.
    bool spent = false;
    bool open = false;
}


//Simple A* implementation using the PlayField.
class PathFinder {

    Map<pInt, Map<pInt, SearchTile@>@>  mKnownTiles;
    Array<Location> mOpenTiles;
    PlayField@ mPlayField;

    PathFinder(PlayField@ playField) {
        @mPlayField = playField;
    }

    Array<Location>@ findPath(Location start, Location target)
    {
        mOpenTiles.clear();
        mKnownTiles.clear();

        //we're already there!
        if (start == target)  {
            return  Array<Location>();
        }
        
        auto@ starttile = getTile(start);
        starttile.gscore = 0.0f;
        starttile.fscore = estimateCost(start, target);
        starttile.open = true;

        mOpenTiles.add(start);

        while (mOpenTiles.length() > 0) {
            pUInt inx = 0;
            Location loc = getLowestCostOpenTile( mOpenTiles, inx);
            if (loc == target)
                return calculatePath(start, target);
            
            mOpenTiles.removeAt(inx);
            SearchTile@ current = getTile(loc);
            current.spent = true;

            auto neighbors = getViableNeighbors(loc);

            for (pUInt i = 0; i < neighbors.length(); ++i)  {
                SearchTile@ tile = getTile(neighbors[i]);
                if (tile.spent) continue;
                
                pInt tscore = current.gscore + 1; //distance between current and neighbor is always one.
                
                if (!tile.open) {
                    mOpenTiles.add(neighbors[i]);
                    tile.open = true;
                } else if (tscore >= tile.gscore) //from current to neighbor is not the most efficient way.
                    continue;
                
                tile.prev = loc;
                tile.gscore = tscore;
                tile.fscore = tile.gscore + estimateCost(neighbors[i], target);
            }
        }

        //

        //No route to target..
        return Array<Location>();
    }


    //This functions tries to find ANY path out of the current location, away from threat with a maximum length.
    Array<Location>@  planEscape(Location threat, Location _current, pUInt maxPathLength) 
    {
        mOpenTiles.clear();
        mKnownTiles.clear();

        auto@ starttile = getTile(_current);
        starttile.gscore = 0.0f;
        //starttile.fscore = estimateCost(start, target);
        starttile.open = true;
        mOpenTiles.add(_current);


        bool found = false;
        Location longestPath; 

        while (mOpenTiles.length() > 0) {

            //let's just advance ALL open tiles.. we're not looking at anything in particular
           
            pInt inx = rand() % mOpenTiles.length();
            Location loc = mOpenTiles[inx];
            mOpenTiles.removeAt(inx);

            SearchTile@ current = getTile(loc);
            current.spent = true;
            
            if (current.gscore == pInt(maxPathLength)) {
                found = true;
                longestPath = loc;
                break;
            }
            auto neighbors = getSafeNeighbors(threat, loc);
            
            for (pUInt i = neighbors.length() -1 ; i != ~0;  --i) {
                SearchTile@ tile = getTile(neighbors[i]);
                if (tile.spent == true) {
                    neighbors.removeAt(i);
                    continue;
                }
                pInt tscore = current.gscore + 1; //distance between current and neighbor is always one.
            
                if (!tile.open) {
                    mOpenTiles.add(neighbors[i]);
                    tile.open = true;
                } else if (tscore >= tile.gscore) //from current to neighbor is not the most efficient way.
                    continue;

                tile.prev = loc;
                tile.gscore = tscore;
            }

            if (neighbors.length() == 0) {//dead end. Let's store how far we got:
                if (!found) {
                    longestPath = loc;
                    found = true;
                } else {
                    auto tile1 = getTile(longestPath);
                    auto tile2 = getTile(loc);
                    if (tile2.gscore > tile1.gscore)
                        longestPath = loc;
                }
                continue;
            }
        }

        if (found) {
            auto path = calculatePath(_current, longestPath);
            return path; 
        }
       
        return Array<Location>();
    }

    private Array<Location>@ calculatePath(Location start, Location target) 
    {
        Array<Location> path;
        while (start != target) {
            path.add(target);
            SearchTile@ tile = getTile(target);
            target = tile.prev;
        }
        return path;
    }

    private SearchTile@ getTile(Location loc) 
    {
        //AATC's MAP is retarded!!
        if (mKnownTiles[loc.x] is null) {
            @mKnownTiles[loc.x] = Map<pInt, SearchTile@>();
        }

        SearchTile@ tile = mKnownTiles[loc.x][loc.y];
        if (tile is null) {
            @tile = SearchTile();
            @mKnownTiles[loc.x][loc.y] = tile;
        }
        return tile;
    }

    private Location getLowestCostOpenTile(Array<Location>@ mOpenTiles, pUInt &out inx)
    {
        pInt score = MAX_PINT;
        for (pUInt i = 0; i < mOpenTiles.length(); ++i) {
            pInt tc = getTile(mOpenTiles[i]).fscore;
            if (tc < score) {
                score = tc;
                inx = i;
            }
        }

        return mOpenTiles[inx];
    }

    Array<Location>@ getSafeNeighbors(Location threat, Location loc) {
         Array<Location> neighbors;

        Location dist(threat.x - loc.x, threat.y - loc.y);

        //x tiles
        Location posX = Location(loc.x + 1, loc.y);
        Location minX = Location(loc.x -1, loc.y);

        //y tiles
        Location posY = Location(loc.x, loc.y + 1);
        Location minY = Location(loc.x, loc.y - 1);

        // If not on the same x, only move away from the threat
        if (dist.x < 0) {
            if (mPlayField.isWalkableTile(posX)) neighbors.add(posX); //right
        } else if (dist.x > 0) {
            if (mPlayField.isWalkableTile(minX)) neighbors.add(minX); //left
        } else {
            //Same x, so may go either way.
            if (mPlayField.isWalkableTile(posX)) neighbors.add(posX); //right
            if (mPlayField.isWalkableTile(minX)) neighbors.add(minX); //left
        }

        // If not on the same y, only move away from the threat
        if (dist.y < 0) {
            if (mPlayField.isWalkableTile(posY)) neighbors.add(posY); // up;
        } else if (dist.y > 0) {
            if (mPlayField.isWalkableTile(minY)) neighbors.add(minY); // down
        } else {
            if (mPlayField.isWalkableTile(posY)) neighbors.add(posY); // up;
            if (mPlayField.isWalkableTile(minY)) neighbors.add(minY); // down
        }
        
         return neighbors;
    }

    Array<Location>@ getViableNeighbors(Location loc) {
        Array<Location> neighbors;

        
        //x tiles
        Location posX = Location(loc.x + 1, loc.y);
        Location minX = Location(loc.x -1, loc.y);

        //y tiles
        Location posY = Location(loc.x, loc.y + 1);
        Location minY = Location(loc.x, loc.y - 1);

        if (mPlayField.isWalkableTile(posX)) neighbors.add(posX); //right
        if (mPlayField.isWalkableTile(posY)) neighbors.add(posY); // up;
        if (mPlayField.isWalkableTile(minX)) neighbors.add(minX); //left
        if (mPlayField.isWalkableTile(minY)) neighbors.add(minY); // down

        return neighbors;
    }

    pInt estimateCost(Location@ loc, Location@ target) 
    {
        pInt dx = abs(loc.x - target.x);
        pInt dy = abs(loc.y - target.y);
        
        //Manhattan distance
        return dx + dy;
    }
    
   
}




