#version 460
layout(local_size_x = 32, local_size_y = 1) in;

#define USE_RANDOM 1
#define USE_MARCHER 1

float scene_with_grad(vec3 p, out int material, out float gradmag);
void scene_init();

#define SCENE_WITH_GRAD scene_with_grad
#define SCENE_INIT scene_init

#includelib

// mandelbox distance function by Rrola (Buddhi's distance estimation)
// http://www.fractalforums.com/index.php?topic=2785.msg21412#msg21412

const float SCALE = 2.3;
const float MR2 = 0.5;
vec4 mbox_scalevec = vec4(SCALE, SCALE, SCALE, abs(SCALE)) / MR2;
float mbox_C1 = abs(SCALE-1.0);

int mandelbox_material(vec3 position, int iters=7) {
    float mbox_C2 = pow(abs(SCALE), float(1-iters));
	// distance estimate
	vec4 p = vec4(position.xyz, 1.0), p0 = vec4(position.xyz, 1.0);  // p.w is knighty's DEfactor
	for (int i=0; i<iters; i++) {
		p.xyz = clamp(p.xyz, -1.0, 1.0) * 2.0 - p.xyz;  // box fold: min3, max3, mad3
		float r2 = dot(p.xyz, p.xyz);  // dp3
        if (r2 > 5.) {
            return i;
        }
		p.xyzw *= clamp(max(MR2/r2, MR2), 0.0, 1.0);  // sphere fold: div1, max1.sat, mul4
		p.xyzw = p*mbox_scalevec + p0;  // mad4
	}
	float d = (length(p.xyz) - mbox_C1) / p.w - mbox_C2;
    return 0;
}

float mandelbox(vec3 position, int iters=7) {
    float mbox_C2 = pow(abs(SCALE), float(1-iters));
	// distance estimate
	vec4 p = vec4(position.xyz, 1.0), p0 = vec4(position.xyz, 1.0);  // p.w is knighty's DEfactor
	for (int i=ZERO; i<iters; i++) {
		p.xyz = clamp(p.xyz, -1.0, 1.0) * 2.0 - p.xyz;  // box fold: min3, max3, mad3
		float r2 = dot(p.xyz, p.xyz);  // dp3
		p.xyzw *= clamp(max(MR2/r2, MR2), 0.0, 1.0);  // sphere fold: div1, max1.sat, mul4
		p.xyzw = p*mbox_scalevec + p0;  // mad4
	}
	float d = (length(p.xyz) - mbox_C1) / p.w - mbox_C2;
    return d;
}

// Original distance estimator: pseudo_knightyan by Knighty (Fractalforums.com/Fractalforums.org)
// katakombit
struct KnightyanParams {
    vec3 CSize; // [(-2,-2,-2),(0,0,0),(2,2,2)]
    float Size; // [-2,0.70968,2]
    float DEfactor; // [0,1,10]
    float TwiddleRXY; // [-2,0.92784,2]
    float TwiddleP;           // [-2,2,5]
    int Iter;// [0,9,100]
    int ColorIterations; //;slider[0,9,100]
};

KnightyanParams catacombs =
{
    0.7*vec3(1., 1., 1.),   // csize
    0.8,                    // size
    1.,                     // DEfactor
    0.92784,                // TwiddleRXY
    2.,                     // TwiddleP
    10,                     // Iter
    9,                      // ColorIterations
};

KnightyanParams stars =
{
    2.0*vec3(1., 1., 1.),   // csize
    2.0,                    // size
    1.0,                     // DEfactor
    1.0,                // TwiddleRXY
    2.,                     // TwiddleP
    10,                     // Iter
    9,                      // ColorIterations
};

KnightyanParams stripes =
{
    1.5*vec3(1., 0.05, 1.),   // csize
    1.1,                    // size
    1.0,                     // DEfactor
    0.9,                // TwiddleRXY
    2.,                     // TwiddleP
    10,                     // Iter
    9,                      // ColorIterations
};

KnightyanParams strings =
{
    0.25*vec3(1., 1.0, 1.),   // csize
    0.95,                    // size
    1.0,                     // DEfactor
    0.8,                // TwiddleRXY
    2.,                     // TwiddleP
    5,                     // Iter
    9,                      // ColorIterations
};

KnightyanParams pseudo =
{
    vec3(0.956,0.8985,1.105772),   // csize
    1.03524229,                    // size
    0.387168145,                     // DEfactor
    1.13538111,                // TwiddleRXY
    2.01793721,                     // TwiddleP
    14, /*14*/                     // Iter
    9,                      // ColorIterations
};

KnightyanParams kni = pseudo;

// vec3 CSize = 0.7*vec3(1., 1., 1.); //;slider[(-2,-2,-2),(0,0,0),(2,2,2)]
// float Size = 0.8; //;slider[-2,0.70968,2]
// float DEfactor = 1.; //;slider[0,1,10]
// float TwiddleRXY = 0.92784; //;slider[-2,0.92784,2]
// int Iter = 10;//;slider[0,9,100]

struct BioCubeParams {
    float Scale; // slider[0.00,1.5,4.00]
    int Iterations; //  slider[0,2,50]
    vec3 Offset; // slider[(-1,-1,-1),(0,1,1),(1,1,1)]
    vec3 Offset2; // slider[(-1,-1,-1),(1,-0.3,-0.3),(1,1,1)]
    float Qube; // slider[-1,0.1,1]
    float Angle1; // slider[-180,0,180]
    vec3 Rot1; // slider[(-1,-1,-1),(1,1,1),(1,1,1)]
    bool fold;
    float foldSym;
    float foldAsym;

};

BioCubeParams templeCube =
{
    1.5,
    12,
    vec3(-0.00825446, 1., 1.),
    vec3(1., -0.2361111, -0.494444),
    0.0451,
    1.5,
    vec3(1., 1., 1.),
    true,
    0.5,
    0.
};


BioCubeParams labyrinthCube =
{
    1.45,
    13,
    vec3(-0.00825446, 1., 1.),
    vec3(1., -0.2361111, -0.494444),
    0.0351,
    1.5,
    vec3(1., 1., 1.),
    true,
    0.5,
    0.
};

BioCubeParams cubep = templeCube;

// void init() {
// 	//fracRotation2 = rotationMatrix3(normalize(Rot2), Angle2);
// 	fracRotation1 = Scale* rotationMatrix3(normalize(Rot1), Angle1);
// }

// Return rotation matrix for rotating around vector v by angle
mat3 rotationMatrix3(vec3 v, float angle)
{
    float c = cos(radians(angle));
    float s = sin(radians(angle));

    return mat3(c + (1.0 - c) * v.x * v.x, (1.0 - c) * v.x * v.y - s * v.z, (1.0 - c) * v.x * v.z + s * v.y,
            (1.0 - c) * v.x * v.y + s * v.z, c + (1.0 - c) * v.y * v.y, (1.0 - c) * v.y * v.z - s * v.x,
            (1.0 - c) * v.x * v.z - s * v.y, (1.0 - c) * v.y * v.z + s * v.x, c + (1.0 - c) * v.z * v.z
            );
}

mat3 cubeFracRotation1 = cubep.Scale * rotationMatrix3(normalize(cubep.Rot1), cubep.Angle1);

vec3 recFold(inout vec3 p) {
    float t;
    p.xy=abs(p.xy);
    t=p.x;
    p.x=p.x+p.y - cubep.foldSym;
    p.y=t-p.y - cubep.foldAsym;
    t=p.x;
    p.x = p.x+p.y;
    p.y = t-p.y;
    return p;
}

// BioCube fractal by DarkBeam
float biocube(vec3 z)
{
    float t; int n = 0;
    float scalep = 1.0;

    // if (cubep.fold) {
    //     z = recFold(z);
    // }

    vec3 z0=z;
    z = abs(z);

    if (z.y>z.x) z.xy =z.yx;
    if (z.z>z.x) z.xz = z.zx;
    if (z.y>z.x) z.xy =z.yx;
    float DE1 =1.0-z.x;
    z = z0;

    // Folds.
    //Dodecahedral
    while (n < cubep.Iterations) {
        z *= cubeFracRotation1;
        z = abs(z);
        z -= cubep.Offset;
        if (z.y>z.x) z.xy =z.yx;
        if (z.z>z.x) z.xz = z.zx;
        if (z.y>z.x) z.xy =z.yx;
        z -= cubep.Offset2;
        if (z.y>z.x) z.xy =z.yx;
        if (z.z>z.x) z.xz = z.zx;
        if (z.y>z.x) z.xy =z.yx;

        n++;  scalep *= cubep.Scale;
        DE1 = abs(min(cubep.Qube/float(n)-DE1,(+z.x)/scalep));
    }

    //Distance to the plane going through vec3(Size,0.,0.) and which normal is plnormal
    return DE1;
}

float knightyan(vec3 p)
{
    float j=kni.DEfactor;
    for(int i=ZERO;i<kni.Iter;i++){
        p = kni.TwiddleP*clamp(p, -kni.CSize, kni.CSize)-p;
        //float k = max(kni.Size/dot(p,p),1.);
        float k = max(kni.Size/dot(p,p), 0.989910313);
        p *= k;
        //j *= k + 0.05;
        j *= k + 0.1;
        float r2 = dot(p, p);
        if (i<kni.ColorIterations) orbitTrap = min(orbitTrap, abs(vec4(p.xyz,r2)));
    }
    float rxy=length(p.xy);
    return max(rxy-kni.TwiddleRXY, abs(rxy*p.z) / length(p))/j;
}


float scene_biocube(vec3 p, out int material) {
	material = MATERIAL_GRASSLAND;
    vec3 p2 = p;
    p2.y = abs(p2.y);
    p2.y -= 1.0;
    vec2 cells = pMod2(p2.xz, vec2(2.1, 2.1));

    float skyplane_dist = p2.y - 1.;

    p2.yz = rot2d(p2.yz, sin(cells.x)*0.05);
    p2.xy = rot2d(p2.xy, sin(cells.y)*0.05);

    orbitTrap = vec4(1e4);
    float d = biocube(p2);
    // d = max(d, skyplane_dist);
    return d;
}


float scene_kleinian(vec3 p, out int material) {
    material = MATERIAL_KLEINIAN;
    vec3 p2 = p;
    p2.y = abs(p2.y);
    p2.y -= 0.2;

    p2.y += 1.5e-1 * length(p.xz);
    float skyplane_dist = p2.y - 0.8;

    p2.zy = p2.yz;

    //p2.yz = rot2d(p2.yz, 0.5);

    orbitTrap = vec4(1e4);
    float d = knightyan(p2);
    d = max(d, skyplane_dist);
    return d;
}

float scene_rotated_brot(vec3 p, out int material) {
	material = MATERIAL_OTHER;
    vec3 p2 = p;
    p2.y = abs(p2.y);
    p2.y -= prop_mandelbox_height;
    vec2 cells = pMod2(p2.xz, vec2(15.0, 15.0));
    p2.zy = p2.yz;
    p2.xz = rot2d(p2.xz, prop_mandelbox_roll);

	return mandelbox(p2, prop_mandelbox_iters);
}


float scene_brot(vec3 p, out int material) {
	material = MATERIAL_OTHER;
    vec3 p2 = p;
    p2.y = abs(p2.y);
    p2.y -= prop_mandelbox_height;
    p2.yz = rot2d(p2.yz, 0.5);
	return mandelbox(p2, prop_mandelbox_iters);
}


float scene_jungle(vec3 p, out int material, out float gradmag) {
    vec3 p2 = p;
    p2.y = abs(p2.y);
    const float scale = 0.1;
    const float height = scale * 20.;
    const float riverdepth = scale * 50.0;
    vec3 h = fbmd_9(p2.xz * 0.025/scale) * height;
    //vec3 h = vec3(0.);
    float width = scale * 60.;

    float spread = sin(saturate(abs((p.x + scale * 50 * sin(-p.z / (scale * 80.))) - width) / width) * PI * .5);
    h.x += riverdepth * min(1., spread);

    float d = p2.y - h.x + riverdepth * .5;
    material = MATERIAL_JUNGLEFLOOR;
    gradmag = length(h.yz);
    return d;
}

float scene_ball(vec3 p, out int material) {
	material = MATERIAL_OTHER;
    float d = length(p - vec3(0., -4., 0.)) - 2.;
    return d;
}

float scene_with_grad(vec3 p, out int material, out float gradmag) {
    switch (prop_geometry) {
     case GEOMETRY_MANDELBOX:
          return scene_rotated_brot(p, material);
     case GEOMETRY_TEMPLECUBE:
     case GEOMETRY_LABYRINTH:
          return scene_biocube(p, material);
     case GEOMETRY_KLEINIAN:
          return scene_kleinian(p, material);
     case GEOMETRY_JUNGLE:
        return scene_jungle(p, material, gradmag);
    default:
        return scene_ball(p, material);
    }
}

void scene_init()
{
    switch (prop_geometry) {
        case GEOMETRY_MANDELBOX:
            break;
        case GEOMETRY_TEMPLECUBE:
            cubep = templeCube;
            break;
        case GEOMETRY_LABYRINTH:
            cubep = labyrinthCube;
            break;
        case GEOMETRY_KLEINIAN:
            KnightyanParams kni = pseudo;
            break;
        default:
    }
}



