#version 430

#extension GL_NV_gpu_shader5 : enable

#include <shaders/materials/commons.glsl>
#include <shaders/commons_hlsl.glsl>
#include <shaders/materials/noise/noise3d.glsl>
#include <shaders/materials/commons_instancing_buffers.h>

layout(std140, row_major) uniform TransformParamsBuffer{
	EntityTransformParams transform_params;
};


layout(std430) buffer VTXCoords {
	float coords[];
} vtx_coords;

vec3 vtx_coords_load(uint idx)
{
	return vec3(vtx_coords.coords[idx * 3 + 0], vtx_coords.coords[idx * 3 + 1], vtx_coords.coords[idx * 3 + 2]);
}

layout(std430) buffer VTXNormals {
	float normals[];
} vtx_normals;

vec3 vtx_normals_load(uint idx)
{
	return vec3(vtx_normals.normals[idx * 3 + 0], vtx_normals.normals[idx * 3 + 1], vtx_normals.normals[idx * 3 + 2]);
}

layout(std430) buffer VTXAlbedo {
	float albedo[];
} vtx_albedo;

vec3 vtx_albedo_load(uint idx)
{
	return vec3(vtx_albedo.albedo[idx * 3 + 0] , vtx_albedo.albedo[idx * 3 + 1], vtx_albedo.albedo[idx * 3 + 2]);
}

layout(std430) buffer VTXUV0 {
	float uv0[];
} vtx_uv0;

vec2 vtx_uv0_load(uint idx)
{
	return vec2(vtx_uv0.uv0[idx * 2 + 0], vtx_uv0.uv0[idx * 2 + 1]);
}


uniform int vtx_builtin_buffers = 0;

struct VertexInput
{
	vec3 pos;
	vec3 norm;
	vec4 color;
	vec2 uv0;
};

out Vertex
{
	vec3 vCoords;
	f16vec3 vNorm;
	f16vec3 vWorldNorm;
	vec3 vLocalPos;
	vec3 vWorldPos;
	f16vec4 vColor;
	f16vec2 vUV0;
} vtx_output;

out uint instanceID;

// Custom vertex modifiers params

#ifdef INLINE_VERTEX_MODIFIERS_PARAMS_BLOCK
#inline <INLINE_VERTEX_MODIFIERS_PARAMS_BLOCK>
#endif

//

void main() {
	
	VertexInput vtx_input;
	vtx_input.pos = vtx_coords_load(gl_VertexID);
	vtx_input.norm = vec3(1.0, 0.0, 0.0);
	vtx_input.color = vec4(1.0);
	vtx_input.uv0 = vtx_input.pos.xy;
	
	if ((vtx_builtin_buffers & VTXBuiltinAttribute_Albedo) != 0)
		vtx_input.color = vec4(vtx_albedo_load(gl_VertexID), 1.0);

	if ((vtx_builtin_buffers & VTXBuiltinAttribute_Normals) != 0)
		vtx_input.norm = vtx_normals_load(gl_VertexID);

	if ((vtx_builtin_buffers & VTXBuiltinAttribute_UV0) != 0)
		vtx_input.uv0 = vtx_uv0_load(gl_VertexID);

	if (instance_params.stride > 0)
	{
		vec4 inst_m0 = instance_transform[gl_InstanceID * instance_params.stride + 0];
		vec4 inst_m1 = instance_transform[gl_InstanceID * instance_params.stride + 1];
		vec4 inst_m2 = instance_transform[gl_InstanceID * instance_params.stride + 2];

		vec4 v = vec4(vtx_input.pos, 1.0);
		vtx_input.pos.x = dot(inst_m0, v);
		vtx_input.pos.y = dot(inst_m1, v);
		vtx_input.pos.z = dot(inst_m2, v);

		vtx_input.norm = normalize(vector_transform_by_mat33(vtx_input.norm, transpose(mat4(inst_m0, inst_m1, inst_m2, vec4(0.0, 0.0, 0.0, 1.0)))));

		vec4 inst_c;
		inst_c.rg = unpackHalf2x16(instance_color[gl_InstanceID * 2 + 0]);
		inst_c.ba = unpackHalf2x16(instance_color[gl_InstanceID * 2 + 1]);
		vtx_input.color.rgb = vtx_input.color.rgb * inst_c.rgb;
	}
	instanceID = gl_InstanceID;

#if 1
#ifdef INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK
	{
		VertexInput vtx_modifier_input_base = vtx_input; // if we need a copy for anything

#inline <INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK>

	}
#endif
#endif

	vtx_output.vLocalPos = vtx_input.pos;

	vec3 vPosModel = vector_transform_by_mat43(vtx_input.pos, transform_params.mModel);
	vtx_output.vWorldNorm = f16vec3(vector_transform_by_mat43(vtx_input.norm, transform_params.mModelNormal));

	// TODO: Make fast path when we can combine model and view. Now we inject us between them because we need
	// model transform for modifiers
#if 0
#ifdef INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK
	vtx_input.pos = vPosModel;
	vtx_input.norm = normalize(vtx_output.vWorldNorm);
#inline <INLINE_VERTEX_MODIFIERS_TRANSFORM_LOCAL_BLOCK>
	vPosModel = vtx_input.pos;
	vtx_output.vWorldNorm = f16vec3(vtx_input.norm);
#endif
#endif

	vec3 vPosModelView = vector_transform_by_mat43(vPosModel.xyz, transform_params.mView);

	vtx_output.vCoords = vPosModelView;

	vtx_output.vWorldPos = vPosModel;
	
	vtx_output.vWorldNorm = normalize(vtx_output.vWorldNorm);

	vtx_output.vNorm.x = float16_t(dot(transform_params.mModelViewInvTrans[0].xyz, vtx_input.norm));
	vtx_output.vNorm.y = float16_t(dot(transform_params.mModelViewInvTrans[1].xyz, vtx_input.norm));
	vtx_output.vNorm.z = float16_t(dot(transform_params.mModelViewInvTrans[2].xyz, vtx_input.norm));
	vtx_output.vNorm = f16vec3(normalize(vtx_output.vNorm));
	vtx_output.vNorm = f16vec3(vtx_input.norm);

	vtx_output.vColor = f16vec4(vtx_input.color);
	vtx_output.vUV0 = f16vec2(vtx_input.uv0);

	gl_Position = transform_params.mProjection * vec4(vtx_output.vCoords, 1.0);

}