#version 430

#include <commons.glsl>

uniform sampler2D s_NoiseRGBA;

in vec3 vNorm;
in vec3 vLocalPos;
in vec4 vCoords;
in vec4 vWorldPos;
in vec3 vWorldNorm;
in vec4 vsColor;

uniform vec4 colorDiffuse;
uniform int materialId;
uniform int gWriteDepth;

// NOTE: we try to always render using base object front faces so we have conservative depth
//       when inside the object we will just pass the define and have 2 shaders and use the second
//       one to not use the conservative depth

uniform mat4 mModel;
uniform mat4 mView;
uniform vec4 vModelview[3];
uniform mat4 mProjection;
uniform float g_global_time;

uniform vec3 vCameraPosition;
uniform vec4 vNearFarPlane; // min, max, 2*min*max, max-min

// output for 2 buffers
#ifndef SHADOWMAP_PASS
layout(location = 0) out vec4 outAlbedo;
layout(location = 1) out uint outNormalMaterial;
#endif

// basic template based on the shadertoy framework template

#if 0 // sphere test
float doModel(vec3 p)
{
	// sphere intersection for testing
	return length(p - vec3(0.0, 0.0, 0.0)) - 10.0;
}

float calcIntersection(in vec3 ro, in vec3 rd, out vec4 color)
{
	const float maxd = 2000.0;         // max trace distance
	const float precis = 0.001;        // precission of the intersection
	float h = precis*2.0;
	float t = 0.0;
	float res = -1.0;
	for( int i=0; i<900; i++ )          // max number of raymarching iterations is 90
	{
		if( h<precis||t>maxd ) break;
		h = doModel( ro+rd*t );
		t += h;
	}

	if( t<maxd )
	{
		res = t;
		color = vec4(1.0);
	}
	return res;
}

vec3 doModelNormal(vec3 pos)
{
	const float eps = 0.002;             // precision of the normal computation

	const vec3 v1 = vec3( 1.0,-1.0,-1.0);
	const vec3 v2 = vec3(-1.0,-1.0, 1.0);
	const vec3 v3 = vec3(-1.0, 1.0,-1.0);
	const vec3 v4 = vec3( 1.0, 1.0, 1.0);

	return normalize( v1*doModel( pos + v1*eps ) + 
					  v2*doModel( pos + v2*eps ) + 
					  v3*doModel( pos + v3*eps ) + 
					  v4*doModel( pos + v4*eps ) );
}
#endif

#if 0 // transparent and infinite object (https://www.shadertoy.com/view/lssGRX)

float noise( in vec3 x )
{
	vec3 p = floor(x);
	vec3 f = fract(x);
	f = f*f*(3.0-2.0*f);
	
	// there's an artefact because the y channel almost, but not exactly, matches the r channel shifted (37,17)
	// this artefact doesn't seem to show up in chrome, so I suspect firefox uses different texture compression.
	vec2 uv = (p.xy+vec2(37.0,17.0)*p.z) + f.xy;
	vec2 rg = textureLod(s_NoiseRGBA, (uv+ 0.5)/256.0, 0.0 ).yx;
	return mix( rg.x, rg.y, f.z );
}

vec4 map( vec3 p )
{
	float den = -1.0 - (abs(p.y-0.5)+0.5)/2.0;

	// clouds
	float f;
	vec3 q = p*.5                          - vec3(0.0,0.0,1.5)*g_global_time + vec3(sin(0.7*gTime),0,0);
	f  = 0.50000*noise( q ); q = q*2.02 - vec3(0.0,0.0,0.0)*g_global_time;
	f += 0.25000*noise( q ); q = q*2.03 - vec3(0.0,0.0,0.0)*g_global_time;
	f += 0.12500*noise( q ); q = q*2.01 - vec3(0.0,0.0,0.0)*g_global_time;
	f += 0.06250*noise( q ); q = q*2.02 - vec3(0.0,0.0,0.0)*g_global_time;
	f += 0.03125*noise( q );

	den = clamp( den + 4.0*f, 0.0, 1.0 );
	
	vec3 col = mix( vec3(1.0, 1.0, 1.0), vec3(0.6,0.5,0.4), den*.5 );// + 0.05*sin(p);
	
	return vec4( col, den*.7 );
}

const vec3 sunDir = vec3(-1,.2,-1);

float testshadow( vec3 p, float dither )
{
	float shadow = 1.0;
	float s = 0.0; // this causes a problem in chrome: .05*dither;
	for ( int j=0; j < 5; j++ )
	{
		vec3 shadpos = p + s*sunDir;
		shadow = shadow - map(shadpos).a*shadow;
		
		s += .05;
	}
	return shadow;
}

vec4 raymarch( in vec3 ro, in vec3 rd, out float intersection )
{
	vec4 sum = vec4( 0 );
	
	float t = 0.0;
	intersection = -1.0;

	// dithering	
	float dither = 0.0;//texture( iChannel0, glFragCoord.xy/iChannelResolution[0].x ).x;
	t += 0.1*dither;
	
	for( int i=0; i<165; i++ )
	{
		if( sum.a > 0.99 ) continue;
		
		vec3 pos = ro + (t+.2*t*t)*rd;
		vec4 col = map( pos );

		float shadow = testshadow(pos, dither);
		col.xyz *= mix( vec3(0.4,0.47,0.6), vec3(1.0,1.0,1.0), shadow );
		
		col.rgb *= col.a;

		sum = sum + col*(1.0 - sum.a);	

		t += 0.1;
	}

	//vec4 bg = mix( vec4(.3,.4,.5,0), vec4(.5,.7,1,0), smoothstep(-.4,.0,rd.y) ); // sky/ocean

	/*// floor
	if ( rd.y < -.2 )
	{
		vec3 pos = ro + rd*(ro.y+1.0)/(-rd.y);
		
		float shadow = testshadow(pos+sunDir/sunDir.y, dither);
		bg.xyz = mix( vec3(0), vec3(.5), shadow*.8+.2 );
	}*/

	//sum += bg*(1.0 - sum.a);
	
	intersection = t;
	return vec4(clamp( sum.xyz, 0.0, 1.0 ), sum.a);
}

float calcIntersection(in vec3 ro, in vec3 rd, out vec4 col)
{
	float intersection;
	col.rgba = raymarch(ro, rd, intersection);

	return intersection;
}

vec3 doModelNormal(vec3 p)
{
	return normalize(vec3(1.0));
}

#endif

#if 1 // solid infinite object (https://www.shadertoy.com/view/XsX3RB)

float noise( in vec3 x )
{
	vec3 p = floor(x);
	vec3 f = fract(x);
	f = f*f*(3.0-2.0*f);
#ifndef HIGH_QUALITY_NOISE
	vec2 uv = (p.xy+vec2(37.0,17.0)*p.z) + f.xy;
	vec2 rg = textureLod( s_NoiseRGBA, (uv+ 0.5)/256.0, 0. ).yx;
#else
	vec2 uv = (p.xy+vec2(37.0,17.0)*p.z);
	vec2 rg1 = textureLod( s_NoiseRGBA, (uv+ vec2(0.5,0.5))/256.0, 0. ).yx;
	vec2 rg2 = textureLod( s_NoiseRGBA, (uv+ vec2(1.5,0.5))/256.0, 0. ).yx;
	vec2 rg3 = textureLod( s_NoiseRGBA, (uv+ vec2(0.5,1.5))/256.0, 0. ).yx;
	vec2 rg4 = textureLod( s_NoiseRGBA, (uv+ vec2(1.5,1.5))/256.0, 0. ).yx;
	vec2 rg = mix( mix(rg1,rg2,f.x), mix(rg3,rg4,f.x), f.y );
#endif	
	return mix( rg.x, rg.y, f.z );
}

float noise( in vec2 x )
{
	vec2 p = floor(x);
	vec2 f = fract(x);
	vec2 uv = p.xy + f.xy*f.xy*(3.0-2.0*f.xy);
	return textureLod( s_NoiseRGBA, (uv+118.4)/256.0, 0. ).x;
}

vec4 texcube( sampler2D sam, in vec3 p, in vec3 n )
{
	vec4 x = texture( sam, p.yz );
	vec4 y = texture( sam, p.zx );
	vec4 z = texture( sam, p.xy );
	return x*abs(n.x) + y*abs(n.y) + z*abs(n.z);
}

//=====================================================================

float lava( vec2 p )
{
	p += vec2(2.0,4.0);
	float f;
	f  = 0.5000*noise( p ); p = p*2.02;
	f += 0.2500*noise( p ); p = p*2.03;
	f += 0.1250*noise( p ); p = p*2.01;
	f += 0.0625*noise( p );
	return f;
}

const mat3 m = mat3( 0.00,  0.80,  0.60,
					-0.80,  0.36, -0.48,
					-0.60, -0.48,  0.64 );

float displacement( vec3 p )
{
	p += vec3(1.0,0.0,0.8);
	
	float f;
	f  = 0.5000*noise( p ); p = m*p*2.02;
	f += 0.2500*noise( p ); p = m*p*2.03;
	f += 0.1250*noise( p ); p = m*p*2.01;
	f += 0.0625*noise( p ); 
	
	float n = noise( p*3.5 );
	f += 0.03*n*n;
	
	return f;
}

float mapTerrain( in vec3 pos )
{
	return pos.y*0.1 + (displacement(pos*vec3(0.8,1.0,0.8)) - 0.4)*(1.0-smoothstep(1.0,3.0,pos.y));
}

float raymarchTerrain( in vec3 ro, in vec3 rd )
{
	float maxd = 30.0;
	float t = 0.1;
	for( int i=0; i<256; i++ )
	{
		float h = mapTerrain( ro+rd*t );
		if( h<(0.001*t) || t>maxd ) break;
		t += h*0.8;
	}

	if( t>maxd ) t=-1.0;
	return t;
}

vec3 calcNormal( in vec3 pos, in float t )
{
	vec3 eps = vec3( max(0.02,0.001*t),0.0,0.0);
	return normalize( vec3(
		   mapTerrain(pos+eps.xyy) - mapTerrain(pos-eps.xyy),
		   mapTerrain(pos+eps.yxy) - mapTerrain(pos-eps.yxy),
		   mapTerrain(pos+eps.yyx) - mapTerrain(pos-eps.yyx) ) );

}

vec3 lig = normalize( vec3(-0.3,0.4,0.7) );
	
vec4 mapClouds( in vec3 pos )
{
	vec3 q = pos*0.5 + vec3(0.0,-g_global_time,0.0);
	
	float d;
	d  = 0.5000*noise( q ); q = q*2.02;
	d += 0.2500*noise( q ); q = q*2.03;
	d += 0.1250*noise( q ); q = q*2.01;
	d += 0.0625*noise( q );
		
	d = d - 0.55;
	d *= smoothstep( 0.5, 0.55, lava(0.1*pos.xz)+0.01 );

	d = clamp( d, 0.0, 1.0 );
	
	vec4 res = vec4( d );

	res.xyz = mix( vec3(1.0,0.8,0.7), 0.2*vec3(0.4,0.4,0.4), res.x );
	res.xyz *= 0.25;
	res.xyz *= 0.5 + 0.5*smoothstep( -2.0, 1.0, pos.y );
	
	return res;
}

vec4 raymarchClouds( in vec3 ro, in vec3 rd, in vec3 bcol, float tmax )
{
	vec4 sum = vec4( 0.0 );

	float sun = pow( clamp( dot(rd,lig), 0.0, 1.0 ),6.0 );
	float t = 0.0;
	for( int i=0; i<60; i++ )
	{
		if( t>tmax || sum.w>0.95 ) break;//continue;
		vec3 pos = ro + t*rd;
		vec4 col = mapClouds( pos );
		
		col.xyz += vec3(1.0,0.7,0.4)*0.4*sun*(1.0-col.w);
		col.xyz = mix( col.xyz, bcol, 1.0-exp(-0.00006*t*t*t) );
		
		col.rgb *= col.a;

		sum = sum + col*(1.0 - sum.a);	

		t += max(0.1,0.05*t);
	}

	sum.xyz /= (0.001+sum.w);

	return clamp( sum, 0.0, 1.0 );
}

float softshadow( in vec3 ro, in vec3 rd, float mint, float k )
{
	float res = 1.0;
	float t = mint;
	for( int i=0; i<64; i++ )
	{
		float h = mapTerrain(ro + rd*t);
		h = max( h, 0.0 );
		res = min( res, k*h/t );
		t += clamp( h, 0.02, 0.5 );
		if( res<0.001 ) break;
	}
	return clamp(res,0.0,1.0);
}

float calcIntersection(in vec3 ro, in vec3 rd, out vec4 color)
{
	vec3 col = vec3(1.0);

	// terrain	
	float t = raymarchTerrain(ro, rd);
	float st = t;
	if( t>0.0 )
	{
		vec3 pos = ro + t*rd;
		vec3 nor = calcNormal( pos, t );
		vec3 ref = reflect( rd, nor );

		vec3 bn = -1.0 + 2.0*texcube( s_NoiseRGBA, 3.0*pos/4.0, nor ).xyz;
		nor = normalize( nor + 0.6*bn );
		
		float hh = 1.0 - smoothstep( -2.0, 1.0, pos.y );

		// lighting
		float sun = 1.0;//clamp( dot( nor, lig ), 0.0, 1.0 );
		float sha = 0.0; if( sun>0.01) sha=softshadow(pos,lig,0.01,32.0);
		float bac = clamp( dot( nor, normalize(lig*vec3(-1.0,0.0,-1.0)) ), 0.0, 1.0 );
		float sky = 0.5 + 0.5*nor.y;
		float lav = smoothstep( 0.5, 0.55, lava(0.1*pos.xz) )*hh*clamp(0.5-0.5*nor.y,0.0,1.0);
		float occ = pow( (1.0-displacement(pos*vec3(0.8,1.0,0.8)))*1.6-0.5, 2.0 );

		float amb = 1.0;

		col.rgb = vec3(0.8);

		vec3 lin = vec3(0.0);
		lin += 1.4*sun*vec3(1.80,1.27,0.99)*pow(vec3(sha),vec3(1.0,1.2,1.5));
		lin += 0.9*sky*vec3(0.16,0.20,0.40)*occ;
		lin += 0.9*bac*vec3(0.40,0.28,0.20)*occ;
		lin += 0.9*amb*vec3(0.15,0.17,0.20)*occ;
		lin += lav*vec3(3.00,0.61,0.00);

		// surface shading/material		
		col = vec3(0.5);//texcube( iChannel1, 0.5*pos, nor ).xyz;
		//col = col*(0.2+0.8*texcube( iChannel2, 4.0*vec3(2.0,8.0,2.0)*pos, nor ).x);
		vec3 verde = vec3(1.0,0.9,0.2);
		//verde *= texture( iChannel2, pos.xz ).xyz;
		col = mix( col, 0.8*verde, hh );
		
		float vv = smoothstep( 0.0, 0.8, nor.y )*smoothstep(0.0, 0.1, pos.y-0.8 );
		verde = vec3(0.2,0.45,0.1);
		//verde *= texture( iChannel2, 30.0*pos.xz ).xyz;
		verde += 0.2; //*texture( iChannel2, 1.0*pos.xz ).xyz;
		vv *= 0.5;//smoothstep( 0.0, 0.5, texture( iChannel2, 0.1*pos.xz + 0.01*nor.x ).x );
		col = mix( col, verde*1.1, vv );
		
		col = lin * col;
		
	}
	
	color.a = 1.0;
	color.rgb = col.rgb;
	return t;
}

vec3 doModelNormal(vec3 pos)
{
	vec3 eps = vec3(0.02,0.0,0.0);
	return normalize( vec3(
		   mapTerrain(pos+eps.xyy) - mapTerrain(pos-eps.xyy),
		   mapTerrain(pos+eps.yxy) - mapTerrain(pos-eps.yxy),
		   mapTerrain(pos+eps.yyx) - mapTerrain(pos-eps.yyx) ) );

}
#endif


vec3 transform_normal(vec3 n)
{
	vec3 tn;
	tn.x = dot(vModelview[0].xyz, n);
	tn.y = dot(vModelview[1].xyz, n);
	tn.z = dot(vModelview[2].xyz, n);
	return normalize(n);
}

void main() {

	float g = vNorm.z * 0.5 + 0.5;
	vec3 worldNorm = vWorldNorm;

	outNormalMaterial = encode_normal_material(normalize(worldNorm), materialId);
	vec4 texture = colorDiffuse;

	vec4 cp = vec4(vCameraPosition, 1.0);
	vec3 rc;
	rc.x = dot(vModelview[0], cp);
	rc.y = dot(vModelview[1], cp);
	rc.z = dot(vModelview[2], cp);
	rc = vec3(0.0);

	// NOTE: when rendering to the shadowmap we should be rendering backfaces
	//       so maybe just mirror the camera and render 'back' towards the real one

	vec3 ro = vWorldPos.xyz;		// start tracing at the cube surface. still should clamp at the "outgoing face"
	vec3 rd = normalize(vWorldPos.xyz - vCameraPosition);
	
	vec4 color = vec4(1.0);
	float intersection = calcIntersection(ro, rd, color);
	if (intersection > 0.0)
	{
		vec3 ri = ro + intersection * rd;
		outNormalMaterial = encode_normal_material(transform_normal(-doModelNormal(ri)), materialId);

		ri = (mView * vec4(ri, 1.0)).xyz;	// move to view space (camera)
	
		outAlbedo = color;

		if (color.a < 0.001)
			discard;
	
		if (gWriteDepth != 0)
		{
			float near = -1.0;	// this is depth range, not the projection
			float far = 1.0;
			float depth = (mProjection * vec4(ri, 1.0)).z / (mProjection * vec4(ri, 1.0)).w;
			depth =  (((far - near) * depth) + near + far) / 2.0;
			gl_FragDepth = depth;
		}
	}
	else
	{
		if (intersection == -1.0)
			outAlbedo = vec4(5.0, 5.0, 0.0, 1.0);
		else
			outAlbedo = vec4(5.0, 0.0, 0.0, 1.0);

		if (gWriteDepth != 0)
		{
			float near = -1.0;	// this is depth range, not the projection
			float far = 1.0;
			float depth = (mProjection * vec4(ro, 1.0)).z / (mProjection * vec4(ro, 1.0)).w;
			depth =  (((far - near) * depth) + near + far) / 2.0;

			gl_FragDepth = depth;
		}

		discard;
	}
}
