#version 430

in vec2 vTexcoord0;
uniform sampler2D s_texture0;		// texture depth
uniform sampler2D s_texture2;		// texture linear depth 
uniform sampler2D s_texture1;		// random texture

uniform vec2 vResolution;
uniform vec2 vResolutionInv;

uniform vec4 projInfo;
uniform vec4 vCameraProjectionParams;
uniform float vFarPlane;
uniform float vNearPlane;


#define projOrtho 0
// radius
const float meters2viewspace = 1.0;
uniform float radius; // 40.0
uniform float R;// = radius * 1.0;
uniform float R2;// = R * R;
uniform float NegInvR2;// = -1.0 / R2;
uniform float RadiusToScreen;// = R * 0.5 * 20.0;   // projScale = 1.0;
uniform float NDotVBias;// = 0.1; // 0...1
uniform float AOMultiplier;// = 1.0 / (1.0 - NDotVBias);
uniform float PowExponent;// = 1.0;

// output for 2 buffers
layout(location = 0) out vec4 outColor;

#define texLinearDepth s_texture0
#define texRandom s_texture1
  
// tweakables
#define AO_RANDOMTEX_SIZE 4
#define M_PI 3.14159265
const float  NUM_STEPS = 16;
const float  NUM_DIRECTIONS = 8; // texRandom/g_Jitter initialization depends on this

void outputColor(vec4 color) {
  outColor = color;
}

//----------------------------------------------------------------------------------

float linearizeDepth(float d, vec2 nf)
{
	float z_n = 2.0 * d - 1.0;
	return 2.0 * nf.x * nf.y / (nf.y + nf.x - z_n * (nf.y - nf.x));
}

vec3 positionFromDepth(vec3 vDirection, float depth)
{
	return vec3(vDirection.xyz * depth);
}

vec3 UVToView(vec2 uv, float depth)
{
	vec3 view_direction;
	view_direction.x = -vCameraProjectionParams.x * 0.5 + vCameraProjectionParams.x * uv.x;
	view_direction.y = -vCameraProjectionParams.y * 0.5 + vCameraProjectionParams.y * uv.y;
	view_direction.z = 1.0;

	vec3 world = positionFromDepth(view_direction, depth);
	return world;
}

vec3 FetchViewPos(vec2 UV)
{
  //float ViewDepth = textureLod(texLinearDepth,UV,0).x;
#if 0
  float ViewDepth = linearizeDepth(texture(texLinearDepth, UV).r, vec2(vNearPlane, vFarPlane));
#else
  float ViewDepth = texture(s_texture2, UV).r;
#endif
  return UVToView(UV, ViewDepth);
}

vec3 MinDiff(vec3 P, vec3 Pr, vec3 Pl)
{
  vec3 V1 = Pr - P;
  vec3 V2 = P - Pl;
  return (dot(V1,V1) < dot(V2,V2)) ? V1 : V2;
}

vec3 ReconstructNormal(vec2 UV, vec3 P)
{
  vec3 Pr = FetchViewPos(UV + vec2(vResolutionInv.x, 0));
  vec3 Pl = FetchViewPos(UV + vec2(-vResolutionInv.x, 0));
  vec3 Pt = FetchViewPos(UV + vec2(0, vResolutionInv.y));
  vec3 Pb = FetchViewPos(UV + vec2(0, -vResolutionInv.y));
  return normalize(cross(MinDiff(P, Pr, Pl), MinDiff(P, Pt, Pb)));
}

//----------------------------------------------------------------------------------
float Falloff(float DistanceSquare)
{
  // 1 scalar mad instruction
  return DistanceSquare * NegInvR2 + 1.0;
}

//----------------------------------------------------------------------------------
// P = view-space position at the kernel center
// N = view-space normal at the kernel center
// S = view-space position of the current sample
//----------------------------------------------------------------------------------
float ComputeAO(vec3 P, vec3 N, vec3 S)
{
  vec3 V = S - P;
  float VdotV = dot(V, V);
  float NdotV = dot(N, V) * 1.0/sqrt(VdotV);

  // Use saturate(x) instead of max(x,0.f) because that is faster on Kepler
  return clamp(NdotV - NDotVBias,0,1) * clamp(Falloff(VdotV),0,1);
}

//----------------------------------------------------------------------------------
vec2 RotateDirection(vec2 Dir, vec2 CosSin)
{
  return vec2(Dir.x*CosSin.x - Dir.y*CosSin.y,
              Dir.x*CosSin.y + Dir.y*CosSin.x);
}

//----------------------------------------------------------------------------------
vec4 GetJitter()
{
  // (cos(Alpha),sin(Alpha),rand1,rand2)
  return textureLod( texRandom, (gl_FragCoord.xy / AO_RANDOMTEX_SIZE), 0);
}

//----------------------------------------------------------------------------------
float ComputeCoarseAO(vec2 FullResUV, float RadiusPixels, vec4 Rand, vec3 ViewPosition, vec3 ViewNormal)
{
  // Divide by NUM_STEPS+1 so that the farthest samples are not fully attenuated
  float StepSizePixels = RadiusPixels / (NUM_STEPS + 1);

  const float Alpha = 2.0 * M_PI / NUM_DIRECTIONS;
  float AO = 0;

  for (float DirectionIndex = 0; DirectionIndex < NUM_DIRECTIONS; ++DirectionIndex)
  {
    float Angle = Alpha * DirectionIndex;

    // Compute normalized 2D direction
    vec2 Direction = RotateDirection(vec2(cos(Angle), sin(Angle)), Rand.xy);

    // Jitter starting sample within the first step
    float RayPixels = (Rand.z * StepSizePixels + 1.0);

    for (float StepIndex = 0; StepIndex < NUM_STEPS; ++StepIndex)
    {
      vec2 SnappedUV = round(RayPixels * Direction) * vResolutionInv.xy + FullResUV;
      vec3 S = FetchViewPos(SnappedUV);

      RayPixels += StepSizePixels;

      AO += ComputeAO(ViewPosition, ViewNormal, S);
    }
  }

  AO *= AOMultiplier / (NUM_DIRECTIONS * NUM_STEPS);
  return clamp(1.0 - AO * 2.0,0,1);
}

void main()
{
  vec2 texCoord = vTexcoord0;

  vec2 uv = texCoord;
  vec3 ViewPosition = FetchViewPos(uv);

  // Reconstruct view-space normal from nearest neighbors
  vec3 ViewNormal = -ReconstructNormal(uv, ViewPosition);

  // Compute projection of disk of radius control.R into screen space
  float RadiusPixels = RadiusToScreen / (projOrtho != 0 ? 1.0 : ViewPosition.z);

  // Get jitter vector for the current full-res pixel
  vec4 Rand = GetJitter();

  float AO = ComputeCoarseAO(uv, RadiusPixels, Rand, ViewPosition, ViewNormal);
 // AO = FetchViewPos(texCoord).y * 0.005;
  //AO = linearizeDepth(texture(texLinearDepth, vTexcoord0).r, vec2(vNearPlane, vFarPlane)) * 0.05;
  //AO = texture(texLinearDepth, vTexcoord0).r;

#if AO_BLUR
  outputColor(vec4(pow(AO, PowExponent), ViewPosition.z, 0, 0));
#else
  outputColor(vec4(pow(AO, PowExponent)));
  //outputColor(vec4(AO));
  //outputColor(abs(vec4(ViewPosition * 0.1, 0.1) * 1.0));
  //outputColor(abs(vec4(ViewNormal * 0.1, 0.1) * 1.0));
#endif
  
}


/*-----------------------------------------------------------------------
  Copyright (c) 2014-2015, NVIDIA. All rights reserved.
  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:
   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Neither the name of its contributors may be used to endorse 
     or promote products derived from this software without specific
     prior written permission.
  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY
  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
  OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
-----------------------------------------------------------------------*/