#version 430

in vec2 vTexcoord0;
uniform sampler2D s_texture0;		// depth
uniform sampler2D s_texture1;		// color

uniform vec2 vResolution;
uniform vec2 vResolutionInv;

uniform vec4 vNearFarPlane; // min, max, 2*min*max, max-min
uniform vec2 vBlurDirection;

// output for 2 buffers
layout(location = 0) out vec4 outColor;

float linearize_depth(in float d)
{
	return vNearFarPlane.z / (vNearFarPlane.y + vNearFarPlane.x - d * vNearFarPlane.w);
}

// bilateral blur
#define KERNEL_RADIUS 3
#define g_Sharpness 20.0
#define LUMA_ONLY 1

vec4 BlurFunction(vec2 uv, float r, vec4 center_c, float center_d, inout float w_total)
{
  vec4  c = texture2D(s_texture1, uv );
  float d = linearize_depth(texture2D(s_texture0, uv).x);
  
  const float BlurSigma = float(KERNEL_RADIUS) * 0.5;
  const float BlurFalloff = 1.0 / (2.0*BlurSigma*BlurSigma);
  
  float ddiff = (d - center_d) * g_Sharpness;
  float w = exp2(-r*r*BlurFalloff - ddiff*ddiff);
  w_total += w;

  return c*w;
}

float BlurFunctionLuma(vec2 uv, float r, float center_c, float center_d, inout float w_total)
{
  float c = texture2D(s_texture1, uv ).x;
  float d = linearize_depth(texture2D(s_texture0, uv).x);
  
  const float BlurSigma = float(KERNEL_RADIUS) * 0.5;
  const float BlurFalloff = 1.0 / (2.0*BlurSigma*BlurSigma);
  
  float ddiff = (d - center_d) * g_Sharpness;
  float w = exp2(-r*r*BlurFalloff - ddiff*ddiff);
  w_total += w;

  return c*w;
}

vec4 fetch_hbao_blur(vec2 texCoord)
{
  vec4  center_c = texture2D(s_texture1, texCoord );
  float center_d = linearize_depth(texture2D(s_texture0, texCoord).x);
  
  vec4  c_total = center_c;
  float w_total = 1.0;
  
  for (float r = 1; r <= KERNEL_RADIUS; ++r)
  {
    vec2 uv = texCoord + vBlurDirection * r;
    c_total += BlurFunction(uv, r, center_c, center_d, w_total);  
  }
  
  for (float r = 1; r <= KERNEL_RADIUS; ++r)
  {
    vec2 uv = texCoord - vBlurDirection * r;
    c_total += BlurFunction(uv, r, center_c, center_d, w_total);  
  }

  return c_total/w_total;
}

float fetch_hbao_blur_luma(vec2 texCoord)
{
  float center_c = texture2D(s_texture1, texCoord ).x;
  float center_d = linearize_depth(texture2D(s_texture0, texCoord).x);
  
  float c_total = center_c;
  float w_total = 1.0;
  
  for (float r = 1; r <= KERNEL_RADIUS; ++r)
  {
    vec2 uv = texCoord + vBlurDirection * r;
    c_total += BlurFunctionLuma(uv, r, center_c, center_d, w_total);  
  }
  
  for (float r = 1; r <= KERNEL_RADIUS; ++r)
  {
    vec2 uv = texCoord - vBlurDirection * r;
    c_total += BlurFunctionLuma(uv, r, center_c, center_d, w_total);  
  }

  return c_total/w_total;
}

void main()
{
#ifdef LUMA_ONLY
	outColor = vec4(fetch_hbao_blur_luma(vTexcoord0));
#else
	outColor = fetch_hbao_blur(vTexcoord0);
#endif
}

