#version 430

in vec2 vTexcoord0;
uniform sampler2D s_texture0;		// color

uniform vec2 vResolution;
uniform vec2 vResolutionInv;

#define Tonemap_Simple 0
#define Tonemap_ACES 1
#define Tonemap_Uncharted 2
#define Tonemap_Reinhardt 3
#define Tonemap_Haarm 4

// color adjustment params
uniform vec4 color_lift;
uniform vec4 color_inv_gamma;
uniform vec4 color_gain;
uniform vec4 color_contrast;
uniform float color_saturation;
uniform float tonemap_blend;
uniform int tonemap_type = Tonemap_Simple;

uniform float exposure;

// output for 2 buffers
layout(location = 0) out vec4 outColor;

// noise (replace this with blue noise!)

// random functions: https://stackoverflow.com/questions/4200224/random-noise-functions-for-glsl
// A single iteration of Bob Jenkins' One-At-A-Time hashing algorithm.
uint hash( uint x ) {
	x += ( x << 10u );
	x ^= ( x >>  6u );
	x += ( x <<  3u );
	x ^= ( x >> 11u );
	x += ( x << 15u );
	return x;
}

// Compound versions of the hashing algorithm I whipped together.
uint hash( uvec2 v ) { return hash( v.x ^ hash(v.y)                         ); }
uint hash( uvec3 v ) { return hash( v.x ^ hash(v.y) ^ hash(v.z)             ); }
uint hash( uvec4 v ) { return hash( v.x ^ hash(v.y) ^ hash(v.z) ^ hash(v.w) ); }

// Construct a float with half-open range [0:1] using low 23 bits.
// All zeroes yields 0.0, all ones yields the next smallest representable value below 1.0.
float floatConstruct( uint m ) {
	const uint ieeeMantissa = 0x007FFFFFu; // binary32 mantissa bitmask
	const uint ieeeOne      = 0x3F800000u; // 1.0 in IEEE binary32

	m &= ieeeMantissa;                     // Keep only mantissa bits (fractional part)
	m |= ieeeOne;                          // Add fractional part to 1.0

	float  f = uintBitsToFloat( m );       // Range [1:2]
	return f - 1.0;                        // Range [0:1]
}



// Pseudo-random value in half-open range [0:1].
float random( float x ) { return floatConstruct(hash(floatBitsToUint(x))); }
float random( vec2  v ) { return floatConstruct(hash(floatBitsToUint(v))); }
float random( vec3  v ) { return floatConstruct(hash(floatBitsToUint(v))); }
float random( vec4  v ) { return floatConstruct(hash(floatBitsToUint(v))); }

// tonemappers

vec3 haarm_approx(vec3 base_color)
{
	vec3 x = max(vec3(0.0), base_color.rgb - 0.004);
	return (x * (6.2 * x + 0.5))/(x * (6.2 * x + 1.7) + 0.06);
}

vec3 reinhard(vec3 base_color)
{
	vec3 x = base_color / (1.0 + base_color);
	return pow(x, vec3(1.0/2.2));
}

vec3 basic(vec3 base_color)
{
	return pow(base_color, vec3(1.0/2.2));
}

float A = 0.15;
float B = 0.50;
float C = 0.10;
float D = 0.20;
float E = 0.02;
float F = 0.30;
float W = 11.2;

vec3 Uncharted2Tonemap(vec3 x)
{
   return ((x*(A*x+C*B)+D*E)/(x*(A*x+B)+D*F))-E/F;
}

vec3 uncharted(vec3 base_color)
{
	base_color = max(vec3(0.0), base_color.rgb);

   float ExposureBias = 2.3f;
   vec3 curr = Uncharted2Tonemap(ExposureBias*base_color);

   vec3 whiteScale = 1.0/Uncharted2Tonemap(vec3(W));
   vec3 color = curr*whiteScale;
      
   return pow(color,vec3(1.0/2.2));
}

// Based on http://www.oscars.org/science-technology/sci-tech-projects/aces
vec3 aces(vec3 color)
{
	mat3 m1 = mat3(
        0.59719, 0.07600, 0.02840,
        0.35458, 0.90834, 0.13383,
        0.04823, 0.01566, 0.83777
	);
	mat3 m2 = mat3(
        1.60475, -0.10208, -0.00327,
        -0.53108,  1.10813, -0.07276,
        -0.07367, -0.00605,  1.07602
	);
	vec3 v = m1 * color;    
	vec3 a = v * (v + 0.0245786) - 0.000090537;
	vec3 b = v * (0.983729 * v + 0.4329510) + 0.238081;
	return pow(clamp(m2 * (a / b), 0.0, 1.0), vec3(1.0 / 2.2));	
}

vec3 EvalLogContrastFunc(vec3 x, float eps, float logMidpoint, vec3 contrast)
{
	vec3 logX = log2(x+eps);
	vec3 adjX = logMidpoint + (logX - logMidpoint) * contrast;
	vec3 ret = max(vec3(0.0), exp2(adjX) - eps);
	return ret;
}

float ApplyLiftInvGammaGain(const float lift, const float invGamma, const float gain, float v)
{
	// lerp gain
	float lerpV = clamp(pow(v, invGamma), 0.0, 1.0);
	float dst = mix(lift, gain, lerpV);
	return dst;
}

vec3 EvalLiftGammaGain(vec3 v)
{
	vec3 ret;
	ret.x = ApplyLiftInvGammaGain(color_lift.x, color_inv_gamma.x, color_gain.x, v.x);
	ret.y = ApplyLiftInvGammaGain(color_lift.y, color_inv_gamma.y, color_gain.y, v.y);
	ret.z = ApplyLiftInvGammaGain(color_lift.z, color_inv_gamma.z, color_gain.z, v.z);
	return ret;
}

//#define APPLY_GRADINT_BEFORE_TONEMAPPING

void main()
{
	vec4 base_color = texelFetch(s_texture0, ivec2(vTexcoord0 * vResolution), 0);
	base_color = base_color * exposure;	// base exposure. make configurable

	//base_color.rgb += color_lift.rgb * color_lift.a;
#ifdef APPLY_GRADING_BEFORE_TONEMAPPING
	base_color.rgb = EvalLogContrastFunc(base_color.rgb, 0.001, 0.18, (vec3(1.0) - color_contrast.rgb) * color_contrast.a);
	base_color.rgb = EvalLiftGammaGain(base_color.rgb);
#endif

	vec3 tonemapped_color = base_color.rgb;
	if (tonemap_type == Tonemap_Simple)
	{
		tonemapped_color.rgb = basic(base_color.rgb);
	}
	if (tonemap_type == Tonemap_Uncharted)
	{
		tonemapped_color.rgb = uncharted(base_color.rgb);
	}
	if (tonemap_type == Tonemap_ACES)
	{
		tonemapped_color.rgb = aces(base_color.rgb);
	}
	if (tonemap_type == Tonemap_Reinhardt)
	{
		tonemapped_color.rgb = reinhard(base_color.rgb);
	}
	if (tonemap_type == Tonemap_Haarm)
	{
		tonemapped_color.rgb = haarm_approx(base_color.rgb);
	}

	vec3 gray = vec3(dot(tonemapped_color.rgb, vec3(0.212, 0.715, 0.072)));
	tonemapped_color.rgb = mix(gray, tonemapped_color.rgb, color_saturation);

	float dither = random(vTexcoord0);

#ifdef APPLY_GRADING_BEFORE_TONEMAPPING
	outColor = vec4(mix(base_color.rgb, tonemapped_color.rgb, tonemap_blend), base_color.a) + dither / 256.0;
#else
	base_color.rgb = mix(base_color.rgb, tonemapped_color, tonemap_blend);
	base_color.rgb = EvalLogContrastFunc(base_color.rgb, 0.001, 0.18, (vec3(1.0) - color_contrast.rgb) * color_contrast.a);
	base_color.rgb = EvalLiftGammaGain(base_color.rgb);
	base_color.rgb += dither / 256.0;
	outColor = base_color;
#endif
}
