#version 410 core

uniform float t;

float time = t * .5 + .05;

vec3 speed = vec3(2, 1.5, 2.5) * time;  // position/rotation change speeds
vec3 glass = vec3(0);

////////////////////////////////////////////////////////////////////////////////
// SIGNED DISTANCE FUNCTIONS                                                  //
////////////////////////////////////////////////////////////////////////////////

// SDFs are taken from https://iquilezles.org/articles/distfunctions
float sdSphere(vec3 p, float s)
{
    return length(p) - s;
}

float sdBox(vec3 p, vec3 b)
{
    vec3 q = abs(p) - b;
    return length(max(q, 0.)) + min(max(q.x, max(q.y, q.z)), 0.);
}

float sdPolygon(vec3 p, vec2 v[9], float t)
{
    float d = dot(p.xy-v[0],p.xy-v[0]);
    float s = 1.;
    for (int i=0, j=8; i<9; j=i, i++) {
        vec2 e = v[j] - v[i];
        vec2 w = p.xy - v[i];
        vec2 b = w - e*clamp(dot(w,e)/dot(e,e), 0., 1.);
        d = min(d, dot(b,b));
        bvec3 c = bvec3(p.y>=v[i].y,p.y<v[j].y,e.x*w.y>e.y*w.x);
        if (all(c) || all(not(c))) s=-s;  
    }
    return max(s*sqrt(d), abs(p.z) - t);
}

float sdPolygon(vec3 p, vec2 v[17], float t)
{
    float d = dot(p.xy-v[0],p.xy-v[0]);
    float s = 1.;
    for (int i=0, j=16; i<17; j=i, i++) {
        vec2 e = v[j] - v[i];
        vec2 w = p.xy - v[i];
        vec2 b = w - e*clamp(dot(w,e)/dot(e,e), 0., 1.);
        d = min(d, dot(b,b));
        bvec3 c = bvec3(p.y>=v[i].y,p.y<v[j].y,e.x*w.y>e.y*w.x);
        if (all(c) || all(not(c))) s=-s;
    }
    return max(s*sqrt(d), abs(p.z) - t);
}

float sdStar5(vec2 p, float r, float rf)
{
    vec2 k1 = vec2(0.809, -0.588);
    vec2 k2 = vec2(-k1.x,k1.y);
    p.x = abs(p.x);
    p -= 2.0*max(dot(k1,p),0.0)*k1;
    p -= 2.0*max(dot(k2,p),0.0)*k2;
    p.x = abs(p.x);
    p.y -= r;
    vec2 ba = rf*vec2(-k1.y,k1.x) - vec2(0,1);
    float h = clamp( dot(p,ba)/dot(ba,ba), 0.0, r );
    return length(p-ba*h) * sign(p.y*ba.x-p.x*ba.y);
}

////////////////////////////////////////////////////////////////////////////////
// EXTRA FUNCTIONS                                                            //
////////////////////////////////////////////////////////////////////////////////

// Random number
float rand(vec2 p)
{
    return fract(sin(dot(p + time, vec2(12.9898, 78.233))) * 43758.5453);
}

// Sine in range from vmin to vmax
float sinr(float x, float vmin, float vmax)
{
    return (sin(x)*.5+.5) * (vmax-vmin) + vmin;
}

// Rotate 2D point p by angle a
vec2 rot2d(vec2 p, float a)
{
    float sin_a = sin(a);
    float cos_a = cos(a);
    return p*mat2(cos_a, -sin_a, sin_a, cos_a);
}

// Rotate 3D point p by angles a
vec3 rot3d(vec3 p, vec3 a)
{
    p.xz = rot2d(p.xz, a.y);  // around axis Y
    p.yz = rot2d(p.yz, a.x);  // around axis X
    p.xy = rot2d(p.xy, a.z);  // around axis Z
    return p;
}

////////////////////////////////////////////////////////////////////////////////
// EFFECT FUNCTIONS                                                           //
////////////////////////////////////////////////////////////////////////////////

// Julia set
vec3 julia(vec2 p)
{
    p = rot2d(p, time) * pow(1.5, sin(time*2.)+.5);

    int iter;
    float x = cos(time) * .5;
    float y = sin(time) * .5;
    for (iter = 0; iter < 64; ++iter) {
        vec2 p2 = p * p;
        if (p2.x + p2.y > 4.) break;
        p = vec2(p2.x - p2.y + x, 2. * p.x * p.y + y);
    }

    x = 1. - pow(float(iter) / 64., .33);
    return vec3(1.-x, (1.-x)*.25, x);
}

vec3 plasma(vec2 p)
{
    return vec3(.3+.3*cos(6.2832 * ((sin(sin(time*1.2+p.x)*5.+p.x*7.) + cos(cos(time*2.+p.y)*5.+p.y*7.)) * .25 + .5 + vec3(0,.33,.67))));  // rainbow colors
}

vec3 cube(vec2 p)
{
    // Ray Marching preparations
    vec3 rp = vec3(0, 0, -2);  // ray origin point coordinate
    vec3 rd = normalize(vec3(p, 1));  // ray direction

    // Ray Marching
    float intensity = 0.;
    for (int i = 0; i < 64; ++i) {
        vec3 p = rot3d(rp + vec3(0, 0, sin(time*2.)), speed);  // rotate ray point for object 1 (rotate object 1 if fact)
        float dist = max(  // get nearest distance to objects, point near object and object index
            sdBox(p, vec3(.5)),
            -sdSphere(p, .6)
        );
        if (dist > 3.) break;  // too far (ray goes out of objects)
        if (dist < .001) {  // close enough (hit to object)
            intensity = 1. - pow(float(i)*.05, 2.2);
            break;
        }
        rp += rd * dist;  // move ray point in direction 'rd' by distance 'dist'
    }
    return vec3(0, max(intensity, .2), .5);
}

vec3 tube(vec2 p)
{
    float angle = (atan(p.y, p.x) + time)*12.;
    return vec3(sinr(20./length(p) + time*20., 0., .75), sinr(angle, 0., .5), sinr(-angle, 0., .75));
}

vec3 colbars(vec2 p)
{
    return (.25 - abs(vec3(sin(time), sin(time*1.2), sin(time*1.5)) - p.y)) * 4.;
}

bool effects;

// Bitmap of text messages
int msg[] = int[](
    1860855986,-1572164183,-1572164183,1716153001,-1572197719,-1572198231,-1371133797,
    63942,8777,8777,8655,8777,8777,8777,
    776304143,692330753,692396401,791224919,690562161,688989201,-384752783,
    64495,2593,494113,310241,494113,35361,494127,
    36683247,35792169,35792169,29507055,8523041,8525089,8529185,
    236743,270632,270632,203046,34081,34081,507087,
    26240547,39356468,38832164,38814882,38830625,38830625,28995191,
    9649353,11837737,11833637,16226787,13928741,13927721,9747753,
    26687783,38831401,38831401,64166183,39067945,39067945,39034057,
    -1907845944,-2125941464,-2125941464,-2042039000,-2008492760,138990889,-2026859322,
    0,-1360237133,-1532320599,-462492231,1152001193,1152003243,0,
    0,5709276,5571924,5444820,5576020,3497309,0);
int text_widths[] = int[](32, 16, 32, 19, 26, 19, 26, 24, 26, 32, 32, 23);

// Man polyline coords
vec2 outline[9] = vec2[](
    vec2(0), vec2(1.1039, 0.0711), vec2(1.5050, -1.1101),
    vec2(1.3115, -1.1463), vec2(1.6974, -2.1151), vec2(1.3365, -2.1733),
    vec2(1.4838, -2.5433), vec2(0.1195, -2.5433), vec2(0));
vec2 face[17] = vec2[](
    vec2(0.0826, -0.0739), vec2(0.1930, -2.4257), vec2(1.3515, -2.4250),
    vec2(1.2565, -2.1862), vec2(0.8567, -2.2504), vec2(0.7462, -2.1774),
    vec2(0.8748, -2.1678), vec2(1.5875, -2.0528), vec2(1.4522, -1.7131),
    vec2(1.1491, -1.7437), vec2(1.2278, -1.6562), vec2(1.4218, -1.6367),
    vec2(1.2328, -1.1622), vec2(1.1709, -1.1731), vec2(1.1178, -1.1021),
    vec2(1.4008, -1.0492), vec2(1.0485, -0.0117));
vec2 brow[9] = vec2[](
    vec2(1.0847, -0.4180), vec2(0.6789, -0.3390), vec2(0.7572, -0.4601),
    vec2(0.8763, -0.4832), vec2(0.8761, -0.5627), vec2(0.9622, -0.4999),
    vec2(0.9907, -0.5054), vec2(0.9997, -0.5799), vec2(1.0972, -0.5213));

// Logo colors
vec3 colors[] = vec3[](
    vec3(0,.2,1), vec3(.7,.1,0.), vec3(1), vec3(.1), vec3(1,1,0), vec3(1,.1,0),
    vec3(.5), vec3(.75), vec3(.3,.5,.7), vec3(.5,.7,1), vec3(.7,.3,.3), vec3(1,.5,.5),
    vec3(.8,.5,.15), vec3(1,.7,.3), vec3(.15,.5,.15), vec3(.3,.7,.3));
int colidx[] = int[](0, 0, 2, 3, 4, 5, 0, 2, 4, 0, 0, 2, 2, 4, 4, 6, 8, 8);

////////////////////////////////////////////////////////////////////////////////
// MAIN SDF                                                                   //
////////////////////////////////////////////////////////////////////////////////

// Signed Distance Function of object(s) with output of point near object and object index
float sdf(vec3 pl, vec3 pt, out vec3 po, out int index)
{
    // Logo
    float obj[8];
    pl += vec3(-.25, -.75, 0);
    obj[0] = max(  // blue
        sdBox(pl - vec3(0, .1, 0), vec3(.85, .85, .15)),  // screen
        -sdStar5(rot2d(pl.xy, -.2) + vec2(-.25, -.08), .55, .38)  // -star
    );
    obj[1] =  // blue
        sdBox(pl + vec3(-.75, .95, 0), vec3(.15, .05, .125));  // button
    obj[2] = max(  // white
        sdBox(pl, vec3(1, 1.1, .05)),  // monitor
        -sdBox(pl - vec3(0, .1, 0), vec3(.85, .85, 1))  // -screen
    );
    obj[3] = min(min(  // black
        max(
            sdBox(pl - vec3(0, .1, 0), vec3(.9, .9, .1)),  // screen frame
            -sdBox(pl - vec3(0, .1, 0), vec3(.85, .85, 1))),  // -screen
        max(
            sdBox(pl, vec3(1.05, 1.15, .1)),  // logo frame
            -sdBox(pl, vec3(1, 1.1, 1)))),  // -monitor
        min(
            sdPolygon(pl - vec3(-1.5, 1.2, -.4), outline, .1),
            sdPolygon(pl - vec3(-1.5, 1.2, -.55), brow, .1))
    );
    obj[4] = sdPolygon(pl - vec3(-1.5, 1.2, -.5), face, .1);  // yellow face
    obj[5] = effects ? sdSphere(pl - vec3(-.56, .60, -.6), .04) : 100.;  // red brow

    // Text
    obj[6] = obj[7] = 100.;
    pt += vec3(0, 1.25, 1);
    int text_index = int(time / 2. ) % 12;
    obj[6] = sdBox(pt, vec3(float(text_widths[text_index])*.05+.05, .4, .1));
    if (obj[6] < .001) {
        glass = vec3(.1, .1, .1);
        obj[6] = 100.;
        for (int j = 0; j < 7; ++j)
            for (int i = 0; i < text_widths[text_index]; ++i)
                if ((msg[j + text_index * 7] >> i & 1) != 0) {
                    vec3 ptt = pt - vec3(float(i*2-text_widths[text_index])*.05+.05, (3.5-float(j))*.1-.05, 0);
                    obj[6] = min(obj[6], sdBox(ptt, vec3(.05)));
                    obj[7] = min(obj[7], sdSphere(ptt, .07));
                }
    }

    // Store point near object and object index    
    int idx = 0;
    for (int i = 1; i < 8; ++i)
        if (obj[i] < obj[idx])
            idx = i;
    po = idx < 6 ? pl : pt;
    index = idx == 1 ? int(effects) : (idx < 6 ? colidx[idx] : colidx[6+text_index] + idx);
    // Return combined SDF
    return obj[idx];
}

////////////////////////////////////////////////////////////////////////////////
// MAIN CODE                                                                  //
////////////////////////////////////////////////////////////////////////////////

void main()
{
    // Normalize pixel coordinates (x = -xres/yres*0.5..xres/yres*0.5, y = -0.5..0.5)
    vec2 uv = gl_FragCoord.xy/vec2(1920., 1080.) - .5;
    uv.x *= 1.777777;

    // Glitches
    if (mod(time, 52.) >= 50. && mod(time, .125) < .0625) uv.x += rand(uv) * sin(time * 3.14159265) * .25;
    uv.x += .00025 / (mod(time*.5, 8.) - 4.5 - uv.y);

    // Background and rays
    float twist = length(uv)*sin(time/200.)*(1.-rand(uv)*.03);
    vec3 col = sinr(atan(uv.y, uv.x)*20. + time*10. + sin(length(uv)*30.)*twist*sin(time*1.57079633), .25, .75) * vec3(.1, .4, .2);

    vec3 rp = vec3(0, 0, -5.);  // ray origin point coordinate
    vec3 rd = normalize(vec3(uv, 1));  // ray direction

    // Position of objects relative to center (applied before rotation)
    vec3 pos = vec3(sin(speed.x)*1.5, sin(speed.y), sin(speed.z)*1.2) * .2;

    // Open man's mouth
    effects = mod(time, 16.) >= 8.;
    if (effects) {
        face[8] -= vec2(-.04, .1);
        face[9].y -= .1;
        face[10].y += .1;
        face[11] += vec2(-.04, .1);
    }

    // Ray Marching
    for (int i = 0; i < 64; ++i) {
        int index;  // object index
        vec3 po; // point near object (almost on object)
        float dist = sdf(
            rot3d(rp - pos, sin(time*vec3(1.,.7,0))*.5 + twist),  // rotate ray point for logo objects
            rot3d(rp - pos, sin(time*vec3(0,3.14159265,1.7))*vec3(0,.3,0)),  // rotate ray point for texts
            po, index);  // get nearest distance to objects, point near object and object index
        if (dist > 7.) break;  // too far (ray goes out of objects)
        if (dist < .001) {  // close enough (hit to object)
            col = colors[index];
            if (effects)
                if (index == 0 && po.z <= -.15) {
                    vec2 p = (po.xy - vec2(0, .1)) / .85;
                    int eff_no = int(mod(time / 16., 5.));
                    if (eff_no == 0) col = julia(p);
                    else if (eff_no == 1) col = plasma(p);
                    else if (eff_no == 2) col = cube(p);
                    else if (eff_no == 3) col = tube(p);
                    else if (eff_no == 4) col = colbars(p);
                    col *= rand(p)*.4 + .8;
                }
            col *= 1. - pow(float(i)*.05, 2.2);
            break;
        }
        rp += rd * dist;  // move ray point in direction 'rd' by distance 'dist'
    }

    // Output to screen
    gl_FragColor = vec4(col + glass, 1);
}