;=========================================================================================
; nurikabe256 
; A puzzle game in 256 bytes for the C64
; by Wil 2022
;
; Sourcecode for ca65 assembler, requires LAMAlib installed
;
; Build command:
; cl65 nurikabe256.s -lib LAMAlib.lib -C c64-asm.cfg --start-addr $802 -o nurikabe256.prg
; 
; To start the game:
; LOAD"NURIKABE256",8,1
; RUN
;
; Controls: Space to mark a water cell
;           X to mark an island cell
;           Cursor Keys to move
;
; The game automatically detects when the puzzle is solved, celebrates this with a green 
; border and then moves to the next puzzle (due to size constraints, only two puzzles are 
; implemented)
;
; The puzzle is played on a rectangular grid of cells, some of which contain numbers.
; The unnumbered cells are initially of unknown content but can only be water or island.
; Cells of the same content are considered connected if they are adjacent vertically or 
; horizontally, but not diagonally.
;
; The challenge is to paint each cell as island or water, subject to the following rules:
;
; - Each numbered cell is an island cell, the number stating the island size in cells
; - Each island must contain exactly one numbered cell.
; - All water cells must be connected
; - 2x2 areas of water cells are forbidden
;
; see https://en.wikipedia.org/wiki/Nurikabe_(puzzle)
;=========================================================================================

.include "LAMAlib.inc"

CR          =$aad7      ;outputs a carriage return, scrolls screen if necessary
CHECKMEM    =$fd68      ;Commodore's memtest routine, takes about a 2 seconds

height=8
width =10

cell        =$cf        ;character used for an undetermined cell
cell_scrcode=$4f        ;undetermined cell as screen code
left        =$a5        ;character used for closing the grid on the right side
up          =$b7        ;character used for closing the grid on the bottom side

key_island  =$58        ;X key
chr_stream  =$a6        ;shaded box character

checksum1   =$62        ;address where first part of checksum is calculated
checksum2   =$63        ;address where second part of checksum is calculated
level       =$02        ;level indicatoer, initialized with 0 by Kernal

;-----------------------------------------------------------------------------------------
; BASIC stub with SYS line
; Code based on makesys macro from lamalib
;-----------------------------------------------------------------------------------------

        ;.byte $0b      ;first byte omitted because it is recreated by BASIC relink anyway
        .byte $08       ;link to next line (end of BASIC program)

        ;the 16 bit line number can double as parameter initialization
        ;this will be copied as current BASIC line number to $0039-$003A
        ;by the BASIC interpreter

        .byte $0b,$1b       ;this is the checksum for solution of level 1

        .byte $9e           ;SYS token
        systarget=$83c      ;this needs to point at game_start
        .byte .sprintf ("%d", systarget)
        .byte $00
;-----------------------------------------------------------------------------------------
; Game code starts here
;-----------------------------------------------------------------------------------------
gameloop:
        .byte $a9           ;this byte can be freely set. we put an lda # opcode here to clear carry
        .byte $00           ;this is the end of the basic line

        jsr $bce9       ;sta into $62 to $65 and tay

        sta $c7 ;turn off reverse mode
;-----------------------------------------------------------------------------------------
; Caclulcate checksum from screen to identify solved puzzle
;-----------------------------------------------------------------------------------------
sum_up:
        lda $540,y
        eor ($03),y     ;$03/04 point to $B1AA, we use ROM data for hashing
        adc checksum1
        sta checksum1
        lda $580,y
        eor ($05),y     ;$05/06 point to $B391, we use ROM data for hashing
        adc checksum2
        sta checksum2
        iny
        bne sum_up

        cmp $3a
        if eq
          lda checksum1
          cmp $39
          if eq
            dec $d020
            dec level    ;make value negative to indicate level 2
            jsr CHECKMEM ;pause for 2 seconds
            inc $d020
;-----------------------------------------------------------------------------------------
; Entry point for SYS call
;-----------------------------------------------------------------------------------------
game_start:
            jsr drawnuri
          endif
        endif 
;-----------------------------------------------------------------------------------------
; Checking and fixing cursor positions
;-----------------------------------------------------------------------------------------
        jsr $e513       ;get cursor X/Y pos
        cpy #width
        if cc
          iny
        endif
        cpy #width+width
        if cs
          dey
        endif
        cpx #height
        if cc
          inx
        endif
        cpx #height+height
        if cs
          dex
        endif
        jsr $e50c       ;place cursor at line X, column Y
;-----------------------------------------------------------------------------------------
; User input with blinking cursor
;-----------------------------------------------------------------------------------------
        dec $cc         ;turn cursor on 
@waitkey:
        jsr GETIN
        beq @waitkey
;-----------------------------------------------------------------------------------------
; Turning the cursor off when it is in its off phase
;-----------------------------------------------------------------------------------------
@waitcursoroff:
        cli
        sei
        ldx $cf         ;cursor phase switch, 0=cursor off
        bne @waitcursoroff      ;only leave loop if cursor phase is off
        inc $cc         ;turn cursor off
        cli             ;reenable irq
;-----------------------------------------------------------------------------------------
; Handling user input
;-----------------------------------------------------------------------------------------
        ldx $ce         ;screen code of character under cursor
        cpx #$b0        ;screen code of inverted "0"
        bcc chk_island_stream_keys   
        ;no placement over numbers
check_moves:
        and #%01110011
        cmp #17         ;any cursor movement?
        bne gameloop
        tya             ;recover pressed key fron GETIN
write_out:
        jsr CHROUT
to_gameloop:
        bcc gameloop

chk_island_stream_keys:
        cmp #key_island
        bne chk_stream
        inc $c7         ;turn on reverse mode
        lda #$20
        bne write_out
chk_stream:
        cmp #$20
        bne check_moves
        lda #chr_stream
        bne write_out	;unconditional jump
;-----------------------------------------------------------------------------------------
; Grid draw routine
;-----------------------------------------------------------------------------------------
.macro place X1,Y1
        stx $0400+height*40+Y1*40+width+X1-41
.endmacro

drawnuri:
        jsr $e544       ;clear screen
        ldx #height
        jsr $e50c       ;place cursor at line X, column Y
printgrid:
        lda #cell
        jsr print_width_elements
        lda #left
        jsr CHROUT
        jsr CR
        dex
        bne printgrid
;-----------------------------------------------------------------------------------------
; Level data
;-----------------------------------------------------------------------------------------
        lda #$b7        ;character for upper line, also used in level 2 data

        ldx #$b1
        bit level
        bmi level2
;data for level 1
        place 3,4
        inx
        place 2,2
        place 10,5
        place 4,7
        inx
        place 2,7
        inx
        place 6,7
        inx
        place 4,3
        place 7,3
        place 5,6
        bne end_level_data
level2:
;data for level 2
        place 5,3
        place 6,5
        inx
        place 4,1
        place 2,3
        place 9,4
        place 4,5
        tax		;using the value $b7 previously set in A
        place 6,2
        dex
        place 3,6
        dex
        place 7,6
        
        ldx #$f5        ;checksum1 for level2 puzzle
        stx $39
        ldx #$d9        ;checksum2 for level2 puzzle
        stx $3a
end_level_data:
;-----------------------------------------------------------------------------------------
; Subroutine for multiple printing of a character
;-----------------------------------------------------------------------------------------
print_width_elements:
        ldy #width
        sty 211         ;indent width chars
@loop:  
        jsr CHROUT
        dey
        bne @loop
        rts
