import QtQuick 2.4

import Qt.labs.settings 1.0

import AdaptDemoSystem 1.0

Item {
  id: presetsEditView

  // each previewed preset listed in array with { name: "..."; instance: "..."; possibleInstances: [...]; tracks: {...}; } objects
  property var previewPresets: ([])
  // map of preset-id to true/false if preview active
  property var previewPresetActive: ({})
  property string previewTargetInstanceName: ""

  property int ensureVisibleSize: 64

  property int visibleHeight: 0

  // effect type -> icons/ folder image (24x24 pix size all now!)
  property var icons: {
    "BgBall": "bg.png",
    "Camera": "camera.png",
    "Mesh": "mesh.png",
    "Mesher": "mesher.png",
    "Vignette": "vignette.png",
    "Tonemap": "tonemap.png",
    "PalaFlow": "dustparticles.png",
    "Pardex": "partsim.png",
    "Ovl": "ovl.png",
    "Godray": "godray.png",
    "BreakIn": "group.png",
    "Reflect": "reflect.png"
  }


  onHeightChanged: {
    if (visibleHeight === 0) {
      visibleHeight = height;
    }
  }

  onXChanged: {
    if (x > demo.windowWidth-ensureVisibleSize) {
      x = demo.windowWidth-ensureVisibleSize;
    }
    if (x < -width+ensureVisibleSize) {
      x = -width+ensureVisibleSize;
    }
    rocket.markUpdateFrequent();
  }
  onYChanged: {
    if (y > demo.windowHeight-64) {
      y = demo.windowHeight-64;
    }
    if (y < -height+ensureVisibleSize) {
      y = -height+ensureVisibleSize;
    }

    visibleHeight = Math.max(0, Math.min(height, demo.windowHeight-y-presetsViewFlick.anchors.topMargin));

    rocket.markUpdateFrequent();
  }

  function getEffectInstances(fxType) {
    var res = [];
    if (root.fxInstancesPerType === undefined) {
      return res;
    }
    if (root.fxInstancesPerType[fxType] === undefined) {
      return res;
    }
    var keys = Object.keys(root.fxInstancesPerType[fxType]);
    for (var i=0; i<keys.length; i++) {
      if (keys[i] !== "") {
        res.push(keys[i]);
      }
    }
    res = res.sort();
    return res;
  }


  function clearPresetPreview() {
    rocket.clearTrackTempValues();

    previewPresets = [];
    previewPresetActive = {};
    previewTargetInstanceName = "";

    clearPreviews();
    root.setCustomText("fx preset", "fx preset title", "<br>------------------------------------");
    root.setCustomText("fx preset", "fx preset active", "");
    root.setCustomText("fx preset", "fx preset footer", "------------------------------------");
    showCustomTextGroup("fx preset", false);
  }

  signal clearPreviews();

  // apply previewed tracks to rocket
  function applyPreviewToRocket() {
    if (previewPresets !== undefined) {

      var trackNames = [];
      var trackValues = [];

      for (var i=0; i<previewPresets.length; i++) {
        var pp = previewPresets[i];
        var tracks = previewPresets[i].tracks;
        var trackKeys = Object.keys(tracks);

        for (var ii=0; ii<trackKeys.length; ii++) {
          trackNames.push(trackKeys[ii]);
          trackValues.push(tracks[trackKeys[ii]])
        }
      }
      rocket.setTrackValues(trackNames, trackValues);

    }
    clearPresetPreview();
  }

  function removePresetPreview(preset) {
    for (var i=0; i<previewPresets.length;) {
      if (previewPresets[i].name === preset.name) {
        // log("remove preset preview:"+preset.name)
        previewPresets.splice(i, 1);
      } else {
        i++;
      }
    }

    if (previewPresetActive[preset.name] !== undefined) {
      previewPresetActive[preset.name] = undefined;
    }

    rocket.clearTrackTempValues();

    for (var i=0; i<previewPresets.length; i++) {
      var tracks = previewPresets[i].tracks;
      var trackKeys = Object.keys(tracks);
      for (var ii=0; ii<trackKeys.length; ii++) {
        rocket.setTrackTempValue(trackKeys[ii], tracks[trackKeys[ii]]);
      }
    }

    if (previewPresets.length === 0) {
      clearPresetPreview();
    }
  }


  function addPresetPreview(preset, pushToList) {
    if (!preset || !preset.params) {
      return -1;
    }

    // focusedEffectBaseName
    // focusedEffectName

    var fen = focusedEffectName;
    if (fen === "") {
      fen = preset.fxName;
    }


    var effectTracks = rocket.getEffectTrackNames(fen);
    var sceneName = "";
    if (effectTracks.length > 0) {
      sceneName = getSceneNameFromTrack(effectTracks[0]);
    } else {
      return -1;
    }

    var previewTargetInstances = [];
    var ppo = {};

    previewTargetInstanceName = preset.fxName;

    var fxInsts = getEffectInstances(preset.fxType);
    for (var i=0; i<fxInsts.length; i++) {
      if (i===0) {
        previewTargetInstanceName = fxInsts[i];
      }
      previewTargetInstances.push(fxInsts);
    }

    if (preset.fxType !== focusedEffectBaseName) {
      // log("<font color='#ff0000'>Rocket not focused to compatible track:"+ preset.fxType+"</font>")
      rocket.findTrack(sceneName+"."+preset.fxName)
      // return false;
    }
    if (focusedEffectName !== "") {
      for (var i=0; i<fxInsts.length; i++) {
        if (focusedEffectName === fxInsts[i]) {
          previewTargetInstanceName = focusedEffectName;
          break;
        }
      }
    }

    ppo.name = preset.name;
    ppo.instance = previewTargetInstanceName;
    ppo.possibleInstances = previewTargetInstances;


    var previewTracks = {};
    var keys = Object.keys(preset.params);
    for (var i=0; i<keys.length; i++) {
      var trackName = sceneName+"."+previewTargetInstanceName+"."+keys[i];
      rocket.setTrackTempValue(trackName, preset.params[keys[i]]);
      previewTracks[trackName] = preset.params[keys[i]];
    }
    ppo.tracks = previewTracks;

    if (pushToList) {
      previewPresets.push(ppo);
    }
    previewPresetActive[preset.name] = true;

    root.setCustomText("fx preset", "fx preset title", "<br>------------------------------------");
    root.setCustomText("fx preset", "fx preset active", "FX preset preview active<br><b>CTRL+R</b>: Sync to rocket");
    root.setCustomText("fx preset", "fx preset footer", "------------------------------------");
    showCustomTextGroup("fx preset", true);
    return previewPresets.length-1;
  }

  Rectangle {
    anchors.fill: parent
    color: "black"
    border.width: 1
    border.color: "white"
  }
  Text {
    anchors.top: parent.top
    anchors.left: parent.left
    anchors.right: parent.right
    anchors.margins: 10
    text: "FX Presets"
    horizontalAlignment: Text.AlignHCenter
    color: "white"
    font.pixelSize: 12
  }
  //  MouseArea {
  //    anchors.fill: parent
  //    onClicked: {
  //      log("pressed on presets edit!")
  //    }
  //  }

  MouseArea {
    anchors.fill: parent
    drag.target: presetsEditView
    drag.threshold: 0

    onClicked: {
      log("pressed on presets edit!")
    }

    //    drag.minimumX: index > 0 ? curveTimes[index-1]-width*0.5+6 : 0-width*0.5+6
    //    drag.maximumX: curveTimes[index+1]+width*0.5-6
  }


  Filer {
    id: filer
    onFileChanged: {
      log("file change:"+path)

      if (path === "demo.json") {

      } else {
        reloadPreset(path);
      }
    }
  }

  property string fxPresetsPath: "fx/fxPresets/"
  property var fxPresets
  property int fxPresetsCount: 0

  function savePreset(name, data) {
    filer.saveTextToFile(fxPresetsPath+name+".json", JSON.stringify(data, null, 2));
    filer.addWatchPath(fxPresetsPath+name+".json");
  }

  function reloadPreset(presetFile) {
    var presetStr = filer.getFileText(presetFile);
    if (presetStr === "") {
      log("ERROR could not reload preset file:"+presetFile)
      return;
    }
    var preset = JSON.parse(presetStr);
    fxPresets.presets[preset.name] = preset;

    if (previewPresetActive[preset.name]) {
      addPresetPreview(preset, false);
    }
  }

  function loadPresets() {
    if (fxPresets === undefined) {
      fxPresets = {};
      fxPresets.presets = {};
      fxPresetsCount = 0;
    }
    var fxPresetFiles = filer.getFiles(fxPresetsPath);
    for (var i=0; i<fxPresetFiles.length; i++) {
      var isJsonFile = (fxPresetFiles[i].indexOf(".json") !== -1);
      if (!isJsonFile) {
        //log("UGH not a json file:"+fxPresetFiles[i])
        continue;
      }

      filer.addWatchPath(fxPresetsPath+fxPresetFiles[i]);

      var presetName = fxPresetFiles[i].replace(".json", "");
      var presetStr = filer.getFileText(fxPresetsPath+fxPresetFiles[i]);
      if (presetStr === "") {
        log("ERROR could not read preset file:"+presetName)
        continue;
      }
      var preset = JSON.parse(presetStr);
      preset.name = presetName;
      //log("loaded preset with name:"+presetName)
      fxPresets.presets[presetName] = preset;
      fxPresetsCount++;
    }
    log("loaded "+fxPresetsCount+" FX presets. P-key to view them.");

  }

  Component.onCompleted: {
    loadPresets();
  }

  property string userId
  property string fxFilter
  property string fxFilterOrig

  Settings {
    property alias userId : presetsEditView.userId
    property alias fxFilter: presetsEditView.fxFilter
    property alias viewSortMode: presetsEditView.viewSortMode
  }

  function numberFrontDigits(n) {
    if (n < 10) {
      return "000"+n;
    }
    if (n < 100) {
      return "00"+n;
    }
    if (n < 1000) {
      return "0"+n;
    }
    return n;
  }

  function addPreset(fxType, fxName) {
    if (fxType === "") {
      log("<font color='#ff0000'>ERROR adding preset - Rocket FX Type empty</font>");
      return;
    }
    if (fxName === "") {
      log("<font color='#ff0000'>ERROR adding preset - Rocket FX Instance empty</font>");
      return;
    }
    if (userId === "") {
      log("<font color='#ff0000'>ERROR adding preset - User ID missing</font>");
      return;
    }

    clearPresetPreview();

    var preset = {};
    preset.fxType = fxType;
    preset.fxName = fxName;
    preset.version = "1.0";

    fxPresetsCount = Object.keys(fxPresets.presets).length;

    // find a not used name
    // by finding biggest used number
    var sortedKeys = Object.keys(fxPresets.presets).sort(function(ka, kb) {
      var kan = ka.substring(ka.length-4, ka.length);
      var kab = kb.substring(kb.length-4, kb.length);
      return kab-kan;
    });
    var presetNumber = 0;
    if (sortedKeys.length > 0) {
      var len = sortedKeys[0].length;
      presetNumber = parseInt(sortedKeys[0].substring(len-4, len))+1;
    }
    preset.name = userId+"-"+fxType+"-preset-"+numberFrontDigits(presetNumber);

    preset.params = {};
    var effectTracks = rocket.getEffectTrackNames(focusedEffectName);
    for (var i=0; i<effectTracks.length; i++) {
      //log(effectTracks[i]+"="+rocket.sync(effectTracks[i]).toFixed(6));
      var saveTrackName = effectTracks[i];
      var saveTrackNameArr = saveTrackName.split(".");
      if (saveTrackNameArr.length > 1) {
        saveTrackName = "";
        for (var ii=2; ii<saveTrackNameArr.length; ii++) {
          saveTrackName += saveTrackNameArr[ii];
          if (ii !== saveTrackNameArr.length-1) {
            saveTrackName += ".";
          }
        }
      }
      if (saveTrackName === "ON") {
        continue;
      }
      preset.params[saveTrackName] = rocket.sync(effectTracks[i]).toFixed(6);
    }
    if (effectTracks.length > 0) {
      log("sceneName: "+getSceneNameFromTrack(effectTracks[0]));
    }

    savePreset(preset.name, preset);
    fxPresets.presets[preset.name] = preset;
    fxPresetsCount = Object.keys(fxPresets.presets).length;
    log("added a "+fxType+" preset from "+fxName);
  }




  Flickable {
    id: presetsViewFlick
    // anchors.fill: parent
    anchors.left: parent.left
    anchors.top: parent.top
    width: parent.width
    height: visibleHeight

    contentHeight: presetsLists.height
    anchors.margins: 10
    anchors.topMargin: 42
    onContentYChanged: {
      rocket.markUpdateFrequent();
    }
    clip: true
    Column {
      id: presetsLists
      anchors.top: parent.top
      anchors.left: parent.left
      anchors.right: parent.right
      PresetsViewSingleList {
        title: (fxFilter === "") ? "All Fx Presets" : "Fx Presets with \""+fxFilter+"\""
        presets: (fxPresets && fxPresets.presets && fxPresetsCount !== -1) ? fxPresets.presets : undefined
        modelSet: fxPresetsCount
        onModelChanged: {
          if (fxPresets && fxPresets.presets) {
            presets = fxPresets.presets;
          }
        }
      }
    }
  }

  Item {
    id: userIdField
    anchors.top: parent.top
    anchors.right: parent.right
    width: 75
    height: 16
    Rectangle {
      anchors.fill: parent
      color: "black"
      border.color: "white"
      border.width: 1
    }
    TextInput {
      anchors.fill: parent
      anchors.leftMargin: 5
      text: userId
      onActiveFocusChanged: {
        if (activeFocus) {
          rocket.markUpdateFrequent();
        }
      }
      selectByMouse: true
      onTextChanged: {
        rocket.markUpdateFrequent();
      }
      onSelectionStartChanged: {
        rocket.markUpdateFrequent();
      }
      onSelectionEndChanged: {
        rocket.markUpdateFrequent();
      }

      onAccepted: {
        userId = text;
        rootFocus.forceActiveFocus();
        rocket.markUpdateFrequent();
      }
      Keys.onEscapePressed: {
        rootFocus.forceActiveFocus();
        text = userId;
        rocket.markUpdateFrequent();
      }
      color: "white"
    }
    Text {
      text: "User ID:"
      horizontalAlignment: Text.AlignRight
      anchors.right: parent.left
      anchors.rightMargin: 5
      width: 70
      height: 16
      color: "white"
    }
  }

  Column {
    id: infoFields
    anchors.top: userIdField.bottom
    anchors.right: parent.right
    Row {
      Text {
        width: 100
        text: "Rocket FX Type:"
        horizontalAlignment: Text.AlignRight
        color: "white"
      }
      Item { width: 5; height: 1 }
      Text {
        width: userIdField.width+0
        text: focusedEffectBaseName
        color: "white"
      }
    }
    Row {
      Text {
        width: 100
        text: "Rocket FX Instance:"
        horizontalAlignment: Text.AlignRight
        color: "white"
      }
      Item { width: 5; height: 1 }
      Text {
        width: userIdField.width+0
        text: focusedEffectName
        color: "white"
      }
    }
  }

  property int viewSortMode: 1
  property var sortModes: [
    "ABC",
    "Latest"
  ]

  Column {
    id: viewOptions
    anchors.left: parent.left
    anchors.top: parent.top
    width: 150
    Row {
      height: 16
      Text {
        text: "sort:"
        horizontalAlignment: Text.AlignRight
        color: "white"
        width: 32
      }
      Item {
        width: 64
        height: 16
        Rectangle {
          id: sortModeRect
          anchors.fill: sortModeText
          color: "black"
          border.color: "white"
          border.width: 1
        }
        MouseArea {
          anchors.fill: sortModeRect
          onClicked: {
            if (viewSortMode < sortModes.length-1) {
              viewSortMode++;
            } else {
              viewSortMode = 0;
            }
            rocket.markUpdateFrequent();
          }
        }
        Text {
          id: sortModeText
          width: 64
          text: sortModes[viewSortMode]
          horizontalAlignment: Text.AlignHCenter
          color: "white"
        }
      }
    }

    Row {
      id: filterEdit
      height: 16

      Text {
        text: "filter:"
        horizontalAlignment: Text.AlignRight
        width: 32
        height: 16
        color: "white"
      }
      Item {
        width: 200
        height: 16
        Rectangle {
          anchors.fill: parent
          color: "black"
          border.color: "white"
          border.width: 1
        }
        TextInput {
          anchors.fill: parent
          anchors.leftMargin: 5
          text: fxFilter
          onActiveFocusChanged: {
            if (activeFocus) {
              rocket.markUpdateFrequent();
            } else {
              fxFilter = text;
            }
          }

          onTextChanged: {
            fxFilter = text;
            rocket.markUpdateFrequent();
          }
          selectByMouse: true
          onSelectionStartChanged: {
            rocket.markUpdateFrequent();
          }
          onSelectionEndChanged: {
            rocket.markUpdateFrequent();
          }
          onAccepted: {
            fxFilter = text;
            rootFocus.forceActiveFocus();
            rocket.markUpdateFrequent();
          }

          Keys.onEscapePressed: {
            rootFocus.forceActiveFocus();
            text = fxFilter;
            rocket.markUpdateFrequent();
          }
          color: "white"
        }
      }
    }

  }



}
