#version 430

#include <commons.glsl>

uniform sampler2D s_NoiseRGBA;
uniform sampler3D s_SDF;

in vec3 vNorm;
in vec3 vLocalPos;
in vec4 vCoords;
in vec4 vWorldPos;
in vec3 vWorldNorm;
in vec4 vsColor;

uniform vec4 colorDiffuse;
uniform int materialId;
uniform int gWriteDepth;

// NOTE: we try to always render using base object front faces so we have conservative depth
//       when inside the object we will just pass the define and have 2 shaders and use the second
//       one to not use the conservative depth

uniform mat4 mModel;
uniform mat4 mView;
uniform vec4 vModelview[3];
uniform mat4 mProjection;

uniform vec3 vCameraPosition;
uniform vec4 vNearFarPlane; // min, max, 2*min*max, max-min

// output for 2 buffers
#ifndef SHADOWMAP_PASS
layout(location = 0) out vec4 outAlbedo;
layout(location = 1) out uint outNormalMaterial;
#endif

// basic template based on the shadertoy framework template

#if 1 // sphere test
float doModel(vec3 p)
{
	// sphere intersection for testing
	return length(p - vec3(0.0, 0.0, 0.0)) - 1.0;
}

float sdBox( vec3 p, vec3 b )
{
	vec3 d = abs(p) - b;
	return length(max(d,0.0))
		 + min(max(d.x,max(d.y,d.z)),0.0); // remove this line for an only partially signed sdf 
}

float sdSphere( vec3 p, float s )
{
	return length(p)-s;
}

float doModelFromTexture(vec3 p)
{
	// TODO: add proper break here. we have to assume we are inside the bbox
	if (p.x < -1.0 || p.y < -1.0 || p.z < -1.0
	 || p.x >  1.0 || p.y >  1.0 || p.z >  1.0)
		return 10.0;
	else
		return texture(s_SDF, (p + 1.0) / 2.0).r;
}

float doModelBounding(vec3 p)
{
	return sdBox(p, vec3(0.90, 0.90, 0.90));
}



float calcIntersection(in vec3 ro, in vec3 rd, out vec4 color)
{
	const float maxd = 2000.0;				// max trace distance

	float t = 0.0;
	float res = -1.0;
	float bbox_h = 0.0;
	
	// bbox intersection. replace this with normal box intersect to calculate distance
	{
		const float precis = 0.001;        // precission of the intersection
		float h = precis*2.0;

		for( int i=0; i<9; i++ )          // max number of raymarching iterations is 90
		{
			if(h<precis || t>maxd)
				break;
			h = doModelBounding(ro+rd*t);
			t += h;
		}

		bbox_h = h;
	}

	// now the actual texture marching
	if (t < maxd)
	{
		const float precis = 0.01;        // precission of the intersection
		float h = precis*2.0;

		res = -1.0;

		for(int i=0; i<90; i++)
		{
			if (h < precis || t > maxd)
				break;
			h = doModelFromTexture( ro+rd*t );
			t += h;
		}

		if (t < maxd && h < 0.01)
		{
			res = t;
			color = vec4(1.0);
		}
		else
		{
			res = -1.0;
		}
	}

	return res;
}

vec3 doModelNormal(vec3 pos)
{
	const float eps = 0.002;             // precision of the normal computation

	const vec3 v1 = vec3( 1.0,-1.0,-1.0);
	const vec3 v2 = vec3(-1.0,-1.0, 1.0);
	const vec3 v3 = vec3(-1.0, 1.0,-1.0);
	const vec3 v4 = vec3( 1.0, 1.0, 1.0);

	return normalize( v1*doModel( pos + v1*eps ) + 
					  v2*doModel( pos + v2*eps ) + 
					  v3*doModel( pos + v3*eps ) + 
					  v4*doModel( pos + v4*eps ) );
}
#endif

vec3 transform_normal(vec3 n)
{
	vec3 tn;
	tn.x = dot(vModelview[0].xyz, n);
	tn.y = dot(vModelview[1].xyz, n);
	tn.z = dot(vModelview[2].xyz, n);
	return normalize(n);
}

void main() {

	float g = vNorm.z * 0.5 + 0.5;
	vec3 worldNorm = vWorldNorm;

	outNormalMaterial = encode_normal_material(normalize(worldNorm), materialId);
	vec4 texture = colorDiffuse;

	vec4 cp = vec4(vCameraPosition, 1.0);
	vec3 rc;
	rc.x = dot(vModelview[0], cp);
	rc.y = dot(vModelview[1], cp);
	rc.z = dot(vModelview[2], cp);
	rc = vec3(0.0);

	// NOTE: when rendering to the shadowmap we should be rendering backfaces
	//       so maybe just mirror the camera and render 'back' towards the real one

	vec3 ro = vWorldPos.xyz;		// start tracing at the cube surface. still should clamp at the "outgoing face"
	vec3 rd = normalize(vWorldPos.xyz - vCameraPosition);
	
	vec4 color = vec4(1.0);
	float intersection = calcIntersection(ro, rd, color);
	if (intersection > 0.0)
	{
		vec3 ri = ro + intersection * rd;
		outNormalMaterial = encode_normal_material(transform_normal(-doModelNormal(ri)), materialId);

		ri = (mView * vec4(ri, 1.0)).xyz;	// move to view space (camera)
	
		outAlbedo = color;
		
		if (color.a < 0.001)
			discard;
	
		if (gWriteDepth != 0)
		{
			float near = -1.0;	// this is depth range, not the projection
			float far = 1.0;
			float depth = (mProjection * vec4(ri, 1.0)).z / (mProjection * vec4(ri, 1.0)).w;
			depth =  (((far - near) * depth) + near + far) / 2.0;
			gl_FragDepth = depth;
		}
	}
	else
	{
		if (intersection == -1.0)
			outAlbedo = vec4(5.0, 5.0, 0.0, 1.0);
		else
			outAlbedo = vec4(5.0, 0.0, 0.0, 1.0);

		if (gWriteDepth != 0)
		{
			float near = -1.0;	// this is depth range, not the projection
			float far = 1.0;
			float depth = (mProjection * vec4(ro, 1.0)).z / (mProjection * vec4(ro, 1.0)).w;
			depth =  (((far - near) * depth) + near + far) / 2.0;

			gl_FragDepth = depth;
		}

		discard;
	}
}
