'use strict'

entityRegistry['module']['falling'] = {
    init: (staticConfig) => {
        const {
            numItems,
            spawnSize,
            spawnHeight,
            frameCount,
            seed,
        } = { ...staticConfig }
        const rng = new Math.seedrandom(seed)

        const world = new CANNON.World()
        world.gravity.set(0, 0, -9.82) // m/s²

        // Materials
        var groundMaterial = new CANNON.Material('groundMaterial')
        // Adjust constraint equation parameters for ground/ground contact
        var ground_ground_cm = new CANNON.ContactMaterial(groundMaterial, groundMaterial, {
            friction: .8,
            restitution: 0.3,
            contactEquationStiffness: 1e8,
            contactEquationRelaxation: 3,
            frictionEquationStiffness: 1e8,
            frictionEquationRegularizationTime: 3,
        });
        // Add contact material to the world
        world.addContactMaterial(ground_ground_cm)

        // // Create a slippery material (friction coefficient = 0.0)
        // var slipperyMaterial = new CANNON.Material("slipperyMaterial");
        // // The ContactMaterial defines what happens when two materials meet.
        // // In this case we want friction coefficient = 0.0 when the slippery material touches ground.
        // var slippery_ground_cm = new CANNON.ContactMaterial(groundMaterial, slipperyMaterial, {
        //     friction: 0,
        //     restitution: 0.3,
        //     contactEquationStiffness: 1e8,
        //     contactEquationRelaxation: 3
        // });
        // // We must add the contact materials to the world
        // world.addContactMaterial(slippery_ground_cm);

        const sphereBody = []
        const sphereContacts = []
        const sphereHitStrength = []
        for (let i = 0; i < numItems; ++i) {
            const body = new CANNON.Body({
                mass: 5.25,
                position: new CANNON.Vec3((rng()-.5)*spawnSize[0], (rng()-.5)*spawnSize[2], spawnHeight + (rng()-.5)*spawnSize[1]),
                quaternion: new CANNON.Quaternion(rng(), rng(), rng(), rng()),
                // shape: new CANNON.Box(new CANNON.Vec3(1,1,1)),
                shape: new CANNON.Sphere(1.5),
                material: groundMaterial
            })
            world.addBody(body)
            body.applyLocalForce(new CANNON.Vec3(rng()*2-1, rng()*2-1, -(rng()*5+5)), new CANNON.Vec3(rng()*2-1, rng()*2-1, rng()*2-1))
            sphereBody.push(body)
            sphereContacts.push(false)
            sphereHitStrength.push(0)
        }

        var groundBody = new CANNON.Body({
            mass: 0, // mass == 0 makes the body static
            material: groundMaterial,
        });
        var groundShape = new CANNON.Plane()
        groundBody.addShape(groundShape)
        world.addBody(groundBody)

        var fixedTimeStep = 1.0 / 100.0 // seconds

        const bodies = []
        for (let i = 0; i < frameCount; ++i) {
            for (let j = 0; j < sphereBody.length; ++j) {
                bodies[j] = bodies[j] || []
                const pos = [sphereBody[j].position.x, -sphereBody[j].position.z, sphereBody[j].position.y]
                const quat = sphereBody[j].quaternion
                const rot = m4.quaternionToEuler([-quat.y, quat.z, -quat.x, quat.w])
                let hit = false
                for (let i = 0; i < world.contacts.length; ++i) {
                    const contact = world.contacts[i]
                    if (contact.bi === sphereBody[j] || contact.bj === sphereBody[j]) {
                        hit = true
                    }
                }
                const newHit = hit === true && sphereContacts[j] === false
                if (newHit) {
                    sphereHitStrength[j] = 1
                } else {
                    sphereHitStrength[j] = Math.max(sphereHitStrength[j] - 0.05, 0)
                }
                bodies[j].push([pos, [-rot[0], -rot[1], -rot[2]], sphereHitStrength[j]])
                sphereContacts[j] = hit
            }
            world.step(fixedTimeStep)
        }

        return {
            bodies
        }
    },
    staticConfig: [
        { paramName: 'numItems', displayName: 'Items', type: 'int', defaultValue: 100, triggerInit: true },
        { paramName: 'spawnSize', displayName: 'Spawn Size', type: 'float3', defaultValue: [50, 300, 50], triggerInit: true },
        { paramName: 'spawnHeight', displayName: 'Spawn Height', type: 'float', defaultValue: 300, triggerInit: true },
        { paramName: 'frameCount', displayName: 'Frame Count', type: 'int', defaultValue: 1000, triggerInit: true },
        { paramName: 'seed', displayName: 'Seed', type: 'string', defaultValue: 'seed', triggerInit: true },
    ],
    dynamicConfig: [
        { paramName: 'model', displayName: 'Model', type: 'model', defaultValue: '' },
        { paramName: 'itemScale', displayName: 'Item Scale', type: 'float', defaultValue: 1 },
        { paramName: 'animation', displayName: 'Animation', type: 'float', defaultValue: 0 },
    ],
    actions: {
        'render': (self, frameTime, config, ctx) => {
            const {
                frameCount,
                model,
                itemScale,
                animation,
            } = { ...config }

            const colorBuffer = renderer.getCurrentBuffer('color')
            const depthBuffer = renderer.getCurrentBuffer('depth')
            const brightnessBuffer = renderer.getCurrentBuffer('brightness')

            const scaleMat = m4.scaling(itemScale, itemScale, itemScale)
            const clampedAnimation = clamp(animation, 0, 1)
            const frame = Math.floor(clampedAnimation * (frameCount-1))
            const frameAlpha = (clampedAnimation * (frameCount-1)) % 1
            self.bodies.forEach(body => {
                const pos1 = body[frame][0]
                const pos2 = body[frame+1][0]
                const pos = m4.lerpVectors(pos1, pos2, frameAlpha)
                if (pos[1] > -50 && pos[1] < 20) {
                    const rot1 = body[frame][1]
                    const rot2 = body[frame+1][1]
                    const rot = [
                        Math.abs(rot1[0]-rot2[0])>Math.PI ? rot1[0] : lerp(rot1[0],rot2[0],frameAlpha),
                        Math.abs(rot1[1]-rot2[1])>Math.PI ? rot1[1] : lerp(rot1[1],rot2[1],frameAlpha),
                        Math.abs(rot1[2]-rot2[2])>Math.PI ? rot1[2] : lerp(rot1[2],rot2[2],frameAlpha),
                    ]

                    const translationMat = m4.translation(pos[0], pos[1], pos[2])
                    const rotationMat = m4.zRotate(m4.yRotate(m4.xRotation(rot[0]), rot[1]), rot[2])
                    const worldMat = m4.multiply(m4.multiply(translationMat, rotationMat), scaleMat)
                    const hs = Math.pow(body[frame][2], .5)

                    // model.objects[0].subObjects[1].diffuse = [0, 0, 0]
                    // model.objects[0].subObjects[1].emissive = [lerp(0.1, .25, hs), lerp(.25, 1, hs), 0]
                    renderer.drawModel(model, worldMat, colorBuffer, depthBuffer, brightnessBuffer)
                }
            })
        }
    }
}
