#include <err.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <zlib.h>

#include <openssl/md5.h>
#include <openssl/sha.h>


void version()
{
    printf("magicrestore v1337-b33 (useless editon)\n");
    printf("by zelest\n");
}


void usage()
{
    printf("Usage: magicrestore [OPTIONS]\n");
    printf("Example: magicrestore -m b4d7d64c550a874da41a4f03c4e56f1f -l 3 -o result.txt\n");
    printf("Options:\n");
    printf("  -c CRC32, --crc32=CRC32      Specify the CRC32 hash to match the restored file\n");
    printf("  -h, --help                   Display this help message\n");
    printf("  -l SIZE, --length=SIZE       Set the size of the file to restore from the void\n");
    printf("  -m MD5, --md5=MD5            Specify the MD5 hash to match the restored file\n");
    printf("  -o FILE, --output=FILE       Specify the output file path (optional)\n");
    printf("  -s SHA256, --sha256=SHA256   Specify the SHA256 hash to match the restored file\n");
    printf("  -v, --version                Displey version\n");
}


void md5_digest(const char* input, char* output)
{
	MD5_CTX context;
	unsigned char digest[MD5_DIGEST_LENGTH];

	MD5_Init(&context);
	MD5_Update(&context, input, strlen(input));
	MD5_Final(digest, &context);

	for (int i = 0; i < MD5_DIGEST_LENGTH; ++i) {
		sprintf(&output[i * 2], "%02x", (unsigned int)digest[i]);
	}
}


void sha256_digest(const char* input, char* output)
{
    SHA256_CTX ctx;
    unsigned char hash[SHA256_DIGEST_LENGTH];

    SHA256_Init(&ctx);
    SHA256_Update(&ctx, input, strlen(input));
    SHA256_Final(hash, &ctx);

    for (int i = 0; i < SHA256_DIGEST_LENGTH; i++) {
        sprintf(output + (i * 2), "%02x", hash[i]);
    }
    output[64] = '\0';
}


void crc32_digest(const char* input, char* output)
{
	unsigned long crc = crc32(0L, Z_NULL, 0);
    crc = crc32(crc, (const Bytef*)input, strlen(input));
    sprintf(output, "%08lx", crc);
}


int main(int argc, char *argv[])
{
	FILE *fh;
	int option;
	char *md5 = NULL;
	char *crc32 = NULL;
	char *sha256 = NULL;
	char *outfile = NULL;
	char digest[128];
	char spinner[] = { '|', '/', '-', '\\' };
	int size=0, printoutput=1, sp=0, i=0, j=0, skip=0, nullskip=0, done=0;

	struct option longOptions[] = {
		{"crc32", required_argument, NULL, 'c'},
		{"sha256", required_argument, NULL, 's'},
		{"md5", required_argument, NULL, 'm'},
		{"length", required_argument, NULL, 'l'},
		{"output", required_argument, NULL, 'o'},
		{"version", required_argument, NULL, 'v'},
		{"help", required_argument, NULL, 'h'},
		{NULL, 0, NULL, 0}
	};

	if ((fh = fopen("/dev/urandom", "r")) == 0) {
		errx(1, "Unable to open /dev/urandom");
	}

	while ((option = getopt_long(argc, argv, "hc:l:m:s:o:v", longOptions, NULL)) != -1) {
		switch (option) {
			case 'l':
				size = atoi(optarg);
				break;
			case 's':
				sha256 = optarg;
				break;
			case 'm':
				md5 = optarg;
				break;
			case 'c':
				crc32 = optarg;
				break;
			case 'o':
				outfile = optarg;
				printoutput = 0;
				break;
			case 'h':
				usage();
				exit(EXIT_SUCCESS);
			case 'v':
				version();
				exit(EXIT_SUCCESS);
			default:
				printf("Unknown option\n");
				exit(EXIT_FAILURE);
		}
	}

	if (size == 0 || (sha256 == NULL && md5 == NULL && crc32 == NULL)) {
		usage();
		exit(EXIT_FAILURE);
	}

	char *buf = malloc(size);

	printf("\n");
	while (done == 0) {
		i++;
		fread(buf, size, 1, fh);
		skip=0;
		nullskip=0;

		for (j=0; j<size; j++) {
			if (buf[j] == '\0') {
				nullskip=1;
			}
		}
		if (nullskip) {
			continue;
		}

		if (skip == 0 && sha256 != NULL) {
			sha256_digest(buf, digest);
			if (strstr(digest, sha256) != 0) {
				done = 1;
			} else {
				skip = 1;
			}
		}

		if (skip == 0 && md5 != NULL) {
			md5_digest(buf, digest);
			if (strstr(digest, md5) != 0) {
				done = 1;
			} else {
				skip = 1;
			}
		} 

		if (skip == 0 && crc32 != NULL) {
			crc32_digest(buf, digest);
			if (strstr(digest, crc32) != 0) {
				done = 1;
			} else {
				skip = 1;
			}
		}

		if (done) {
			fclose(fh);

			printf("\n\nWoohoo! \\o/\n\n");
			printf("After %d attempts, your file was successfully restored from the void!\n\n", i);

			if (sha256 != NULL) {
				printf("File matching the SHA256 hash: %s\n", sha256);
			}
			if (md5 != NULL) {
				printf("File matching the MD5 hash: %s\n", md5);
			}
			if (crc32 != NULL) {
				printf("File matching the CRC32 hash: %s\n", crc32);
			}

			printf("\n");

			for (i=0; i<size; i++) {
				printf("[pos: %d, hex: 0x%02x, chr: %d] %c\n", i, (int)buf[i], (int)buf[i], buf[i]);
			}

			printf("\n");

			if (outfile != NULL) {
				fh = fopen(outfile, "w");
				if (fh != NULL) {
					if (fwrite(buf, 1, size, fh) == 0) {
						printf("Unable to save data to %s\n", outfile);
						printoutput=1;
					} else {
						printf("Your file was stored at %s\n", outfile);
					}
					fclose(fh);
				} else {
					printf("Unable to open %s\n", outfile);
					printoutput=1;
				}
			}

			if (printoutput) {
				printf("Data:\n%s\n\n", buf);
			}

			fflush(stdout);
			free(buf);
			exit(EXIT_SUCCESS);
		}

		if (i % 75000 == 0) {
			if (sp > 3) {
				sp = 0;
			}
		    printf("Read %dMb (%d million tries) from entropy to find your lost file... %c\r", (i*size) / 1024 / 1024, i / 1000000, spinner[sp % 4]);
			fflush(stdout);
			sp++;
		}
	}

	return 0;
}
