layout (location = 0) in vec2 v_uv;
layout (location = 1) in vec3 v_normal;
layout (location = 2) in vec3 v_model_pos;
layout (location = 3) in vec3 v_tangent;
layout (location = 4) in vec3 v_light_dir;
layout (location = 5) in vec3 v_world_normal;
layout (location = 6) in vec3 v_world_pos;
layout (location = 7) in vec3 v_world_eye;
layout (location = 8) in vec3 v_world_tangent;
layout (location = 9) in vec3 v_camera_pos;
layout (location = 10) in vec3 v_color;

layout (location = 0) out vec4 f_color;

layout (set = 1, binding = 0) uniform Uniforms {
    mat4 g_light_projection_from_world;
    mat4 g_camera_from_world;
    mat4 g_projection_from_camera;
    float g_chart_time;
    float g_app_time;
    vec3 g_light_dir;

    vec4 color;
    float bias;
    float ambient;
    float normal_factor;

    vec3 specular_color;
    vec4 foo;
    float cut_scale;
    float tex_scale;
    vec3 cut_color;

    float obj_sphere_radius;
    vec3 obj_center;
    vec3 obj_rotate;
    float obj_transition;
};

layout (set = 1, binding = 1) uniform sampler2D tex;
layout (set = 1, binding = 2) uniform sampler2D shadow;
layout (set = 1, binding = 3) uniform sampler2D ambient_map;
layout (set = 1, binding = 4) uniform sampler2D normal_map;
layout (set = 1, binding = 5) uniform sampler2D env_map;
layout (set = 1, binding = 6) uniform sampler2D cut_map;

const float DISTANCE_MAX = 100;
const float SURFACE_PROXIMITY = 0.0001;

float calculate_light(vec3 world_pos) {
    vec3 lightspace_pos = (g_light_projection_from_world * vec4(world_pos, 1.0)).xyz;
    lightspace_pos.xy = lightspace_pos.xy * 0.5 + 0.5;
    float shadow_z = texture(shadow, lightspace_pos.xy).r;
    return (shadow_z + bias*0.01 > lightspace_pos.z) ? 1.0 : 0.0;
}

vec3 calculate_normal(vec3 normal, vec3 tangent, vec3 normal_map_sample, float intensity) {
    vec3 c_normal = normalize(normal);
    vec3 c_tangent = normalize(tangent);
    vec3 c_bitangent = cross(c_normal, c_tangent);
    vec3 mapped_normal = mat3(c_tangent, c_bitangent, c_normal) * normal_map_sample;
    vec3 scaled_normal = c_normal + intensity * (mapped_normal - c_normal);
    return scaled_normal;
}

vec3 light_pixel(vec3 pos, vec3 normal, float ambient_occlusion, float ambient, vec3 base_color, float light_min) {
    float light = calculate_light(pos);
    light = max(light, light_min);
//    light *= ambient_occlusion * ambient_occlusion;

    // Calculate environment map
    vec3 eye_dir = normalize(pos - v_world_eye);
    vec3 reflect_dir = reflect(eye_dir, normal);
    vec2 reflect_uv = reflect_dir.xy * 0.5 + 0.5;
    vec3 env_color = vec3(1); //texture(env_map, reflect_uv).rgb;

    // Light components
    vec3 light_dir = normalize(g_light_dir);
    float diffuse = max(dot(normal, light_dir), 0) * light;
    float specular = max(pow(dot(normal, light_dir), 5.0), 0.00) * (light + 0.1);

    vec3 final_color = base_color * v_color * (ambient + (1-ambient) * diffuse) + specular * env_color * specular_color;

    return final_color;
}



// https://iquilezles.org/articles/normalsSDF
vec3 calc_sdf_normal(vec3 pos_cs) {
    vec2 e = vec2(1.0, -1.0)*0.5773;
    const float eps = 0.0005;
    return normalize(e.xyy*sd(pos_cs + e.xyy*eps) +
    e.yyx*sd(pos_cs + e.yyx*eps) +
    e.yxy*sd(pos_cs + e.yxy*eps) +
    e.xxx*sd(pos_cs + e.xxx*eps));
}


void setup_sdf() {
    mat4 world_from_sdf_model = translate(-obj_center) * rotate(obj_rotate);
    mat4 camera_from_sdf_model = g_camera_from_world * world_from_sdf_model;
    sdf_model_from_camera = inverse(camera_from_sdf_model);
    l_obj_sphere_radius = obj_sphere_radius;
    l_obj_transition = obj_transition;
    if (sd(v_camera_pos) < 0) discard;
}

void main() {
    setup_sdf();

    float ao_map_sample = texture(ambient_map, v_uv).r;

//    vec3 base_color = texture(tex, v_uv * (1 + tex_scale)).rgb * color.rgb;
    vec3 base_color = color.rgb;

    // Calculate normal
    vec3 normal_map_sample = texture(normal_map, v_uv).xyz * 2 - 1;
    vec3 normal = calculate_normal(v_normal, v_tangent, normal_map_sample, normal_factor);
    vec3 world_normal = calculate_normal(v_world_normal, v_world_tangent, normal_map_sample, normal_factor);

    // Calculate final color
    vec3 color = light_pixel(v_world_pos, world_normal, ao_map_sample, ambient, base_color, 0.0);
    f_color = vec4(color, 1.0);
}
