#version 330 core

in vec2 texCoord;
out vec4 fragColor;

uniform sampler2D texture0;
uniform sampler2D texture1;
uniform sampler2D texture2;
uniform vec4 color = vec4(1,1,1,1);
uniform vec2 iResolution = vec2(1920.0, 1080.0);
uniform float time = 0.0;

// More 2D shapes here: https://iquilezles.org/articles/distfunctions2d

mat2 rotateZ(float angleDeg)
{
  // "radians — convert a quantity in degrees to radians"
  float angleRad = radians(angleDeg);
  // http://mathworld.wolfram.com/RotationMatrix.html
  mat2 rot = mat2(vec2(cos(angleRad), sin(angleRad)), vec2(-sin(angleRad), cos(angleRad)));
  return rot;
}

float circle(vec2 coord, float size)
{
  // "length — calculate the length of a vector"
  return length(coord)-size;
}

float rect(vec2 p, vec2 b)
{
  // "abs — return the absolute value of the parameter"
  vec2 d = abs(p) - b;
  // "min — return the lesser of two values"
  // "max — return the greater of two values"
  return min(max(d.x,d.y),0.0) + length(max(d,0.0));
}

// Note that IQ provides much more efficient way to do this
// however this is to illustrate that more complex shapes
// can be created from simpler shapes
float xcross(vec2 p, vec2 b)
{
  p *= rotateZ(45.0);
  float dist = rect(p, b);

  p *= rotateZ(90.0);
  dist = min(dist, rect(p, b));

  return dist;
}

float triangle(vec2 p, vec2 h)
{
  // https://en.wikipedia.org/wiki/Steiner_tree_problem
  // "sqrt — return the square root of the parameter"
  float steinerRatio = sqrt(3.0)/2.0;
  return max(-h.y,max(abs(p).x*steinerRatio-p.y*0.5,p.y)-h.x*0.5);
}

uniform float rotation = 0.0;
uniform float scale = 1.0;
uniform float x = 0.0;
uniform float y = 0.0;

void mainImage( vec2 fragCoord )
{
    vec2 uv = fragCoord.xy / iResolution.xy;
    vec2 aspectRatio = vec2(iResolution.x/iResolution.y, -1.0);
    vec2 coord = aspectRatio * (uv - vec2(0.5));

    // define shapes
    // "d" is for distance.
    // Shape is drawn in later part, when the value is negative :-)
    float d = xcross((coord - vec2(x,-y))*rotateZ(rotation), vec2(0.1*scale,0.3*scale));
	
    // Time varying pixel color
    vec3 col = vec3(0);//0.5 + 0.5*cos(time+uv.xyx+vec3(0,2,4));

    // Apply shapes
    // " sign returns -1.0 if x is less than 0.0, 0.0 if x is equal to 0.0, and +1.0 if x is greater than 0.0. "
    col = sign(d)*col;
    vec3 shapeColor = vec3(1.0,1.0,1.0);

    //antialiased version
    // "mix — linearly interpolate between two values"
    // "smoothstep performs smooth Hermite interpolation between 0 and 1 when edge0 < x < edge1. This is useful in cases where a threshold function with a smooth transition is desired."
    col = mix(col, shapeColor, 1.0-smoothstep(0.0,0.003,d));

	  vec4 tex1Col = texture(texture1, texCoord);
	  vec4 tex2Col = texture(texture2, texCoord);

    col = vec3((1.0-tex2Col.r)*col.r,(1.0-tex2Col.g)*col.g,(1.0-tex2Col.b)*col.b);

    //not antialiased version
    //col = mix(col, shapeColor, d > 0.0 ? 0.0 : 1.0);

    fragColor = vec4(col,col.r*(0.9+tex1Col.a*0.1))*color;
}

void main()
{
    vec2 fragCoord = texCoord * iResolution;
    mainImage(fragCoord);
}
