;===============================================================================
; $f1 racer
; a c64 nanogame with background graphics, sprites, joystick control and sound
; in less than 256 bytes
;
; by Wil, Pararaum, and Freeze
; presented at Lovebyte 2023
;===============================================================================

.include "LAMAlib.inc"

.MACPACK cbm

;-------------------------------------------------------------------------------
; constants
;
DELAY      =8
STREETWIDTH=6

;LINENUMBER is a 16 bit value and is the line number of the BASIC SYS line
;the BASIC interpreter copies the value into ZP addresses $39 and $3a
;low byte of line number should be $ff to ensure score display is drawn immediately
;higbyte should be $80 to ensure sprite starting in the middle of the screen
LINENUMBER=$80ff

;-------------------------------------------------------------------------------
; zero page address labels
;
delaycnt   =$02
street_xpos=$03
scorecnt   =$39	;zero page address, initialized with the low byte of line number
spr_xpos   =$3A	;zero page address, initialized with the high byte of line number

;-------------------------------------------------------------------------------
; memory address labels
;
SOUNDSRC   =$e2dd	;a good pattern in ROM to set the SID registers
SPRITEADDR =$340	;where we put our sprites, must be $0d*$40 since we reuse the value
SCREEN_BASE=$400
SCORE	   =SCREEN_BASE + scorestr_end - scorestr - 2	;since score is a string, the first byte is the tenths and the second the ones
SYSTARGET  =$80b	;address of gamestart, but needs to be given as a constant here

;-------------------------------------------------------------------------------
; BASIC stub code with SYS line
; the line number and the byte before the end are reused as data and program code
;
	;.byte $0b	;we skip this byte because we start our file at address $802. 
			;The content of $801 is regenerated by BASIC relink routine
			;which is run automatically after loading
	.byte $08
        .byte <LINENUMBER,>LINENUMBER
        .byte $9e   ;SYS token
        .byte $30+SYSTARGET/1000
        .byte $30+SYSTARGET .mod 1000/100
        .byte $30+SYSTARGET .mod 100/10
        .byte $30+(SYSTARGET .mod 10)

	.byte $00	;end of line marker

;-------------------------------------------------------------------------------
; game code starts here
;
gamestart:	
	.assert gamestart = SYSTARGET, error, "you need to adjust the SYSTARGET address"

	;play sfx and set X to 0 afterwards
	;first time the game is run we hear a softer sound, it's a feature!

	ldx $eb00	;the second byte of this MUST be 0 to serve as end of BASIC program marker
	;X contains now $0c
soundloop:
	lda SOUNDSRC-1,x
	sta $d400+14-1,x
	dex
	bne soundloop
	lda #$80
        sta $d412

	;x contains 0 because of previous loop or clean entry from BASIC
	stx $d020	;black border
	inx		;X contains now 1
	stx 646		;set cursor color to white
	stx $d015	;activate sprite
	jsr $e544

	ldx #'0'
	stx SCORE
	stx SCORE+1

	;install sprite
	ldx #spritedata_end - spritedata - 1
	.assert        (spritedata_end - spritedata - 1) = $0d, error, "Value not $0d"
	stx street_xpos	;initialitize position of street with $0d
	stx 2040	;set sprite pointer to $0d (sprite at $340)	
copyloop:
	lda spritedata,x	; Load sprite data.
	sta SPRITEADDR+1,x	; Store upper half.
	sta SPRITEADDR+1+15,x	; Store bottom half.
	dex
	bpl copyloop

	lda #220		; $DC
	sta $d021		; $C is grey 2..., luckily.
	sta $d001

gameloop:
	ldx spr_xpos
	;check joystick
	lda $dc00
	lsr
	lsr
	lsr
	if cc
	  dex
	  cpx #12
	  if cc
  	    inx
	  endif
	endif
	lsr
	if cc
	  inx
	  cpx #160
	  if cs
  	    dex
	  endif
	endif
 
	;set sprite X coordinate from doubled X register
	stx spr_xpos
	txa
	asl
	sta $d000
	rol $d010

	;wait end of frame
waitsync:
	ldx $d012		;load current rasterline
        inx			;this results to zero if rasterline is $ff
	bne waitsync		;redo if rasterline was not $ff
	
	lda $d01f	    	;check sprite-sprite collision
	bne gamestart

	inc scorecnt
	if eq
	  lda #<scorestr
	  ldy #>scorestr
	  sta scorecnt 	    	;this is a value around $ea, so the delay until the next score update is not too long 
	  jsr STROUT

          ldx SCORE+1
          inx
          cpx #'9'+1
          if cs
            ldx #'0'
            inc SCORE		;increase tenths
          endif
          stx SCORE+1
	endif

	dec delaycnt
	bpl gameloop
	lda #DELAY
	sta delaycnt

	;fill whole line
	ldy #39
fill_line_loop:	
	lda (street_xpos),y	;this points into ROM ($B1xx)
	; Use symbols like ▚, ▞, ▖, ▜.
	ora #%01111011
	sta SCREEN_BASE+40,y
	and #%11111101
	sta $d800+40,y
	dey
	bpl fill_line_loop
	
	;change position of street
	ldx street_xpos
	lda $dc04 ;using low value of CIA timer as random number source
	dex
	if eq
	  inx
	endif
	lsr
	if cc
	  inx
	  inx
	  cpx #30
	  if cs
	    dex
	    dex
	  endif
	endif
	stx street_xpos

	;punch in the hole (=street)
	lda #32
	ldy #STREETWIDTH
draw_street:
	sta SCREEN_BASE+40,X
	inx
	dey
	bne draw_street
	;y is now 0

	;draw next game line
	iny 		;y is now 1
	sty 677
	jsr 59777	;scroll down screen lines 1..24
	jmp gameloop

	.rodata

;-------------------------------------------------------------------------------
;string for score display, change this to translate the game to other languages :-)

scorestr:
	.byte 19,"score:",0	;19 = home
scorestr_end:

;-------------------------------------------------------------------------------
; sprite data
; the sprite is assumed to be symmetrical around the horizontal axis, so only
; the upper half of the sprite is stored here

spritedata:
       ;.byte %00000000
	.byte            %01111110, %00000000
	.byte %00000001, %01111110, %10000000
	.byte %00000001, %11111111, %10000000
	.byte %00000001, %11111111, %10000000
	.byte %00000001, %01111110, %10000000
spritedata_end:
