//***************************************************************************
// "background.c"
// Background code, responsible for parsing the background script and drawing
// the background (also used for parallax in levels).
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stdint.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include "main.h"
#include "file.h"
#include "ingame.h"
#include "parser.h"
#include "settings.h"
#include "video.h"

// Background color (used where nothing else renders)
static uint32_t bgcolor = 0xFF000000;

// Color to use in CGA mode
// If NOCGACOLOR, then it wasn't choosen (falls back to bgcolor in that case)
// Note this goes through the CGA color filters, it's a raw RGBA value
#define NOCGACOLOR 0xFFFFFFFF
static uint32_t cgacolor;

// Information about a piece of parallax (a "layer")
typedef struct {
   // Positioning
   int32_t x;                 // Base X coordinate
   int32_t y;                 // Base Y coordinate
   int32_t ratio_x;           // Scrolling X ratio
   int32_t ratio_y;           // Scrolling Y ratio
   int32_t repeat_x;          // Repeat offset in the X axis (0 = disable)
   int32_t repeat_y;          // Repeat offset in the Y axis (0 = disable)
   int32_t speed_x;           // Horizontal speed
   int32_t speed_y;           // Vertical speed
   unsigned reverse_y: 1;     // Set if positioned from the bottom

   // Graphics
   AnimState anim;            // Animation
} BGLayer;

// Where layers are stored
static size_t num_layers = 0;
static BGLayer *layers = NULL;

// Using our own counter since we can't rely on game_anim to be updated
static unsigned bg_anim = 0;

//***************************************************************************
// load_background
// Loads the background data from the specified script file
//---------------------------------------------------------------------------
// param filename: file to read from
// param gfxset: graphics set with the sprites to use
//---------------------------------------------------------------------------
// Amusing note: if you call this function without unloading the previous
// background, it *will merge the backgrounds*. Literally. This was not
// intentional, but we don't pretend to remove this behavior, so feel free to
// rely on it. This may be useful later!
//***************************************************************************

void load_background(const char *filename, const GraphicsSet *gfxset) {
   // Set a default background color
   bgcolor = 0x404040;
   cgacolor = NOCGACOLOR;

   // Open script
   File *file = open_file(filename, FILE_READ);
   if (file == NULL)
      abort_program(ERR_LOADBG, filename);

   // Go through all lines
   while (!end_of_file(file)) {
      // Get parameters for this line
      char *line = read_line(file);
      if (line == NULL)
         abort_program(ERR_LOADBG, filename);
      Args *args = parse_args(line);
      if (args == NULL)
         abort_program(ERR_LOADBG, filename);
      free(line);

      // Empty line?
      if (args->count == 0) {
         free_args(args);
         continue;
      }

      // Background color?
      if (!strcmp(args->list[0], "bgcolor")) {
         // Check parameters
         // To-do: explain the error?
         if (args->count != 2)
            abort_program(ERR_LOADBG, filename);
         if (!is_color(args->list[1]))
            abort_program(ERR_LOADBG, filename);

         // Retrieve new background color
         bgcolor = strtoul(args->list[1], NULL, 16);
      }

      // CGA mode background?
      if (!strcmp(args->list[0], "cgacolor")) {
         // Check parameters
         // To-do: explain the error?
         if (args->count != 2)
            abort_program(ERR_LOADBG, filename);
         if (!is_color(args->list[1]))
            abort_program(ERR_LOADBG, filename);

         // Retrieve new background color
         cgacolor = strtoul(args->list[1], NULL, 16);
      }

      // Layer?
      else if (!strcmp(args->list[0], "layer")) {
         // Check parameters
         // To-do: explain the error?
         if (args->count < 4)
            abort_program(ERR_LOADBG, filename);
         if (!is_valid_id(args->list[1]))
            abort_program(ERR_LOADBG, filename);
         if (!is_integer(args->list[2]))
            abort_program(ERR_LOADBG, filename);
         if (!is_integer(args->list[3]))
            abort_program(ERR_LOADBG, filename);

         // Create a new layer
         layers = (BGLayer *) realloc(layers, sizeof(BGLayer) *
            (num_layers + 1));
         if (layers == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         BGLayer *ptr = &layers[num_layers];
         num_layers++;

         // Retrieve layer basic data
         ptr->anim.frame = get_anim(gfxset, args->list[1]);
         if (ptr->anim.frame != NULL)
            ptr->anim.frame_time = ptr->anim.frame->duration;
         ptr->x = atoi(args->list[2]);
         ptr->y = atoi(args->list[3]);

         // Set reasonable defaults for optional parameters
         ptr->ratio_x = 0;
         ptr->ratio_y = 0;
         ptr->repeat_x = 0;
         ptr->repeat_y = 0;
         ptr->speed_x = 0;
         ptr->speed_y = 0;
         ptr->reverse_y = 0;

         // Check for extra arguments
         for (size_t i = 4; i < args->count; ) {
            // Scrolling ratio?
            if (!strcmp(args->list[i], "ratio")) {
               // Check parameters
               // To-do: explain the error?
               if (i + 3 > args->count)
                  abort_program(ERR_LOADBG, filename);
               if (!is_integer(args->list[i+1]))
                  abort_program(ERR_LOADBG, filename);
               if (!is_integer(args->list[i+2]))
                  abort_program(ERR_LOADBG, filename);

               // Get scrolling ratios
               ptr->ratio_x = atoi(args->list[i+1]);
               ptr->ratio_y = atoi(args->list[i+2]);

               // Skip parameters
               i += 3;
            }

            // Repeating?
            else if (!strcmp(args->list[i], "repeat")) {
               // Check parameters
               // To-do: explain the error?
               if (i + 3 > args->count)
                  abort_program(ERR_LOADBG, filename);
               if (!is_uinteger(args->list[i+1]))
                  abort_program(ERR_LOADBG, filename);
               if (!is_uinteger(args->list[i+2]))
                  abort_program(ERR_LOADBG, filename);

               // Get repeat offsets
               ptr->repeat_x = atoi(args->list[i+1]);
               ptr->repeat_y = atoi(args->list[i+2]);

               // Skip parameters
               i += 3;
            }

            // Speed?
            else if (!strcmp(args->list[i], "speed")) {
               // Check parameters
               // To-do: explain the error?
               if (i + 3 > args->count)
                  abort_program(ERR_LOADBG, filename);
               if (!is_integer(args->list[i+1]))
                  abort_program(ERR_LOADBG, filename);
               if (!is_integer(args->list[i+2]))
                  abort_program(ERR_LOADBG, filename);

               // Get speeds
               ptr->speed_x = atoi(args->list[i+1]);
               ptr->speed_y = atoi(args->list[i+2]);

               // Skip parameters
               i += 3;
            }

            // Spawn from the bottom?
            else if (!strcmp(args->list[i], "bottom")) {
               ptr->reverse_y = 1;
               i++;
            }

            // Huh...
            // Can't recognize parameter, skip it
            // Otherwise this thing will hang
            else {
               i++;
            }
         }
      }

      // Go for next line
      free_args(args);
   }

   // If no color was chosen for the CGA mode, default to the background
   // color. The whole point of that setting is to provide a different color
   // in case the background one isn't particularly fitting.
   if (cgacolor == NOCGACOLOR)
      cgacolor = bgcolor;

   // Done!
   close_file(file);
}

//***************************************************************************
// update_background
// Update the background animation (call once per frame)
//***************************************************************************

void update_background(void) {
   // Determine subpixel offset
   int32_t comma_offset = (bg_anim & 0x80) >> 7 |
                          (bg_anim & 0x40) >> 5 |
                          (bg_anim & 0x20) >> 3 |
                          (bg_anim & 0x10) >> 1 |
                          (bg_anim & 0x08) << 1 |
                          (bg_anim & 0x04) << 3 |
                          (bg_anim & 0x02) << 5 |
                          (bg_anim & 0x01) << 7;

   // Process all layers
   for (size_t i = 0; i < num_layers; i++) {
      // Make the layers move as needed
      layers[i].x += (layers[i].speed_x + comma_offset) >> 8;
      layers[i].y += (layers[i].speed_y + comma_offset) >> 8;

      // Update animation
      update_anim(&layers[i].anim);
   }

   // Update subpixel precision offset
   bg_anim++;
}

//***************************************************************************
// draw_background
// Draws the background (parallax in case of the levels)
//***************************************************************************

void draw_background(void) {
   // In audiovideo mode the background must *not* be rendered!
   if (settings.audiovideo) {
      clear_screen(AV_EMPTY);
      return;
   }

   // In CGA mode just draw the background with the CGA color and quit
   // Yeah, no parallax, not like CGA games usually had parallax :P
   if (settings.cga_mode) {
      clear_screen(settings.dark_bg ? 0x000000 : cgacolor);
      return;
   }

   // Draw background color
   clear_screen(bgcolor);

   // If background is disabled, stop here
   // (make sure to dim the color if needed)
   if (!settings.background) {
      if (settings.dark_bg) dim_screen(0x80);
      return;
   }

   // Determine position of the camera, this is needed for parallax in-game
   // However, these background functions can also be used outside a level,
   // and in such cases we want to behave as if the camera never moved.
   int32_t cam_x, cam_y, cam_rev_y;
   if (game_mode == GAMEMODE_INGAME) {
      cam_x = camera.x;
      cam_y = camera.y;
      cam_rev_y = camera.y - (camera.limit_bottom - (screen_h - 1));
   } else {
      cam_x = 0;
      cam_y = 0;
      cam_rev_y = screen_h - 1;
   }

   // Draw all layers
   for (size_t i = 0; i < num_layers; i++) {
      // Er...
      if (layers[i].anim.frame == NULL)
         continue;

      // Get sprite to use
      Sprite *spr = layers[i].anim.frame->sprite;
      if (spr == NULL || spr->width == 0 || spr->height == 0)
         continue;

      // Get sprite flags for this frame
      unsigned flags = layers[i].anim.frame->flags;

      // Determine horizontal position of sprite
      int32_t x = layers[i].x - (cam_x * layers[i].ratio_x >> 8);
      if (flags & SPR_HFLIP) {
         x += layers[i].anim.frame->x_offset;
         x -= spr->width - 1;
      } else
         x -= layers[i].anim.frame->x_offset;

      // Determine vertical position of sprite
      int32_t y;
      if (layers[i].reverse_y) {
         y = screen_h - layers[i].y - 1;
         y -= (cam_rev_y * layers[i].ratio_y) >> 8;
      } else {
         y = layers[i].y;
         y -= (cam_y * layers[i].ratio_y) >> 8;
      }

      if (flags & SPR_VFLIP) {
         y += layers[i].anim.frame->y_offset;
         y -= spr->height - 1;
      } else
         y -= layers[i].anim.frame->y_offset;

      // Off screen? (if repeat is disabled)
      if (!layers[i].repeat_x) {
         if (x > screen_w) continue;
         if (x <= -spr->width) continue;
      }
      if (!layers[i].repeat_y) {
         if (y > screen_h) continue;
         if (y <= -spr->height) continue;
      }

      // Retrieve the distance between each repeated sprite
      // If no repeating is done then we set the number high enough so the
      // first "repeat" is off-screen (thereby ending the loop early). Easier
      // than making dedicated code...
      int32_t delta_x = layers[i].repeat_x;
      int32_t delta_y = layers[i].repeat_y;
      if (!delta_x) delta_x = 0x10000;
      if (!delta_y) delta_y = 0x10000;

      // Determine where the tiling starts
      x += delta_x;
      y += delta_y;
      x = (x >= 0) ? (x % delta_x) : (delta_x + x % delta_x);
      y = (y >= 0) ? (y % delta_y) : (delta_y + y % delta_y);
      x -= delta_x;
      y -= delta_y;

      // Put sprite on screen
      for (int32_t yr = y; yr < screen_h; yr += delta_y)
      for (int32_t xr = x; xr < screen_w; xr += delta_x)
         draw_sprite(spr, xr, yr, flags);
   }

   // Darken the background if the relevant setting is set
   if (settings.dark_bg)
      dim_screen(0x80);
}

//***************************************************************************
// unload_background
// Gets rid of background information
//***************************************************************************

void unload_background(void) {
   // Get rid of all layers
   if (layers) free(layers);
   layers = NULL;
   num_layers = 0;

   // Reset background color
   bgcolor = 0x404040;
   cgacolor = NOCGACOLOR;
}
