//***************************************************************************
// "cutscene.c"
// Code for the cutscenes.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "main.h"
#include "background.h"
#include "file.h"
#include "input.h"
#include "loading.h"
#include "parser.h"
#include "reader.h"
#include "savegame.h"
#include "scene.h"
#include "settings.h"
#include "sound.h"
#include "text.h"

// Used to determine the loading bar length (since we don't know beforehand
// how many things we need to load)
static size_t load_count;
static size_t load_total;

// List of graphics set to load
typedef struct {
   char *name;
   GraphicsSet *gfxset;
} GfxSetList;
static GfxSetList *gfxset_list = NULL;
static size_t num_gfxsets = 0;

// Background data
static size_t bg_gfxset;

// Which BGMs do we need?
static int using_bgm[NUM_BGM];

// Element data
// I'd call it objects, but it'd be confused with in-game objects (and don't
// get me started on sprites, that'd outright conflict with video.h)
#define NUM_ELEMENTS 0x100
typedef struct {
   int32_t x;              // Horizontal position
   int32_t y;              // Vertical position
   int32_t xspeed;         // Horizontal speed
   int32_t yspeed;         // Vertical speed
   int32_t yaccel;         // Vertical acceleration
   uint32_t timer;         // Timer used for events
   int32_t limit;          // Limit position
   unsigned flags;         // Sprite flags
   AnimState anim;         // Animation
} Element;
static Element elements[NUM_ELEMENTS];

// Possible commands
enum {
   COM_SETANIM,            // Set element animation
   COM_UNSETANIM,          // Remove element animation
   COM_SETFLIP,            // Set element flipping
   COM_SETPOS,             // Set element position
   COM_MOVE,               // Make element move
   COM_JUMP,               // Make element jump

   COM_TEXT,               // Show text globe (normal)
   COM_AUTOTEXT,           // Show text globe (autoadvances)
   COM_SETVOICE,           // Set voice for text globes

   COM_MUSIC,              // Play music
   COM_PLAY,               // Play sound effect
   COM_PLAYEX,             // Play sound effect at position
   COM_CLUE,               // Like play but audiovideo only
   COM_CLUEEX,             // Like playex but audiovideo only
   COM_FILTER,             // Set filter

   COM_WAIT,               // Wait for all events to finish
   COM_DELAY               // Wait a specified amount of time
};

// Command list
typedef struct {
   unsigned type;          // Command type (see above)
   int id;                 // Element or sound ID
   int32_t x;              // Horizontal coordinate
   int32_t y;              // Vertical coordinate
   int32_t accel;          // Acceleration
   uint32_t timer;         // Timer value
   int32_t limit;          // Limit value
   unsigned gfxset;        // Graphics set
   char *anim;             // Animation name
} Command;
static Command *commands = NULL;
static size_t num_commands = 0;
static size_t curr_command;

// Used to make timed delays (see "delay" command)
static unsigned delay;

// Text UI positions
#define GLOBE_X (screen_cx - 144)
#define GLOBE_Y (screen_cy - 96)
#define GLOBE_TX (GLOBE_X + 8)
#define GLOBE_TY (GLOBE_Y + 8)

// Possible types of globes
enum {
   GTYPE_NONE,          // No globe
   GTYPE_NOWHERE,       // Globe without arrow
   GTYPE_LEFT,          // Globe pointing left
   GTYPE_MIDDLE,        // Globe pointing down
   GTYPE_RIGHT,         // Globe pointing right
   NUM_GTYPES
};

// Used to show the text globes
static int in_text;                 // Set when showing the globe
static unsigned text_delay;         // How long before next character
static int typing;                  // Set while still typing the text
static int text_beep;               // Used to beep every other char only
static SfxID text_voice;            // Which beep to use for globes

static char *globe_text = NULL;     // Text shown in the globe right now
static unsigned globe_pos;          // Where we are in the string above
static unsigned globe_type;         // Kind of globe (see GTYPE_*)
static char *reader_text = NULL;    // As globe_text but for screen reader

static char *script = NULL;         // Entire dialogue script
static const char *script_ptr;      // Where in the script we are right now

// Timer used for autotext
// If 0, it means autotext is disabled
// (autotext is used only in the credits)
static unsigned text_timer;

// How fast does the text get typed
// This is the gap between each character in frames
// (in other words: smaller = faster)
#define TEXT_SPEED 3

// Filter stuff
enum {
   FILTER_NONE,
   FILTER_GRAYSCALE
};
static unsigned filter;

// Cutscene-specific graphics
static GraphicsSet *gfxset = NULL;
static Sprite *spr_globe[NUM_GTYPES];

// Private function prototypes
static void load_cutscene(void);
static void parse_cutscene(void);
static void load_cutscene_script(void);
static Command *new_command(unsigned);
static size_t add_gfxset_to_list(const char *);

//***************************************************************************
// init_cutscene
// Initializes a cutscene. Loads resources as needed and such.
//***************************************************************************

void init_cutscene(void) {
   // Update savegame
   set_savegame_scene(get_curr_scene());

   // Reset elements
   for (size_t i = 0; i < NUM_ELEMENTS; i++) {
      elements[i].x = 0;
      elements[i].y = 0;
      elements[i].xspeed = 0;
      elements[i].yspeed = 0;
      elements[i].yaccel = 0;
      elements[i].timer = 0;
      elements[i].limit = 0;
      elements[i].flags = 0;
      elements[i].anim.frame = NULL;
   }

   // Reset commands
   commands = NULL;
   num_commands = 0;
   curr_command = 0;

   // Reset delay
   delay = 0;

   // Not showing text
   in_text = 0;
   globe_text = NULL;
   reader_text = NULL;
   text_voice = SFX_SPEAKROBOT;

   // Reset filter
   filter = FILTER_NONE;

   // Load assets
   loading_screen(load_cutscene);

   // Reset script
   script_ptr = script;

   // Make screen visible
   fade_on();
}


//***************************************************************************
// load_cutscene [internal]
// Loads the assets for a cutscene. Run during the loading screen.
//***************************************************************************

static void load_cutscene(void) {
   // We don't have a background by default
   bg_gfxset = SIZE_MAX;

   // Nor background music...
   for (int i = 0; i < NUM_BGM; i++)
      using_bgm[i] = 0;

   // Reset loading bar counters
   // The initial load total accounts for loading the script and the graphics
   // common to all cutscenes (i.e. the globe)
   load_count = 0;
   load_total = 2;

   // Parse the cutscene
   parse_cutscene();
   set_loading_total(load_count, load_total);

   // Load the common graphics
   gfxset = load_graphics_set("graphics/cutscenes");
#define SPR(name) get_sprite(gfxset, name)
   spr_globe[GTYPE_NONE] = NULL;
   spr_globe[GTYPE_NOWHERE] = SPR("globe_nowhere");
   spr_globe[GTYPE_LEFT] = SPR("globe_left");
   spr_globe[GTYPE_MIDDLE] = SPR("globe_middle");
   spr_globe[GTYPE_RIGHT] = SPR("globe_right");
#undef SPR
   load_count++;
   set_loading_total(load_count, load_total);

   // Load script
   load_cutscene_script();
   load_count++;
   set_loading_total(load_count, load_total);

   // Load all graphics sets
   for (size_t i = 0; i < num_gfxsets; i++) {
      // Load graphics set
      char buffer[0x80];
      sprintf(buffer, "graphics/%s", gfxset_list[i].name);
      gfxset_list[i].gfxset = load_graphics_set(buffer);

      // Update progress bar
      load_count++;
      set_loading_total(load_count, load_total);
   }

   // Load background
   if (bg_gfxset != SIZE_MAX) {
      char buffer[0x80];
      sprintf(buffer, "graphics/%s/background", gfxset_list[bg_gfxset].name);
      load_background(buffer, gfxset_list[bg_gfxset].gfxset);
      load_count++;
      set_loading_total(load_count, load_total);
   }

   // Load music
   for (int i = 0; i < NUM_BGM; i++) {
      if (using_bgm[i]) {
         load_bgm(i);
         load_count++;
         set_loading_total(load_count, load_total);
      }
   }
}

//***************************************************************************
// parse_cutscene [internal]
// Parses the cutscene file to get the cutscene data.
//***************************************************************************

static void parse_cutscene(void) {
   // Open the file
   char filename[0x80];
   sprintf(filename, "cutscenes/%s", get_scene_id());
   File *file = open_file(filename, FILE_READ);
   if (file == NULL)
      abort_program(ERR_LOADCUTSCENE, filename);

   // Used to store the filename:line string
   char fileline[0xC0];

   // To keep track of the current line
   unsigned curr_line = 0;

   // Go through all lines
   while (!end_of_file(file)) {
      // Get filename:line for this line
      curr_line++;
      sprintf(fileline, "%s:%u", filename, curr_line);

      // Get parameters for this line
      char *line = read_line(file);
      if (line == NULL)
         abort_program(ERR_LOADCUTSCENE, filename);
      Args *args = parse_args(line);
      if (args == NULL)
         abort_program(ERR_LOADCUTSCENE, filename);
      free(line);

      // Empty line?
      if (args->count == 0) {
         free_args(args);
         continue;
      }

      // Set background?
      if (strcmp(args->list[0], "background") == 0) {
         // Check arguments
         if (args->count != 2)
            abort_program(ERR_NUMARGS, fileline);
         if (!is_valid_id(args->list[1]))
            abort_program(ERR_GFXSETID, fileline);

         // Get graphics set
         bg_gfxset = add_gfxset_to_list(args->list[1]);

         // We need to load the background later
         load_total++;
      }

      // Set element animation?
      else if (strcmp(args->list[0], "setanim") == 0) {
         // Check arguments
         if (args->count != 4)
            abort_program(ERR_NUMARGS, fileline);
         if (!is_uinteger(args->list[1]))
            abort_program(ERR_ELEMENTID, fileline);
         if (!is_valid_id(args->list[2]))
            abort_program(ERR_GFXSETID, fileline);
         if (!is_valid_id(args->list[3]))
            abort_program(ERR_ANIMID, fileline);

         // Create command
         Command *cmd = new_command(COM_SETANIM);

         // Get ID of element which is affected
         cmd->id = atoi(args->list[1]);
         if (cmd->id >= NUM_ELEMENTS)
            abort_program(ERR_ELEMENTID, fileline);

         // Get graphics set
         cmd->gfxset = add_gfxset_to_list(args->list[2]);

         // Get animation name
         // I'd just fetch the animation itself if it wasn't because the
         // graphics set isn't loaded yet...
         cmd->anim = (char *) malloc(strlen(args->list[3]) + 1);
         if (cmd->anim == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         strcpy(cmd->anim, args->list[3]);
      }

      // Remove element animation?
      else if (strcmp(args->list[0], "unsetanim") == 0) {
         // Check arguments
         if (args->count != 2)
            abort_program(ERR_NUMARGS, fileline);
         if (!is_uinteger(args->list[1]))
            abort_program(ERR_ELEMENTID, fileline);

         // Create command
         Command *cmd = new_command(COM_UNSETANIM);

         // Get ID of element which is affected
         cmd->id = atoi(args->list[1]);
         if (cmd->id >= NUM_ELEMENTS)
            abort_program(ERR_ELEMENTID, fileline);
      }

      // Set element flipping?
      else if (strcmp(args->list[0], "setflip") == 0) {
         // Check arguments
         if (args->count != 3)
            abort_program(ERR_NUMARGS, fileline);
         if (!is_uinteger(args->list[1]))
            abort_program(ERR_ELEMENTID, fileline);

         // Create command
         Command *cmd = new_command(COM_SETFLIP);

         // Get ID of element which is affected
         cmd->id = atoi(args->list[1]);
         if (cmd->id >= NUM_ELEMENTS)
            abort_program(ERR_ELEMENTID, fileline);

         // Get flipping
         // To-do: make better error code
         if (strcmp(args->list[2], "none") == 0)
            cmd->x = SPR_NOFLIP;
         else if (strcmp(args->list[2], "hflip") == 0)
            cmd->x = SPR_HFLIP;
         else if (strcmp(args->list[2], "vflip") == 0)
            cmd->x = SPR_VFLIP;
         else if (strcmp(args->list[2], "hvflip") == 0)
            cmd->x = SPR_HVFLIP;
         else
            abort_program(ERR_UNKNOWN, fileline);
      }

      // Set element position?
      else if (strcmp(args->list[0], "setpos") == 0) {
         // Check arguments
         if (args->count != 4)
            abort_program(ERR_NUMARGS, fileline);
         if (!is_uinteger(args->list[1]))
            abort_program(ERR_ELEMENTID, fileline);
         if (!is_integer(args->list[2]))
            abort_program(ERR_COORDVAL, fileline);
         if (!is_integer(args->list[3]))
            abort_program(ERR_COORDVAL, fileline);

         // Create command
         Command *cmd = new_command(COM_SETPOS);

         // Get ID of element which is affected
         cmd->id = atoi(args->list[1]);
         if (cmd->id >= NUM_ELEMENTS)
            abort_program(ERR_ELEMENTID, fileline);

         // Get new coordinates
         cmd->x = atoi(args->list[2]);
         cmd->y = atoi(args->list[3]);
      }

      // Make element move?
      else if (strcmp(args->list[0], "move") == 0) {
         // Check arguments
         if (args->count != 5)
            abort_program(ERR_NUMARGS, fileline);
         if (!is_uinteger(args->list[1]))
            abort_program(ERR_ELEMENTID, fileline);
         if (!is_integer(args->list[2]))
            abort_program(ERR_SPEEDVAL, fileline);
         if (!is_integer(args->list[3]))
            abort_program(ERR_SPEEDVAL, fileline);
         if (!is_uinteger(args->list[4]))
            abort_program(ERR_TIMERVAL, fileline);

         // Create command
         Command *cmd = new_command(COM_MOVE);

         // Get ID of element which is affected
         cmd->id = atoi(args->list[1]);
         if (cmd->id >= NUM_ELEMENTS)
            abort_program(ERR_ELEMENTID, fileline);

         // Get new speeds
         cmd->x = atoi(args->list[2]);
         cmd->y = atoi(args->list[3]);

         // Get duration
         cmd->timer = atoi(args->list[4]);
         if (cmd->timer <= 0)
            abort_program(ERR_TIMERVAL, fileline);
      }

      // Make element jump?
      else if (strcmp(args->list[0], "jump") == 0) {
         // Check arguments
         if (args->count != 6)
            abort_program(ERR_NUMARGS, fileline);
         if (!is_uinteger(args->list[1]))
            abort_program(ERR_ELEMENTID, fileline);
         if (!is_integer(args->list[2]))
            abort_program(ERR_SPEEDVAL, fileline);
         if (!is_integer(args->list[3]))
            abort_program(ERR_SPEEDVAL, fileline);
         if (!is_uinteger(args->list[4]))
            abort_program(ERR_SPEEDVAL, fileline);
         if (!is_integer(args->list[5]))
            abort_program(ERR_COORDVAL, fileline);

         // Create command
         Command *cmd = new_command(COM_JUMP);

         // Get ID of element which is affected
         cmd->id = atoi(args->list[1]);
         if (cmd->id >= NUM_ELEMENTS)
            abort_program(ERR_ELEMENTID, fileline);

         // Get initial speed
         cmd->x = atoi(args->list[2]);
         cmd->y = atoi(args->list[3]);

         // Get falling acceleration
         cmd->accel = atoi(args->list[4]);
         if (cmd->accel <= 0)
            abort_program(ERR_SPEEDVAL, fileline);

         // Get bottom limit at which the element can fall (i.e. floor)
         cmd->limit = atoi(args->list[5]);
      }

      // Show text globe?
      else if (strcmp(args->list[0], "text") == 0 ||
      strcmp(args->list[0], "autotext") == 0) {
         // Check arguments
         if (args->count != 2)
            abort_program(ERR_NUMARGS, fileline);

         // Create command
         Command *cmd = new_command(args->list[0][0] == 'a' ?
            COM_AUTOTEXT : COM_TEXT);

         // Get globe type
         // To-do: handle error properly
         if (strcmp(args->list[1], "noglobe") == 0)
            cmd->id = GTYPE_NONE;
         else if (strcmp(args->list[1], "nowhere") == 0)
            cmd->id = GTYPE_NOWHERE;
         else if (strcmp(args->list[1], "left") == 0)
            cmd->id = GTYPE_LEFT;
         else if (strcmp(args->list[1], "middle") == 0)
            cmd->id = GTYPE_MIDDLE;
         else if (strcmp(args->list[1], "right") == 0)
            cmd->id = GTYPE_RIGHT;
         else
            abort_program(ERR_UNKNOWN, fileline);
      }

      // Set voice used by text globes?
      else if (strcmp(args->list[0], "voice") == 0) {
         // Check arguments
         if (args->count != 2)
            abort_program(ERR_NUMARGS, fileline);

         // Create command
         Command *cmd = new_command(COM_SETVOICE);

         // Get voice ID
         if (strcmp(args->list[1], "robot") == 0)
            cmd->id = SFX_SPEAKROBOT;
         else if (strcmp(args->list[1], "sol") == 0)
            cmd->id = SFX_SPEAKSOL;
         else if (strcmp(args->list[1], "ruby") == 0)
            cmd->id = SFX_SPEAKRUBY;
         else if (strcmp(args->list[1], "evil") == 0)
            cmd->id = SFX_SPEAKEVIL;
         else
            abort_program(ERR_VOICEID, fileline);
      }

      // Set music?
      else if (strcmp(args->list[0], "music") == 0) {
         // Check arguments
         if (args->count != 2)
            abort_program(ERR_NUMARGS, fileline);

         // Create command
         Command *cmd = new_command(COM_MUSIC);

         // Get BGM ID
         cmd->id = get_bgm_by_name(args->list[1]);
         if (cmd->id == BGM_NONE)
            abort_program(ERR_BGMID, fileline);

         // Check if we need to load this BGM yet
         if (!using_bgm[cmd->id]) {
            using_bgm[cmd->id] = 1;
            load_total++;
         }
      }

      // Play sound effect?
      else if (strcmp(args->list[0], "play") == 0 ||
      strcmp(args->list[0], "clue") == 0) {
         // Check arguments
         if (args->count != 2)
            abort_program(ERR_NUMARGS, fileline);

         // Create command
         Command *cmd = new_command(args->list[0][0] == 'p' ?
            COM_PLAY : COM_CLUE);

         // Get sound ID
         cmd->id = get_sfx_by_name(args->list[1]);
         if (cmd->id == SFX_NONE)
            abort_program(ERR_SFXID, fileline);
      }

      // Play sound effect with position?
      else if (strcmp(args->list[0], "playex") == 0 ||
      strcmp(args->list[0], "clueex") == 0) {
         // Check arguments
         if (args->count != 4)
            abort_program(ERR_NUMARGS, fileline);
         if (!is_integer(args->list[2]))
            abort_program(ERR_COORDVAL, fileline);
         if (!is_integer(args->list[3]))
            abort_program(ERR_COORDVAL, fileline);

         // Create command
         Command *cmd = new_command(args->list[0][0] == 'p' ?
            COM_PLAYEX : COM_CLUEEX);

         // Get sound ID
         cmd->id = get_sfx_by_name(args->list[1]);
         if (cmd->id == SFX_NONE)
            abort_program(ERR_SFXID, fileline);
         cmd->x = atoi(args->list[2]);
         cmd->y = atoi(args->list[2]);
      }

      // Set filter type?
      else if (strcmp(args->list[0], "filter") == 0) {
         // Check arguments
         if (args->count != 2)
            abort_program(ERR_NUMARGS, fileline);

         // Create command
         Command *cmd = new_command(COM_FILTER);

         // Get filter type
         if (strcmp(args->list[1], "none") == 0)
            cmd->id = FILTER_NONE;
         else if (strcmp(args->list[1], "grayscale") == 0 ||
         strcmp(args->list[1], "greyscale") == 0)
            cmd->id = FILTER_GRAYSCALE;
         else
            abort_program(ERR_VOICEID, fileline);
      }

      // Wait for events to finish?
      else if (strcmp(args->list[0], "wait") == 0) {
         // Check arguments
         if (args->count != 1)
            abort_program(ERR_NUMARGS, fileline);

         // Create command
         new_command(COM_WAIT);
      }

      // Wait a specified amount of time?
      else if (strcmp(args->list[0], "delay") == 0) {
         // Check arguments
         if (args->count != 2)
            abort_program(ERR_NUMARGS, fileline);
         if (!is_uinteger(args->list[1]))
            abort_program(ERR_TIMERVAL, fileline);

         // Create command
         Command *cmd = new_command(COM_DELAY);

         // Store delay
         cmd->timer = atoi(args->list[1]);
         if (cmd->timer <= 0)
            abort_program(ERR_TIMERVAL, fileline);
      }

      // Invalid command?
      else {
         // To-do: describe what's wrong?
         abort_program(ERR_BADCOMMAND, fileline);
      }

      // Go for next line
      free_args(args);
   }

   // Done
   close_file(file);

   // We loaded something! :P
   load_count++;
   load_total++;
}

//***************************************************************************
// load_cutscene_script [internal]
// Loads the script dialogue for this cutscene.
//***************************************************************************

static void load_cutscene_script(void) {
   // Open the file
   char filename[0x80];
   sprintf(filename, "cutscenes/%s.%s", get_scene_id(), get_language_id());
   File *file = open_file(filename, FILE_READ);
   if (file == NULL)
      return;

   // Allocate initial buffer
   // The byte allocated initially is to make room for the terminating null
   // character once we're done loading the string
   size_t size = 1;
   size_t pos = 0;
   script = (char *) malloc(size);
   if (script == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Keep reading the file
   while (!end_of_file(file)) {
      // Get next line
      char *line = read_line(file);
      if (line == NULL) break;

      // End of screen?
      if (strcmp(line, "--") == 0) {
         // Make room for it
         size++;
         script = (char *) realloc(script, size);
         if (script == NULL)
            abort_program(ERR_NOMEMORY, NULL);

         // Add end of screen marker
         script[pos] = '\xFF';
         pos++;
      }

      // Normal line of text?
      else {
         // Get length of string
         size_t len = strlen(line);

         // Make room for it
         size += len + 1;
         script = (char *) realloc(script, size);
         if (script == NULL)
            abort_program(ERR_NOMEMORY, NULL);

         // Store line
         // To-do: avoid spurious newlines? (they don't have any effect on
         // the displayed outcome, but they will play an extra beep)
         strcpy(&script[pos], line);
         pos += len;
         script[pos] = '\n';
         pos++;
      }

      // Done with the line
      free(line);
   }

   // Terminate string
   script[pos] = '\0';

   // Done
   close_file(file);
}

//***************************************************************************
// new_command [internal]
// Allocates room for a new command.
//---------------------------------------------------------------------------
// param type: command type
// return: pointer to new command
//***************************************************************************

static Command *new_command(unsigned type) {
   // Allocate room for new command
   num_commands++;
   commands = (Command *) realloc(commands, sizeof(Command) * num_commands);
   if (commands == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   Command *ptr = &commands[num_commands-1];

   // Clear command
   ptr->type = type;
   ptr->id = 0;
   ptr->x = 0;
   ptr->y = 0;
   ptr->accel = 0;
   ptr->timer = 0;
   ptr->limit = 0;
   ptr->gfxset = 0;
   ptr->anim = NULL;

   // Ready!
   return ptr;
}

//***************************************************************************
// add_to_gfxset_list [internal]
// Adds a graphics set to the list to load.
//---------------------------------------------------------------------------
// param name: name of graphics set
//***************************************************************************

static size_t add_gfxset_to_list(const char *name) {
   // Check if it was already requested
   for (size_t i = 0; i < num_gfxsets; i++) {
      if (strcmp(gfxset_list[i].name, name) == 0)
         return i;
   }

   // Allocate room in the list for it
   num_gfxsets++;
   gfxset_list = (GfxSetList *) realloc(gfxset_list,
      sizeof(GfxSetList) * num_gfxsets);
   if (gfxset_list == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Fill in data
   gfxset_list[num_gfxsets-1].name = (char *) malloc(strlen(name) + 1);
   if (gfxset_list[num_gfxsets-1].name == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(gfxset_list[num_gfxsets-1].name, name);
   gfxset_list[num_gfxsets-1].gfxset = NULL;

   // More stuff to load...
   load_total++;

   // Return ID of this graphics set in the list
   return num_gfxsets-1;
}

//***************************************************************************
// run_cutscene
// Processes a logic frame of a cutscene
//***************************************************************************

void run_cutscene(void) {
   // Update background
   update_background();

   // Process elements
   int during_event = 0;
   for (size_t i = 0; i < NUM_ELEMENTS; i++) {
      // Event in progress?
      if (elements[i].timer > 0 || elements[i].yaccel) {
         // Mark in-event flag
         during_event = 1;

         // Move object as needed
         elements[i].x += elements[i].xspeed;
         elements[i].y += elements[i].yspeed;
         elements[i].yspeed += elements[i].yaccel;

         // Stop falling?
         if (elements[i].yaccel > 0 && elements[i].yspeed >= 0 &&
         elements[i].y >= elements[i].limit) {
            elements[i].y = elements[i].limit;
            elements[i].xspeed = 0;
            elements[i].yspeed = 0;
            elements[i].yaccel = 0;
         }

         // Reduce event timer
         if (elements[i].timer > 0)
            elements[i].timer--;
      }

      // Animate sprite
      update_anim(&elements[i].anim);
   }

   // Processing text?
   if (in_text) {
      // Type characters
      if (typing) {
         text_delay--;
         if (text_delay == 0) {
            // Copy next character to the displayed string
            // Oh god this piece of code is a mess...
            size_t count = get_utf8_charlen(script_ptr);
            globe_text = (char *) realloc(globe_text, globe_pos + count + 1);
            if (globe_text == NULL)
               abort_program(ERR_NOMEMORY, NULL);
            memcpy(&globe_text[globe_pos], script_ptr, count);
            script_ptr += count;
            globe_pos += count;
            globe_text[globe_pos] = '\0';

            // End of script?
            if (*script_ptr == '\0' || *script_ptr == '\xFF')
               typing = 0;

            // Make noise, unless the globe isn't shown
            // The toggling of text_beep is used to do a beep only every
            // other character (otherwise it's too fast and it gets really
            // annoying, especially with the SFX getting cut off)
            if (globe_type != GTYPE_NONE && !settings.reader) {
               if (text_beep) play_sfx(text_voice);
               text_beep = !text_beep;
            }

            // Wait for next character
            text_delay = TEXT_SPEED;
         }
      }

      // Autoadvance?
      else if (text_timer > 0) {
         text_timer--;
         if (text_timer == 0)
            input.menu.accept = 1;
      }

      // Skip text
      if (input.menu.accept) {
         // Skip remaining text
         while (*script_ptr != '\0' && *script_ptr != '\xFF')
            script_ptr++;

         // Hide globe
         in_text = 0;
         if (globe_text != NULL) {
            free(globe_text);
            free(reader_text);
            globe_text = NULL;
            reader_text = NULL;
         }
      }

      // Quick hack to prevent the cutscene engine to keep waiting on events
      // that may have not finished yet (can happen if the player skips the
      // text too quickly)
      if (!in_text)
         during_event = 0;
   }

   // Waiting?
   if (delay > 0) {
      delay--;
      during_event = 0;
   }

   // Process commands
   if (!during_event && !delay && !in_text) {
      while (curr_command < num_commands) {
         // To make my life easier
         const Command *cmd = &commands[curr_command];
         curr_command++;

         // Check command type
         switch (cmd->type) {
            // Set element animation
            case COM_SETANIM: {
               set_anim(&elements[cmd->id].anim,
                  get_anim(gfxset_list[cmd->gfxset].gfxset, cmd->anim));
            } break;

            // Remove element animation
            case COM_UNSETANIM: {
               elements[cmd->id].anim.frame = NULL;
            } break;

            // Set element flipping
            case COM_SETFLIP: {
               elements[cmd->id].flags = cmd->x;
            } break;

            // Set element position
            case COM_SETPOS: {
               elements[cmd->id].x = cmd->x << 8;
               elements[cmd->id].y = cmd->y << 8;
            } break;

            // Make element move
            case COM_MOVE: {
               elements[cmd->id].xspeed = cmd->x;
               elements[cmd->id].yspeed = cmd->y;
               elements[cmd->id].timer = cmd->timer;
            } break;

            // Make element jump
            case COM_JUMP: {
               elements[cmd->id].xspeed = cmd->x;
               elements[cmd->id].yspeed = cmd->y;
               elements[cmd->id].yaccel = cmd->accel;
               elements[cmd->id].limit = cmd->limit << 8;
               elements[cmd->id].timer = 0;
            } break;

            // Show text globe
            case COM_TEXT:
            case COM_AUTOTEXT: {
               // Yes, this can happen (i.e. by trying to do a text command
               // while not providing script data, the game would crash
               // without this check)
               if (script_ptr == NULL)
                  break;

               // Huh, no more script data...
               if (*script_ptr == '\0')
                  break;

               // If this isn't the first text then most likely we're at the
               // end-of-text marker from the previous one, so skip it (we
               // won't skip over 0x00 though because that means there isn't
               // any more data)
               if (*script_ptr == '\xFF')
                  script_ptr++;

               // Set text to be output to the screen reader
               size_t len = strcspn(script_ptr, "\xFF");
               reader_text = (char *) malloc(len + 1);
               if (reader_text == NULL)
                  abort_program(ERR_NOMEMORY, NULL);
               strncpy(reader_text, script_ptr, len);
               reader_text[len] = '\0';

               // Start typing
               in_text = 1;
               text_delay = 1;
               typing = 1;
               text_beep = 1;
               globe_pos = 0;
               globe_type = cmd->id;

               // Set autotext timer (if relevant)
               text_timer = (cmd->type == COM_AUTOTEXT && !settings.reader)
                  ? 180 : 0;

               // Stop processing commands here
               goto for_break;
            } break;

            // Set text globe beeping sound
            case COM_SETVOICE: {
               text_voice = cmd->id;
            } break;

            // Play music
            case COM_MUSIC: {
               play_bgm(cmd->id);
            } break;

            // Play sound effect
            case COM_PLAY: {
               play_sfx(cmd->id);
            } break;
            case COM_PLAYEX: {
               play_2d_sfx(cmd->id, cmd->x, cmd->y);
            } break;

            // Play audio clue
            case COM_CLUE: {
               if (settings.audiovideo)
                  play_sfx(cmd->id);
            } break;
            case COM_CLUEEX: {
               if (settings.audiovideo)
                  play_2d_sfx(cmd->id, cmd->x, cmd->y);
            } break;

            // Set graphics filter
            case COM_FILTER: {
               filter = cmd->id;
            } break;

            // Wait for events to finish
            case COM_WAIT: {
               // Stop processing commands here
               goto for_break;
            } break;

            // Wait a specified amount of time
            case COM_DELAY: {
               // Set amount of delay
               delay = cmd->timer;

               // Stop processing commands here
               goto for_break;
            } break;
         }
      }

      // Finished?
      input.menu.cancel = 1;
   }

   // To work around switch taking up break for itself
   for_break:

   // Done with cutscene?
   if (input.menu.cancel)
      switch_to_next_scene();
}

//***************************************************************************
// draw_ingame
// Draws everything in a cutscene
//***************************************************************************

void draw_cutscene(void) {
   // Draw background
   draw_background();

   // Draw all elements
   for (size_t i = 0; i < NUM_ELEMENTS; i++) {
      // Get animation
      const AnimFrame *anim = elements[i].anim.frame;
      if (anim == NULL)
         continue;

      // Get sprite
      const Sprite *spr = anim->sprite;
      if (spr == NULL)
         continue;

      // Get sprite flags for this frame
      unsigned flags = elements[i].flags ^ anim->flags;

      // Get horizontal coordinate of sprite
      int32_t x = elements[i].x >> 8;
      if (flags & SPR_HFLIP) {
         x += anim->x_offset;
         x -= spr->width - 1;
      } else {
         x -= anim->x_offset;
      }

      // Get vertical coordinate of sprite
      int32_t y = elements[i].y >> 8;
      if (flags & SPR_VFLIP) {
         y += anim->y_offset;
         y -= spr->width - 1;
      } else {
         y -= anim->y_offset;
      }

      // Offsets are from the screen center (had to do this to cope with
      // multiple screen ratios, this wasn't much of an issue with
      // backgrounds due to their tiling nature)
      x += screen_cx;
      y += screen_cy;

      // Draw sprite
      draw_sprite(spr, x, y, flags);
   }

   // Apply filter
   switch (filter) {
      // Make everything grayscale
      case FILTER_GRAYSCALE:
         grayscale_filter();
         break;

      // No filter or unrecognized
      default:
         break;
   }

   // Draw text globe
   if (in_text) {
      // Draw globe
      draw_sprite(spr_globe[globe_type], GLOBE_X, GLOBE_Y, SPR_NOFLIP);
      draw_text(globe_text, GLOBE_TX, GLOBE_TY, FONT_LIT, ALIGN_TOPLEFT);

      // Output to screen reader as well
      set_reader_text(reader_text);
   }
}

//***************************************************************************
// deinit_cutscene
// Deinitializes a cutscene. Unloads all the allocated resources.
//***************************************************************************

void deinit_cutscene(void) {
   // Unload music
   stop_bgm();
   unload_all_bgm();

   // Unload background
   unload_background();

   // Unload text data
   if (globe_text != NULL) {
      free(globe_text);
      globe_text = NULL;
   }
   if (reader_text != NULL) {
      free(reader_text);
      reader_text = NULL;
   }
   if (script != NULL) {
      free(script);
      script = NULL;
   }

   // Unload graphics sets
   if (gfxset_list != NULL) {
      for (size_t i = 0; i < num_gfxsets; i++) {
         if (gfxset_list[i].name != NULL)
            free(gfxset_list[i].name);
         if (gfxset_list[i].gfxset != NULL)
            destroy_graphics_set(gfxset_list[i].gfxset);
      }
      free(gfxset_list);
      gfxset_list = NULL;
      num_gfxsets = 0;
   }
   if (gfxset != NULL) {
      destroy_graphics_set(gfxset);
      gfxset = NULL;
   }

   // Unload command list
   if (commands != NULL) {
      // Scan all commands for data they may have
      for (size_t i = 0; i < num_commands; i++) {
         Command *cmd = &commands[i];
         if (cmd->anim != NULL)
            free(cmd->anim);
      }

      // Get rid of list itself
      free(commands);
      commands = NULL;
      num_commands = 0;
   }
}
