//***************************************************************************
// "enemies.c"
// Code for enemy objects.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdlib.h>
#include "ingame.h"
#include "level.h"
#include "objects.h"
#include "physics.h"
#include "settings.h"
#include "sound.h"
#include "tables.h"
#include "tally.h"
#include "video.h"

// Flamer behavior parameters
#define FLAMER_SPEED    0x100    // Speed at which the flamer moves
#define FLAMER_MOVE_E   0x20     // For how long to move in easy
#define FLAMER_MOVE_N   0x40     // For how long to move in normal
#define FLAMER_MOVE_H   0x60     // For how long to move in hard
#define FLAMER_FIREDIST 0x18     // Offset at which the fireballs appear
#define FLAMER_FIRESPD  0x180    // Speed at which the fireballs move
#define FLAMER_FIRETIME 0x40     // For how long to shoot fire
#define FLAMER_WEIGHT   0x40     // Weight if not on the ground
#define FLAMER_RANGE_X  0x40     // Maximum X distance where it triggers
#define FLAMER_RANGE_Y  0x30     // Maximum Y distance where it triggers

// Sprayer behavior parameters
#define SPRAYER_SPEED   0x100    // Speed at which the sprayer moves
#define SPRAYER_DROP    0x200    // Speed at which toxic gas falls
#define SPRAYER_WAVE    2        // How fast it waves vertically
#define SPRAYER_TIME_E  0x3F     // For how long to spray in easy
#define SPRAYER_TIME_N  0x5F     // For how long to spray in normal
#define SPRAYER_TIME_H  0x7F     // For how long to spray in hard
#define SPRAYER_LIMIT   0x40     // How far can get from the spawn position
#define SPRAYER_RANGE_X 0x28     // Maximum X distance where it triggers
#define SPRAYER_RANGE_Y 0x50     // Maximum Y distance where it triggers

// Turret behavior parameters
#define TURRET_XSPEED   0x280    // How far the mortars are shot
#define TURRET_YSPEED   0x380    // How high the mortars are shot
#define TURRET_TIME_E   0x80     // How often it shoots in easy
#define TURRET_TIME_N   0x60     // How often it shoots in normal
#define TURRET_TIME_H   0x40     // How often it shoots in hard
#define TURRET_WEIGHT   0x40     // Weight if not on the ground
#define TURRET_RANGE_X  0x70     // Maximum X distance where it triggers
#define TURRET_RANGE_Y  0x70     // Maximum Y distance where it triggers

// Roller behavior parameters
#define ROLLER_ACCEL_E  0x0C     // How fast the roller accelerates (easy)
#define ROLLER_ACCEL_N  0x10     // How fast the roller accelerates (normal)
#define ROLLER_ACCEL_H  0x18     // How fast the roller accelerates (hard)
#define ROLLER_BOUNCE   0x380    // How much it bounces when hitting a wall
#define ROLLER_WEIGHT   0x40     // How fast the roller falls

// Grabber behavior parameters
#define GRABBER_SPEED   0x100    // Speed at which the grabber moves
#define GRABBER_WAVE    2        // How fast it waves vertically
#define GRABBER_LIMIT   0x40     // How far can get from the spawn position
#define GRABBER_RANGE_X 0x10     // How near the player must be to grab (X)
#define GRABBER_RANGE_Y 0x70     // How near the player must be to grab (Y)
#define GRABBER_MAX_Y   0x40     // How far down can it try to grab
#define GRABBER_FALL_E  0x200    // Falling speed in easy
#define GRABBER_FALL_N  0x300    // Falling speed in normal
#define GRABBER_FALL_H  0x400    // Falling speed in hard
#define GRABBER_RISE_E  0x180    // Rising speed in easy
#define GRABBER_RISE_N  0x240    // Rising speed in normal
#define GRABBER_RISE_H  0x300    // Rising speed in hard

// Spider behavior parameters
#define SPIDER_SPEED_E  0x100    // How fast it moves in easy
#define SPIDER_SPEED_N  0x180    // How fast it moves in normal
#define SPIDER_SPEED_H  0x200    // How fast it moves in hard
#define SPIDER_WAIT_E   0x40     // How much to wait before turning (easy)
#define SPIDER_WAIT_N   0x30     // How much to wait before turning (normal)
#define SPIDER_WAIT_H   0x20     // How much to wait before turning (hard)
#define SPIDER_WEIGHT   0x40     // Weight if not on a wall
#define SPIDER_RANGE_X  0x40     // Maximum X distance where it triggers
#define SPIDER_RANGE_Y  0x60     // Maximum Y distance where it triggers

// Heater behavior parameters
#define HEATER_FIRESPD  0x200    // Speed at which fire rises
#define HEATER_WAVE     3        // How much the fire waves to the sides
#define HEATER_MARGIN   2        // How much the fire is off to the sides
#define HEATER_TIME_E   0x30     // For how long to fire in easy
#define HEATER_TIME_N   0x40     // For how long to fire in normal
#define HEATER_TIME_H   0x50     // For how long to fire in hard
#define HEATER_WEIGHT   0x40     // Weight if not on the ground
#define HEATER_RANGE_X  0x40     // Maximum X distance where it triggers
#define HEATER_RANGE_Y  0x40     // Maximum Y distance where it triggers

// Bomb behavior parameters
#define BOMB_SPEED      0x80     // How fast it walks
#define BOMB_RANGE_X    0x40     // Maximum X distance where it triggers
#define BOMB_RANGE_Y    0x40     // Maximum Y distance where it triggers
#define BOMB_MORTAR_X1  0x90     // X speed for the inner mortars
#define BOMB_MORTAR_Y1  0x3C0    // Y speed for the inner mortars
#define BOMB_MORTAR_X2  0x180    // X speed for the middle mortars
#define BOMB_MORTAR_Y2  0x300    // Y speed for the middle mortars
#define BOMB_MORTAR_X3  0x240    // X speed for the outer mortars
#define BOMB_MORTAR_Y3  0x200    // Y speed for the outer mortars
#define BOMB_WEIGHT     0x20     // Weight if not on the ground

// Speed of the scrap when killing an enemy
#define SCRAP_SPEED_X1  0xF0     // X speed for the upper scrap
#define SCRAP_SPEED_Y1  0x3C0    // Y speed for the upper scrap
#define SCRAP_SPEED_X2  0x240    // X speed for the middle scrap
#define SCRAP_SPEED_Y2  0x220    // Y speed for the middle scrap
#define SCRAP_SPEED_X3  0xF0     // X speed for the lower scrap
#define SCRAP_SPEED_Y3  0X40     // Y speed for the lower scrap

// Possible animations for an enemy
typedef enum {
   EN_ANIM_FLAMERMOVEDIM,  // Flamer: moving (dim)
   EN_ANIM_FLAMERMOVELIT,  // Flamer: moving (lit)
   EN_ANIM_FLAMERIDLE,     // Flamer: idle
   EN_ANIM_FLAMERTRIGGER,  // Flamer: trigger

   EN_ANIM_SPRAYERDIM,     // Sprayer: dim
   EN_ANIM_SPRAYERLIT,     // Sprayer: lit

   EN_ANIM_TURRETLIT,      // Turret: lit
   EN_ANIM_TURRETSHOOT,    // Turret: shoot

   EN_ANIM_ROLLERIDLE,     // Roller: idle
   EN_ANIM_ROLLERSLOW,     // Roller: rolling (slow)
   EN_ANIM_ROLLERMED,      // Roller: rolling (medium)
   EN_ANIM_ROLLERFAST,     // Roller: rolling (fast)

   EN_ANIM_GRABBERDIM,     // Grabber: dim
   EN_ANIM_GRABBERLIT,     // Grabber: lit
   EN_ANIM_GRABBERGRAB,    // Grabber: grabbing

   EN_ANIM_SPIDERIDLE_U,   // Spider: idle (up)
   EN_ANIM_SPIDERSLOW_U,   // Spider: slow (up)
   EN_ANIM_SPIDERMED_U,    // Spider: medium (up)
   EN_ANIM_SPIDERFAST_U,   // Spider: fast (up)
   EN_ANIM_SPIDERIDLE_D,   // Spider: idle (down)
   EN_ANIM_SPIDERSLOW_D,   // Spider: slow (down)
   EN_ANIM_SPIDERMED_D,    // Spider: medium (down)
   EN_ANIM_SPIDERFAST_D,   // Spider: fast (down)

   EN_ANIM_HEATERIDLE,     // Heater: idle
   EN_ANIM_HEATERLIT,      // Heater: lit
   EN_ANIM_HEATERFIRE,     // Heater: firing

   EN_ANIM_BOMBWALK,       // Bomb: walk
   EN_ANIM_BOMBLIT,        // Bomb: lit

   NUM_EN_ANIM             // Number of animations
} EnemyAnim;

// How many enemies are in the level?
unsigned num_enemies = 0;

// Where enemy graphics are stored
static GraphicsSet *gfxset_enemy = NULL;
static AnimFrame *anim_enemy[NUM_EN_ANIM];

//***************************************************************************
// load_enemies
// Loads the enemies assets.
//***************************************************************************

void load_enemies(void) {
   // Load graphics
   gfxset_enemy = load_graphics_set("graphics/enemies");

   // Get a list of all the animations we need
   // To-do: replace with a loop
#define ANIM(x) get_anim(gfxset_enemy, x)
   anim_enemy[EN_ANIM_FLAMERMOVEDIM] = ANIM("flamer_move_dim");
   anim_enemy[EN_ANIM_FLAMERMOVELIT] = ANIM("flamer_move_lit");
   anim_enemy[EN_ANIM_FLAMERIDLE] = ANIM("flamer_idle");
   anim_enemy[EN_ANIM_FLAMERTRIGGER] = ANIM("flamer_trigger");
   anim_enemy[EN_ANIM_SPRAYERDIM] = ANIM("sprayer_dim");
   anim_enemy[EN_ANIM_SPRAYERLIT] = ANIM("sprayer_lit");
   anim_enemy[EN_ANIM_TURRETLIT] = ANIM("turret_lit");
   anim_enemy[EN_ANIM_TURRETSHOOT] = ANIM("turret_shoot");
   anim_enemy[EN_ANIM_ROLLERIDLE] = ANIM("roller_idle");
   anim_enemy[EN_ANIM_ROLLERSLOW] = ANIM("roller_slow");
   anim_enemy[EN_ANIM_ROLLERMED] = ANIM("roller_medium");
   anim_enemy[EN_ANIM_ROLLERFAST] = ANIM("roller_fast");
   anim_enemy[EN_ANIM_GRABBERDIM] = ANIM("grabber_dim");
   anim_enemy[EN_ANIM_GRABBERLIT] = ANIM("grabber_lit");
   anim_enemy[EN_ANIM_GRABBERGRAB] = ANIM("grabber_grab");
   anim_enemy[EN_ANIM_SPIDERIDLE_U] = ANIM("spider_idle_up");
   anim_enemy[EN_ANIM_SPIDERSLOW_U] = ANIM("spider_slow_up");
   anim_enemy[EN_ANIM_SPIDERMED_U] = ANIM("spider_medium_up");
   anim_enemy[EN_ANIM_SPIDERFAST_U] = ANIM("spider_fast_up");
   anim_enemy[EN_ANIM_SPIDERIDLE_D] = ANIM("spider_idle_down");
   anim_enemy[EN_ANIM_SPIDERSLOW_D] = ANIM("spider_slow_down");
   anim_enemy[EN_ANIM_SPIDERMED_D] = ANIM("spider_medium_down");
   anim_enemy[EN_ANIM_SPIDERFAST_D] = ANIM("spider_fast_down");
   anim_enemy[EN_ANIM_HEATERIDLE] = ANIM("heater_idle");
   anim_enemy[EN_ANIM_HEATERLIT] = ANIM("heater_lit");
   anim_enemy[EN_ANIM_HEATERFIRE] = ANIM("heater_fire");
   anim_enemy[EN_ANIM_BOMBWALK] = ANIM("bomb_walk");
   anim_enemy[EN_ANIM_BOMBLIT] = ANIM("bomb_lit");
#undef ANIM
}

//***************************************************************************
// unload_enemies
// Frees up the resources taken up by enemies assets.
//***************************************************************************

void unload_enemies(void) {
   // Unload graphics
   if (gfxset_enemy) {
      destroy_graphics_set(gfxset_enemy);
      gfxset_enemy = NULL;
   }
}

//***************************************************************************
// init_flamer
// Initializes a flamer enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_flamer(Object *obj) {
   // Set the hitbox
   set_hitbox(obj, -13, 13, -13, 15);

   // Reset timer
   obj->timer = 0x7F;
}

//***************************************************************************
// run_flamer
// Game logic for a flamer enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//---------------------------------------------------------------------------
// States:
// * Beeping: timer is between 0x00 and 0x1F
// * Shooting: timer is between 0x20 and 0x5F
// * Moving: timer is between 0x20 and 0x7F
//   - In easy, it's moving between 0x60 and 0x7F
//   - In normal, it's moving between 0x40 and 0x7F
//   - In hard, it's moving between 0x20 and 0x7F
//---------------------------------------------------------------------------
// Changes made by difficulty: how much to idle/move
//***************************************************************************

void run_flamer(Object *obj) {
   // Are we too far from the screen to bother?
   if (is_too_far(obj))
      return;

   // Set for how long we idle based on difficulty
   // The harder the game, the less we idle
   uint16_t move_time;
   switch (get_difficulty()) {
      case DIFF_EASY: move_time = 0x80 - FLAMER_MOVE_E; break;
      case DIFF_HARD: move_time = 0x80 - FLAMER_MOVE_H; break;
      default:        move_time = 0x80 - FLAMER_MOVE_N; break;
   }

   // Update timer
   // Our logic is almost entirely timing-based
   if (obj->timer < 0x7F || settings.flamer_ai == 2) {
      obj->timer++;
      obj->timer &= 0x7F;
   }

   // See if there's any player near enough, and get lit if so
   if (obj->timer == 0x7F && settings.flamer_ai == 1) {
      for (Object *player = get_first_object(OBJGROUP_PLAYER);
      player != NULL; player = player->next) {
         if (!obj->dir && (player->x < obj->x ||
         player->x - obj->x >= FLAMER_RANGE_X))
            continue;
         if (obj->dir && (player->x > obj->x ||
         obj->x - player->x >= FLAMER_RANGE_X))
            continue;
         if (abs(player->y - obj->y) >= FLAMER_RANGE_Y)
            continue;
         obj->timer = 0;
         break;
      }
   }

   // Start moving or stay idle?
   if (obj->timer >= move_time) {
      obj->speed = obj->dir ? -FLAMER_SPEED : FLAMER_SPEED;
   } else
      obj->speed = 0;

   // Move around
   obj->gravity += FLAMER_WEIGHT;
   apply_physics(obj);
   if (obj->timer >= move_time && (obj->on_wall || obj->on_cliff))
      obj->dir = ~obj->dir;

   // Spawn fireballs after our warning was given
   if (obj->timer >= 0x20 && obj->timer < 0x20 + FLAMER_FIRETIME &&
   !(obj->timer & 0x07)) {
      // Create fireball object
      Object *ptr = add_object(OBJ_FIREBALL, obj->x +
         (obj->dir ? -FLAMER_FIREDIST : FLAMER_FIREDIST), obj->y, 0);
      ptr->speed = obj->dir ? -FLAMER_FIRESPD : FLAMER_FIRESPD;
      ptr->speed += obj->speed;
   }

   // Set object animation depending on what is it doing
   if (obj->timer < 0x20)
      set_object_anim(obj, anim_enemy[EN_ANIM_FLAMERTRIGGER]);
   else if (obj->timer < 0x20 + FLAMER_FIRETIME)
      set_object_anim(obj, anim_enemy[EN_ANIM_FLAMERMOVELIT]);
   else if (obj->timer >= move_time)
      set_object_anim(obj, anim_enemy[EN_ANIM_FLAMERMOVEDIM]);
   else
      set_object_anim(obj, anim_enemy[EN_ANIM_FLAMERIDLE]);
}

//***************************************************************************
// init_sprayer
// Initializes a sprayer enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_sprayer(Object *obj) {
   // Set the hitbox
   set_hitbox(obj, -13, 13, -13, 13);

   // Reset timer
   obj->timer = 0x7F;
}

//***************************************************************************
// run_sprayer
// Game logic for a sprayer enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//---------------------------------------------------------------------------
// States:
// * Beeping: timer is between 0x00 and 0x1F
// * Spraying: timer is between 0x20 and 0x7F
//   - In easy, toxic gas is sprayed between 0x20 and 0x3F
//   - In normal, toxic gas is sprayed between 0x20 and 0x5F
//   - In hard, toxic gas is sprayed between 0x20 and 0x7F
//---------------------------------------------------------------------------
// Changes made by difficulty: how much to spray each time
//***************************************************************************

void run_sprayer(Object *obj) {
   // Are we too far from the screen to bother?
   if (is_too_far(obj))
      return;

   // Make our vertical position unstable, so it looks like we're floating
   // or something like that. Using gravity here so collision can be applied
   // and the enemy can't get stuck into the floor or ceiling.
   obj->gravity = (obj->base_y + (sines[game_anim << SPRAYER_WAVE & 0xFF]
      >> 6) - obj->y) << 8;

   // Move around
   obj->speed = obj->dir ? -SPRAYER_SPEED : SPRAYER_SPEED;
   apply_physics(obj);
   if (obj->on_wall)
      obj->dir = ~obj->dir;
   if (obj->x <= obj->base_x - SPRAYER_LIMIT)
      obj->dir = 0;
   if (obj->x >= obj->base_x + SPRAYER_LIMIT)
      obj->dir = 1;
   if (obj->on_ground)
      obj->base_y -= speed_to_int(obj->gravity);

   // See if there's any player near enough, and get lit if so
   if (obj->timer == 0x7F && settings.sprayer_ai == 1) {
      for (Object *player = get_first_object(OBJGROUP_PLAYER);
      player != NULL; player = player->next) {
         if (abs(player->x - obj->x) >= SPRAYER_RANGE_X)
            continue;
         if (player->y < obj->y)
            continue;
         if (player->y - obj->y >= SPRAYER_RANGE_Y)
            continue;
         obj->timer = 0;
         break;
      }
   }

   // Determine for how long to keep spraying based on difficulty
   uint16_t spray_limit;
   switch (get_difficulty()) {
      case DIFF_EASY: spray_limit = SPRAYER_TIME_E; break;
      case DIFF_HARD: spray_limit = SPRAYER_TIME_H; break;
      default:        spray_limit = SPRAYER_TIME_N; break;
   }

   // Spray toxic gas every so often
   if (obj->timer >= 0x20 && obj->timer < spray_limit &&
   !(obj->timer & 0x07)) {
      // Create gas object
      Object *ptr = add_object(OBJ_TOXICGAS, obj->x, obj->y + 0x10, 0);
      ptr->gravity = SPRAYER_DROP;
   }

   // Set animation
   set_object_anim(obj, obj->timer < 0x20 ?
      anim_enemy[EN_ANIM_SPRAYERLIT] :
      anim_enemy[EN_ANIM_SPRAYERDIM]);

   // Update timer
   if (settings.sprayer_ai == 2 || obj->timer < 0x7F) {
      obj->timer++;
      obj->timer &= 0x7F;
   }
}

//***************************************************************************
// init_turret
// Initializes a turret enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_turret(Object *obj) {
   // Set the hitbox
   set_hitbox(obj, -18, 18, -12, 15);
}

//***************************************************************************
// run_turret
// Game logic for a turret enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//---------------------------------------------------------------------------
// States:
// * Beeping: timer is between 0x00 and 0x1F
// * Shooting: timer reaches 0x20
// * Waiting: timer is between 0x20 and 0x7F
//   - In easy, waits up to 0x7F before beeping again
//   - In normal, waits up to 0x5F before beeping again
//   - In hard, waits up to 0x3F before beeping again
//---------------------------------------------------------------------------
// Changes made by difficulty: how often to fire
//***************************************************************************

void run_turret(Object *obj) {
   // Are we too far from the screen to bother?
   if (is_too_far(obj))
      return;

   // Calculate delay between shots
   int max_delay;
   switch (get_difficulty()) {
      case DIFF_EASY: max_delay = TURRET_TIME_E; break;
      case DIFF_HARD: max_delay = TURRET_TIME_H; break;
      default:        max_delay = TURRET_TIME_N; break;
   }

   // Update timer
   if (settings.turret_ai == 0)
      obj->timer = max_delay - 1;
   if (obj->timer < max_delay && settings.turret_ai != 0)
      obj->timer++;
   if (obj->timer == max_delay && settings.turret_ai == 2)
      obj->timer = 0;

   // See if there's any player near enough, and get lit if so
   if (obj->timer == max_delay && settings.turret_ai == 1) {
      for (Object *player = get_first_object(OBJGROUP_PLAYER);
      player != NULL; player = player->next) {
         if (!obj->dir && (player->x < obj->x ||
         player->x - obj->x >= TURRET_RANGE_X))
            continue;
         if (obj->dir && (player->x > obj->x ||
         obj->x - player->x >= TURRET_RANGE_X))
            continue;
         if (abs(player->y - obj->y) >= TURRET_RANGE_Y)
            continue;
         obj->timer = 0;
         break;
      }
   }

   // Shoot a mortar?
   if (obj->timer == 0x20) {
      // Create mortar
      Object *other = add_object(OBJ_MORTAR, obj->x, obj->y - 0x08,
         obj->dir);

      // Set mortar speed
      if (obj->dir) {
         other->x -= 0x08;
         other->speed = -TURRET_XSPEED;
      } else {
         other->x += 0x08;
         other->speed = TURRET_XSPEED;
      }
      other->gravity = -TURRET_YSPEED;

      // Play sound effect
      play_2d_sfx(SFX_SHOOT, obj->x, obj->y);
   }

   // Let it fall if not on the ground
   obj->gravity += TURRET_WEIGHT;
   apply_physics(obj);

   // Set animation
   set_object_anim(obj, anim_enemy[obj->timer < 0x20 ?
      EN_ANIM_TURRETLIT : EN_ANIM_TURRETSHOOT]);
}

//***************************************************************************
// init_roller
// Initializes a roller enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_roller(Object *obj) {
   // Set the hitbox
   set_hitbox(obj, -12, 12, -12, 15);
}

//***************************************************************************
// run_roller
// Game logic for a roller enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//---------------------------------------------------------------------------
// States:
// * Rolling: obj->jumping is clear
// * Bouncing: obj->jumping is set
//---------------------------------------------------------------------------
// Changes made by difficulty: acceleration speed
//***************************************************************************

void run_roller(Object *obj) {
   // Are we too far from the screen to bother?
   if (is_too_far(obj))
      return;

   // Bounce if we hit an obstacle
   if (obj->on_wall && !obj->jumping) {
      // Stop enemy here
      obj->speed = 0;

      // Bounce because of impact
      obj->gravity = -ROLLER_BOUNCE;
      obj->on_ground = 0;
      obj->jumping = 1;

      // Play sound effect
      play_2d_sfx(SFX_CRASH, obj->x, obj->y);
   } else if (obj->on_ground && obj->jumping) {
      // Face the other way
      obj->dir = ~obj->dir;

      // Done bouncing, keep going
      obj->jumping = 0;
   }

   // Accelerate when we're on the ground
   if (obj->on_ground) {
      // Determine acceleration speed based on difficulty
      int accel;
      switch (get_difficulty()) {
         case DIFF_EASY: accel = ROLLER_ACCEL_E; break;
         case DIFF_HARD: accel = ROLLER_ACCEL_H; break;
         default:        accel = ROLLER_ACCEL_N; break;
      }

      obj->speed += obj->dir ? -accel : accel;
   }

   // Apply physics
   obj->gravity += ROLLER_WEIGHT;
   apply_physics(obj);

   // Release toxic gas (unless we're bouncing)
   obj->timer++;
   if (!obj->jumping && !(obj->timer & 0x07)) {
      Object *ptr = add_object(OBJ_TOXICGAS, obj->x, obj->y + 0x08, 0);
      ptr->gravity = -0x80;

      // Audiovideo clue too while we're at it
      // This needs to be playing constantly
      play_2d_sfx(SFX_ROBOTROLL, obj->x, obj->y);
   }

   // Determine which animation to use based on how fast we're moving
   EnemyAnim anim_id;
   if (obj->speed == 0)
      anim_id = EN_ANIM_ROLLERIDLE;
   else if (abs(obj->speed) < 0x180)
      anim_id = EN_ANIM_ROLLERSLOW;
   else if (abs(obj->speed) < 0x300)
      anim_id = EN_ANIM_ROLLERMED;
   else
      anim_id = EN_ANIM_ROLLERFAST;

   // Set animation
   set_object_anim(obj, anim_enemy[anim_id]);
}

//***************************************************************************
// run_grabber
// Game logic for a grabber enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//---------------------------------------------------------------------------
// States:
// * Crouching set when going down
// * Jumping set when going up
// * Low byte of timer controls the waving
// * High byte of timer == 0x00 when moving around
// * High byte of timer > 0x00 when going to grab
//---------------------------------------------------------------------------
// Changes made by difficulty: grabbing speed (LAME)
//***************************************************************************

void run_grabber(Object *obj) {
   // Are we too far from the screen to bother?
   if (is_too_far(obj))
      return;

   // Going up?
   if (obj->jumping) {
      // Reached the limit? (start wandering around as usual)
      if (obj->y <= obj->base_y) {
         obj->jumping = 0;
         obj->gravity = 0;
         obj->timer = 0;
      }

      // Keep moving
      // The speed depends on the difficulty
      else switch (get_difficulty()) {
         case DIFF_EASY: obj->gravity = -GRABBER_RISE_E; break;
         case DIFF_HARD: obj->gravity = -GRABBER_RISE_H; break;
         default:        obj->gravity = -GRABBER_RISE_N; break;
      }
   }

   // Going down?
   else if (obj->crouching) {
      // Reached the limit? (start going up)
      if (obj->on_ground || obj->y >= obj->base_y + GRABBER_MAX_Y) {
         obj->crouching = 0;
         obj->jumping = 1;
         obj->gravity = 0;
      }

      // Keep moving
      // The speed depends on the difficulty
      else switch (get_difficulty()) {
         case DIFF_EASY: obj->gravity = GRABBER_FALL_E; break;
         case DIFF_HARD: obj->gravity = GRABBER_FALL_H; break;
         default:        obj->gravity = GRABBER_FALL_N; break;
      }
   }

   // Lurking around?
   else {
      if ((obj->timer & 0xFF) < 0x7F)
         obj->timer++;

      // Make our vertical position unstable, so it looks like we're floating
      // or something like that. Using gravity here so collision can be
      // applied and the enemy can't get stuck into the floor or ceiling.
      obj->gravity = (obj->base_y - (sines[game_anim << GRABBER_WAVE & 0xFF]
         >> 6) - obj->y) << 8;

      // Ready to grab?
      if (obj->timer >= 0x100) {
         obj->timer += 0x100;
         if (obj->timer >= 0x2000)
            obj->crouching = 1;
      }

      // Nope, still looking for players
      else {
         // Determine which direction to move
         obj->speed = obj->dir ? -GRABBER_SPEED : GRABBER_SPEED;

         // Look for players
         if (settings.grabber_ai == 1) {
            for (Object *player = get_first_object(OBJGROUP_PLAYER);
            player != NULL; player = player->next) {
               // If this player is near enough then get ready to grab it
               if (player->y >= obj->y &&
               abs(player->x - obj->x) < GRABBER_RANGE_X &&
               player->y <= obj->y + GRABBER_RANGE_Y) {
                  obj->speed = 0;
                  obj->gravity = 0;
                  obj->timer = 0x100;
                  play_2d_sfx(SFX_WARNING, obj->x, obj->y);
               }
            }
         }

         // Timer reaching its limit can also trigger it automatically
         if (settings.grabber_ai == 2 && obj->timer == 0x7F) {
            obj->speed = 0;
            obj->gravity = 0;
            obj->timer = 0x100;
            play_2d_sfx(SFX_WARNING, obj->x, obj->y);
         }
      }
   }

   // Move around
   apply_physics(obj);

   // If we're lurking and hit an obstacle, move away from it
   if (obj->timer < 0x100) {
      if (obj->on_wall)
         obj->dir = ~obj->dir;
      if (obj->x <= obj->base_x - GRABBER_LIMIT)
         obj->dir = 0;
      if (obj->x >= obj->base_x + GRABBER_LIMIT)
         obj->dir = 1;
      if (obj->on_ground)
         obj->base_y -= speed_to_int(obj->gravity);
   }

   // Set animation
   if (obj->jumping)
      set_object_anim(obj, anim_enemy[EN_ANIM_GRABBERDIM]);
   else if (obj->crouching)
      set_object_anim(obj, anim_enemy[EN_ANIM_GRABBERGRAB]);
   else if (obj->timer >= 0x100)
      set_object_anim(obj, anim_enemy[EN_ANIM_GRABBERLIT]);
   else
      set_object_anim(obj, anim_enemy[EN_ANIM_GRABBERDIM]);
}

//***************************************************************************
// init_spider
// Initializes a spider enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_spider(Object *obj) {
   // This trips the AI, believe it or not
   // Probably because physics are not running all the time
   // (default is on_ground is set!)
   obj->on_ground = 0;

   // Set the hitbox
   set_hitbox(obj, -11, 11, -13, 13);

   // Reset timer
   if (settings.spider_ai != 2)
      obj->timer = 1;
}

//***************************************************************************
// run_spider
// Game logic for a spider enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//---------------------------------------------------------------------------
// States:
// * Direction is 0 for going up
// * Direction is 1 for going down
// * Timer counts for how long we have been waiting to keep moving
//---------------------------------------------------------------------------
// Changes made by difficulty: how fast the spider moves
//***************************************************************************

void run_spider(Object *obj) {
   // Are we too far from the screen to bother?
   if (is_too_far(obj))
      return;

   // Not even grabbed onto a wall?
   if (get_tile_by_pixel(obj->x, obj->abs_hitbox.y1)->collision !=
   TILE_EMPTY_BG) {
      // Make spider fall
      obj->gravity += SPIDER_WEIGHT;
      apply_physics(obj);

      // Some tweaks to ensure it works fine
      // The base_y change is needed to ensure the spider adapts to its new
      // movement range (since it won't be near its origin anymore and it
      // should attempt to adapt whatever new wall it lands on)
      obj->dir = 0;
      obj->timer = 0;
      obj->base_y = obj->y + TILE_SIZE*2;

      // Set the proper animation
      // Make the spider look like it's trying to move if it has fallen to
      // the ground and hasn't landed on a wall yet
      set_object_anim(obj, obj->on_ground ?
         anim_enemy[EN_ANIM_SPIDERSLOW_U] :
         anim_enemy[EN_ANIM_SPIDERIDLE_U]);

      // Done here, don't run AI...
      return;
   }

   // Stuck?
   // To-do: find a way to make this condition look saner
   if (obj->timer ||
   (!obj->dir && obj->blocked) || (obj->dir && obj->on_ground) ||
   (!obj->dir && (obj->y < obj->base_y - TILE_SIZE*2 ||
      get_tile_by_pixel(obj->x, obj->y - TILE_SIZE/2)->collision
      != TILE_EMPTY_BG)) ||
   (obj->dir && (obj->y > obj->base_y + TILE_SIZE*2 ||
      get_tile_by_pixel(obj->x, obj->y + TILE_SIZE/2)->collision
      != TILE_EMPTY_BG)))
   {
      // Figure out for how long we can wait
      int max_wait;
      switch (get_difficulty()) {
         case DIFF_EASY: max_wait = SPIDER_WAIT_E; break;
         case DIFF_HARD: max_wait = SPIDER_WAIT_H; break;
         default:        max_wait = SPIDER_WAIT_N; break;
      }

      // Keep waiting
      obj->timer++;

      // Done waiting? (turn around?)
      if (obj->timer >= max_wait) {
         if (settings.spider_ai == 2) {
            obj->timer = 0;
            obj->dir = ~obj->dir;
         } else {
            obj->timer = max_wait;
         }
      }

      // Player close enough?
      if (obj->timer == max_wait && settings.spider_ai == 1) {
         for (Object *player = get_first_object(OBJGROUP_PLAYER);
         player != NULL; player = player->next) {
            if (abs(player->x - obj->x) >= SPIDER_RANGE_X)
               continue;
            if (abs(player->y - obj->y) >= SPIDER_RANGE_Y)
               continue;
            obj->timer = 0;
            obj->dir = ~obj->dir;
            break;
         }
      }

      // Don't move while waiting
      obj->speed = 0;
      obj->gravity = 0;
      obj->blocked = 0;

   // Not stuck, keep going as usual
   } else {
      // Not waiting...
      obj->timer = 0;

      // Calculate the speed at which we move
      int speed;
      switch (get_difficulty()) {
         case DIFF_EASY: speed = -SPIDER_SPEED_E; break;
         case DIFF_HARD: speed = -SPIDER_SPEED_H; break;
         default:        speed = -SPIDER_SPEED_N; break;
      }

      // Flip the direction if going down
      if (obj->dir) speed = -speed;

      // There, done, move in that direction
      obj->gravity = speed;
      obj->speed = 0;
      apply_physics(obj);
   }

   // Determine which animation to use based on how fast we're moving
   EnemyAnim anim_id;
   if (obj->timer)
      anim_id = obj->dir ? EN_ANIM_SPIDERIDLE_D : EN_ANIM_SPIDERIDLE_U;
   else if (get_difficulty() == DIFF_EASY)
      anim_id = obj->dir ? EN_ANIM_SPIDERSLOW_D : EN_ANIM_SPIDERSLOW_U;
   else if (get_difficulty() == DIFF_HARD)
      anim_id = obj->dir ? EN_ANIM_SPIDERFAST_D : EN_ANIM_SPIDERFAST_U;
   else
      anim_id = obj->dir ? EN_ANIM_SPIDERMED_D : EN_ANIM_SPIDERMED_U;

   // Set animation
   set_object_anim(obj, anim_enemy[anim_id]);
}

//***************************************************************************
// init_heater
// Initializes a heater enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_heater(Object *obj) {
   // Set the hitbox
   set_hitbox(obj, -12, 12, -15, 15);

   // Reset timer
   obj->timer = 0x7F;
}

//***************************************************************************
// run_heater
// Game logic for a heater enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//---------------------------------------------------------------------------
// States:
// * Beeping: timer is between 0x00 and 0x1F
// * Spraying: timer is between 0x20 and 0x7F
//   - In easy, toxic gas is sprayed between 0x20 and 0x3F
//   - In normal, toxic gas is sprayed between 0x20 and 0x5F
//   - In hard, toxic gas is sprayed between 0x20 and 0x7F
//---------------------------------------------------------------------------
// Changes made by difficulty: how much to fire each time
//***************************************************************************

void run_heater(Object *obj) {
   // Are we too far from the screen to bother?
   if (is_too_far(obj))
      return;

   // Determine for how long to keep spraying based on difficulty
   uint16_t fire_limit;
   switch (get_difficulty()) {
      case DIFF_EASY: fire_limit = HEATER_TIME_E; break;
      case DIFF_HARD: fire_limit = HEATER_TIME_H; break;
      default:        fire_limit = HEATER_TIME_N; break;
   }

   // Let it fall if not on the ground
   obj->gravity += HEATER_WEIGHT;
   apply_physics(obj);

   // See if there's any player near enough, and get lit if so
   if (obj->timer == 0x7F && settings.heater_ai == 1) {
      for (Object *player = get_first_object(OBJGROUP_PLAYER);
      player != NULL; player = player->next) {
         if (abs(player->x - obj->x) >= SPRAYER_RANGE_X)
            continue;
         if (player->y > obj->y)
            continue;
         if (obj->y - player->y >= SPRAYER_RANGE_Y)
            continue;
         obj->timer = 0;
         break;
      }
   }

   // Release fire every so often
   if (obj->timer >= 0x20 && obj->timer < fire_limit &&
   !(obj->timer & 0x07)) {
      // Create fireball
      Object *ptr = add_object(OBJ_FIREBALL, obj->x, obj->y - 0x10, 0);
      ptr->speed = sines[obj->timer << HEATER_WAVE & 0xFF] >> HEATER_MARGIN;
      ptr->gravity = -HEATER_FIRESPD;

      // Play sound effect
      play_2d_sfx(SFX_FIRE, obj->x, obj->y);
   }

   // Set animation
   if (obj->timer < 0x20)
      set_object_anim(obj, anim_enemy[EN_ANIM_HEATERLIT]);
   else if (obj->timer < fire_limit)
      set_object_anim(obj, anim_enemy[EN_ANIM_HEATERFIRE]);
   else
      set_object_anim(obj, anim_enemy[EN_ANIM_HEATERIDLE]);

   // Update timer
   if (obj->timer < 0x7F || settings.heater_ai == 2) {
      obj->timer++;
      obj->timer &= 0x7F;
   }
}

//***************************************************************************
// init_bomb
// Initializes a bomb enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_bomb(Object *obj) {
   // Set the hitbox
   set_hitbox(obj, -11, 11, -11, 15);
}

//***************************************************************************
// run_bomb
// Game logic for a bomb enemy object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//---------------------------------------------------------------------------
// States:
// * Walking: timer < 0x30
// * Lit: timer >= 0x100
// * Explodes: timer == 0x120
//---------------------------------------------------------------------------
// Changes made by difficulty: number of mortars when exploding
//***************************************************************************

void run_bomb(Object *obj) {
   // Are we too far from the screen to bother?
   // Unless we're about to explode, then just die
   if (is_too_far(obj)) {
      if (obj->timer >= 0x100)
         delete_object(obj);
      return;
   }

   // Walk around
   if (obj->timer < 0x100) {
      // Determine our current speed. Due to our crappy mechanics, we can't
      // be moving constantly, so we're bound to how fast the feet can move.
      obj->timer++;
      obj->timer %= 0x30;
      obj->speed = obj->timer < 0x10 ?
         (obj->dir ? -BOMB_SPEED : BOMB_SPEED) : 0;

      // See if there's any player near enough, and get lit if so
      for (Object *player = get_first_object(OBJGROUP_PLAYER);
      player != NULL; player = player->next) {
         if (abs(player->x - obj->x) < BOMB_RANGE_X &&
         abs(player->y - obj->y) < BOMB_RANGE_Y) {
            // Turn on the timer
            obj->timer = 0x100;

            // Emit a warning
            play_2d_sfx(SFX_WARNING, obj->x, obj->y);
            break;
         }
      }
   }

   // Ready to explode?
   else {
      // Update timer
      obj->timer++;

      // Timer expired?
      if (obj->timer == 0x120) {
         // Spawn projectiles all around
         Object *ptr;

         if (get_difficulty() != DIFF_EASY) {
            ptr = add_object(OBJ_MORTAR, obj->x, obj->y, 1);
            ptr->speed = -BOMB_MORTAR_X1;
            ptr->gravity = -BOMB_MORTAR_Y1;

            ptr = add_object(OBJ_MORTAR, obj->x, obj->y, 0);
            ptr->speed = BOMB_MORTAR_X1;
            ptr->gravity = -BOMB_MORTAR_Y1;
         }

         ptr = add_object(OBJ_MORTAR, obj->x, obj->y, 1);
         ptr->speed = -BOMB_MORTAR_X2;
         ptr->gravity = -BOMB_MORTAR_Y2;

         ptr = add_object(OBJ_MORTAR, obj->x, obj->y, 0);
         ptr->speed = BOMB_MORTAR_X2;
         ptr->gravity = -BOMB_MORTAR_Y2;

         if (get_difficulty() == DIFF_HARD) {
            ptr = add_object(OBJ_MORTAR, obj->x, obj->y, 1);
            ptr->speed = -BOMB_MORTAR_X3;
            ptr->gravity = -BOMB_MORTAR_Y3;

            ptr = add_object(OBJ_MORTAR, obj->x, obj->y, 0);
            ptr->speed = BOMB_MORTAR_X3;
            ptr->gravity = -BOMB_MORTAR_Y3;
         }

         // Spawn an explosion
         add_object(OBJ_BIGEXPLOSION, obj->x, obj->y, 0);

         // Destroy ourselves
         delete_object(obj);
         return;
      }

      // Stop moving around
      obj->speed = 0;
   }

   // Run physics (momentum, map collision, etc.)
   obj->gravity += BOMB_WEIGHT;
   apply_physics(obj);

   // Turn around if we stumble upon an obstacle or we're about to fall
   // Only turn around when we're walking, of course...
   if ((obj->on_wall ||
   (!obj->dir && obj->on_cliff_r) ||
   (obj->dir && obj->on_cliff_l))
   && obj->timer < 0x10)
      obj->dir = ~obj->dir;

   // Set animation
   set_object_anim(obj, obj->timer < 0x100 ?
      anim_enemy[EN_ANIM_BOMBWALK] :
      anim_enemy[EN_ANIM_BOMBLIT]);
}

//***************************************************************************
// destroy_enemy
// Destroys an enemy object (like when the player attacks it). Makes scrap
// appear and such (and of course, the object is deleted).
//---------------------------------------------------------------------------
// param obj: pointer to enemy to destroy
//***************************************************************************

void destroy_enemy(Object *obj) {
   // Exception to the rule: the player can't kill the bomb, trying to do so
   // will result in the bomb exploding immediately instead.
   if (obj->type == OBJ_BOMB) {
      obj->timer = 0x11F;
      return;
   }

   // Spawn scrap
   static const struct {
      ObjType type;        // Type of object to spawn (affects sprites)
      int32_t x_speed;     // Initial horizontal momentum
      int32_t y_speed;     // Initial vertical momentum
   } scrap_data[] = {
      { OBJ_SCRAPGEAR,   -SCRAP_SPEED_X1, -SCRAP_SPEED_Y1 }, // Top-left
      { OBJ_SCRAPSPRING,  SCRAP_SPEED_X1, -SCRAP_SPEED_Y1 }, // Top-right
      { OBJ_SCRAPSPRING, -SCRAP_SPEED_X2, -SCRAP_SPEED_Y2 }, // Middle-left
      { OBJ_SCRAPGEAR,    SCRAP_SPEED_X2, -SCRAP_SPEED_Y2 }, // Middle-right
      { OBJ_SCRAPGEAR,   -SCRAP_SPEED_X3, -SCRAP_SPEED_Y3 }, // Bottom-left
      { OBJ_SCRAPSPRING,  SCRAP_SPEED_X3, -SCRAP_SPEED_Y3 }, // Bottom-right
      { NUM_OBJTYPES, 0, 0 }
   };

   for (unsigned i = 0; scrap_data[i].type != NUM_OBJTYPES; i++) {
      Object *ptr = add_object(scrap_data[i].type, obj->x, obj->y,
         scrap_data[i].x_speed >= 0 ? 0 : 1);
      ptr->speed = scrap_data[i].x_speed;
      ptr->gravity = scrap_data[i].y_speed;
   }

   // Spawn smoke
   add_object(OBJ_SMOKE, obj->x, obj->y, 0);

   // Add bonus
   add_enemy_bonus(obj->type == OBJ_BOMB ? 2 : 1);

   // Delete enemy
   delete_object(obj);

   // Play sound effect
   play_sfx(SFX_DESTROY);

   // Reduce enemy count (check added for safety)
   if (num_enemies > 0) {
      num_enemies--;
      if (settings.clear_enemies && num_enemies == 0)
         finish_level();
   }
}
