//***************************************************************************
// "framerate.c"
// Framerate control system.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stdint.h>
#include <SDL2/SDL.h>
#include "main.h"
#include "framerate.h"

// Timestamp of the last frame
static uint64_t last_stamp;
static uint64_t last_second;

// Step table
// Used to determine the timestamps for each frame within a second
// Done this way to work around integer rounding errors
static uint64_t steps[FRAMERATE+1];
static unsigned curr_step;

//***************************************************************************
// init_framerate
// Initializes the framerate control subsystem
//***************************************************************************

void init_framerate(void) {
#ifdef DEBUG
   fputs("Initializing framerate\n", stderr);
#endif

   // Reset the framerate, just in case...
   reset_framerate();

   // Initialize step table
   curr_step = 0;
   uint64_t frequency = SDL_GetPerformanceFrequency();
   for (unsigned i = 0; i < FRAMERATE+1; i++)
      steps[i] = i * frequency / FRAMERATE;
}

//***************************************************************************
// reset_framerate
// Resets the framerate counter, discarding all ellapsed frames. Meant to be
// used after loading times to prevent lots of frames from being skipped.
//***************************************************************************

void reset_framerate(void) {
   // Set the timestamp of the last frame to the current one, as if the frame
   // had just started.
   last_stamp = SDL_GetPerformanceCounter();
   last_second = last_stamp;
}

//***************************************************************************
// get_num_frames
// Gets the number of frames ellapsed since the last call (or reset). This
// function also imposes a cap on how many frames can be skipped to prevent
// the game from becoming impossible to control if it goes too slow.
//---------------------------------------------------------------------------
// return: number of ellapsed frames
//***************************************************************************

unsigned get_num_frames(void) {
   /*
   // Calculate how long does a frame last
   uint64_t frame_time = SDL_GetPerformanceFrequency() / 60;

   // Calculate how many *complete* frames ellapsed since the last call to
   // this function, and update the last frame timestamp accordingly.
   uint64_t num_frames = SDL_GetPerformanceCounter() - last_stamp;
   num_frames /= frame_time;
   last_stamp += num_frames * frame_time;
   */

   // Get the current time stamp
   uint64_t curr_stamp = SDL_GetPerformanceCounter();

   // Determine how many frames advanced
   uint64_t num_frames = 0;
   while (steps[curr_step+1] < curr_stamp - last_second) {
      num_frames++;
      curr_step++;
      if (curr_step == FRAMERATE) {
         last_second += steps[FRAMERATE];
         curr_step = 0;
      }
   }

   // Impose a cap on the amount of frames we can skip
   // If we skip too many frames then the game could become effectively
   // unplayable because of how much is skipped (not smooth enough, not
   // responsive to input). We'd rather slow down than become unplayable.
   if (num_frames > 6)
      num_frames = 6;

   // Return amount of ellapsed frames
   return (unsigned) num_frames;
}

//***************************************************************************
// deinit_framerate
// Deinitializes the framerate control subsystem
//***************************************************************************

void deinit_framerate(void) {
   // Do nothing, will get deinitialized along with SDL...
   // If for some reason later we need to deinitialize anything the relevant
   // code will go inside this function.
}
