//***************************************************************************
// "ingame.c"
// Skeleton code for the in-game mode.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdio.h>
#include "main.h"
#include "background.h"
#include "bosses.h"
#include "editor.h"
#include "enemies.h"
#include "ingame.h"
#include "input.h"
#include "level.h"
#include "loading.h"
#include "menu.h"
#include "objects.h"
#include "player.h"
#include "reader.h"
#include "replay.h"
#include "savegame.h"
#include "scene.h"
#include "settings.h"
#include "sound.h"
#include "tally.h"
#include "text.h"
#include "title_card.h"
#include "video.h"

// Where the camera information is stored
Camera camera;

// Thresholds at which the camera starts moving
// If the player is away this many pixels from the center then the camera
// will scroll to catch up (respecting the limits, that is)
#define CAMERA_X_THRESHOLD 0x08
#define CAMERA_Y_THRESHOLD 0x08

// Used to control the camera offset while zoomed in
// The max offset must be a multiple of the offset speed
#define CAMERA_OFFSET_X_SPEED 0x02
#define CAMERA_OFFSET_Y_SPEED 0x02
#define CAMERA_MAX_OFFSET_X 0x20
#define CAMERA_MAX_OFFSET_Y 0x30

// Generic animation counter for in-game stuff
// Can be used to trigger objects in sync and such
uint32_t game_anim = 0;

// Counter used to control the title card
static unsigned title_card_timer;

// Counter used to control the game speed (see settings.game_speed)
static uint16_t skip_timer;
static uint16_t skip_step;

// Variables to keep track of whether a tap happened in one-switch mode if a
// logic frame was skipped (because the game is running below 100% speed)
static struct {
   unsigned tap: 1;
   unsigned tap2: 1;
} oneswitch_latch;

// Offset applied to 24.8 fixed point values in speed_to_int before
// truncating the value to an integer. Changes every frame.
static uint8_t comma_offset = 0;

// Seed for the pseudo random number generator
// We use our own to ensure we get the same sequence every time
static uint64_t prng_seed;

// Bitfield containing which switches have been triggered in the map
static uint8_t map_switches;

// Flag to know if the game is paused or not
static int paused;

// IDs for the options in the pause menu
enum {
   PAUSE_CONTINUE,      // "Continue"
   PAUSE_RESTART,       // "Restart"
   PAUSE_QUIT           // "Quit"
};

// Position of the pause button in mouse switch mode
#define PAUSE_X1     (screen_cx - 0x10)
#define PAUSE_Y1     (screen_h - 0x20)
#define PAUSE_X2     (screen_cx + 0x0F)
#define PAUSE_Y2     (screen_h - 0x09)

// Graphics set with HUD graphics
static GraphicsSet *gfxset_hud = NULL;
static Sprite *spr_hudhealth[2];
static Sprite *spr_hudstar[2];
static Sprite *spr_pause[2];

// Default BGM to use when something wears off
static BgmID default_bgm;

// Private function prototypes
static void init_pause_menu(void);
static void load_ingame(void);
static void draw_hud(void);
static void pause_continue(void);
static void pause_restart(void);
static void pause_quit(void);

//***************************************************************************
// init_ingame
// Initializes the in-game mode. Loads resources as needed and such.
//***************************************************************************

void init_ingame(void) {
   // Reset all synchronized animation
   game_anim = 0;

   // Reset PRNG seed
   reset_random();

   // Coming from the editor or replaying?
   if (get_editor_map() != NULL ||
   settings.replay || settings.record) {
      // Hide title card
      title_card_timer = 0;
   }

   // Nope, let's initialize stuff normally
   else {
      // Make this the current scene in the savegame
      set_savegame_scene(get_curr_scene());

      // Initialize title card
      title_card_timer = 180;
   }

   // Reset game speed counter
   skip_timer = 0;
   switch (settings.game_speed) {
      case 0:  skip_step = 0x40;  break;
      case 1:  skip_step = 0x60;  break;
      case 2:  skip_step = 0x80;  break;
      case 3:  skip_step = 0xC0;  break;
      case 4:  skip_step = 0x100; break;
      case 5:  skip_step = 0x140; break;
      case 6:  skip_step = 0x180; break;
      case 7:  skip_step = 0x1C0; break;
      case 8:  skip_step = 0x200; break;
      default: skip_step = 0x100; break;
   }

   // Reset latching
   oneswitch_latch.tap = 0;
   oneswitch_latch.tap2 = 0;

   // Reset score tally
   reset_tally();

   // Don't count time spent in the level unless we're playing from the
   // beginning (don't cheat your time score!)
   {
      SaveGame savegame;
      get_savegame(&savegame);
      if (savegame.checkpoint_x != 0xFFFF &&
      savegame.checkpoint_y != 0xFFFF)
         disable_level_time();
   }

   // Reset all switches
   map_switches = 0;

   // Initialize objects manager
   init_objects();

   // Load assets
   loading_screen(load_ingame);

   // Make screen visible
   fade_on();

   // Start unpaused, d'oh!
   paused = 0;

   // Initialize menu for the pause menu
   init_pause_menu();

   // Start playing music, unless replaying
   if (settings.replay)
      stop_bgm();
   else
      play_ingame_bgm(0);

   // Start recording?
   if (settings.record)
      start_replay_recording();
}

//***************************************************************************
// init_pause_menu [internal]
// Sets up the pause menu.
//***************************************************************************

static void init_pause_menu(void) {
   init_menu();
   set_reinit_menu_func(init_pause_menu);
   menu.defoption.down = PAUSE_CONTINUE;
   menu.defoption.up = PAUSE_QUIT;

   // "Continue" option
   unsigned offset;
   offset = calc_text_len(text.ingame.opt_continue) / 2 + 4;
   menu.options[PAUSE_CONTINUE].box.x1 = screen_cx - offset;
   menu.options[PAUSE_CONTINUE].box.x2 = screen_cx + offset;
   menu.options[PAUSE_CONTINUE].box.y1 = screen_cy - 0x18;
   menu.options[PAUSE_CONTINUE].box.y2 = screen_cy - 0x08;
   menu.options[PAUSE_CONTINUE].move.up = PAUSE_QUIT;
   menu.options[PAUSE_CONTINUE].move.down = PAUSE_RESTART;
   menu.options[PAUSE_CONTINUE].move.oneswitch = PAUSE_RESTART;
   menu.options[PAUSE_CONTINUE].action.accept = pause_continue;

   // "Restart" option
   offset = calc_text_len(text.ingame.opt_restart) / 2 + 4;
   menu.options[PAUSE_RESTART].box.x1 = screen_cx - offset;
   menu.options[PAUSE_RESTART].box.x2 = screen_cx + offset;
   menu.options[PAUSE_RESTART].box.y1 = screen_cy - 0x08;
   menu.options[PAUSE_RESTART].box.y2 = screen_cy + 0x08;
   menu.options[PAUSE_RESTART].move.up = PAUSE_CONTINUE;
   menu.options[PAUSE_RESTART].move.down = PAUSE_QUIT;
   menu.options[PAUSE_RESTART].move.oneswitch = PAUSE_QUIT;
   menu.options[PAUSE_RESTART].action.accept = pause_restart;

   // "Quit" option
   offset = calc_text_len(text.ingame.opt_quit) / 2 + 4;
   menu.options[PAUSE_QUIT].box.x1 = screen_cx - offset;
   menu.options[PAUSE_QUIT].box.x2 = screen_cx + offset;
   menu.options[PAUSE_QUIT].box.y1 = screen_cy + 0x08;
   menu.options[PAUSE_QUIT].box.y2 = screen_cy + 0x18;
   menu.options[PAUSE_QUIT].move.up = PAUSE_RESTART;
   menu.options[PAUSE_QUIT].move.down = PAUSE_CONTINUE;
   menu.options[PAUSE_QUIT].move.oneswitch = PAUSE_CONTINUE;
   menu.options[PAUSE_QUIT].action.accept = pause_quit;
   menu.options[PAUSE_QUIT].sfx = SFX_CANCEL;
}

//***************************************************************************
// load_ingame [internal]
// Loads the assets for the in-game mode. Run during the loading screen.
//***************************************************************************

#define NUM_LOAD 8
static void load_ingame(void) {
   // Load object assets
   load_player();
   set_loading_total(1, NUM_LOAD);
   load_enemies();
   set_loading_total(2, NUM_LOAD);
   load_bosses();
   set_loading_total(3, NUM_LOAD);
   load_objects();
   set_loading_total(4, NUM_LOAD);
   load_level();
   set_loading_total(5, NUM_LOAD);

   // Load the HUD sprites if needed
   // (if not needed, then we for some reason we left them around, e.g. we
   // temporarily switched to another mode and we expected to come back)
   // <Sik> ...why are these kept?
   if (gfxset_hud == NULL) {
      gfxset_hud = load_graphics_set("graphics/hud");
#define SPR(x) get_sprite(gfxset_hud, x)
      spr_hudhealth[0] = SPR("health_off");
      spr_hudhealth[1] = SPR("health_on");
      spr_hudstar[0] = SPR("star_off");
      spr_hudstar[1] = SPR("star_on");
      spr_pause[0] = SPR("pause_dim");
      spr_pause[1] = SPR("pause_lit");
#undef SPR
   }
   set_loading_total(6, NUM_LOAD);

   // Load BGMs we need (we don't want to halt the game to open a file!)
   // Don't load them during demo mode, that's a waste of time! :P
   if (!settings.replay) {
      load_bgm(get_level_theme());
      load_bgm(BGM_INVINCIBILITY);
      for (Object *obj = get_first_object(OBJGROUP_BOSS);
      obj != NULL; obj = obj->next) {
         if (obj->type == OBJ_BOSS6_SPAWN) load_bgm(BGM_FINALBOSS);
         else load_bgm(BGM_BOSS);
      }
      for (Object *obj = get_first_object(OBJGROUP_PLATFORM);
      obj != NULL; obj = obj->next) {
         if (obj->type == OBJ_SHIP) {
            load_bgm(BGM_BOSS);
            break;
         }
      }
      load_bgm(BGM_FINISH);
   }
   set_loading_total(7, NUM_LOAD);

   // Load title card
   load_title_card();
   set_loading_total(8, NUM_LOAD);

   // Set default BGM to the level theme
   default_bgm = get_level_theme();
}
#undef NUM_LOAD

//***************************************************************************
// run_ingame
// Processes a logic frame in the in-game mode
//***************************************************************************

void run_ingame(void) {
   // Emulate pausing in mouse-switch mode
   if (settings.mouse_switch && !paused) {
      if (input.cursor.click &&
      input.cursor.x >= PAUSE_X1 && input.cursor.y >= PAUSE_Y1 &&
      input.cursor.x <= PAUSE_X2 && input.cursor.y <= PAUSE_Y2)
         input.player.press[PL_INPUT_PAUSE] = 1;
      if (input.mouseswitch.pause & 2)
         input.player.press[PL_INPUT_PAUSE] = 1;
   }

   // Update title card
   if (title_card_timer > 0)
      title_card_timer--;
   if (title_card_timer >= 90)
      return;

   // Determine how many frames to execute (for game speed regulation)
   unsigned num_frames;
   if (settings.replay) {
      num_frames = 1;
   } else {
      skip_timer += skip_step;
      num_frames = skip_timer >> 8;
      skip_timer &= 0xFF;
   }

   // Pause/unpause game?
   if (input.player.press[PL_INPUT_PAUSE])
      toggle_pause();

   // Update the pause menu if paused. Also, don't run the game logic while
   // paused either (d'oh).
   if (paused) {
      update_menu();
      return;
   }

   // Debug mode controls?
   if (settings.debug) {
      // These controls affect the main player
      // Get which object is the main player
      Object *player = get_first_object(OBJGROUP_PLAYER);

      // Only allow this if the player is alive...
      if (player != NULL && !player->dead) {
         // Set player power-up
         if (input.debug.wings)
            player->type = OBJ_PLAYERWINGS;
         if (input.debug.spider)
            player->type = OBJ_PLAYERSPIDER;
         if (input.debug.hammer)
            player->type = OBJ_PLAYERHAMMER;
         if (input.debug.parasol)
            player->type = OBJ_PLAYERPARASOL;
      }
   }

   // Calculate rounding offset used by speed_to_int based on the current
   // frame. This should give us sub-pixel precision for momentum values
   // while still using integers.
   // Before you ask: the offset is the last byte of the animation counter
   // with the bits reversed, so there's an offset of 1/2 every two frames,
   // an offset of 1/4 every four frames, etc.
   comma_offset = (game_anim & 0x80) >> 7 |
                  (game_anim & 0x40) >> 5 |
                  (game_anim & 0x20) >> 3 |
                  (game_anim & 0x10) >> 1 |
                  (game_anim & 0x08) << 1 |
                  (game_anim & 0x04) << 3 |
                  (game_anim & 0x02) << 5 |
                  (game_anim & 0x01) << 7;

   // Apply game speed :P
   for (unsigned i = 0; i < num_frames; i++) {
      // Fake input if replaying
      if (settings.replay)
         update_replay_playback();

      // Update replay if recording
      if (settings.record)
         update_replay_recording();

      // Apply latched taps in one-switch mode if needed
      input.oneswitch.tap |= oneswitch_latch.tap;
      input.oneswitch.tap2 |= oneswitch_latch.tap2;
      oneswitch_latch.tap = 0;
      oneswitch_latch.tap2 = 0;

      // Process the logic for all objects
      run_objects();
      update_level_time();
      game_anim++;

      // Determine whether to use the minified viewport
      int zoomed = settings.zoom/* || settings.audiovideo */;

      // Check if there's a player we can chase with the camera
      // If there is no player then leave the camera static
      Object *cam_target = get_first_object(OBJGROUP_PLAYER);
      if (cam_target != NULL && !cam_target->dead) {

         // Determine where the camera would end up if it just tried to
         // center the player directly. Though easier, we don't do this
         // because we have much subtler "physics" for the camera instead.
         int target_x = cam_target->x - screen_cx;
         int target_y = cam_target->y - screen_cy +
            (zoomed ? 0x00 : 0x10);

         // Determine the automated offset of the camera
         // Used by the helicopter boss to make it playable
         int32_t auto_target = 0;
         for (Object *boss = get_first_object(OBJGROUP_BOSS);
         boss != NULL; boss = boss->next) {
            if (boss->type == OBJ_BOSS3 ||
            boss->type == OBJ_BOSS3_SPAWN) {
               auto_target = (boss->y - cam_target->y) >> 1;
               auto_target += 0x08;
               break;
            }
         }

         // Audiovideo behavior? (no thresholds, though we mess with the
         // vertical position a bit to avoid being confused)
         if (settings.audiovideo) {
            camera.x = target_x;
            camera.y = target_y - 0x10;
         }

         // Normal camera behavior?
         else if (!zoomed) {
            // Don't exaggerate autotargetting
            auto_target /= 2;

            // If the player has moved a certain threshold past the screen
            // center, then adjust the camera so the player can't go further
            // than that.
            if (camera.x > target_x + CAMERA_X_THRESHOLD)
               camera.x = target_x + CAMERA_X_THRESHOLD;
            if (camera.x < target_x - CAMERA_X_THRESHOLD)
               camera.x = target_x - CAMERA_X_THRESHOLD;
            if (camera.y > target_y + CAMERA_Y_THRESHOLD)
               camera.y = target_y + CAMERA_Y_THRESHOLD;
            if (camera.y < target_y - CAMERA_Y_THRESHOLD)
               camera.y = target_y - CAMERA_Y_THRESHOLD;
         }

         // Zoomed camera behavior?
         else {
            // Offset the camera to the direction of the player
            // Do it only when moving though, to avoid needless motion
            if (cam_target->speed) {
               if (cam_target->dir) {
                  if (camera.offset_x > -CAMERA_MAX_OFFSET_X)
                     camera.offset_x -= CAMERA_OFFSET_X_SPEED;
               } else {
                  if (camera.offset_x < CAMERA_MAX_OFFSET_X)
                     camera.offset_x += CAMERA_OFFSET_X_SPEED;
               }
            }

            // Offset the camera to look up and down
            if (input.player.hold[PL_INPUT_UP] &&
            !input.player.hold[PL_INPUT_DOWN]) {
               if (camera.offset_y > -CAMERA_MAX_OFFSET_Y)
                  camera.offset_y -= CAMERA_OFFSET_Y_SPEED;
            } else if (input.player.hold[PL_INPUT_DOWN] &&
            !input.player.hold[PL_INPUT_UP]) {
               if (camera.offset_y < CAMERA_MAX_OFFSET_Y)
                  camera.offset_y += CAMERA_OFFSET_Y_SPEED;
            } else {
               if (camera.offset_y < 0)
                  camera.offset_y += CAMERA_OFFSET_Y_SPEED;
               if (camera.offset_y > 0)
                  camera.offset_y -= CAMERA_OFFSET_Y_SPEED;
            }
         }

         // Adjust autooffset as needed
         if (camera.auto_offset > auto_target) {
            camera.auto_offset -= CAMERA_OFFSET_Y_SPEED;
            if (camera.auto_offset < auto_target)
               camera.auto_offset = auto_target;
         } else if (camera.auto_offset < auto_target) {
            camera.auto_offset += CAMERA_OFFSET_Y_SPEED;
            if (camera.auto_offset > auto_target)
               camera.auto_offset = auto_target;
         }

         // Apply camera offset
         camera.x = target_x + camera.offset_x;
         camera.y = target_y + camera.offset_y;
         camera.y += camera.auto_offset;
      }

      // Quake effect?
      if (camera.quake) {
         // Offset the camera (only if shaking is enabled)
         if (settings.shaking && !zoomed) {
            int32_t offset = camera.quake >> 2;
            switch (game_anim & 3) {
               case 0: break;
               case 1: camera.y += offset;
               case 2: break;
               case 3: camera.y -= offset;
            }
         }

         // Reduce quake time
         camera.quake--;
      }

      // Ensure the camera doesn't go outbounds
      // Except in audiovideo mode where the player is always centered
      if (!settings.audiovideo) {
         // Check what are the limits of the screen
         int32_t top, left, bottom, right;
         if (!zoomed) {
            top = 0;
            left = 0;
            bottom = screen_h - 1;
            right = screen_w - 1;
         } else {
            top = screen_cy >> 1;
            left = screen_cx >> 1;
            bottom = screen_h - 1 - top;
            right = screen_w - 1 - left;
         }

         // Correct camera position if needed
         if (camera.x > camera.limit_right - right)
            camera.x = camera.limit_right - right;
         if (camera.x < camera.limit_left - left)
            camera.x = camera.limit_left - left;
         if (camera.y > camera.limit_bottom - bottom)
            camera.y = camera.limit_bottom - bottom;
         if (camera.y < camera.limit_top - top)
            camera.y = camera.limit_top - top;
      }

      // Update level stuff
      update_background();
      update_level();

      // Prevent repeats from one-shot input
      for (unsigned i = 0; i < PL_INPUT_PAUSE; i++)
         input.player.press[i] = 0;
      input.oneswitch.tap = 0;
      input.oneswitch.tap2 = 0;
   }

   // If no frames were processed, hack the input system to ensure one-shot
   // input is not missed (otherwise it'd be missed if entered in this frame,
   // as the next one would show only hold, no press)
   if (num_frames == 0) {
      for (unsigned i = 0; i < PL_INPUT_PAUSE; i++) {
         if (input.player.press[i]) input.player.hold[i] = 0;
      }

      oneswitch_latch.tap |= input.oneswitch.tap;
      oneswitch_latch.tap2 |= input.oneswitch.tap2;
   }

   // Update the score tally
   run_tally();
}

//***************************************************************************
// draw_ingame
// Draws everything in the in-game mode
//***************************************************************************

void draw_ingame(void) {
   // Show the cursor if the game is paused (since the pause menu appears),
   // but hide it when not paused (since the menu is hidden)
   if (!paused)
      set_cursor(settings.mouse_switch ? CURSOR_CROSS : CURSOR_NONE);

   if (title_card_timer < 90) {
      // Draw the gameplay area
      draw_background();
      draw_far_objects();
      draw_level_low();
      draw_objects();
      draw_obstructions();
      draw_level_high();
      draw_near_objects();

      // Draw the HUD
      draw_hud();

      // Zoom in now if relevant
      if (settings.zoom/* || settings.audiovideo */)
         zoom_screen();

      // Draw score tally if relevant
      draw_tally();

      // If the game is paused, make the screen darker for better contrast
      if (paused)
         dim_screen(0xC0);

      // If not paused and the mouse-switch grid is enabled (assuming we're
      // in mouse-switch mode), show the grid
      else if (settings.mouse_switch && settings.mousesw_grid) {
         draw_vline(screen_cx - settings.mousesw_x,
            0, screen_h - 1, 0xFFFFFF);
         draw_vline(screen_cx + settings.mousesw_x,
            0, screen_h - 1, 0xFFFFFF);
         draw_hline(0, screen_cy - settings.mousesw_y,
            screen_w - 1, 0xFFFFFF);
         draw_hline(0, screen_cy + settings.mousesw_y,
            screen_w - 1, 0xFFFFFF);
      }

      // Fading in from title card?
      if (title_card_timer > 90 - 31)
         dim_screen(((90 - title_card_timer) << 3));
   }

   // Keep the screen black during the title card
   else {
      clear_screen(0x000000);
   }

   // Draw the title card if needed
   if (title_card_timer > 0) {
      if (title_card_timer > 60)
         draw_title_card(0);
      else
         draw_title_card(60 - title_card_timer);
   }

   // Draw the pause menu if paused
   if (paused) {
      // Draw menu title
      draw_text(text.ingame.pause_title, screen_cx, screen_cy - 0x30,
         FONT_LIT, ALIGN_CENTER);

      // Draw options
      draw_text(text.ingame.opt_continue, screen_cx, screen_cy - 0x10,
         menu.selected == PAUSE_CONTINUE ? FONT_LIT:FONT_DIM, ALIGN_CENTER);
      draw_text(text.ingame.opt_restart, screen_cx, screen_cy,
         menu.selected == PAUSE_RESTART ? FONT_LIT:FONT_DIM, ALIGN_CENTER);
      draw_text(text.ingame.opt_quit, screen_cx, screen_cy + 0x10,
         menu.selected == PAUSE_QUIT ? FONT_LIT:FONT_DIM, ALIGN_CENTER);

      // Output selected option on screen reader
      switch (menu.selected) {
         case PAUSE_CONTINUE: set_reader_text(text.ingame.opt_continue); break;
         case PAUSE_RESTART: set_reader_text(text.ingame.opt_restart); break;
         case PAUSE_QUIT: set_reader_text(text.ingame.opt_quit); break;
         default: break;
      }
   }
}

//***************************************************************************
// draw_hud [internal]
// Draws the HUD
//***************************************************************************

static void draw_hud(void) {
   // Make sure the player exists before trying to show its info
   Object *player = get_first_object(OBJGROUP_PLAYER);
   if (player == NULL)
      return;

   // In audiovideo mode we need to resort to outputting text instead
   if (settings.audiovideo) {
      // Draw the player's health
      // To-do: localization
      set_reader_text_int("Health: {param}", player->health);

      // No graphics to render
      return;
   }

   // Draw current health
   // Make it flash if the player is low on health, though make sure it shows
   // during the pause menu since all animation is stopped while paused. Also
   // don't show it if the "no damage" or "no health" cheats are enabled
   // (since health points aren't relevant in those modes).
   // To-do: find a way to make this condition look more readable?
   int in_danger =
      player->health == (settings.collect_health ? 0 : 1);
   if (player->type != OBJ_PLAYER && settings.power_shield)
      in_danger = 0;
   if (player->shield)
      in_danger = 0;

   if (!(settings.no_damage || settings.no_health) &&
   (!in_danger || paused || (game_anim & 0x08))) {
      // Determine position of the hearts
      int32_t x, y;
      if (settings.zoom) {
         x = screen_cx >> 1;
         y = screen_cy >> 1;
      } else {
         x = 0x10;
         y = 0x08;
      };

      // Collectible health? Draw how much we collected
      if (settings.collect_health) {
         draw_sprite(spr_hudhealth[1], x, y, SPR_NOFLIP);
         draw_text_int("{param}", player->health, x + 0x14, y,
            FONT_LIT, ALIGN_TOPLEFT);
      }

      // Draw health bar
      else {
         for (unsigned i = 1; i <= (settings.extra_health ? 5 : 3); i++) {
            draw_sprite(spr_hudhealth[player->health >= i ? 1 : 0],
            x + (i-1) * 0x10, y, SPR_NOFLIP);
         }
      }
   }

   // Draw enemies left when the goal is to clear enemies
   if (settings.clear_enemies) {
      draw_text_int("[{param}]", num_enemies, screen_cx, 0x08,
         FONT_LIT, ALIGN_TOP);
   }

   // Draw collected blue stars (if any)
   size_t total_blue_stars = get_total_blue_stars();
   if (total_blue_stars > settings.blue_limit)
      total_blue_stars = settings.blue_limit;
   size_t num_blue_stars = get_num_blue_stars();

   if (num_blue_stars != total_blue_stars || (game_anim & 0x08)) {
      // Determine position of stars
      int32_t x, y;
      if (settings.zoom) {
         x = screen_w - (screen_cx >> 1);
         y = screen_cy >> 1;
      } else {
         x = screen_w - 0x10;
         y = 0x08;
      }

      // Draw stars
      for (size_t i = 0; i < total_blue_stars; i++) {
         x -= 0x10;
         draw_sprite(spr_hudstar[i < num_blue_stars ? 1 : 0],
            x, y, SPR_NOFLIP);
      }
   }

   // Draw debug coordinates if requested
   if (settings.show_coord) {
      // Buffer used to write the temporary strings
      char buffer[0x80];

      // Draw Sol's coordinates
      Object *player = get_first_object(OBJGROUP_PLAYER);
      if (player == NULL)
         strcpy(buffer, "Sol: -");
      else
         sprintf(buffer, "Sol: %04X, %04X", player->x, player->y);
      draw_text(buffer, 0x10, 0x18, FONT_LIT, ALIGN_TOPLEFT);

      // Draw camera's coordinates
      sprintf(buffer, "Cam: %04X, %04X", camera.x, camera.y);
      draw_text(buffer, 0x10, 0x28, FONT_LIT, ALIGN_TOPLEFT);

      // Draw level size
      uint16_t level_w, level_h;
      get_level_size(&level_w, &level_h);
      sprintf(buffer, "Lev: %04X×%04X",
         level_w * TILE_SIZE, level_h * TILE_SIZE);
      draw_text(buffer, 0x10, 0x38, FONT_LIT, ALIGN_TOPLEFT);
   }

   // Draw arrows inside free movement mode to make it clear it's active
   if (settings.free_move) {
      // Get player first
      Object *player = get_first_object(OBJGROUP_PLAYER);
      if (player != NULL) {
         // Calculate its on-screen coordinates
         int32_t x = player->x - camera.x;
         int32_t y = player->y - camera.y;

         // Some dumb offset for some dumb animation
         int32_t offset = (game_anim & 0x08) ? -1 : 1;

         // Draw arrows around it
         draw_text("↑", x, y - 0x20 - offset, FONT_LIT, ALIGN_CENTER);
         draw_text("↓", x, y + 0x18 + offset, FONT_LIT, ALIGN_CENTER);
         draw_text("←", x - 0x18 - offset, y - 0x04, FONT_LIT, ALIGN_CENTER);
         draw_text("→", x + 0x18 + offset, y - 0x04, FONT_LIT, ALIGN_CENTER);
      }
   }

   // Draw object hitboxes if requested
   if (settings.show_hitbox) {
      // Scan every object
      for (unsigned list_id = 0; list_id < NUM_OBJGROUPS; list_id++)
      for (Object *obj = get_first_object(list_id); obj != NULL;
      obj = obj->next) {
         // No hitbox? (don't bother)
         if (!obj->has_hitbox)
            continue;

         // Get on-screen coordinates of hitbox
         int32_t x1 = obj->abs_hitbox.x1 - camera.x;
         int32_t y1 = obj->abs_hitbox.y1 - camera.y;
         int32_t x2 = obj->abs_hitbox.x2 - camera.x;
         int32_t y2 = obj->abs_hitbox.y2 - camera.y;

         // Calculate break points so it looks better
         int32_t xb1 = x1 + ((x2 - x1) >> 2) - 1;
         int32_t yb1 = y1 + ((y2 - y1) >> 2) - 1;
         int32_t xb2 = x2 - ((x2 - x1) >> 2);
         int32_t yb2 = y2 - ((y2 - y1) >> 2);

         // Draw hitbox
         draw_hline(x1, y1, xb1, 0xFFFFFF);
         draw_hline(xb2, y1, x2, 0xFFFFFF);
         draw_hline(x1, y2, xb1, 0xFFFFFF);
         draw_hline(xb2, y2, x2, 0xFFFFFF);
         draw_vline(x1, y1, yb1, 0xFFFFFF);
         draw_vline(x1, yb2, y2, 0xFFFFFF);
         draw_vline(x2, y1, yb1, 0xFFFFFF);
         draw_vline(x2, yb2, y2, 0xFFFFFF);

         // Thicker bottom lines to indicate on_cliff_* flags
         if (obj->on_cliff_l)
            draw_hline(x1, y2-1, xb1, 0xFFFFFF);
         if (obj->on_cliff_r)
            draw_hline(xb2, y2-1, x2, 0xFFFFFF);
      }
   }

   // Show time spent in the level if enabled
   if (settings.show_time)
      draw_level_time();

   // Inside one-switch mode? Draw current input status
   if (settings.one_switch) {
      // Determine which arrow set to use
      int alt = player->type == OBJ_PLAYERSPIDER && player->active;

      // Draw which direction the player is going
      draw_text("X", 0x16, screen_h - 8,
         input.oneswitch.dir == 0 ? FONT_LIT : FONT_DIM, ALIGN_BOTTOM);
      draw_text(alt ? "↑" : "→", 0x26, screen_h - 8,
         input.oneswitch.dir == 1 ? FONT_LIT : FONT_DIM, ALIGN_BOTTOM);
      draw_text("X", 0x36, screen_h - 8,
         input.oneswitch.dir == 2 ? FONT_LIT : FONT_DIM, ALIGN_BOTTOM);
      draw_text(alt ? "↓" : "←", 0x46, screen_h - 8,
         input.oneswitch.dir == 3 ? FONT_LIT : FONT_DIM, ALIGN_BOTTOM);
   }

   // Draw pause button if inside mouse-switch mode
   if (settings.mouse_switch && !paused) {
      // Determine if the mouse is hovering the button
      int inside = input.cursor.x >= PAUSE_X1 &&
                   input.cursor.y >= PAUSE_Y1 &&
                   input.cursor.x <= PAUSE_X2 &&
                   input.cursor.y <= PAUSE_Y2
                   ? 1 : 0;

      // Draw button
      draw_sprite(spr_pause[inside], PAUSE_X1, PAUSE_Y1, SPR_NOFLIP);

      // Tell "pause" over the screen reader if hovering over the button
      if (inside)
         set_reader_text(text.ingame.pause);
   }

   // Draw "demo mode" if replaying
   if (settings.replay && (game_anim & 0x10))
      draw_text(text.ingame.demo_mode,
      screen_cx, (screen_cy + screen_h) >> 1,
      FONT_LIT, ALIGN_CENTER);

   // Draw replay counter if relevant
   if (settings.record)
      draw_replay_hud();
}

//***************************************************************************
// deinit_ingame
// Deinitializes the in-game mode. Unloads all the allocated resources.
//***************************************************************************

void deinit_ingame(void) {
   // Not replaying anymore
   settings.replay = 0;

   // Save recorded replay if needed
   if (settings.record)
      save_replay();

   // Unload music
   stop_bgm();
   unload_all_bgm();

   // Unload HUD graphics
   destroy_graphics_set(gfxset_hud);
   gfxset_hud = NULL;

   // Deinitialize object manager
   deinit_objects();

   // Unload assets
   unload_title_card();
   unload_background();
   unload_level();
   unload_objects();
   unload_bosses();
   unload_enemies();
   unload_player();

#if defined(DEBUG) || defined(DUMB_MEM)
   // If we were testing a level then it means the map data is still loaded,
   // so we need to tell the level editor code to unload the map if we're
   // quitting the program.
   if (next_game_mode == GAMEMODE_QUIT && get_editor_map() != NULL)
      deinit_editor();
#endif
}

//***************************************************************************
// speed_to_int
// Converts a 24.8 fixed point value into an integer. The rounding is
// adjusted based on the amount of frames elapsed since the game started,
// so e.g. 1.25 would return 1 for 75% of the time but 2 for 25% of the time.
// Meant to be used with speed-like values.
//---------------------------------------------------------------------------
// param value: original value
// return: integer value
//***************************************************************************

int32_t speed_to_int(int32_t value) {
   // Apply offset to value and truncate it to an integer
   // The calculation for the offset is done in run_ingame for performance
   // reasons (no need to recalculate it more than once per frame!)
   return (value + comma_offset) >> 8;
}

//***************************************************************************
// reset_random
// Resets the pseudo random number generator
//***************************************************************************

void reset_random(void) {
   prng_seed = 0x5AA5A55A5AA5A55A;
}

//***************************************************************************
// get_random
// Returns a random number between 0 and max
//---------------------------------------------------------------------------
// param max: maximum value to return
// return: number between 0 and max
//***************************************************************************

uint16_t get_random(uint16_t max) {
   // Mess with the seed
   prng_seed ^= prng_seed >> 21;
   prng_seed ^= prng_seed << 35;
   prng_seed ^= prng_seed >> 4;
   prng_seed = ~prng_seed;

   // Return random number
   return prng_seed % (max + 1);
}

//***************************************************************************
// apply_quake
// Applies a quake effect to the camera
//---------------------------------------------------------------------------
// param strength: how strong is the quake
//***************************************************************************

void apply_quake(unsigned strength) {
   // Apply quake only if it's stronger than the current one
   if (camera.quake < strength)
      camera.quake = strength;
}

//***************************************************************************
// set_map_switch
// Triggers a switch in the map
//---------------------------------------------------------------------------
// param which: switch ID
//***************************************************************************

void set_map_switch(unsigned which) {
   map_switches |= 1 << which;
}

//***************************************************************************
// get_map_switch
// Check if a switch in the map was triggered
//---------------------------------------------------------------------------
// param which: switch ID
// return: non-zero if triggered, zero otherwise
//***************************************************************************

int get_map_switch(unsigned which) {
   return (map_switches & 1 << which) ? 1 : 0;
}

//***************************************************************************
// finish_level
// Finishes the level (triggers the score tally)
//***************************************************************************

void finish_level(void) {
   // Start the score tally unless disabled
   if (settings.score_tally) {
      start_tally();
      return;
   }

   // Quit level immediately otherwise
   if (get_editor_map() != NULL)
      fade_off_and_switch(GAMEMODE_EDITOR);
   else
      switch_to_next_scene();
}

//***************************************************************************
// play_ingame_bgm
// Wrapper for play_bgm to be used in-game. It's main purpose is to help play
// BGMs only when relevant (e.g. don't change BGM during tally, etc.), as
//---------------------------------------------------------------------------
// param id: ID of BGM to play (0 for default music)
//***************************************************************************

void play_ingame_bgm(BgmID id) {
   // Don't play music during demo mode!
   if (settings.replay)
      return;

   // Don't mess with the score tally music either
   if (default_bgm == BGM_FINISH)
      return;

   // These IDs permanently replace the level theme
   if (id == BGM_BOSS || id == BGM_FINALBOSS || id == BGM_FINISH)
      default_bgm = id;

   // Play selected BGM
   play_bgm(id == 0 ? default_bgm : id);
}

//***************************************************************************
// toggle_pause
// Toggles pause on and off.
//***************************************************************************

void toggle_pause(void) {
   // Toggle state
   paused = !paused;

   // Pause or resume BGM accordingly
   if (paused) pause_bgm();
   if (!paused) resume_bgm();

   // Reset latching
   oneswitch_latch.tap = 0;
   oneswitch_latch.tap2 = 0;
}

//***************************************************************************
// pause_continue [internal]
// Code for the "Continue" option in the pause menu. Just unpauses the game.
// Same as pressing the pause key again.
//***************************************************************************

static void pause_continue(void) {
   // Just unpause the game...
   paused = 0;

   // Remember to resume music too!
   resume_bgm();
}

//***************************************************************************
// pause_restart [internal]
// Code for the "Restart" option in the pause menu. Restarts the current
// level, sending the player back to the beginning.
//***************************************************************************

static void pause_restart(void) {
   // Kill checkpoint (to force a restart from the beginning)
   reset_spawn_point();

   // Reset savegame too
   SaveGame save;
   get_savegame(&save);
   save.curr_scene = get_curr_scene();
   save.checkpoint_x = 0xFFFF;
   save.checkpoint_y = 0xFFFF;
   save_savegame(&save);

   // Restart level
   fade_off_and_switch(GAMEMODE_INGAME);
}

//***************************************************************************
// pause_quit [internal]
// Code for the "Quit" option in the pause menu. Quits the current game and
// returns back to the title screen or level editor.
//***************************************************************************

static void pause_quit(void) {
   // If we were recording this means we give up
   if (settings.record)
      fade_off_and_switch(GAMEMODE_QUIT);

   // If we were editing a level return back to the editor
   else if (get_editor_map() != NULL)
      fade_off_and_switch(GAMEMODE_EDITOR);

   // Otherwise that means it's the main game, return back to the title
   // screen in that case
   else
      fade_off_and_switch(GAMEMODE_TITLE);
}

//***************************************************************************
// get_difficulty
// Gets the current difficulty. Handles the differences between the setting
// to use in-game and the difficulty at which a replay was recorded.
//***************************************************************************

Difficulty get_difficulty(void) {
   // If in-game, return the value from the settings
   if (!settings.replay)
      return settings.difficulty;

   // Otherwise, return the one from the replay
   else
      return get_replay_difficulty();
}
