//***************************************************************************
// "launcher.c"
// Game launcher, integrated because trying to figure out how to parse the
// configuration without going crazy was too much effort.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// We need to know if we want the launcher
#include "main.h"
#if HAS_LAUNCHER

// Required headers
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#include <iup/iup.h>
#include "file.h"
#include "launcher.h"
#include "parser.h"
#include "settings.h"
#include "text.h"

// Argh, apparently IUP isn't taking margins into account when calculating
// tab sizes in the version I'm using right now, so this is here as a
// workaround until I figure out what to do
// 0 = don't have margins (ugly but works), 1 = have them (nice but broken)
// Change this if IUP ever gets fixed
#define HAS_MARGINS 1
#define HAS_PADDINGS 1
#define MAP_WORKAROUND 1

// Enable this to allow resizing. Normally this should *not* be set to true
// (it's a dialog box after all) but this may be needed for debugging.
#define ALLOW_RESIZE 0

// Enable these to show certain buttons
#define HAS_MINISOL 1            // Play Minisol!
#define HAS_MANUAL 0             // Read manual

// GUI elements used in the launcher
static Ihandle *dialog = NULL;               // Dialog window itself
static Ihandle *tabs = NULL;                 // Tabs container
static Ihandle *image = NULL;                // Image in the Play! tab

static Ihandle *play_tab = NULL;             // Play! tab
static Ihandle *play_image = NULL;           // Image in the Play! tab
static Ihandle *play_version = NULL;         // Current version
static Ihandle *play_group = NULL;           // (vbox holding buttons)
static Ihandle *play_play = NULL;            // Play Sol!
#if HAS_MINISOL
static Ihandle *play_minisol = NULL;         // Play Minisol!
#endif
#if HAS_MANUAL
static Ihandle *play_manual = NULL;          // Read manual
#endif
static Ihandle *play_saves = NULL;           // Saves folder
static Ihandle *play_quit = NULL;            // Quit game

static Ihandle *set_tab = NULL;              // Settings tab
static Ihandle *set_fullscreen = NULL;       // Fullscreen
static Ihandle *set_software = NULL;         // No hardware acceleration
static Ihandle *set_nosound = NULL;          // Disable sound
static Ihandle *set_nojoystick = NULL;       // Disable joystick

static Ihandle *access_tab = NULL;           // Accessibility tab
static Ihandle *access_reader = NULL;        // Screen reader (frame)
static Ihandle *access_readertype = NULL;    // Screen reader type
static Ihandle *access_audiovideo = NULL;    // Audiovideo mode
static Ihandle *access_onesw = NULL;         // One-switch (frame)
static Ihandle *access_oneswen = NULL;       // One-switch enable
static Ihandle *access_oneswtap = NULL;      // One-switch tap duration
static Ihandle *access_oneswmenu = NULL;     // One-switch menu delay
static Ihandle *access_oneswfive = NULL;     // One-switch five-tap delay

static Ihandle *mod_tab = NULL;              // Modding tab
static Ihandle *mod_frame = NULL;            // Mod list (frame)
static Ihandle *mod_list = NULL;             // Mod list itself
static Ihandle *mod_buttons = NULL;          // Mod list (buttons group)
static Ihandle *mod_add = NULL;              // Add mod
static Ihandle *mod_rem = NULL;              // Remove mod
static Ihandle *mod_args = NULL;             // Arguments list (frame)
static Ihandle *mod_argsfield = NULL;        // Arguments list

// Action to do when the dialog closes
typedef enum {
   ACTION_PLAY,               // Play Sol
   ACTION_MINISOL,            // Play Minisol
   ACTION_MANUAL,             // Show manual
   ACTION_SAVES,              // Show savegame folder
   ACTION_QUIT                // Quit the game
} Action;
static Action action = ACTION_QUIT;

// Arguments list to be passed back to the game
// If NULL that means no arguments to be passed
static Args *args = NULL;

// Private function prototypes
static void init_dialog(void);
static void create_image(void);
static void close_dialog(Action);
static int callback_play(Ihandle *);
#if HAS_MINISOL
static int callback_minisol(Ihandle *);
#endif
#if HAS_MANUAL
static int callback_manual(Ihandle *);
#endif
static int callback_saves(Ihandle *);
static int callback_quit(Ihandle *);
static int callback_add(Ihandle *);
static int callback_rem(Ihandle *);
static void query_language(void);
static int callback_language(Ihandle *);

//***************************************************************************
// launcher
// Main loop function for the game launcher.
//***************************************************************************

Args *launcher(void) {
   // We need to ensure we have some data!
   init_languages();
   load_config();
   load_launcher_config();
   load_language();

   // Initialize IUP
   IupOpen(NULL, NULL);
   IupSetGlobal("UTF8MODE", "YES");
   IupSetGlobal("UTF8MODE_FILE", "YES");

   // Oh um, we need a language I suppose...
   if (settings.language == -1)
      query_language();
   if (settings.language == -1) {
      IupClose();
      quit_program();
      return NULL;
   }

   // Show dialog, etc.
   init_dialog();
   IupShowXY(dialog, IUP_CENTER, IUP_CENTER);
   IupMainLoop();
   IupDestroy(dialog);
   IupDestroy(image);

   // We're done with IUP
   IupClose();

   // Save any changes to the configuration
   // Apparently this breaks stuff later otherwise
   // (something else tries to reload it...)
   // http://www.gamedev.net/topic/657090-best-comment-ever/page-5#entry5281768
   save_config();

   // Determine what to do now
   switch (action) {
      // Play game? :D
      case ACTION_PLAY:
         return args;

      case ACTION_MINISOL:
         add_local_mod("minisol.solm");
         return args;

      // Show manual?
      case ACTION_MANUAL: {
         char *dir = get_manual_dir();
         char *path = (char *) malloc(strlen(dir) + 0x20);
         if (path == NULL)
            abort_program(ERR_NOMEMORY, NULL);

         sprintf(path, "%s/index.html", dir);
#ifdef _WIN32
         run_program_with_param("explorer", path);
#else
         run_program_with_param("xdg-open", path);
#endif

         free(path);
         free(dir);
         quit_program();
      } return NULL;

      // Show savegame folder?
      case ACTION_SAVES:
         return parse_args("-c");

      // Quit game? :(
      default:
         quit_program();
         return NULL;
   }
}

//***************************************************************************
// init_dialog [internal]
// Initializes the launcher dialog.
//***************************************************************************

static void init_dialog(void) {
   // Create dialog
   dialog = IupDialog(NULL);
   IupSetAttribute(dialog, "TITLE", GAME_NAME);
#if !ALLOW_RESIZE
   IupSetAttribute(dialog, "DIALOGFRAME", "YES");
#endif

   // Create image to be shown in the Play! tab
   create_image();

   // Create tabs
   tabs = IupTabs(NULL);
   IupAppend(dialog, tabs);

   play_tab = IupHbox(NULL);
   IupSetAttribute(play_tab, "TABTITLE", text.launcher.play);
   IupAppend(tabs, play_tab);

   mod_tab = IupVbox(NULL);
   IupSetAttribute(mod_tab, "TABTITLE", text.launcher.modding);
   IupSetAttribute(mod_tab, "NORMALIZESIZE", "HORIZONTAL");
#if HAS_MARGINS
   IupSetAttribute(mod_tab, "CMARGIN", "2x2");
#endif
   IupAppend(tabs, mod_tab);

   set_tab = IupVbox(NULL);
   IupSetAttribute(set_tab, "TABTITLE", text.launcher.settings);
   IupSetAttribute(set_tab, "NORMALIZESIZE", "HORIZONTAL");
#if HAS_MARGINS
   IupSetAttribute(set_tab, "CMARGIN", "2x2");
#endif
   IupAppend(tabs, set_tab);

   access_tab = IupVbox(NULL);
   IupSetAttribute(access_tab, "TABTITLE", text.launcher.accessibility);
   IupSetAttribute(access_tab, "NORMALIZESIZE", "HORIZONTAL");
#if HAS_MARGINS
   IupSetAttribute(access_tab, "CMARGIN", "2x2");
#endif
   IupAppend(tabs, access_tab);

   // Image to show in the Play! tab
   play_image = IupLabel(NULL);
   IupSetAttributeHandle(play_image, "IMAGE", image);

   // Version number
   char verbuffer[0x40];
   sprintf(verbuffer, "Ver. %d.%02d%s", VER_MAJOR, VER_MINOR, VER_SUFFIX);
   play_version = IupLabel(verbuffer);

   // Put both :P
   IupAppend(play_tab, IupVbox(play_image, play_version, NULL));

   // Play! tab buttons
   play_play = IupButton(text.launcher.play_sol, NULL);
   IupSetCallback(play_play, "ACTION", callback_play);
   IupSetAttribute(play_play, "EXPAND", "YES");

#if HAS_MINISOL
   play_minisol = IupButton(text.launcher.play_minisol, NULL);
   IupSetCallback(play_minisol, "ACTION", callback_minisol);
   IupSetAttribute(play_minisol, "EXPAND", "YES");
#endif

#if HAS_MANUAL
   play_manual = IupButton(text.launcher.manual, NULL);
   IupSetCallback(play_manual, "ACTION", callback_manual);
   IupSetAttribute(play_manual, "EXPAND", "YES");
#endif

   play_saves = IupButton(text.launcher.saves, NULL);
   IupSetCallback(play_saves, "ACTION", callback_saves);
   IupSetAttribute(play_saves, "EXPAND", "YES");

   play_quit = IupButton(text.launcher.quit, NULL);
   IupSetCallback(play_quit, "ACTION", callback_quit);
   IupSetAttribute(play_quit, "EXPAND", "YES");

   play_group = IupVbox(play_play,
#if HAS_MINISOL
      play_minisol,
#endif
#if HAS_MANUAL
      play_manual,
#endif
      play_saves, play_quit, NULL);

   IupSetAttribute(play_group, "NORMALIZESIZE", "HORIZONTAL");
#if HAS_MARGINS
   IupSetAttribute(play_group, "CMARGIN", "2x2");
#endif
#if HAS_PADDINGS
   IupSetAttribute(play_group, "PADDING", "4x0");
#endif
   IupAppend(play_tab, play_group);

   // Failsafe settings
   set_fullscreen = IupToggle(text.launcher.fullscreen, NULL);
   IupSetAttribute(set_fullscreen, "VALUE", settings.fullscreen ? "ON" : "OFF");
   IupAppend(set_tab, set_fullscreen);

   {
      Ihandle *separator = IupLabel("");
      IupSetAttribute(separator, "SEPARATOR", "HORIZONTAL");
      IupAppend(set_tab, separator);
   }

   set_software = IupToggle(text.launcher.software, NULL);
   IupSetAttribute(set_software, "VALUE", settings.no_hwaccel ? "ON" : "OFF");
   IupAppend(set_tab, set_software);

   set_nosound = IupToggle(text.launcher.no_sound, NULL);
   IupSetAttribute(set_nosound, "VALUE", settings.no_sound ? "ON" : "OFF");
   IupAppend(set_tab, set_nosound);

   set_nojoystick = IupToggle(text.launcher.no_joystick, NULL);
   IupSetAttribute(set_nojoystick, "VALUE", settings.no_joystick ? "ON" : "OFF");
   IupAppend(set_tab, set_nojoystick);

   // Screen reader setting
   access_readertype = IupList(NULL);
   IupSetAttribute(access_readertype, "1", text.launcher.reader.disabled);
   IupSetAttribute(access_readertype, "2", text.launcher.reader.native);
   IupSetAttribute(access_readertype, "3", text.launcher.reader.clipboard);
   IupSetAttribute(access_readertype, "4", text.launcher.reader.titlebar);
   IupSetAttribute(access_readertype, "DROPDOWN", "YES");
   IupSetInt(access_readertype, "VALUE", settings.reader + 1);

   access_audiovideo = IupToggle(text.launcher.audiovideo, NULL);
   IupSetAttribute(access_audiovideo, "VALUE", settings.audiovideo ? "ON" : "OFF");

   access_reader = IupFrame(IupVbox(IupHbox(access_readertype, NULL),
      access_audiovideo, NULL));
   IupSetAttribute(access_reader, "TITLE", text.launcher.reader.title);
#if HAS_MARGINS
   IupSetAttribute(access_reader, "CMARGIN", "2x2");
#endif
   IupAppend(access_tab, access_reader);

   // One-switch settings
   access_oneswen = IupToggle(text.launcher.onesw.enable, NULL);
   IupSetAttribute(access_oneswen, "VALUE", settings.one_switch ? "ON" : "OFF");

   access_oneswtap = IupText(NULL);
   IupSetAttribute(access_oneswtap, "SPIN", "YES");
   IupSetAttribute(access_oneswtap, "SPINMIN", "1");
   IupSetAttribute(access_oneswtap, "SPINMAX", "300");
   IupSetInt(access_oneswtap, "VALUE", settings.tap_delay);

   access_oneswmenu = IupText(NULL);
   IupSetAttribute(access_oneswmenu, "SPIN", "YES");
   IupSetAttribute(access_oneswmenu, "SPINMIN", "1");
   IupSetAttribute(access_oneswmenu, "SPINMAX", "300");
   IupSetInt(access_oneswmenu, "VALUE", settings.menu_delay);

   access_oneswfive = IupText(NULL);
   IupSetAttribute(access_oneswfive, "SPIN", "YES");
   IupSetAttribute(access_oneswfive, "SPINMIN", "1");
   IupSetAttribute(access_oneswfive, "SPINMAX", "300");
   IupSetInt(access_oneswfive, "VALUE", settings.fivetap_delay);

   access_onesw = IupFrame(NULL);
   IupSetAttribute(access_onesw, "TITLE", text.launcher.onesw.title);
#if HAS_MARGINS
   IupSetAttribute(access_onesw, "CMARGIN", "2x2");
#endif

   {
      Ihandle *hbox;
      Ihandle *vbox = IupVbox(NULL);
      IupAppend(vbox, access_oneswen);

      hbox = IupHbox(access_oneswtap,
         IupLabel(text.launcher.onesw.tap), NULL);
      IupSetAttribute(hbox, "NORMALIZESIZE", "VERTICAL");
#if HAS_PADDINGS
      IupSetAttribute(hbox, "PADDING", "4x0");
#endif
      IupAppend(vbox, hbox);

      hbox = IupHbox(access_oneswmenu,
         IupLabel(text.launcher.onesw.menu), NULL);
      IupSetAttribute(hbox, "NORMALIZESIZE", "VERTICAL");
#if HAS_PADDINGS
      IupSetAttribute(hbox, "PADDING", "4x0");
#endif
      IupAppend(vbox, hbox);

      hbox = IupHbox(access_oneswfive,
         IupLabel(text.launcher.onesw.fivetap), NULL);
      IupSetAttribute(hbox, "NORMALIZESIZE", "VERTICAL");
#if HAS_PADDINGS
      IupSetAttribute(hbox, "PADDING", "4x0");
#endif
      IupAppend(vbox, hbox);

      IupAppend(vbox, IupLabel(text.launcher.onesw.comment));
      IupAppend(access_onesw, vbox);
   }

   IupAppend(access_tab, access_onesw);

   // Mods list
   mod_list = IupList(NULL);
   IupSetAttribute(mod_list, "EXPAND", "YES");

   mod_add = IupButton(text.launcher.mod.add, NULL);
   IupSetCallback(mod_add, "ACTION", callback_add);

   mod_rem = IupButton(text.launcher.mod.rem, NULL);
   IupSetCallback(mod_rem, "ACTION", callback_rem);

   mod_buttons = IupVbox(mod_add, mod_rem, NULL);
   IupSetAttribute(mod_buttons, "NORMALIZESIZE", "HORIZONTAL");

   mod_frame = IupFrame(IupHbox(mod_list, mod_buttons, NULL));
   IupSetAttribute(mod_frame, "TITLE", text.launcher.mod.list);
#if HAS_PADDINGS
   IupSetAttribute(mod_frame, "PADDING", "4x0");
#endif

   // Arguments list
   mod_argsfield = IupText(NULL);
   IupSetAttribute(mod_argsfield, "EXPAND", "HORIZONTAL");

   mod_args = IupFrame(mod_argsfield);
   IupSetAttribute(mod_args, "TITLE", text.launcher.mod.args);
#if HAS_MARGINS
   IupSetAttribute(mod_args, "CMARGIN", "2x2");
#endif

   IupAppend(mod_tab, IupVbox(mod_frame, mod_args, NULL));

#if MAP_WORKAROUND
   // Workaround for the IUP margin bug
   // Provided by Antonio Scuri while he was looking into the bug
   IupMap(dialog);
   IupSetAttribute(dialog, "RASTERSIZE", NULL);
#endif
}

//***************************************************************************
// create_image [internal]
// Loads the image we'll show in the Play! tab and converts it into an IUP
// resource.
//***************************************************************************

static void create_image(void) {
   // Get sprite for the image
   Sprite *sprite = load_sprite("graphics/launcher/image.png");
   if (sprite == NULL)
      return;

   // Allocate buffer to hold the converted image
   uint8_t *buffer = (uint8_t *) malloc(3 * sprite->width * sprite->height);
   if (buffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Convert sprite to whatever format IUP expects
   uint32_t *src = sprite->data;
   uint8_t *dest = buffer;
   for (unsigned y = 0; y < sprite->height; y++)
   for (unsigned x = 0; x < sprite->width; x++) {
      uint32_t color = *src++;
      *dest++ = color >> 16;
      *dest++ = color >> 8;
      *dest++ = color;
   }

   // Create an IUP image using this buffer
   image = IupImageRGB(sprite->width, sprite->height, buffer);

   // Done with these
   free(buffer);
   destroy_sprite(sprite);
}

//***************************************************************************
// close_dialog [internal]
// Function called whenever the dialog is about to close. Retrieves anything
// that needs to be retrieved from the UI controls.
//---------------------------------------------------------------------------
// param what: what action to take
//***************************************************************************

static void close_dialog(Action what) {
   // Get screen reader settings
   settings.reader = IupGetInt(access_readertype, "VALUE") - 1;
   settings.audiovideo = IupGetInt(access_audiovideo, "VALUE");

   // Get one-switch mode settings
   settings.one_switch = IupGetInt(access_oneswen, "VALUE");
   settings.tap_delay = IupGetInt(access_oneswtap, "VALUE");
   settings.menu_delay = IupGetInt(access_oneswmenu, "VALUE");
   settings.fivetap_delay = IupGetInt(access_oneswfive, "VALUE");

   // Get arguments to be passed directly to the game
   args = parse_args(IupGetAttribute(mod_argsfield, "VALUE"));

   // Handle failsafe settings
   settings.fullscreen = IupGetInt(set_fullscreen, "VALUE");
   settings.no_hwaccel = IupGetInt(set_software, "VALUE");
   settings.no_sound = IupGetInt(set_nosound, "VALUE");
   settings.no_joystick = IupGetInt(set_nojoystick, "VALUE");

   // Store updated settings (seems annoying, but fixes some trouble later)
   // We also need to store them in case the user just plain quits
   save_config();

   // Make sure no more option arguments can be passed
   int has_option_break = 0;
   for (size_t i = 0; i < args->count; i++)
      if (strcmp(args->list[i], "--") == 0) {
         has_option_break = 1;
         break;
      }
   if (!has_option_break)
      add_arg(args, "--");

   // Add mods!
   for (int i = 1; ; i++) {
      // Retrieve attribute
      char attr[0x20];
      sprintf(attr, "%d", i);
      const char *filename = IupGetAttribute(mod_list, attr);
      if (filename == NULL)
         break;

      // Add it to the argument list
      add_arg(args, filename);
   }

   // Set up action to do now
   action = what;
}

//***************************************************************************
// callback_play [internal]
// Callback for the Play Sol button.
//---------------------------------------------------------------------------
// param button: pointer to button
// return: IUP_CLOSE (to close the dialog)
//***************************************************************************

static int callback_play(Ihandle *button) {
   // Unused...
   (void)(button);

   // Tell game to start playing
   close_dialog(ACTION_PLAY);
   return IUP_CLOSE;
}

//***************************************************************************
// callback_minisol [internal]
// Callback for the Play Minisol button.
//---------------------------------------------------------------------------
// param button: pointer to button
// return: IUP_CLOSE (to close the dialog)
//***************************************************************************

#if HAS_MINISOL
static int callback_minisol(Ihandle *button) {
   // Unused...
   (void)(button);

   // Tell game to start playing
   close_dialog(ACTION_MINISOL);
   return IUP_CLOSE;
}
#endif

//***************************************************************************
// callback_manual [internal]
// Callback for the Read Manual button.
//---------------------------------------------------------------------------
// param button: pointer to button
// return: IUP_CLOSE (to close the dialog)
//***************************************************************************

#if HAS_MANUAL
static int callback_manual(Ihandle *button) {
   // Unused...
   (void)(button);

   // Tell game to show the manual and shut down
   close_dialog(ACTION_MANUAL);
   return IUP_CLOSE;
}
#endif

//***************************************************************************
// callback_saves [internal]
// Callback for the Saves Folder button.
//---------------------------------------------------------------------------
// param button: pointer to button
// return: IUP_CLOSE (to close the dialog)
//***************************************************************************

static int callback_saves(Ihandle *button) {
   // Unused...
   (void)(button);

   // Tell game to show the saves folder and shut down
   close_dialog(ACTION_SAVES);
   return IUP_CLOSE;
}

//***************************************************************************
// callback_quit [internal]
// Callback for the Quit button.
//---------------------------------------------------------------------------
// param button: pointer to button
// return: IUP_CLOSE (to close the dialog)
//***************************************************************************

static int callback_quit(Ihandle *button) {
   // Unused...
   (void)(button);

   // Tell game to shut down
   close_dialog(ACTION_QUIT);
   return IUP_CLOSE;
}

//***************************************************************************
// callback_add [internal]
// Callback for the Add Mod button.
//---------------------------------------------------------------------------
// param button: pointer to button
// return: IUP_DEFAULT (to do nothing else)
//***************************************************************************

static int callback_add(Ihandle *button) {
   // Unused...
   (void)(button);

   // Show up the file browser
   Ihandle *filedialog = IupFileDlg();
   IupSetAttribute(filedialog, "DIALOGTYPE", "OPEN");
   IupSetAttribute(filedialog, "FILTER", "*.solm");
   IupSetAttribute(filedialog, "TITLE", text.launcher.mod.add);
   IupPopup(filedialog, IUP_CENTER, IUP_CENTER);

   // Check if a file was selected
   if (IupGetInt(filedialog, "STATUS") == 0) {
      const char *filename = IupGetAttribute(filedialog, "VALUE");
      IupSetStrAttribute(mod_list, "APPENDITEM", filename);
   }

   // There, we're done
   IupDestroy(filedialog);
   return IUP_DEFAULT;
}

//***************************************************************************
// callback_rem [internal]
// Callback for the Remove Mod button.
//---------------------------------------------------------------------------
// param button: pointer to button
// return: IUP_DEFAULT (to do nothing else)
//***************************************************************************

static int callback_rem(Ihandle *button) {
   // Unused...
   (void)(button);

   // Check which item is to be removed
   // If nothing selected then do nothing...
   int id = IupGetInt(mod_list, "VALUE");
   if (id != 0) {
      IupSetInt(mod_list, "REMOVEITEM", id);
   }

   // There, we're done
   return IUP_DEFAULT;
}

//***************************************************************************
// query_language [internal]
// Shows a dialog to select a language. Used when a language hasn't been
// selected yet (since reloading the entire launcher when the language
// changes is just too much of a mess).
//***************************************************************************

static void query_language(void) {
   // Create a dialog to hold the options
   Ihandle *dialog = IupDialog(NULL);
   Ihandle *radio = IupRadio(NULL);
   Ihandle *button = IupButton("OK", NULL);
   Ihandle *vbox1 = IupVbox(NULL);
   Ihandle *vbox2 = IupVbox(NULL);

   IupSetAttribute(vbox1, "NORMALIZESIZE", "HORIZONTAL");
   IupSetAttribute(vbox1, "CMARGIN", "4x0");

   for (unsigned i = 0; i < get_num_languages(); i++) {
      char name[0x30];
      sprintf(name, "lang%u", i);
      Ihandle *toggle = IupToggle(get_language_name(i), NULL);
      IupSetAttribute(toggle, "VALUE", i == 0 ? "YES" : "NO");
      IupSetAttribute(toggle, "MINSIZE", "128x1");
      IupSetHandle(name, toggle);
      IupAppend(vbox1, toggle);
   }

   IupSetHandle("langradio", radio);
   IupAppend(radio, vbox1);

   IupSetCallback(button, "ACTION", callback_language);

   IupSetAttribute(vbox2, "NORMALIZESIZE", "HORIZONTAL");
   IupSetAttribute(vbox2, "CMARGIN", "2x2");
   IupSetAttribute(vbox2, "NCGAP", "2");
   IupAppend(vbox2, radio);
   IupAppend(vbox2, button);

   IupSetAttribute(dialog, "TITLE", GAME_NAME);
   IupSetAttribute(dialog, "DIALOGFRAME", "YES");
   IupAppend(dialog, vbox2);

   // Show dialog
   IupShowXY(dialog, IUP_CENTER, IUP_CENTER);
   IupMainLoop();
   IupDestroy(dialog);
}

//***************************************************************************
// callback_language [internal]
// Callback for the "OK" button in the language dialog.
//---------------------------------------------------------------------------
// param button: pointer to button
// return: IUP_CLOSE (to close the dialog)
//***************************************************************************

static int callback_language(Ihandle *button) {
   // Unused...
   (void)(button);

   // Set the language that has been chosen
   Ihandle *radio = IupGetHandle("langradio");
   const char *value = IupGetAttribute(radio, "VALUE");
   if (value != NULL && strncmp(value, "lang", 4) == 0) {
      unload_language();
      settings.language = atoi(&value[4]);
      load_language();
   }

   // Free up the handles
   // (is this really needed?)
   IupSetHandle("langradio", NULL);
   for (unsigned i = 0; i < get_num_languages(); i++) {
      char name[0x30];
      sprintf(name, "lang%u", i);
      IupSetHandle(name, NULL);
   }

   // Tell dialog to close
   return IUP_CLOSE;
}

// See #if HAS_LAUNCHER above
#endif
