//***************************************************************************
// "level.c"
// Level code, including things like loading all the level-specific data and
// drawing the tilemap.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "main.h"
#include "background.h"
#include "editor.h"
#include "enemies.h"
#include "file.h"
#include "ingame.h"
#include "level.h"
#include "objects.h"
#include "savegame.h"
#include "scene.h"
#include "settings.h"
#include "tally.h"
#include "video.h"

// Level information
// <Sik> ...if this data isn't public then why is this definition here?
typedef struct {
   // Level properties
   uint16_t width;         // Level width
   uint16_t height;        // Level height
   uint32_t spawn_x;       // Player spawn X coordinate
   uint32_t spawn_y;       // Player spawn Y coordinate
   Theme theme;            // Level theme
   LevelTile *data;        // Tilemap data
} Level;

// Where the level map information is stored
static Level level;

// List of pieces used to make up the tile graphics
// Piece IDs must fit within a byte if we pretend to save in memory...
enum {
   PIECE_SOLID_INSIDE_1,      // Solid tile, inside, top-left
   PIECE_SOLID_INSIDE_2,      // Solid tile, inside, top-right
   PIECE_SOLID_INSIDE_3,      // Solid tile, inside, bottom-left
   PIECE_SOLID_INSIDE_4,      // Solid tile, inside, bottom-right
   PIECE_SOLID_FLOOR_1,       // Solid tile, floor corner, left
   PIECE_SOLID_FLOOR_2,       // Solid tile, floor, left
   PIECE_SOLID_FLOOR_3,       // Solid tile, floor, right
   PIECE_SOLID_FLOOR_4,       // Solid tile, floor corner, right
   PIECE_SOLID_CEILING_1,     // Solid tile, ceiling corner, left
   PIECE_SOLID_CEILING_2,     // Solid tile, ceiling, left
   PIECE_SOLID_CEILING_3,     // Solid tile, ceiling, right
   PIECE_SOLID_CEILING_4,     // Solid tile, ceiling corner, right
   PIECE_SOLID_WALL_1,        // Solid tile, left wall, top
   PIECE_SOLID_WALL_2,        // Solid tile, left wall, bottom
   PIECE_SOLID_WALL_3,        // Solid tile, right wall, top
   PIECE_SOLID_WALL_4,        // Solid tile, right wall, bottom
   PIECE_SOLID_CORNER_1,      // Solid tile, L-corner, top-left
   PIECE_SOLID_CORNER_2,      // Solid tile, L-corner, top-right
   PIECE_SOLID_CORNER_3,      // Solid tile, L-corner, bottom-left
   PIECE_SOLID_CORNER_4,      // Solid tile, L-corner, bottom-right

   PIECE_NWSE_FLOOR_1,        // 2:1 NWSE slope (left), floor, top-left
   PIECE_NWSE_FLOOR_2,        // 2:1 NWSE slope (left), floor, top-right
   PIECE_NWSE_FLOOR_3,        // 2:1 NWSE slope (left), floor, bottom-left
   PIECE_NWSE_FLOOR_4,        // 2:1 NWSE slope (left), floor, bottom-right
   PIECE_NWSE_FLOOR_5,        // 2:1 NWSE slope (right), floor, top-left
   PIECE_NWSE_FLOOR_6,        // 2:1 NWSE slope (right), floor, top-right
   PIECE_NWSE_FLOOR_7,        // 2:1 NWSE slope (right), floor, bottom-left
   PIECE_NWSE_FLOOR_8,        // 2:1 NWSE slope (right), floor, bottom-right
   PIECE_NWSE_WALL_1,         // 2:1 NWSE slope (left), wall, top-left
   PIECE_NWSE_WALL_2,         // 2:1 NWSE slope (left), wall, top-right
   PIECE_NWSE_WALL_3,         // 2:1 NWSE slope (left), wall, bottom-left
   PIECE_NWSE_WALL_4,         // 2:1 NWSE slope (left), wall, bottom-right
   PIECE_NWSE_WALL_5,         // 2:1 NWSE slope (right), wall, top-left
   PIECE_NWSE_WALL_6,         // 2:1 NWSE slope (right), wall, top-right
   PIECE_NWSE_WALL_7,         // 2:1 NWSE slope (right), wall, bottom-left
   PIECE_NWSE_WALL_8,         // 2:1 NWSE slope (right), wall, bottom-right
   PIECE_NWSE_CEILING_1,      // 2:1 NWSE slope (left), ceil., leftmost
   PIECE_NWSE_CEILING_2,      // 2:1 NWSE slope (left), ceil., left
   PIECE_NWSE_CEILING_3,      // 2:1 NWSE slope (left), ceil., right
   PIECE_NWSE_CEILING_4,      // 2:1 NWSE slope (left), ceil., rightmost
   PIECE_NWSE_CEILING_5,      // 2:1 NWSE slope (right), ceil., leftmost
   PIECE_NWSE_CEILING_6,      // 2:1 NWSE slope (right), ceil., left
   PIECE_NWSE_CEILING_7,      // 2:1 NWSE slope (right), ceil., right
   PIECE_NWSE_CORNER_1,       // 2:1 NWSE slope (left), L-corner
   PIECE_NWSE_CORNER_2,       // 2:1 NWSE slope (right), L-corner

   PIECE_NESW_FLOOR_1,        // 2:1 NESW slope (right), floor, top-left
   PIECE_NESW_FLOOR_2,        // 2:1 NESW slope (right), floor, top-right
   PIECE_NESW_FLOOR_3,        // 2:1 NESW slope (right), floor, bottom-left
   PIECE_NESW_FLOOR_4,        // 2:1 NESW slope (right), floor, bottom-right
   PIECE_NESW_FLOOR_5,        // 2:1 NESW slope (left), floor, top-left
   PIECE_NESW_FLOOR_6,        // 2:1 NESW slope (left), floor, top-right
   PIECE_NESW_FLOOR_7,        // 2:1 NESW slope (left), floor, bottom-left
   PIECE_NESW_FLOOR_8,        // 2:1 NESW slope (left), floor, bottom-right
   PIECE_NESW_WALL_1,         // 2:1 NESW slope (right), wall, top-left
   PIECE_NESW_WALL_2,         // 2:1 NESW slope (right), wall, top-right
   PIECE_NESW_WALL_3,         // 2:1 NESW slope (right), wall, bottom-left
   PIECE_NESW_WALL_4,         // 2:1 NESW slope (right), wall, bottom-right
   PIECE_NESW_WALL_5,         // 2:1 NESW slope (left), wall, top-left
   PIECE_NESW_WALL_6,         // 2:1 NESW slope (left), wall, top-right
   PIECE_NESW_WALL_7,         // 2:1 NESW slope (left), wall, bottom-left
   PIECE_NESW_WALL_8,         // 2:1 NESW slope (left), wall, bottom-right
   PIECE_NESW_CEILING_1,      // 2:1 NESW slope (right), ceil., leftmost
   PIECE_NESW_CEILING_2,      // 2:1 NESW slope (right), ceil., left
   PIECE_NESW_CEILING_3,      // 2:1 NESW slope (right), ceil., right
   PIECE_NESW_CEILING_4,      // 2:1 NESW slope (right), ceil., rightmost
   PIECE_NESW_CEILING_5,      // 2:1 NESW slope (left), ceil., left
   PIECE_NESW_CEILING_6,      // 2:1 NESW slope (left), ceil., right
   PIECE_NESW_CEILING_7,      // 2:1 NESW slope (left), ceil., rightmost
   PIECE_NESW_CORNER_1,       // 2:1 NESW slope (right), L-corner
   PIECE_NESW_CORNER_2,       // 2:1 NESW slope (left), L-corner

   PIECE_BGWALL_INSIDE_1,     // Background wall, inside, top-left
   PIECE_BGWALL_INSIDE_2,     // Background wall, inside, top-right
   PIECE_BGWALL_INSIDE_3,     // Background wall, inside, bottom-left
   PIECE_BGWALL_INSIDE_4,     // Background wall, inside, bottom-right
   PIECE_BGWALL_TOP_1,        // Background wall, top corner, left
   PIECE_BGWALL_TOP_2,        // Background wall, top, left
   PIECE_BGWALL_TOP_3,        // Background wall, top, right
   PIECE_BGWALL_TOP_4,        // Background wall, top corner, right
   PIECE_BGWALL_BOTTOM_1,     // Background wall, bottom corner, left
   PIECE_BGWALL_BOTTOM_2,     // Background wall, bottom, left
   PIECE_BGWALL_BOTTOM_3,     // Background wall, bottom, right
   PIECE_BGWALL_BOTTOM_4,     // Background wall, bottom corner, right
   PIECE_BGWALL_SIDE_1,       // Background wall, left side, top
   PIECE_BGWALL_SIDE_2,       // Background wall, left side, bottom
   PIECE_BGWALL_SIDE_3,       // Background wall, right side, top
   PIECE_BGWALL_SIDE_4,       // Background wall, right side, bottom
   PIECE_BGWALL_CORNER_1,     // Background wall, L-corner, top-left
   PIECE_BGWALL_CORNER_2,     // Background wall, L-corner, top-right
   PIECE_BGWALL_CORNER_3,     // Background wall, L-corner, bottom-left
   PIECE_BGWALL_CORNER_4,     // Background wall, L-corner, bottom-right

   PIECE_BGFLOOR_TOP_1,       // Floor w/background, top border, left
   PIECE_BGFLOOR_TOP_2,       // Floor w/background, top, left
   PIECE_BGFLOOR_TOP_3,       // Floor w/background, top, right
   PIECE_BGFLOOR_TOP_4,       // Floor w/background, top border, right
   PIECE_BGFLOOR_INSIDE_1,    // Floor w/background, inside border, left
   PIECE_BGFLOOR_INSIDE_2,    // Floor w/background, inside, left
   PIECE_BGFLOOR_INSIDE_3,    // Floor w/background, inside, right
   PIECE_BGFLOOR_INSIDE_4,    // Floor w/background, inside border, right
   PIECE_BGFLOOR_SIDE_1,      // Floor w/background, wall, top-left
   PIECE_BGFLOOR_SIDE_2,      // Floor w/background, wall, middle-left
   PIECE_BGFLOOR_SIDE_3,      // Floor w/background, wall, top-right
   PIECE_BGFLOOR_SIDE_4,      // Floor w/background, wall, middle-right

   PIECE_BRIDGE_INSIDE_1,     // Bridge, inside, left
   PIECE_BRIDGE_INSIDE_2,     // Bridge, inside, right
   PIECE_BRIDGE_SIDE_1,       // Bridge, border, left
   PIECE_BRIDGE_SIDE_2,       // Bridge, border, right

   PIECE_BELT_B_LEFT,         // Conveyor belt bottom, left side
   PIECE_BELT_B_INSIDE_1,     // Conveyor belt bottom, inside (left half)
   PIECE_BELT_B_INSIDE_2,     // Conveyor belt bottom, inside (right half)
   PIECE_BELT_B_RIGHT,        // Conveyor belt bottom, right side

   // Any tiles modified dynamically go below

   PIECE_BELT_L_LEFT,         // Conveyor belt to the left, left end
   PIECE_BELT_L_MIDDLE,       // Conveyor belt to the left, middle
   PIECE_BELT_L_RIGHT,        // Conveyor belt to the left, right end
   PIECE_BELT_R_LEFT,         // Conveyor belt to the right, left end
   PIECE_BELT_R_MIDDLE,       // Conveyor belt to the right, middle
   PIECE_BELT_R_RIGHT,        // Conveyor belt to the right, right end

   // Special tiles

   PIECE_NONE,                // No piece
   NUM_PIECES                 // Number of possible piece IDs
};

// Possible conveyor belt directions
enum {
   BELTDIR_LEFT,              // Going to the left
   BELTDIR_RIGHT,             // Going to the right
   NUM_BELTDIR
};

// Used to keep track of blue stars
static size_t num_blue_stars;
static size_t total_blue_stars;

// Where level-specific graphics are stored
static GraphicsSet *gfxset_level = NULL;
static AnimFrame *anim_level[NUM_LV_ANIM];
static Sprite *spr_pieces[NUM_PIECES];
static Sprite *spr_belt[NUM_BELTDIR][8][3];

// Animations we want to remain in sync
static AnimState anim_obstruction;
static AnimState anim_liquid;

// Private function prototypes
static int load_level_from_file(const char *);
static int load_level_from_editor(void);
static int spawn_object_from_level(LevelObjID, unsigned, unsigned, uint8_t);
static void draw_tile(int, int, uint8_t, uint8_t, uint8_t, uint8_t);
static void draw_clues(int, int, uint8_t, uint8_t, uint8_t, uint8_t);

// Graphic set directories for each theme
static const char *theme_dirnames[NUM_THEMES] = {
   "graphics/level_virtual",
   "graphics/level_park",
   "graphics/level_sewer",
   "graphics/level_harbor",
   "graphics/level_desert",
   "graphics/level_cave",
   "graphics/level_factory",
   "graphics/level_carnival"
};

// Where to get the sprite names for each piece
static struct {
   const char *section;
   const char *variable;
} piece_names[NUM_PIECES] = {
   { "solid", "inside_1" },
   { "solid", "inside_2" },
   { "solid", "inside_3" },
   { "solid", "inside_4" },
   { "solid", "floor_1" },
   { "solid", "floor_2" },
   { "solid", "floor_3" },
   { "solid", "floor_4" },
   { "solid", "ceiling_1" },
   { "solid", "ceiling_2" },
   { "solid", "ceiling_3" },
   { "solid", "ceiling_4" },
   { "solid", "wall_1" },
   { "solid", "wall_2" },
   { "solid", "wall_3" },
   { "solid", "wall_4" },
   { "solid", "corner_1" },
   { "solid", "corner_2" },
   { "solid", "corner_3" },
   { "solid", "corner_4" },

   { "nwse", "floor_1" },
   { "nwse", "floor_2" },
   { "nwse", "floor_3" },
   { "nwse", "floor_4" },
   { "nwse", "floor_5" },
   { "nwse", "floor_6" },
   { "nwse", "floor_7" },
   { "nwse", "floor_8" },
   { "nwse", "wall_1" },
   { "nwse", "wall_2" },
   { "nwse", "wall_3" },
   { "nwse", "wall_4" },
   { "nwse", "wall_5" },
   { "nwse", "wall_6" },
   { "nwse", "wall_7" },
   { "nwse", "wall_8" },
   { "nwse", "ceiling_1" },
   { "nwse", "ceiling_2" },
   { "nwse", "ceiling_3" },
   { "nwse", "ceiling_4" },
   { "nwse", "ceiling_5" },
   { "nwse", "ceiling_6" },
   { "nwse", "ceiling_7" },
   { "nwse", "corner_1" },
   { "nwse", "corner_2" },

   { "nesw", "floor_1" },
   { "nesw", "floor_2" },
   { "nesw", "floor_3" },
   { "nesw", "floor_4" },
   { "nesw", "floor_5" },
   { "nesw", "floor_6" },
   { "nesw", "floor_7" },
   { "nesw", "floor_8" },
   { "nesw", "wall_1" },
   { "nesw", "wall_2" },
   { "nesw", "wall_3" },
   { "nesw", "wall_4" },
   { "nesw", "wall_5" },
   { "nesw", "wall_6" },
   { "nesw", "wall_7" },
   { "nesw", "wall_8" },
   { "nesw", "ceiling_1" },
   { "nesw", "ceiling_2" },
   { "nesw", "ceiling_3" },
   { "nesw", "ceiling_4" },
   { "nesw", "ceiling_5" },
   { "nesw", "ceiling_6" },
   { "nesw", "ceiling_7" },
   { "nesw", "corner_1" },
   { "nesw", "corner_2" },

   { "bgwall", "inside_1" },
   { "bgwall", "inside_2" },
   { "bgwall", "inside_3" },
   { "bgwall", "inside_4" },
   { "bgwall", "top_1" },
   { "bgwall", "top_2" },
   { "bgwall", "top_3" },
   { "bgwall", "top_4" },
   { "bgwall", "bottom_1" },
   { "bgwall", "bottom_2" },
   { "bgwall", "bottom_3" },
   { "bgwall", "bottom_4" },
   { "bgwall", "side_1" },
   { "bgwall", "side_2" },
   { "bgwall", "side_3" },
   { "bgwall", "side_4" },
   { "bgwall", "corner_1" },
   { "bgwall", "corner_2" },
   { "bgwall", "corner_3" },
   { "bgwall", "corner_4" },

   { "bgfloor", "top_1" },
   { "bgfloor", "top_2" },
   { "bgfloor", "top_3" },
   { "bgfloor", "top_4" },
   { "bgfloor", "inside_1" },
   { "bgfloor", "inside_2" },
   { "bgfloor", "inside_3" },
   { "bgfloor", "inside_4" },
   { "bgfloor", "side_1" },
   { "bgfloor", "side_2" },
   { "bgfloor", "side_3" },
   { "bgfloor", "side_4" },

   { "bridge", "inside_1" },
   { "bridge", "inside_2" },
   { "bridge", "side_1" },
   { "bridge", "side_2" },

   { "belt_bottom", "left" },
   { "belt_bottom", "inside_1" },
   { "belt_bottom", "inside_2" },
   { "belt_bottom", "right" },

   { "", "" }
};

// How each tile looks like in audiovideo mode
// One bit per pixel (each tile being 4×4)
static const uint16_t tiles_av[] = {
   /*
   0x0000000000000000ULL,       // TILE_EMPTY
   0xFFFFFFFFFFFFFFFFULL,       // TILE_SOLID
   0xC0F0FCFFFFFFFFFFULL,       // TILE_NWSE_1
   0x00000000C0F0FCFFULL,       // TILE_NWSE_2
   0x030F3FFFFFFFFFFFULL,       // TILE_NESW_1
   0x00000000030F3FFFULL,       // TILE_NESW_2
   0x0000000000000000ULL,       // TILE_BG_EMPTY
   0xFF00000000000000ULL,       // TILE_BG_FLOOR
   0xC0F0FCFFFFFFFFFFULL,       // TILE_BG_NWSE_1
   0x00000000C0F0FCFFULL,       // TILE_BG_NWSE_2
   0x030F3FFFFFFFFFFFULL,       // TILE_BG_NESW_1
   0x00000000030F3FFFULL,       // TILE_BG_NESW_2
   0xFFFFFFFFFFFFFFFFULL,       // TILE_HIDDEN
   0xFF00000000000000ULL,       // TILE_BRIDGE
   0xFFFFFFFFFFFFFFFFULL,       // TILE_BELT_LEFT
   0xFFFFFFFFFFFFFFFFULL        // TILE_BELT_RIGHT
   */

   /*
   0x0000,     // TILE_EMPTY
   0xFFFF,     // TILE_SOLID
   0xCFFF,     // TILE_NWSE_1
   0x00CF,     // TILE_NWSE_2
   0x3FFF,     // TILE_NESW_1
   0x003F,     // TILE_NESW_2
   0x0000,     // TILE_BG_EMPTY
   0xF000,     // TILE_BG_FLOOR
   0xCFFF,     // TILE_BG_NWSE_1
   0x00CF,     // TILE_BG_NWSE_2
   0x3FFF,     // TILE_BG_NESW_1
   0x003F,     // TILE_BG_NESW_2
   0xFFFF,     // TILE_HIDDEN
   0xF000,     // TILE_BRIDGE
   0xFFFF,     // TILE_BELT_LEFT
   0xFFFF,     // TILE_BELT_RIGHT
   */

   0x0000,     // TILE_EMPTY
   0x6666,     // TILE_SOLID
   0x4666,     // TILE_NWSE_1
   0x0046,     // TILE_NWSE_2
   0x2666,     // TILE_NESW_1
   0x0026,     // TILE_NESW_2
   0x0000,     // TILE_BG_EMPTY
   0x6000,     // TILE_BG_FLOOR
   0x4666,     // TILE_BG_NWSE_1
   0x0046,     // TILE_BG_NWSE_2
   0x2666,     // TILE_BG_NESW_1
   0x0026,     // TILE_BG_NESW_2
   0x6666,     // TILE_HIDDEN
   0x6000,     // TILE_BRIDGE
   0x6666,     // TILE_BELT_LEFT
   0x6666,     // TILE_BELT_RIGHT
};

//***************************************************************************
// load_level
// Loads all the resources needed for this level (e.g. things like the level
// map and the assets such as sprites).
//***************************************************************************

void load_level(void) {
   // Reset enemy count first of all
   num_enemies = 0;

   // If we just came from the level editor, load the level from the map
   // being currently edited in the level editor instead of looking for it
   // inside a file.
   if (get_editor_map() != NULL) {
      if (load_level_from_editor())
         abort_program(ERR_LOADLEVELED, NULL);
   }

   // If we're in a normal playthrough, look for the next level in a file
   // instead (because the levels are there, right?).
   else {
      // Load data from file
      char filename[0x80];
      sprintf(filename, "levels/%s.sol", get_scene_id());
      if (load_level_from_file(filename))
         abort_program(ERR_LOADLEVEL, filename);

      // If it's a checkpoint save, restore checkpoint
      // It's safe to assume the checkpoint belongs to this level since the
      // savegame would get reset otherwise when entering the in-game mode
      SaveGame savedata;
      get_savegame(&savedata);
      if (savedata.checkpoint_x != 0xFFFF && savedata.checkpoint_y != 0xFFFF)
      {
         set_spawn_point(savedata.checkpoint_x * TILE_SIZE + TILE_SIZE/2,
                         savedata.checkpoint_y * TILE_SIZE + TILE_SIZE/2);
      }
   }

   // If the "force virtual" cheat is on, force all levels to use the virtual
   // theme regardless of their properties
   if (settings.force_virtual)
      level.theme = THEME_VIRTUAL;

   // Determine which pieces to use for all tiles
   // If a tile changes mid-game then we can just call update_tile_pieces for
   // the affected tiles when that happens. Do NOT do this to all tiles all
   // the time though, that's wasteful and slow...
   for (unsigned y = 0; y < level.height; y++)
   for (unsigned x = 0; x < level.width; x++)
      update_tile_pieces(x, y);

   // Spawn player
   add_object(OBJ_PLAYER, level.spawn_x, level.spawn_y, 0);

   // Set the initial camera coordinates
   camera.x = level.spawn_x - screen_cx;
   camera.y = level.spawn_y - screen_cy;

   // Set the initial camera limits
   camera.limit_left = 0;
   camera.limit_right = (level.width << TILE_SIZE_BIT) - 1;
   camera.limit_top = 0;
   camera.limit_bottom = (level.height << TILE_SIZE_BIT) - 1;

   // Reset quake
   camera.quake = 0;

   // Reset camera offsets
   camera.offset_x = 0;
   camera.offset_y = 0;
   camera.auto_offset = 0;

   // Load graphics set. Note that the graphics set to load depends on which
   // theme the current level uses.
   gfxset_level = load_graphics_set(theme_dirnames[level.theme]);

   // Get a list of all the level-specific animations we need
   // To-do: turn into a loop with a list
#define ANIM(id, name) anim_level[id] = get_anim(gfxset_level, name)
   ANIM(LV_ANIM_PUSHABLE, "pushable");
   ANIM(LV_ANIM_PLATFORMSTD, "platform_standard");
   ANIM(LV_ANIM_PLATFORMBRK, "platform_breakable");
   ANIM(LV_ANIM_PLATFORMBRK2, "platform_breaking");
   ANIM(LV_ANIM_PLATFORMLEFT, "platform_leftover");
   ANIM(LV_ANIM_OBSTRUCTION, "obstruction");
   ANIM(LV_ANIM_OBSTRUCTLEFT, "obstruction_leftover");
   ANIM(LV_ANIM_SMALLSCENERY, "scenery_small");
   ANIM(LV_ANIM_BIGSCENERY, "scenery_big");
   ANIM(LV_ANIM_CRUSHER, "crusher");
   ANIM(LV_ANIM_LIQUIDHAZARD, "liquid_hazard");
   ANIM(LV_ANIM_BOUNCINGHAZARD, "bouncing_hazard");
   ANIM(LV_ANIM_FIRE, "fire");
   ANIM(LV_ANIM_STALACTITE, "stalactite");
   ANIM(LV_ANIM_DOOR, "door");
   ANIM(LV_ANIM_BALLOON_IDLE, "balloon_idle");
   ANIM(LV_ANIM_BALLOON_POP, "balloon_pop");
#undef ANIM

   // Get a list of all the piece sprites
   char name[0x100];
   sprintf(name, "%s/tiles", theme_dirnames[level.theme]);
   Ini *ini = load_ini(name);
   for (unsigned i = 0; i < PIECE_NONE; i++) {
      spr_pieces[i] = get_sprite(gfxset_level, get_ini_var(ini,
      piece_names[i].section, piece_names[i].variable, NULL));
   }
   spr_pieces[PIECE_NONE] = NULL;

   // Belts are handled separately since they're animated
   for (unsigned i = 0; i < 8; i++) {
      // Get left end of belt
      sprintf(name, "left_%u", i+1);
      spr_belt[BELTDIR_LEFT][i][0] = get_sprite(gfxset_level,
         get_ini_var(ini, "belt_left", name, NULL));
      spr_belt[BELTDIR_RIGHT][i][0] = get_sprite(gfxset_level,
         get_ini_var(ini, "belt_right", name, NULL));

      // Get middle of belt
      sprintf(name, "middle_%u", i+1);
      spr_belt[BELTDIR_LEFT][i][1] = get_sprite(gfxset_level,
         get_ini_var(ini, "belt_left", name, NULL));
      spr_belt[BELTDIR_RIGHT][i][1] = get_sprite(gfxset_level,
         get_ini_var(ini, "belt_right", name, NULL));

      // Get right end of belt
      sprintf(name, "right_%u", i+1);
      spr_belt[BELTDIR_LEFT][i][2] = get_sprite(gfxset_level,
         get_ini_var(ini, "belt_left", name, NULL));
      spr_belt[BELTDIR_RIGHT][i][2] = get_sprite(gfxset_level,
         get_ini_var(ini, "belt_right", name, NULL));
   }

   // OK, done with pieces now
   destroy_ini(ini);

   // Load the background data
   sprintf(name, "%s/background", theme_dirnames[level.theme]);
   load_background(name, gfxset_level);

   // Start animation for things that should stay in sync
   set_anim(&anim_obstruction, retrieve_level_anim(LV_ANIM_OBSTRUCTION));
   set_anim(&anim_liquid, retrieve_level_anim(LV_ANIM_LIQUIDHAZARD));

   // Initialize blue stars counter
   num_blue_stars = 0;
   total_blue_stars = 0;
   for (Object *other = get_first_object(OBJGROUP_ITEM);
   other != NULL; other = other->next) {
      if (other->type == OBJ_BLUESTAR)
         total_blue_stars++;
   }
}

//***************************************************************************
// load_level_from_file [internal]
// Loads the tilemap data for a level from a file.
//---------------------------------------------------------------------------
// param filename: name of file to read from
// return: zero on success, non-zero on failure
//***************************************************************************

static int load_level_from_file(const char *filename) {
   // Open file
   File *file = open_file(filename, FILE_READ);
   if (file == NULL)
      return -1;

   // Check identifier
   uint8_t buffer[4];
   if (read_file(file, buffer, 4)) goto error;
   if (memcmp(buffer, "\x1ASol", 4)) goto error;

   // Check version
   if (read_file(file, buffer, 2)) goto error;
   uint16_t version = buffer[0] << 8 | buffer[1];
   if (version != 0x0100) goto error;

   // Get level dimensions
   if (read_file(file, buffer, 4)) goto error;
   level.width = buffer[0] << 8 | buffer[1];
   level.height = buffer[2] << 8 | buffer[3];
   if (level.width < 12) goto error;
   if (level.height < 8) goto error;

   // Get player spawn coordinates
   if (read_file(file, buffer, 4)) goto error;
   int spawn_x = buffer[0] << 8 | buffer[1];
   int spawn_y = buffer[2] << 8 | buffer[3];
   if (spawn_x >= level.width) goto error;
   if (spawn_y >= level.height) goto error;
   if (level.spawn_x == UINT32_MAX && level.spawn_y == UINT32_MAX) {
      set_spawn_point((spawn_x * TILE_SIZE) + TILE_SIZE/2,
         (spawn_y * TILE_SIZE) + TILE_SIZE/2);
   }

   // Get level theme
   if (read_file(file, buffer, 1)) goto error;
   level.theme = (Theme) buffer[0];
   if (level.theme >= NUM_THEMES) goto error;

   // Allocate memory for the tilemap
   level.data = (LevelTile *) malloc(sizeof(LevelTile) *
      level.width * level.height);
   if (level.data == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Read tilemap data from the file
   LevelTile *ptr = level.data;
   for (unsigned y = 0; y < level.height; y++)
   for (unsigned x = 0; x < level.width; x++, ptr++) {
      // Read next tile
      if (read_file(file, buffer, 3)) goto error;

      // Retrieve collision type
      ptr->collision = buffer[0];
      ptr->force_solid = 0;
      ptr->obstruction = 0;

      // Spawn object in this tile
      if (spawn_object_from_level(buffer[1], x, y, buffer[2]))
         goto error;
   }

   // Done!
   close_file(file);
   return 0;

   // Boo...
error:
   if (level.data) {
      free(level.data);
      level.data = NULL;
   }
   close_file(file);
   return -1;
}

//***************************************************************************
// load_level_from_editor [internal]
// Loads the tilemap data for a level from the map being currently edited in
// the level editor.
//---------------------------------------------------------------------------
// return: zero on success, non-zero on failure
//***************************************************************************

static int load_level_from_editor(void) {
   // Get the map information
   const Map *map = get_editor_map();
   level.width = map->width;
   level.height = map->height;
   level.theme = map->theme;

   // Set the player spawn point if the level just started
   if (level.spawn_x == UINT32_MAX && level.spawn_y == UINT32_MAX) {
      set_spawn_point(map->spawn_x * TILE_SIZE + 0x10,
                      map->spawn_y * TILE_SIZE + 0x10);
   }

   // Allocate memory for the tilemap
   level.data = (LevelTile *) malloc(sizeof(LevelTile) *
      level.width * level.height);
   if (level.data == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Read tilemap information
   LevelTile *ptr = level.data;
   for (unsigned y = 0; y < level.height; y++)
   for (unsigned x = 0; x < level.width; x++, ptr++) {
      // Get collision of this tile
      ptr->collision = map->data[x][y].coll;
      ptr->force_solid = 0;
      ptr->obstruction = 0;

      // Spawn object in this tile
      if (spawn_object_from_level(map->data[x][y].obj, x, y,
      map->data[x][y].flags))
         goto error;
   }

#ifdef DEBUG
   // Print level properties in the debug build
   printf("*** LEVEL INFORMATION ***\n");
   printf("  Size: %ux%u tiles\n", level.width, level.height);
   printf("  Spawn: %u;%u\n", level.spawn_x, level.spawn_y);
   printf("  Theme: %u\n", level.theme);
#endif

   // Done!
   return 0;

   // Boo...
error:
   if (level.data) {
      free(level.data);
      level.data = NULL;
   }
   return -1;
}

//***************************************************************************
// spawn_object_from_level [internal]
// Helper function used to spawn an object based on information from the
// source tilemap.
//---------------------------------------------------------------------------
// param id: level object ID
// param x: X coordinate (in tiles!)
// param y: Y coordinate (in tiles!)
// param flags: flags to alter the way the object spawns
// return: zero on success, non-zero on failure (corrupt file?)
//***************************************************************************

static int spawn_object_from_level(LevelObjID id, unsigned x, unsigned y,
uint8_t flags) {
   // Catch invalid object flags
   if (flags & MAPFLAG_RESERVED)
      return -1;

   // Determine what kind of object to spawn, if any
   ObjType objtype;
   switch (id) {
      // No object! (use a sentinel value to mark this)
      case LVOBJ_NONE: return 0;

      // Enemies
      case LVOBJ_FLAMER:         objtype = OBJ_FLAMER;         break;
      case LVOBJ_SPRAYER:        objtype = OBJ_SPRAYER;        break;
      case LVOBJ_TURRET:         objtype = OBJ_TURRET;         break;
      case LVOBJ_ROLLER:         objtype = OBJ_ROLLER;         break;
      case LVOBJ_GRABBER:        objtype = OBJ_GRABBER;        break;
      case LVOBJ_SPIDER:         objtype = OBJ_SPIDER;         break;
      case LVOBJ_HEATER:         objtype = OBJ_HEATER;         break;
      case LVOBJ_BOMB:           objtype = OBJ_BOMB;           break;

      // Boss spawn points
      case LVOBJ_BOSS1:          objtype = OBJ_BOSS1_SPAWN;    break;
      case LVOBJ_BOSS2:          objtype = OBJ_BOSS2_SPAWN;    break;
      case LVOBJ_BOSS3:          objtype = OBJ_SHIP;           break;
      case LVOBJ_BOSS4:          objtype = OBJ_BOSS4_SPAWN;    break;
      case LVOBJ_BOSS5:          objtype = OBJ_BOSS5_SPAWN;    break;
      case LVOBJ_BOSS6:          objtype = OBJ_BOSS6_SPAWN;    break;

      // Stuff to move around
      case LVOBJ_SPRING:         objtype = OBJ_SPRING;         break;
      case LVOBJ_SSPRING:        objtype = OBJ_SSPRING;        break;
      case LVOBJ_BALLOON:        objtype = OBJ_BALLOON;        break;
      case LVOBJ_PUSHABLE:       objtype = OBJ_PUSHABLE;       break;
      case LVOBJ_PLATFORM:       objtype = OBJ_PLATFORM;       break;
      case LVOBJ_PLATFORM_H:     objtype = OBJ_PLATFORM_H;     break;
      case LVOBJ_PLATFORM_V:     objtype = OBJ_PLATFORM_V;     break;
      case LVOBJ_PLATFORM_B:     objtype = OBJ_PLATFORM_B;     break;
      case LVOBJ_OBSTRUCTION:    objtype = OBJ_OBSTRUCTION;    break;
      case LVOBJ_DOOR_1:         objtype = OBJ_DOOR_1;         break;
      case LVOBJ_DOOR_2:         objtype = OBJ_DOOR_2;         break;
      case LVOBJ_DOOR_3:         objtype = OBJ_DOOR_3;         break;
      case LVOBJ_DOOR_4:         objtype = OBJ_DOOR_4;         break;
      case LVOBJ_DOOR_5:         objtype = OBJ_DOOR_5;         break;
      case LVOBJ_DOOR_6:         objtype = OBJ_DOOR_6;         break;
      case LVOBJ_DOOR_7:         objtype = OBJ_DOOR_7;         break;
      case LVOBJ_DOOR_8:         objtype = OBJ_DOOR_8;         break;
      case LVOBJ_GENERATOR:      objtype = OBJ_GENERATOR;      break;
      case LVOBJ_MAINGENERATOR:  objtype = OBJ_MAINGENERATOR;  break;

      // Collectibles
      case LVOBJ_HEART:          objtype = OBJ_HEART;          break;
      case LVOBJ_INVINCIBILITY:  objtype = OBJ_INVINCIBILITY;  break;
      case LVOBJ_CHECKPOINT:     objtype = OBJ_CHECKPOINT;     break;
      case LVOBJ_GOAL:           objtype = OBJ_GOAL;           break;
      case LVOBJ_BLUESTAR:       objtype = OBJ_BLUESTAR;       break;
      case LVOBJ_SHIELD:         objtype = OBJ_SHIELD_ITEM;    break;
      case LVOBJ_WINGS:          objtype = OBJ_WINGS_ITEM;     break;
      case LVOBJ_SPIDERPOW:      objtype = OBJ_SPIDER_ITEM;    break;
      case LVOBJ_HAMMER:         objtype = OBJ_HAMMER_ITEM;    break;
      case LVOBJ_PARASOL:        objtype = OBJ_PARASOL_ITEM;   break;
      case LVOBJ_DANCER:         objtype = OBJ_DANCER;         break;

      // Triggers
      case LVOBJ_SWITCH_1:       objtype = OBJ_SWITCH_1;       break;
      case LVOBJ_SWITCH_2:       objtype = OBJ_SWITCH_2;       break;
      case LVOBJ_SWITCH_3:       objtype = OBJ_SWITCH_3;       break;
      case LVOBJ_SWITCH_4:       objtype = OBJ_SWITCH_4;       break;
      case LVOBJ_SWITCH_5:       objtype = OBJ_SWITCH_5;       break;
      case LVOBJ_SWITCH_6:       objtype = OBJ_SWITCH_6;       break;
      case LVOBJ_SWITCH_7:       objtype = OBJ_SWITCH_7;       break;
      case LVOBJ_SWITCH_8:       objtype = OBJ_SWITCH_8;       break;

      // Hazards
      case LVOBJ_SPIKES_U:       objtype = OBJ_SPIKES_U;       break;
      case LVOBJ_SPIKES_D:       objtype = OBJ_SPIKES_D;       break;
      case LVOBJ_SPIKES_L:       objtype = OBJ_SPIKES_L;       break;
      case LVOBJ_SPIKES_R:       objtype = OBJ_SPIKES_R;       break;
      case LVOBJ_CRUSHER:        objtype = OBJ_CRUSHER;        break;
      case LVOBJ_LIQUIDHAZARD:   objtype = OBJ_LIQUIDHAZARD;   break;
      case LVOBJ_BOUNCINGHAZARD: objtype = OBJ_BOUNCINGHAZARD; break;
      case LVOBJ_FIRE:           objtype = OBJ_FIRE;           break;
      case LVOBJ_STALACTITE:     objtype = OBJ_STALACTITE;     break;
      case LVOBJ_COIL_F:         objtype = OBJ_COIL_F;         break;
      case LVOBJ_COIL_C:         objtype = OBJ_COIL_C;         break;
      case LVOBJ_BUZZSAW:        objtype = OBJ_BUZZSAW;        break;
      case LVOBJ_BUZZSAW_C:      objtype = OBJ_BUZZSAW_C;      break;
      case LVOBJ_BUZZSAW_F:      objtype = OBJ_BUZZSAW_F;      break;

      // Scenery
      case LVOBJ_SCENERYSMALL:   objtype = OBJ_SCENERYSMALL;   break;
      case LVOBJ_SCENERYBIG:     objtype = OBJ_SCENERYBIG;     break;

      // Unknown object
      default: return -1;
   }

   // Spawn object
   add_object(objtype,
      x * TILE_SIZE + TILE_SIZE/2,
      y * TILE_SIZE + TILE_SIZE/2,
      (flags & MAPFLAG_FLIP) ? 1 : 0);

   // Done
   return 0;
}

//***************************************************************************
// unload_level
// Deallocates all the level resources (including the level map data and the
// level assets).
//***************************************************************************

void unload_level(void) {
   // Unload graphics
   if (gfxset_level) {
      destroy_graphics_set(gfxset_level);
      gfxset_level = NULL;
   }

   // Unload tilemap
   if (level.data) {
      free(level.data);
      level.data = NULL;
   }
}

//***************************************************************************
// update_level
// Updates some level stuff (e.g. animations). Call once per frame.
//***************************************************************************

void update_level(void) {
   // Update animations
   update_anim(&anim_obstruction);
   update_anim(&anim_liquid);
}

//***************************************************************************
// draw_level_high
// Draws the high layer of the level tilemap (stuff that appears above the
// objects, including the main layout).
//***************************************************************************

void draw_level_high(void) {
   // We only draw a single layer in audiovideo mode
   if (settings.audiovideo)
      return;

   // Update conveyor belt animation
   unsigned belt_anim = speed_to_int(game_anim * BELT_SPEED/2) & 7;

   spr_pieces[PIECE_BELT_L_LEFT]   = spr_belt[BELTDIR_LEFT][belt_anim][0];
   spr_pieces[PIECE_BELT_L_MIDDLE] = spr_belt[BELTDIR_LEFT][belt_anim][1];
   spr_pieces[PIECE_BELT_L_RIGHT]  = spr_belt[BELTDIR_LEFT][belt_anim][2];

   spr_pieces[PIECE_BELT_R_LEFT]   = spr_belt[BELTDIR_RIGHT][belt_anim][0];
   spr_pieces[PIECE_BELT_R_MIDDLE] = spr_belt[BELTDIR_RIGHT][belt_anim][1];
   spr_pieces[PIECE_BELT_R_RIGHT]  = spr_belt[BELTDIR_RIGHT][belt_anim][2];

   // Go through all visible tiles
   for (int y = -(camera.y & TILE_SIZE_MASK) - TILE_SIZE,
   yr = (camera.y >> TILE_SIZE_BIT) - 1; y < screen_h;
   y += TILE_SIZE, yr++) {
      for (int x = -(camera.x & TILE_SIZE_MASK) - TILE_SIZE,
      xr = (camera.x >> TILE_SIZE_BIT) - 1; x < screen_w;
      x += TILE_SIZE, xr++) {
         // Retrieve pointer to this tile
         const LevelTile *tile = get_tile(xr, yr);

         // Draw tile
         draw_tile(x, y,
            tile->piece_tl, tile->piece_tr,
            tile->piece_bl, tile->piece_br);
      }
   }
}

//***************************************************************************
// draw_level_low
// Draws the low layer of the level tilemap (stuff that appears below all
// objects, e.g. background walls used for near scenery).
//***************************************************************************

void draw_level_low(void) {
   // Go through all visible tiles
   for (int y = -(camera.y & TILE_SIZE_MASK),
   yr = camera.y >> TILE_SIZE_BIT; y < screen_h;
   y += TILE_SIZE, yr++) {
      for (int x = -(camera.x & TILE_SIZE_MASK),
      xr = camera.x >> TILE_SIZE_BIT; x < screen_w;
      x += TILE_SIZE, xr++) {
         // Retrieve pointer to this tile
         const LevelTile *tile = get_tile(xr, yr);

         // Draw tile
         if (!settings.audiovideo) {
            draw_tile(x, y,
            tile->bgpiece_tl, tile->bgpiece_tr,
            tile->bgpiece_bl, tile->bgpiece_br);
         } else {
            draw_clues(x, y,
            tile->piece_tl, tile->piece_tr,
            tile->piece_bl, tile->piece_br);
         }
      }
   }
}

//***************************************************************************
// draw_tile [internal]
// Draws a tile from the tilemap on screen using the specified pieces.
//---------------------------------------------------------------------------
// param x: X coordinate of tile (in pixels)
// param y: Y coordinate of tile (in pixels)
// param tl: ID of top-left piece
// param tr: ID of top-right piece
// param bl: ID of bottom-left piece
// param br: ID of bottom-right piece
//***************************************************************************

static void draw_tile(int x, int y, uint8_t tl, uint8_t tr, uint8_t bl,
uint8_t br) {
   const Sprite *spr;

   // Get the center coordinates for the tile
   // All sprites are drawn based off this position
   x += TILE_SIZE/2;
   y += TILE_SIZE/2;

   // Draw top-left piece
   spr = spr_pieces[tl];
   if (spr != NULL)
      draw_sprite(spr, x - spr->width, y - spr->height, SPR_NOFLIP);

   // Draw top-right piece
   spr = spr_pieces[tr];
   if (spr != NULL)
      draw_sprite(spr, x, y - spr->height, SPR_NOFLIP);

   // Draw bottom-left piece
   spr = spr_pieces[bl];
   if (spr != NULL)
      draw_sprite(spr, x - spr->width, y, SPR_NOFLIP);

   // Draw bottom-right piece
   spr = spr_pieces[br];
   if (spr != NULL)
      draw_sprite(spr, x, y, SPR_NOFLIP);
}

//***************************************************************************
// draw_clues [internal]
// Like draw_tile but for audiovideo mode.
//---------------------------------------------------------------------------
// param x: X coordinate of tile (in pixels)
// param y: Y coordinate of tile (in pixels)
// param tl: ID of top-left piece
// param tr: ID of top-right piece
// param bl: ID of bottom-left piece
// param br: ID of bottom-right piece
//***************************************************************************

static void draw_clues(int x, int y, uint8_t tl, uint8_t tr, uint8_t bl,
uint8_t br) {
   int32_t x1, y1, x2, y2;

   // Top left corner
   if (tl != AV_EMPTY) {
      x1 = x; x2 = x1 + 7;
      y1 = y; y2 = y1 + 7;
      fill_rectangle(x1, y1, x2, y2, tl);
   }

   // Top right corner
   if (tr != AV_EMPTY) {
      x2 = x + TILE_SIZE-1; x1 = x2 - 7;
      y1 = y; y2 = y1 + 7;
      fill_rectangle(x1, y1, x2, y2, tr);
   }

   // Bottom left corner
   if (bl != AV_EMPTY) {
      x1 = x; x2 = x1 + 7;
      y2 = y + TILE_SIZE-1; y1 = y2 - 7;
      fill_rectangle(x1, y1, x2, y2, bl);
   }

   // Bottom right corner
   if (br != AV_EMPTY) {
      x2 = x + TILE_SIZE-1; x1 = x2 - 7;
      y2 = y + TILE_SIZE-1; y1 = y2 - 7;
      fill_rectangle(x1, y1, x2, y2, br);
   }
}

//***************************************************************************
// draw_obstructions
// Draws all the obstruction objects in the level. Handled separately since
// in reality they aren't objects at all but they're part of the tilemap.
//---------------------------------------------------------------------------
// Fun fact: obstructions actually *used* to be objects (which is why they're
// placed like objects in levels in the first place), but the problem is that
// the cave level has lots of them, and the object manager is sorta lazy when
// handling objects so this could cause performance issues on older systems.
//***************************************************************************

void draw_obstructions(void) {
   // Go through all visible tiles
   for (int y = -(camera.y & TILE_SIZE_MASK),
   yr = camera.y >> TILE_SIZE_BIT; y < screen_h;
   y += TILE_SIZE, yr++) {
      for (int x = -(camera.x & TILE_SIZE_MASK),
      xr = camera.x >> TILE_SIZE_BIT; x < screen_w;
      x += TILE_SIZE, xr++) {
         // Check if there's an obstruction here
         if (!(get_tile(xr, yr)->obstruction))
            continue;

         // Draw object
         if (!settings.audiovideo) {
            const AnimFrame *frame = anim_obstruction.frame;
            draw_sprite(frame->sprite,
               x + TILE_SIZE/2 - frame->x_offset,
               y + TILE_SIZE/2 - frame->y_offset,
               SPR_NOFLIP);
         } else {
            int32_t x1 = x;
            int32_t y1 = y;
            int32_t x2 = x1 + TILE_SIZE-1;
            int32_t y2 = y1 + TILE_SIZE-1;
            fill_rectangle(x1, y1, x2, y2, AV_LEVEL);
         }
      }
   }
}

//***************************************************************************
// get_tile
// Retrieves the pointer to the tile at the specified tile coordinates (*not*
// pixel coordinates!). Note that if the coordinates lie outside the level
// dimensions that this function will take that into account and still return
// a valid tile (to allow things to work properly when going off-screen).
//---------------------------------------------------------------------------
// param x: X coordinate (in tiles)
// param y: Y coordinate (in tiles)
// return: tile at those coordinates
//***************************************************************************

LevelTile *get_tile(int32_t x, int32_t y) {
   // Clamp tile coordinates so they lie within the game world
   if (x < 0) x = 0;
   if (y < 0) y = 0;
   if (x >= level.width) x = level.width - 1;
   if (y >= level.height) y = level.height - 1;

   // Return tile at this coordinates
   return &level.data[x + y * level.width];
}

//***************************************************************************
// get_tile_by_pixel
// Retrieves the pointer to the tile at the specified coordinates. Note that
// if the coordinates lie outside the level dimensions that this function
// will take that into account and still return a valid tile (to allow things
// to work properly when going off-screen).
//---------------------------------------------------------------------------
// param x: X coordinate (in pixels)
// param y: Y coordinate (in pixels)
// return: tile at those coordinates
//***************************************************************************

LevelTile *get_tile_by_pixel(int32_t x, int32_t y) {
   // Turn pixel coordinates into tile coordinates
   x >>= TILE_SIZE_BIT;
   y >>= TILE_SIZE_BIT;

   // Clamp tile coordinates so they lie within the game world
   if (x < 0) x = 0;
   if (y < 0) y = 0;
   if (x >= level.width) x = level.width - 1;
   if (y >= level.height) y = level.height - 1;

   // Return tile at this coordinates
   return &level.data[x + y * level.width];
}

//***************************************************************************
// get_map_collision
// Similar to get_tile_by_pixel, but returns the kind of collision of that
// tile (used by physics code). This takes into account any special flags the
// tile may have set and changes the collision accordingly.
//---------------------------------------------------------------------------
// param x: X coordinate (in pixels)
// param y: Y coordinate (in pixels)
// return: collision at those coordinates
//***************************************************************************

uint8_t get_map_collision(int32_t x, int32_t y) {
   // Retrieve the tile at these coordinates
   const LevelTile *tile = get_tile_by_pixel(x, y);

   // Normal tile? (return normal collision)
   if (!tile->force_solid)
      return tile->collision;

   // Forced solid? (behave like it was a solid tile)
   else
      return TILE_SOLID;
}

//***************************************************************************
// update_tile_pieces
// Determines what pieces should a given tile use and updates it accordingly.
// Doing this constantly to the entire tilemap while drawing would be
// expensive, so instead we just do it only whenever the tile changes!
//---------------------------------------------------------------------------
// param x: X coordinate of tile
// param y: Y coordinate of tile
//***************************************************************************

// Some quick inline functions to make our life easier
// (checks that are used multiple times, etc.)
static inline int looks_solid(uint8_t type)
   { return type == TILE_SOLID || type == TILE_HIDDEN; }

static inline int has_slope(uint8_t type)
   { return (type >= TILE_NWSE_1 && type <= TILE_NESW_2) ||
            (type >= TILE_NWSE_1_BG && type <= TILE_NESW_2_BG); }
static inline int has_slope_top(uint8_t type)
   { return type == TILE_NWSE_1 || type == TILE_NWSE_1_BG ||
            type == TILE_NESW_1 || type == TILE_NESW_1_BG; }
static inline int has_slope_bottom(uint8_t type)
   { return type == TILE_NWSE_2 || type == TILE_NWSE_2_BG ||
            type == TILE_NESW_2 || type == TILE_NESW_2_BG; }

static inline int has_nwse_top(uint8_t type)
   { return type == TILE_NWSE_1 || type == TILE_NWSE_1_BG; }
static inline int has_nwse_bottom(uint8_t type)
   { return type == TILE_NWSE_2 || type == TILE_NWSE_2_BG; }
static inline int has_nwse(uint8_t type)
   { return has_nwse_top(type) || has_nwse_bottom(type); }

static inline int has_nesw_top(uint8_t type)
   { return type == TILE_NESW_1 || type == TILE_NESW_1_BG; }
static inline int has_nesw_bottom(uint8_t type)
   { return type == TILE_NESW_2 || type == TILE_NESW_2_BG; }
static inline int has_nesw(uint8_t type)
   { return has_nesw_top(type) || has_nesw_bottom(type); }

static inline int has_bgwall(uint8_t type)
   { return type >= TILE_EMPTY_BG && type <= TILE_NESW_2_BG; }

void update_tile_pieces(uint16_t x, uint16_t y) {
   // Use clues in audiovideo mode instead!
   if (settings.audiovideo) {
      update_tile_clues(x, y);
      return;
   }

   // Get the tile to update
   LevelTile *tile = get_tile(x, y);

   // Set all pieces to none by default
   tile->piece_tl = PIECE_NONE;
   tile->piece_tr = PIECE_NONE;
   tile->piece_bl = PIECE_NONE;
   tile->piece_br = PIECE_NONE;
   tile->bgpiece_tl = PIECE_NONE;
   tile->bgpiece_tr = PIECE_NONE;
   tile->bgpiece_bl = PIECE_NONE;
   tile->bgpiece_br = PIECE_NONE;

   // Get what kind of collision this type has
   // We'll determine what pieces to use based on that
   switch (tile->collision) {
      // Solid?
      case TILE_SOLID:
      case TILE_HIDDEN:
         // [solid tiles]
         // Determine top-left piece
         {
            uint8_t above = get_tile(x, y-1)->collision;
            uint8_t side = get_tile(x-1, y)->collision;
            uint8_t corner = get_tile(x-1, y-1)->collision;

            if (looks_solid(above)) above = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (has_slope_top(above))
               above = TILE_SOLID;
            if (has_slope_top(corner) ||
            corner == TILE_NESW_2 || corner == TILE_NESW_2_BG)
               corner = TILE_SOLID;
            if (side == TILE_NESW_1 || side == TILE_NESW_1_BG) {
               side = TILE_SOLID;
               corner = TILE_EMPTY;
            }

            if (has_nwse_bottom(above)) {
               tile->piece_tl = (side == TILE_SOLID) ?
               PIECE_NWSE_FLOOR_7 : PIECE_NWSE_WALL_7;
            } else if (has_nesw_bottom(above)) {
               tile->piece_tl = (side == TILE_SOLID) ?
               PIECE_NESW_FLOOR_7 : PIECE_NESW_WALL_7;
            } else if (above == TILE_SOLID && side == TILE_SOLID) {
               tile->piece_tl = (corner == TILE_SOLID) ?
               PIECE_SOLID_INSIDE_1 : PIECE_SOLID_CORNER_1;
            } else if (above != TILE_SOLID && side == TILE_SOLID) {
               tile->piece_tl = PIECE_SOLID_FLOOR_2;
            } else if (above == TILE_SOLID && side != TILE_SOLID) {
               tile->piece_tl = PIECE_SOLID_WALL_1;
            } else {
               tile->piece_tl = PIECE_SOLID_FLOOR_1;
            }
         }

         // [solid tiles]
         // Determine top-right piece
         {
            uint8_t above = get_tile(x, y-1)->collision;
            uint8_t side = get_tile(x+1, y)->collision;
            uint8_t corner = get_tile(x+1, y-1)->collision;

            if (looks_solid(above)) above = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (has_slope_top(above))
               above = TILE_SOLID;
            if (has_slope_top(corner) ||
            corner == TILE_NWSE_2 || corner == TILE_NWSE_2_BG)
               corner = TILE_SOLID;
            if (side == TILE_NWSE_1 || side == TILE_NWSE_1_BG) {
               side = TILE_SOLID;
               corner = TILE_EMPTY;
            }

            if (has_nwse_bottom(above)) {
               tile->piece_tr = (side == TILE_SOLID) ?
               PIECE_NWSE_FLOOR_8 : PIECE_NWSE_WALL_8;
            } else if (has_nesw_bottom(above)) {
               tile->piece_tr = (side == TILE_SOLID) ?
               PIECE_NESW_FLOOR_8 : PIECE_NESW_WALL_8;
            } else if (above == TILE_SOLID && side == TILE_SOLID) {
               tile->piece_tr = (corner == TILE_SOLID) ?
               PIECE_SOLID_INSIDE_2 : PIECE_SOLID_CORNER_2;
            } else if (above != TILE_SOLID && side == TILE_SOLID) {
               tile->piece_tr = PIECE_SOLID_FLOOR_3;
            } else if (above == TILE_SOLID && side != TILE_SOLID) {
               tile->piece_tr = PIECE_SOLID_WALL_3;
            } else {
               tile->piece_tr = PIECE_SOLID_FLOOR_4;
            }
         }

         // [solid tiles]
         // Determine bottom-left piece
         {
            uint8_t below = get_tile(x, y+1)->collision;
            uint8_t side = get_tile(x-1, y)->collision;
            uint8_t corner = get_tile(x-1, y+1)->collision;

            if (looks_solid(below)) below = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (has_nwse_top(side) || has_nesw(side))
               side = TILE_SOLID;

            if (below == TILE_SOLID && side == TILE_SOLID) {
               tile->piece_bl = (corner == TILE_SOLID) ?
               PIECE_SOLID_INSIDE_3 : PIECE_SOLID_CORNER_3;
            } else if (below != TILE_SOLID && side == TILE_SOLID) {
               tile->piece_bl = PIECE_SOLID_CEILING_2;
            } else if (below == TILE_SOLID && side != TILE_SOLID) {
               tile->piece_bl = PIECE_SOLID_WALL_2;
            } else {
               tile->piece_bl = PIECE_SOLID_CEILING_1;
            }
         }

         // [solid tiles]
         // Determine bottom-right piece
         {
            uint8_t below = get_tile(x, y+1)->collision;
            uint8_t side = get_tile(x+1, y)->collision;
            uint8_t corner = get_tile(x+1, y+1)->collision;

            if (looks_solid(below)) below = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (has_nesw_top(side) || has_nwse(side))
               side = TILE_SOLID;

            if (below == TILE_SOLID && side == TILE_SOLID) {
               tile->piece_br = (corner == TILE_SOLID) ?
               PIECE_SOLID_INSIDE_4 : PIECE_SOLID_CORNER_4;
            } else if (below != TILE_SOLID && side == TILE_SOLID) {
               tile->piece_br = PIECE_SOLID_CEILING_3;
            } else if (below == TILE_SOLID && side != TILE_SOLID) {
               tile->piece_br = PIECE_SOLID_WALL_4;
            } else {
               tile->piece_br = PIECE_SOLID_CEILING_4;
            }
         }

         // Done!
         break;

      // 2:1 NWSE slope? (left half)
      case TILE_NWSE_1:
      case TILE_NWSE_1_BG:
         // [2:1 NWSE slope (left half)]
         // Determine top-left piece
         {
            uint8_t side = get_tile(x-1, y)->collision;
            //if (side == TILE_NESW_1 || side == TILE_NESW_1_BG)
            if (has_nesw_top(side))
               side = TILE_SOLID;

            if (looks_solid(side))
               tile->piece_tl = PIECE_NWSE_FLOOR_1;
            else
               tile->piece_tl = PIECE_NWSE_WALL_1;
         }

         // [2:1 NWSE slope (left half)]
         // Determine top-right piece
         {
            uint8_t side = get_tile(x+1, y)->collision;
            if (has_nwse(side) || has_nesw_top(side))
               side = TILE_SOLID;

            if (looks_solid(side))
               tile->piece_tr = PIECE_NWSE_FLOOR_2;
            else
               tile->piece_tr = PIECE_NWSE_WALL_2;
         }

         // [2:1 NWSE slope (left half)]
         // Determine bottom-left piece
         {
            uint8_t side = get_tile(x-1, y)->collision;
            uint8_t bottom = get_tile(x, y+1)->collision;
            uint8_t corner = get_tile(x-1, y+1)->collision;

            if (looks_solid(bottom)) bottom = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (has_nwse_top(side) || has_nesw(side))
               side = TILE_SOLID;

            if (bottom == TILE_SOLID && side == TILE_SOLID) {
               tile->piece_bl = (corner == TILE_SOLID) ?
               PIECE_NWSE_FLOOR_3 : PIECE_NWSE_CORNER_1;
            } else if (bottom == TILE_SOLID) {
               tile->piece_bl = PIECE_NWSE_WALL_3;
            } else {
               tile->piece_bl = (side == TILE_SOLID) ?
               PIECE_NWSE_CEILING_2 : PIECE_NWSE_CEILING_1;
            }
         }

         // [2:1 NWSE slope (left half)]
         // Determine bottom-right piece
         {
            uint8_t side = get_tile(x+1, y)->collision;
            uint8_t bottom = get_tile(x, y+1)->collision;
            uint8_t corner = get_tile(x+1, y+1)->collision;

            if (looks_solid(bottom)) bottom = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (has_slope(side))
               side = TILE_SOLID;
            if (has_nwse_top(corner))
               corner = TILE_SOLID;

            if (bottom == TILE_SOLID) {
               tile->piece_br = (side == TILE_SOLID)
               ? PIECE_NWSE_FLOOR_4 : PIECE_NWSE_WALL_4;
            } else {
               tile->piece_br = (side == TILE_SOLID) ?
               PIECE_NWSE_CEILING_3 : PIECE_NWSE_CEILING_4;
            }
         }

         break;

      // 2:1 NWSE slope? (right half)
      case TILE_NWSE_2:
      case TILE_NWSE_2_BG:
         // [2:1 NWSE slope (right half)]
         // Determine bottom-left piece
         {
            uint8_t side = get_tile(x-1, y)->collision;
            uint8_t bottom = get_tile(x, y+1)->collision;
            uint8_t corner = get_tile(x-1, y+1)->collision;

            if (looks_solid(bottom)) bottom = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (has_slope(side))
               side = TILE_SOLID;

            if (bottom == TILE_SOLID && side == TILE_SOLID) {
               tile->piece_bl = (corner == TILE_SOLID) ?
               PIECE_NWSE_FLOOR_5 : PIECE_NWSE_CORNER_2;
            } else if (bottom == TILE_SOLID) {
               tile->piece_bl = PIECE_NWSE_WALL_5;
            } else {
               tile->piece_bl = (side == TILE_SOLID) ?
               PIECE_NWSE_CEILING_6 : PIECE_NWSE_CEILING_5;
            }
         }

         // [2:1 NWSE slope (right half)]
         // Determine bottom-right piece
         {
            uint8_t side = get_tile(x+1, y)->collision;
            uint8_t bottom = get_tile(x, y+1)->collision;

            if (looks_solid(bottom)) bottom = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;

            if (has_slope(side))
               side = TILE_SOLID;

            if (bottom == TILE_SOLID) {
               tile->piece_br = (side == TILE_SOLID) ?
               PIECE_NWSE_FLOOR_6 : PIECE_NWSE_WALL_6;
            } else {
               tile->piece_br = PIECE_NWSE_CEILING_7;
            }
         }

         break;

      // 2:1 NESW slope? (right half)
      case TILE_NESW_1:
      case TILE_NESW_1_BG:
         // [2:1 NESW slope (right half)]
         // Determine top-left piece
         {
            uint8_t side = get_tile(x-1, y)->collision;
            if (has_nesw(side) || has_nwse_top(side))
               side = TILE_SOLID;

            if (looks_solid(side))
               tile->piece_tl = PIECE_NESW_FLOOR_1;
            else
               tile->piece_tl = PIECE_NESW_WALL_1;
         }

         // [2:1 NESW slope (right half)]
         // Determine top-right piece
         {
            uint8_t side = get_tile(x+1, y)->collision;
            if (side == TILE_NWSE_1 || side == TILE_NWSE_1_BG)
               side = TILE_SOLID;

            if (looks_solid(side))
               tile->piece_tr = PIECE_NESW_FLOOR_2;
            else
               tile->piece_tr = PIECE_NESW_WALL_2;
         }

         // [2:1 NESW slope (right half)]
         // Determine bottom-left piece
         {
            uint8_t side = get_tile(x-1, y)->collision;
            uint8_t bottom = get_tile(x, y+1)->collision;

            if (looks_solid(bottom)) bottom = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;

            if (has_slope(side))
               side = TILE_SOLID;

            if (bottom == TILE_SOLID) {
               tile->piece_bl = (side == TILE_SOLID) ?
               PIECE_NESW_FLOOR_3 : PIECE_NESW_WALL_3;
            } else {
               tile->piece_bl = (side == TILE_SOLID) ?
               PIECE_NESW_CEILING_2 : PIECE_NESW_CEILING_1;
            }
         }

         // [2:1 NESW slope (right half)]
         // Determine bottom-right piece
         {
            uint8_t side = get_tile(x+1, y)->collision;
            uint8_t bottom = get_tile(x, y+1)->collision;
            uint8_t corner = get_tile(x+1, y+1)->collision;

            if (looks_solid(bottom)) bottom = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (has_nesw_top(side) || has_nwse(side))
               side = TILE_SOLID;

            if (bottom == TILE_SOLID && side == TILE_SOLID) {
               tile->piece_br = (corner == TILE_SOLID) ?
               PIECE_NESW_FLOOR_4 : PIECE_NESW_CORNER_1;
            } else if (bottom == TILE_SOLID) {
               tile->piece_br = PIECE_NESW_WALL_4;
            } else {
               tile->piece_br = (side == TILE_SOLID) ?
               PIECE_NESW_CEILING_3 : PIECE_NESW_CEILING_4;
            }
         }

         break;

      // 2:1 NESW slope? (left half)
      case TILE_NESW_2:
      case TILE_NESW_2_BG:
         // [2:1 NWSE slope (left half)]
         // Determine bottom-left piece
         {
            uint8_t side = get_tile(x-1, y)->collision;
            uint8_t bottom = get_tile(x, y+1)->collision;
            uint8_t corner = get_tile(x-1, y+1)->collision;

            if (looks_solid(bottom)) bottom = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (has_slope(side))
               side = TILE_SOLID;
            if (has_nesw_top(corner))
               corner = TILE_SOLID;

            if (bottom == TILE_SOLID) {
               tile->piece_bl = (side == TILE_SOLID || corner == TILE_SOLID)
               ? PIECE_NESW_FLOOR_5 : PIECE_NESW_WALL_5;
            } else {
               tile->piece_bl = PIECE_NESW_CEILING_5;
            }
         }

         // [2:1 NWSE slope (left half)]
         // Determine bottom-right piece
         {
            uint8_t side = get_tile(x+1, y)->collision;
            uint8_t bottom = get_tile(x, y+1)->collision;
            uint8_t corner = get_tile(x+1, y+1)->collision;

            if (looks_solid(bottom)) bottom = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (has_slope(side))
               side = TILE_SOLID;

            if (bottom == TILE_SOLID && side == TILE_SOLID) {
               tile->piece_br = (corner == TILE_SOLID) ?
               PIECE_NESW_FLOOR_6 : PIECE_NESW_CORNER_2;
            } else if (bottom == TILE_SOLID) {
               tile->piece_br = PIECE_NESW_WALL_6;
            } else {
               tile->piece_br = (side == TILE_SOLID) ?
               PIECE_NESW_CEILING_6 : PIECE_NESW_CEILING_7;
            }
         }

         break;

      // Bridge?
      case TILE_BRIDGE:
         // [2:1 bridge]
         // Determine top-left piece
         {
            if (get_tile(x-1, y)->collision == TILE_BRIDGE)
               tile->piece_tl = PIECE_BRIDGE_INSIDE_1;
            else
               tile->piece_tl = PIECE_BRIDGE_SIDE_1;
         }

         // [2:1 bridge]
         // Determine top-right piece
         {
            if (get_tile(x+1, y)->collision == TILE_BRIDGE)
               tile->piece_tr = PIECE_BRIDGE_INSIDE_2;
            else
               tile->piece_tr = PIECE_BRIDGE_SIDE_2;
         }

         break;

      // Conveyor belt to the left?
      case TILE_BELT_LEFT:
         // [Conveyor belt left]
         // Determine top-left piece
         {
            if (get_tile(x-1, y)->collision == TILE_BELT_LEFT)
               tile->piece_tl = PIECE_BELT_L_MIDDLE;
            else
               tile->piece_tl = PIECE_BELT_L_LEFT;
         }

         // [Conveyor belt left]
         // Determine top-right piece
         {
            if (get_tile(x+1, y)->collision == TILE_BELT_LEFT)
               tile->piece_tr = PIECE_BELT_L_MIDDLE;
            else
               tile->piece_tr = PIECE_BELT_L_RIGHT;
         }

         // [Conveyor belt left]
         // Determine bottom-left piece
         {
            if (get_tile(x-1, y)->collision == TILE_BELT_LEFT)
               tile->piece_bl = PIECE_BELT_B_INSIDE_1;
            else
               tile->piece_bl = PIECE_BELT_B_LEFT;
         }

         // [Conveyor belt left]
         // Determine bottom-right piece
         {
            if (get_tile(x+1, y)->collision == TILE_BELT_LEFT)
               tile->piece_br = PIECE_BELT_B_INSIDE_2;
            else
               tile->piece_br = PIECE_BELT_B_RIGHT;
         }

         break;

      // Conveyor belt to the right?
      case TILE_BELT_RIGHT:
         // [Conveyor belt right]
         // Determine top-left piece
         {
            if (get_tile(x-1, y)->collision == TILE_BELT_RIGHT)
               tile->piece_tl = PIECE_BELT_R_MIDDLE;
            else
               tile->piece_tl = PIECE_BELT_R_LEFT;
         }

         // [Conveyor belt right]
         // Determine top-right piece
         {
            if (get_tile(x+1, y)->collision == TILE_BELT_RIGHT)
               tile->piece_tr = PIECE_BELT_R_MIDDLE;
            else
               tile->piece_tr = PIECE_BELT_R_RIGHT;
         }

         // [Conveyor belt right]
         // Determine bottom-left piece
         {
            if (get_tile(x-1, y)->collision == TILE_BELT_RIGHT)
               tile->piece_bl = PIECE_BELT_B_INSIDE_1;
            else
               tile->piece_bl = PIECE_BELT_B_LEFT;
         }

         // [Conveyor belt right]
         // Determine bottom-right piece
         {
            if (get_tile(x+1, y)->collision == TILE_BELT_RIGHT)
               tile->piece_br = PIECE_BELT_B_INSIDE_2;
            else
               tile->piece_br = PIECE_BELT_B_RIGHT;
         }

         break;

      // Nothing to do...
      default:
         break;
   }

   // Now do the same, but for background (low layer) pieces
   switch (tile->collision) {
      // Background walls
      case TILE_SOLID:
      case TILE_HIDDEN:
      case TILE_EMPTY_BG:
      case TILE_NWSE_1_BG:
      case TILE_NWSE_2_BG:
      case TILE_NESW_1_BG:
      case TILE_NESW_2_BG:
         // [background wall tiles]
         // Determine top-left piece
         {
            uint8_t above = get_tile(x, y-1)->collision;
            uint8_t side = get_tile(x-1, y)->collision;
            uint8_t corner = get_tile(x-1, y-1)->collision;

            if (looks_solid(above)) above = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (!(looks_solid(tile->collision) &&
            !has_bgwall(above) && !has_bgwall(side))) {
               if (has_bgwall(above) || above == TILE_SOLID)
                  above = TILE_EMPTY_BG;
               if (has_bgwall(side) || side == TILE_SOLID)
                  side = TILE_EMPTY_BG;
               if (has_bgwall(corner) || corner == TILE_SOLID)
                  corner = TILE_EMPTY_BG;

               if (above == TILE_EMPTY_BG && side == TILE_EMPTY_BG) {
                  tile->bgpiece_tl = (corner == TILE_EMPTY_BG) ?
                  PIECE_BGWALL_INSIDE_1 : PIECE_BGWALL_CORNER_1;
               } else if (above == TILE_EMPTY_BG && side != TILE_EMPTY_BG) {
                  tile->bgpiece_tl = PIECE_BGWALL_SIDE_1;
               } else {
                  tile->bgpiece_tl = (side == TILE_EMPTY_BG) ?
                  PIECE_BGWALL_TOP_2 : PIECE_BGWALL_TOP_1;
               }
            }
         }

         // [background wall tiles]
         // Determine top-right piece
         {
            uint8_t above = get_tile(x, y-1)->collision;
            uint8_t side = get_tile(x+1, y)->collision;
            uint8_t corner = get_tile(x+1, y-1)->collision;

            if (looks_solid(above)) above = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (!(looks_solid(tile->collision) &&
            !has_bgwall(above) && !has_bgwall(side))) {
               if (has_bgwall(above) || above == TILE_SOLID)
                  above = TILE_EMPTY_BG;
               if (has_bgwall(side) || side == TILE_SOLID)
                  side = TILE_EMPTY_BG;
               if (has_bgwall(corner) || corner == TILE_SOLID)
                  corner = TILE_EMPTY_BG;

               if (above == TILE_EMPTY_BG && side == TILE_EMPTY_BG) {
                  tile->bgpiece_tr = (corner == TILE_EMPTY_BG) ?
                  PIECE_BGWALL_INSIDE_2 : PIECE_BGWALL_CORNER_2;
               } else if (above == TILE_EMPTY_BG && side != TILE_EMPTY_BG) {
                  tile->bgpiece_tr = PIECE_BGWALL_SIDE_3;
               } else {
                  tile->bgpiece_tr = (side == TILE_EMPTY_BG) ?
                  PIECE_BGWALL_TOP_3 : PIECE_BGWALL_TOP_4;
               }
            }
         }

         // [background wall tiles]
         // Determine bottom-left piece
         {
            uint8_t below = get_tile(x, y+1)->collision;
            uint8_t side = get_tile(x-1, y)->collision;
            uint8_t corner = get_tile(x-1, y+1)->collision;

            if (looks_solid(below)) below = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (!(looks_solid(tile->collision) &&
            !has_bgwall(below) && !has_bgwall(side))) {
               if (has_bgwall(below) || below == TILE_SOLID)
                  below = TILE_EMPTY_BG;
               if (has_bgwall(side) || side == TILE_SOLID)
                  side = TILE_EMPTY_BG;
               if (has_bgwall(corner) || corner == TILE_SOLID)
                  corner = TILE_EMPTY_BG;

               if (below == TILE_EMPTY_BG && side == TILE_EMPTY_BG) {
                  tile->bgpiece_bl = (corner == TILE_EMPTY_BG) ?
                  PIECE_BGWALL_INSIDE_3 : PIECE_BGWALL_CORNER_3;
               } else if (below == TILE_EMPTY_BG && side != TILE_EMPTY_BG) {
                  tile->bgpiece_bl = PIECE_BGWALL_SIDE_2;
               } else {
                  tile->bgpiece_bl = (side == TILE_EMPTY_BG) ?
                  PIECE_BGWALL_BOTTOM_2 : PIECE_BGWALL_BOTTOM_1;
               }
            }
         }

         // [background wall tiles]
         // Determine bottom-right piece
         {
            uint8_t below = get_tile(x, y+1)->collision;
            uint8_t side = get_tile(x+1, y)->collision;
            uint8_t corner = get_tile(x+1, y+1)->collision;

            if (looks_solid(below)) below = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (!(looks_solid(tile->collision) &&
            !has_bgwall(below) && !has_bgwall(side))) {
               if (has_bgwall(below) || below == TILE_SOLID)
                  below = TILE_EMPTY_BG;
               if (has_bgwall(side) || side == TILE_SOLID)
                  side = TILE_EMPTY_BG;
               if (has_bgwall(corner) || corner == TILE_SOLID)
                  corner = TILE_EMPTY_BG;

               if (below == TILE_EMPTY_BG && side == TILE_EMPTY_BG) {
                  tile->bgpiece_br = (corner == TILE_EMPTY_BG) ?
                  PIECE_BGWALL_INSIDE_4 : PIECE_BGWALL_CORNER_4;
               } else if (below == TILE_EMPTY_BG && side != TILE_EMPTY_BG) {
                  tile->bgpiece_br = PIECE_BGWALL_SIDE_4;
               } else {
                  tile->bgpiece_br = (side == TILE_EMPTY_BG) ?
                  PIECE_BGWALL_BOTTOM_3 : PIECE_BGWALL_BOTTOM_4;
               }
            }
         }

         break;

      // Thin floor on background wall
      case TILE_FLOOR_BG:
         // [thin floor w/background]
         // Determine top-left piece
         {
            uint8_t above = get_tile(x, y-1)->collision;
            uint8_t side = get_tile(x-1, y)->collision;

            if (looks_solid(above)) above = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;

            if (has_bgwall(above) || above == TILE_SOLID)
               above = TILE_EMPTY_BG;
            if (side != TILE_FLOOR_BG &&
            (has_bgwall(side) || side == TILE_SOLID))
               side = TILE_EMPTY_BG;

            if (side == TILE_FLOOR_BG) {
               tile->bgpiece_tl = (above == TILE_EMPTY_BG) ?
               PIECE_BGFLOOR_INSIDE_2 : PIECE_BGFLOOR_TOP_2;
            } else if (side == TILE_EMPTY_BG) {
               tile->bgpiece_tl = (above == TILE_EMPTY_BG) ?
               PIECE_BGFLOOR_INSIDE_1 : PIECE_BGFLOOR_TOP_1;
            } else {
               tile->bgpiece_tl = (above == TILE_EMPTY_BG) ?
               PIECE_BGFLOOR_SIDE_2 : PIECE_BGFLOOR_SIDE_1;
            }
         }

         // [thin floor w/background]
         // Determine top-right piece
         {
            uint8_t above = get_tile(x, y-1)->collision;
            uint8_t side = get_tile(x+1, y)->collision;

            if (looks_solid(above)) above = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;

            if (has_bgwall(above) || above == TILE_SOLID)
               above = TILE_EMPTY_BG;
            if (side != TILE_FLOOR_BG &&
            (has_bgwall(side) || side == TILE_SOLID))
               side = TILE_EMPTY_BG;

            if (side == TILE_FLOOR_BG) {
               tile->bgpiece_tr = (above == TILE_EMPTY_BG) ?
               PIECE_BGFLOOR_INSIDE_3 : PIECE_BGFLOOR_TOP_3;
            } else if (side == TILE_EMPTY_BG) {
               tile->bgpiece_tr = (above == TILE_EMPTY_BG) ?
               PIECE_BGFLOOR_INSIDE_4 : PIECE_BGFLOOR_TOP_4;
            } else {
               tile->bgpiece_tr = (above == TILE_EMPTY_BG) ?
               PIECE_BGFLOOR_SIDE_4 : PIECE_BGFLOOR_SIDE_3;
            }
         }

         // [thin floor w/background]
         // Determine bottom-left piece
         {
            uint8_t below = get_tile(x, y+1)->collision;
            uint8_t side = get_tile(x-1, y)->collision;
            uint8_t corner = get_tile(x-1, y+1)->collision;

            if (looks_solid(below)) below = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (has_bgwall(below) || below == TILE_SOLID)
               below = TILE_EMPTY_BG;
            if (has_bgwall(side) || side == TILE_SOLID)
               side = TILE_EMPTY_BG;
            if (has_bgwall(corner) || corner == TILE_SOLID)
               corner = TILE_EMPTY_BG;

            if (below == TILE_EMPTY_BG && side == TILE_EMPTY_BG) {
               tile->bgpiece_bl = (corner == TILE_EMPTY_BG) ?
               PIECE_BGWALL_INSIDE_3 : PIECE_BGWALL_CORNER_3;
            } else if (below == TILE_EMPTY_BG && side != TILE_EMPTY_BG) {
               tile->bgpiece_bl = PIECE_BGWALL_SIDE_2;
            } else {
               tile->bgpiece_bl = (side == TILE_EMPTY_BG) ?
               PIECE_BGWALL_BOTTOM_2 : PIECE_BGWALL_BOTTOM_1;
            }
         }

         // [thin floor w/background]
         // Determine bottom-right piece
         {
            uint8_t below = get_tile(x, y+1)->collision;
            uint8_t side = get_tile(x+1, y)->collision;
            uint8_t corner = get_tile(x+1, y+1)->collision;

            if (looks_solid(below)) below = TILE_SOLID;
            if (looks_solid(side)) side = TILE_SOLID;
            if (looks_solid(corner)) corner = TILE_SOLID;

            if (has_bgwall(below) || below == TILE_SOLID)
               below = TILE_EMPTY_BG;
            if (has_bgwall(side) || side == TILE_SOLID)
               side = TILE_EMPTY_BG;
            if (has_bgwall(corner) || corner == TILE_SOLID)
               corner = TILE_EMPTY_BG;

            if (below == TILE_EMPTY_BG && side == TILE_EMPTY_BG) {
               tile->bgpiece_br = (corner == TILE_EMPTY_BG) ?
               PIECE_BGWALL_INSIDE_4 : PIECE_BGWALL_CORNER_4;
            } else if (below == TILE_EMPTY_BG && side != TILE_EMPTY_BG) {
               tile->bgpiece_br = PIECE_BGWALL_SIDE_4;
            } else {
               tile->bgpiece_br = (side == TILE_EMPTY_BG) ?
               PIECE_BGWALL_BOTTOM_3 : PIECE_BGWALL_BOTTOM_4;
            }
         }

         break;

      // Nothing to do...
      default:
         break;
   }
}

//***************************************************************************
// update_tile_clues
// Like update_tile_pieces, but for audiovideo mode.
//---------------------------------------------------------------------------
// param x: X coordinate of tile
// param y: Y coordinate of tile
//***************************************************************************

// Quick inline stuff again
static inline int solid_from_top(TileType type) {
   return type != TILE_EMPTY && type != TILE_EMPTY_BG &&
          type != TILE_HIDDEN;
}

static inline int solid_from_bottom(TileType type) {
   return type != TILE_EMPTY && type != TILE_EMPTY_BG &&
          type != TILE_FLOOR_BG && type != TILE_BRIDGE &&
          type != TILE_HIDDEN;
}

void update_tile_clues(uint16_t x, uint16_t y) {
   // Get the tile to update
   LevelTile *tile = get_tile(x, y);

   // Clear all clues by default
   tile->piece_tl = AV_EMPTY;
   tile->piece_tr = AV_EMPTY;
   tile->piece_bl = AV_EMPTY;
   tile->piece_br = AV_EMPTY;

   // Determine what to do depending on the solidness of the tile
   switch (tile->collision) {
      // Solid stuff
      case TILE_SOLID:
      case TILE_BELT_LEFT:
      case TILE_BELT_RIGHT: {
         if (!solid_from_top(get_tile(x-1, y)->collision) &&
         !solid_from_top(get_tile(x, y-1)->collision))
            tile->piece_tl = AV_GAP;
         if (!solid_from_top(get_tile(x+1, y)->collision) &&
         !solid_from_top(get_tile(x, y-1)->collision))
            tile->piece_tr = AV_GAP;
         if (!solid_from_top(get_tile(x-1, y)->collision) &&
         !solid_from_top(get_tile(x, y+1)->collision))
            tile->piece_bl = AV_CEIL;
         if (!solid_from_top(get_tile(x+1, y)->collision) &&
         !solid_from_top(get_tile(x, y+1)->collision))
            tile->piece_br = AV_CEIL;
      } break;

      // Solid from the top only
      case TILE_FLOOR_BG:
      case TILE_BRIDGE: {
         if (!solid_from_top(get_tile(x-1, y)->collision))
            tile->piece_tl = AV_GAP;
         if (!solid_from_top(get_tile(x+1, y)->collision))
            tile->piece_tr = AV_GAP;
      } break;

      // Slopes
      case TILE_NWSE_1:
      case TILE_NWSE_1_BG: {
         TileType other = get_tile(x-1, y-1)->collision;
         if (other != TILE_NWSE_2 && other != TILE_NWSE_2_BG)
            tile->piece_tl = AV_SLOPE;
      } break;

      case TILE_NWSE_2:
      case TILE_NWSE_2_BG: {
         TileType other = get_tile(x+1, y+1)->collision;
         if (other != TILE_NWSE_1 && other != TILE_NWSE_1_BG)
            tile->piece_br = AV_SLOPE;
      } break;

      case TILE_NESW_1:
      case TILE_NESW_1_BG: {
         TileType other = get_tile(x+1, y-1)->collision;
         if (other != TILE_NESW_2 && other != TILE_NESW_2_BG)
            tile->piece_tr = AV_SLOPE;
      } break;

      case TILE_NESW_2:
      case TILE_NESW_2_BG: {
         TileType other = get_tile(x-1, y+1)->collision;
         if (other != TILE_NESW_1 && other != TILE_NESW_1_BG)
            tile->piece_bl = AV_SLOPE;
      } break;

      // Leave everything else purposefully empty
      default:
         break;
   }
}

//***************************************************************************
// retrieve_level_anim
// Retrieves an animation from the level-specific graphics set.
//---------------------------------------------------------------------------
// param id: ID of animation to retrieve
// return: pointer to animation
//***************************************************************************

const AnimFrame *retrieve_level_anim(LevelAnim id) {
   return anim_level[id];
}

//***************************************************************************
// retrieve_liquid_anim
// Retrieves the current animation frame for liquid hazards.
//---------------------------------------------------------------------------
// return: pointer to animation
//***************************************************************************

const AnimFrame *retrieve_liquid_anim(void) {
   return anim_liquid.frame;
}

//***************************************************************************
// set_spawn_point
// Sets where the player spawns when a new life is started. Initially set to
// the level beginning, can be changed by triggering checkpoints.
//---------------------------------------------------------------------------
// param x: horizontal coordinate
// param y: vertical coordinate
//***************************************************************************

void set_spawn_point(int32_t x, int32_t y) {
   level.spawn_x = x;
   level.spawn_y = y;
}

//***************************************************************************
// reset_spawn_point
// Resets the spawn point. Used to tell the game to force to reset the
// checkpoint.
//***************************************************************************

void reset_spawn_point(void) {
   level.spawn_x = UINT32_MAX;
   level.spawn_y = UINT32_MAX;
}

//***************************************************************************
// get_level_theme
// Gets the theme used for this level.
//---------------------------------------------------------------------------
// return: current theme (see THEME_*)
//***************************************************************************

Theme get_level_theme(void) {
   return level.theme;
}

//***************************************************************************
// get_level_size
// Gets the size of the level.
//---------------------------------------------------------------------------
// param width: where to store width
// param height: where to store height
//***************************************************************************

void get_level_size(uint16_t *width, uint16_t *height) {
   *width = level.width;
   *height = level.height;
}

//***************************************************************************
// get_num_blue_stars
// Returns how many blue stars were collected.
//---------------------------------------------------------------------------
// return: count of collected blue stars
//***************************************************************************

size_t get_num_blue_stars(void) {
   return num_blue_stars;
}

//***************************************************************************
// get_total_blue_stars
// Returns how many blue stars existed when the level started.
//---------------------------------------------------------------------------
// return: total number of blue stars
//***************************************************************************

size_t get_total_blue_stars(void) {
   return total_blue_stars;
}

//***************************************************************************
// inc_blue_stars
// Increments the amount of collected blue stars.
//***************************************************************************

void inc_blue_stars(void) {
   num_blue_stars++;

   // Oh, reached the limit?
   if (num_blue_stars == settings.blue_limit) {
      num_blue_stars = 0;
      add_item_bonus(1);
   }
}
