//***************************************************************************
// "main.c"
// Core code of the program. Includes the entry point (where the game starts)
// as well as some common functionality used about everywhere in the program.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <SDL2/SDL.h>
#include "main.h"
#include "cutscene.h"
#include "editor.h"
#include "file.h"
#include "framerate.h"
#include "ingame.h"
#include "input.h"
#include "langselect.h"
#include "launcher.h"
#include "loading.h"
#include "logo.h"
#include "menu.h"
#include "options.h"
#include "options_gameplay.h"
#include "options_input.h"
#include "options_mouse.h"
#include "options_sound.h"
#include "options_video.h"
#include "reader.h"
#include "recording.h"
#include "settings.h"
#include "savegame.h"
#include "saveselect.h"
#include "scene.h"
#include "sound.h"
#include "text.h"
#include "title.h"
#include "video.h"

// For Windows-specific stuff
#ifdef _WIN32
#include <windows.h>
#endif

// For Linux-specific stuff
#ifdef __linux__
#include <unistd.h>
#include <sys/types.h>
#endif

// Structure holding pointers to the callbacks for each game mode
// 'init' is used when entering the game mode (allocate resources)
// 'run' is used to process each logic frame (frameskip already handled)
// 'draw' is used to update the screen contents (frameskip already handled)
// 'deinit' is used when quitting the game mode (deallocate resources)
static struct {
   void (*init)(void);        // Initialization
   void (*run)(void);         // Process a logic frame
   void (*draw)(void);        // Update the screen
   void (*deinit)(void);      // Deinitialization
} game_mode_func[NUM_GAMEMODES] = {
   { init_ingame, run_ingame, draw_ingame, deinit_ingame },
   { init_cutscene, run_cutscene, draw_cutscene, deinit_cutscene },
   { init_editor, run_editor, draw_editor, deinit_editor },
   { init_title, run_title, draw_title, deinit_title },
   { init_saveselect, run_saveselect, draw_saveselect, deinit_saveselect },
   { init_options, run_options, draw_options, deinit_options },
   { init_options_gameplay,run_options_gameplay,draw_options_gameplay,NULL },
   { init_options_video, run_options_video, draw_options_video,
      deinit_options_video },
   { init_options_sound, run_options_sound, draw_options_sound,
      deinit_options_sound },
   { init_options_input, run_options_input, draw_options_input,
      deinit_options_input },
   { init_options_input, run_options_input, draw_options_input,
      deinit_options_input },
   { init_options_mouse, run_options_mouse, draw_options_mouse, NULL },
   { init_logo, run_logo, draw_logo, deinit_logo },
   { init_language_select, run_language_select, draw_language_select,
      deinit_language_select }
};

// Information used to keep track of the game modes
GameMode game_mode;           // Currently running game mode
GameMode prev_game_mode;      // Game mode run before this one
GameMode next_game_mode;      // Game mode to switch to

// Private function prototypes
static void parse_arguments(int, char **);
static void clean_up(void);

// Used to determine which timing method we're using right now
// No, we can't use settings.timing since that may include TIMING_AUTO (this
// variable contains the *real* method we're using). We can't just query the
// power state every time since that slows down things really hard (hint: on
// the oergolaptop it results in slideshow framerate, despite TIMING_SPEED
// and TIMING_POWER both running at 60FPS).
static Timing timing;

// This flag gets set whenever the user wants to quit (e.g. by pressing
// Alt+F4 or something). The program must quit *immediately* when this flag
// gets set (i.e. don't ask confirmation or anything, just *quit*).
static int quit;

// On MinGW we don't really need SDL_main...
#ifdef __MINGW32__
#undef main
#endif

//***************************************************************************
//***************************************************************************

#ifdef _WIN32
static void log_callback(void *userdata, int category,
SDL_LogPriority priority, const char *message) {
   // We don't care about these, sorry
   (void) userdata;
   (void) category;
   (void) priority;
   (void) message;
}
#endif

//***************************************************************************
// main
// Where the program starts.
//---------------------------------------------------------------------------
// param argc: number of command line arguments
// param argv: list of command line argumnets
// return: EXIT_SUCCESS when quitting normally
//         EXIT_FAILURE if there was an error
//***************************************************************************

int main(int argc, char **argv) {
#ifdef _WIN32
   // Used to work around a timing bug on some systems
   // Disabled since the reason it was originally tried for turned out to be
   // something else entirely, but if there's somebody out there with one of
   // those rare systems where this *can* happen, this can be reenabled.
   /*
   {
      DWORD mask, mask2;
      GetProcessAffinityMask(GetCurrentProcess(), &mask, &mask2);
      for (uint32_t i = 1; i != 0; i <<= 1)
         if (mask & i) { mask &= i; break; }
      if (i) SetThreadAffinityMask(GetCurrentThread(), mask);
   }
   */

   // Override SDL's logging function since it steals back the console :|
   // We need to do this before SDL even gets initialized because that
   // alone causes logging to happen and that will break what we just
   // did right now
   SDL_LogSetOutputFunction(log_callback, NULL);
#endif

   // Testing the console
   //printf("*** SANDWICH-MAAAAAAAAAAAAAAN ***\n");

   // DON'T RUN THE GAME AS ROOT >_<
   // We can't localize this string since we want this to run before
   // absolutely anything else (otherwise we actually risk the game doing
   // some operation that can break the system if it was manipulated)
   // To-do: find the equivalent check for Windows
#ifdef __linux__
   if (geteuid() == 0)
   {
      // Drop all privileges again if we can, JUST IN CASE
      // The needless check is to shut up GCC, we really don't care if this
      // fails because we're just trying to reduce risks as much as possible.
      // It's not like we can't do anything if we're root...
      if (!setuid(getuid())) { /* shut up, GCC */ }

      // Yell at the player
      SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, "Error",
         "(-_-;)\n\n"
         "You're trying to run the game as root. If something goes\n"
         "wrong, YOU CAN BREAK THE SYSTEM, and this game doesn't need\n"
         "root privileges. Sorry, but not going to allow that.\n\n"
         "Try again without running as root.", NULL);
      return EXIT_FAILURE;
   }
#endif

   // Initialize the settings
   set_default_err_msg();
   init_config();

   // Sol only uses the query timers, so we don't need to increase the
   // system timer for framerate to be good enough
   // Not using override priority because I suppose a power user may have a
   // good reason to mess with it (maybe if this setting backfires?)
   //SDL_SetHint(SDL_HINT_TIMER_RESOLUTION, "0");

#ifdef DEBUG
   // Enabling the internal IME renderer in the debug build. Normally the
   // game will render the placeholder text by itself, but this is here just
   // in case it's broken or something. Note that this can be overriden by
   // setting environment variables anyway.
   //SDL_SetHint(SDL_HINT_IME_INTERNAL_EDITING, "1");
   //SDL_SetHint("SDL_IME_INTERNAL_EDITING", "1");
#endif

   // Ugly hack but because I could not gain console access properly at all:
   // check if the console is owned by us, and if so, get rid of it (if we
   // don't own it then it means we are meant to use it). Results in the
   // console flashing for a split second but eh (we'll just let players
   // think it's a bogus window from the initialization process)
   // Note that this is *not* done with a debug build since we definitely
   // want console output no matter what in that case.
#ifdef _WIN32
#ifndef DEBUG
   {
      HWND console = GetConsoleWindow();
      if (console != NULL) {
         DWORD pid;
         GetWindowThreadProcessId(console, &pid);
         if (pid == GetCurrentProcessId())
            FreeConsole();
      }
   }
#endif
#endif

   // Initialize the game
   init_filesystem(argv[0]);
   parse_arguments(argc, argv);
   load_mods();
   init_languages();
   load_config();
   load_language();
   if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_TIMER))
      abort_program(ERR_INITSDL, NULL);
   init_video();
   init_reader();
   init_framerate();
   init_input();
   init_sound();
   load_font();
   load_scene_list();

   // Determine initial timing method
   timing = settings.timing;
   if (timing == TIMING_AUTO) {
      timing = (SDL_GetPowerInfo(NULL, NULL) == SDL_POWERSTATE_ON_BATTERY) ?
      TIMING_POWER : TIMING_SPEED;
   }

   // Set the default mappings for the joystick if we didn't already
   // We need to do this after detecting the joystick to account for mappings
   // (why doesn't everybody just use a standard layout so I don't have to
   // bother with this and can just hardcode the default values?)
#ifdef DEBUG
   fputs("Initial event poll\n", stderr);
#endif
   update_program();
   set_joystick_defaults();

#ifdef DEBUG
   fputs("Initialization complete! :)\n", stderr);
#endif

   // Test
   //abort_program(ERR_COORDVAL, "SÉGA MÉGA DRÍVE¡¡");

#ifdef DEBUG
   // Show SDL current version
   {
      SDL_version sdlver;
      SDL_GetVersion(&sdlver);
      printf("SDL version %d.%d.%d-%d\n",
         sdlver.major, sdlver.minor, sdlver.patch,
         SDL_GetRevisionNumber());
   }
#endif

   // Get savegame loaded
   load_savegame();

   // Prevent screen saver from triggering
   SDL_DisableScreenSaver();

   // Set initial game mode
   if (settings.record) {
      if (settings.language == -1)
         settings.language = 0;
      settings.level_select = 1;
      switch_game_mode(GAMEMODE_SAVESLOT);
   } else if (settings.attract) {
      if (settings.language == -1)
         settings.language = 0;
      switch_game_mode(GAMEMODE_TITLE);
   } else if (settings.language == -1) {
      switch_game_mode(GAMEMODE_LANGSELECT);
   } else {
      switch_game_mode(GAMEMODE_LOGO);
   }
   game_mode = NUM_GAMEMODES;

   // Main loop of the program
   quit = 0;
   while (!quit) {
      // Set the new game mode as the current one
      prev_game_mode = game_mode;
      game_mode = next_game_mode;
      next_game_mode = NUM_GAMEMODES;

      // Huh, quit game?
      if (game_mode == GAMEMODE_QUIT)
         break;

      // Start up game mode
      set_cursor(CURSOR_NONE);
      set_reinit_menu_func(NULL);
      if (game_mode_func[game_mode].init != NULL)
         game_mode_func[game_mode].init();

      // Main loop
      reset_framerate();
      reset_input();
      while (!quit && next_game_mode == NUM_GAMEMODES) {
         // Update subsystems
         unsigned num_frames = update_program();

         // Update logic as needed
         // Don't update anything if the program has been paused through
         // the debug pause key, though
         if (!settings.pause) {
            while (num_frames-- && next_game_mode == NUM_GAMEMODES) {
               if (game_mode_func[game_mode].run != NULL)
                  game_mode_func[game_mode].run();
               update_input();
               update_fading();

               // These two should be one-off but they don't have any release
               // events to go with, nor are they handled with the usual
               // system, so...
               // To-do: handle mouse differently?
               //input.cursor.wheel_up = 0;
               //input.cursor.wheel_down = 0;
               handle_mouse_button(-1, 0);
               handle_mouse_button(-2, 0);
            }
         }

         // If the program is paused we still need to make sure to keep
         // processing new input, otherwise the program will hang as it's
         // impossible to unpause it.
         else {
            update_input();
         }

         // Update the screen output
         reset_reader();
         if (game_mode_func[game_mode].draw != NULL)
            game_mode_func[game_mode].draw();
         draw_fading();
         draw_cursor();
      }

      // Make direct quitting behave the same as quitting the usual way
      if (quit) next_game_mode = GAMEMODE_QUIT;

      // Shut down game mode
      if (is_recording())
         stop_recording();
      if (game_mode_func[game_mode].deinit != NULL)
         game_mode_func[game_mode].deinit();
   }

   // Quit program
   quit_program();

   // Shut up compiler! (this point will never be reached since quit_program
   // won't return, but the compiler will not get it)
   return EXIT_SUCCESS;
}

//***************************************************************************
// update_program
// Waits until the next frame starts. Updates I/O and such.
//---------------------------------------------------------------------------
// return: how many frames to process (always > 0)
//***************************************************************************

unsigned update_program(void) {
   // Update screen
   update_video();
   update_reader();
   if (timing == TIMING_SPEED)
      SDL_Delay(1);

   // Wait until next frame starts
   unsigned num_frames = 0;
   while (!num_frames) {
      if (timing == TIMING_POWER) SDL_Delay(1);
      num_frames = get_num_frames();
   }

   // If we're recording, disable frameskip
   // Game will slow down, but the final video will be full framerate :)
   if (is_recording())
      num_frames = 1;

   // Release mouse wheel "buttons"
   //handle_mouse_button(3, 0);
   //handle_mouse_button(4, 0);

   // Poll all events as needed
   SDL_Event event;
   while (SDL_PollEvent(&event)) {
      switch (event.type) {
         // Window focus?
         case SDL_WINDOWEVENT:
            if (event.window.event == SDL_WINDOWEVENT_FOCUS_GAINED)
               handle_gain_focus();
            else if (event.window.event == SDL_WINDOWEVENT_FOCUS_LOST)
               handle_lose_focus();
            break;

         // Keyboard changed status?
         case SDL_KEYDOWN:
         case SDL_KEYUP:
            handle_key_event(event.key.keysym.scancode,
               event.type == SDL_KEYDOWN);
            break;

         // Mouse moved?
         case SDL_MOUSEMOTION:
            if (SDL_GetRelativeMouseMode()) {
               handle_mouse_motion(event.motion.xrel, event.motion.yrel, 1);
            } else {
               handle_mouse_motion(event.motion.x, event.motion.y,
               event.motion.windowID);
            }
            break;

         // Mouse button?
         case SDL_MOUSEBUTTONUP:
         case SDL_MOUSEBUTTONDOWN:
            if (!event.button.windowID)
               break;
            handle_mouse_button(event.button.button,
               event.type == SDL_MOUSEBUTTONDOWN);
            break;

         // Mouse wheel?
         case SDL_MOUSEWHEEL:
            if (!event.wheel.windowID)
               break;
            if (event.wheel.y > 0)
               handle_mouse_button(-1, 1);
            if (event.wheel.y < 0)
               handle_mouse_button(-2, 1);
            break;

         // Joystick axis?
         case SDL_JOYAXISMOTION:
            handle_joy_axis(event.jaxis.axis, event.jaxis.value,
               event.jaxis.which);
            break;

         // Joystick hat?
         case SDL_JOYHATMOTION:
            handle_joy_hat(event.jhat.hat, event.jhat.value,
               event.jhat.which);
            break;

         // Joystick button?
         case SDL_JOYBUTTONDOWN:
         case SDL_JOYBUTTONUP:
            handle_joy_button(event.jbutton.button,
               event.type == SDL_JOYBUTTONDOWN,
               event.jbutton.which);
            break;

         // Joystick connected?
         case SDL_JOYDEVICEADDED:
            add_joystick(event.jdevice.which);
            break;

         // Joystick disconnected?
         case SDL_JOYDEVICEREMOVED:
            remove_joystick(event.jdevice.which);
            break;

         // Controller axis?
         case SDL_CONTROLLERAXISMOTION:
            handle_controller_axis(event.caxis.axis,
               event.caxis.value, event.caxis.which);
            break;

         // Controller button?
         case SDL_CONTROLLERBUTTONDOWN:
         case SDL_CONTROLLERBUTTONUP:
            handle_controller_button(event.cbutton.button,
               event.type == SDL_CONTROLLERBUTTONDOWN);
            break;

         // Text entered?
         case SDL_TEXTINPUT:
            handle_ime_entered(event.text.text);
            break;

         // Entering text? (see IMEs, etc.)
         case SDL_TEXTEDITING:
            // To-do: handle it
            break;

         // Quit program?
         case SDL_QUIT:
            quit = 1;
            break;

         // Unhandled, ignore
         default:
            break;
      }
   }

   // Return how many frames to process
   return num_frames;
}

//***************************************************************************
// switch_game_mode
// Tells the main loop to switch to a different game mode. The game mode will
// be switched as soon as the logical frame stops being processed and the
// current mode deinitialization function will be called when done.
//---------------------------------------------------------------------------
// param mode: game mode to switch to
//***************************************************************************

void switch_game_mode(GameMode mode) {
   next_game_mode = mode;
}

//***************************************************************************
// quit_program
// Performs all the clean up functions then quits the game. Use this to quit
// the program under normal circumstances (i.e. not due to an error).
//---------------------------------------------------------------------------
// THIS FUNCTION DOES NOT RETURN
//***************************************************************************

void quit_program(void) {
   // Store settings
   save_config();

   // Shut down subsystems
   clean_up();

   // Shut down program
#if defined(DEBUG) || defined(DUMB_MEM)
   unload_language();
#endif
   exit(EXIT_SUCCESS);
}

//***************************************************************************
// abort_program
// Call this when there's a fatal error and the program can't continue. It
// shuts down all subsystems and shows an error message, then quits the
// program indicating a failure status.
//---------------------------------------------------------------------------
// param code: error code
// param details: string with extra information (e.g. filename), can be NULL
//                if the specific error doesn't need it
//---------------------------------------------------------------------------
// THIS FUNCTION DOES NOT RETURN. On GCC, the "noreturn" attribute is used so
// the compiler knows to not generate useless return code (it isn't needed
// though, it just prevents bloat, feel free to remove it if you want).
//***************************************************************************

#ifdef __GNUC__
void __attribute__((__noreturn__))
abort_program (ErrCode code, const char *details) {
#else
void abort_program(ErrCode code, const char *details) {
#endif
   // RACE CONDITION! If we're loading, the main thread is probably trying to
   // access the video system to show the progress bar! We need to halt it so
   // we can safely shut down the game.
   // Note that this call does nothing if we're not loading.
   abort_loading();

   // Ack, no language selected yet! Resort to the default language then
   // (only if we don't have an error message already)
   if (text.error[code] == NULL && settings.language == -1) {
      settings.language = 0;
      load_language();
   }

   // Shut down subsystems
   clean_up();

   // Buffer where the message will be stored
   // I know, fixed size, but we want this to be in the stack in case the
   // heap isn't working properly (which can happen if the error is
   // ERR_NOMEMORY), though this may be somewhat useless if we can't invoke
   // the message box...
#define MSGBUF_SIZE 0x1000
   char msgbuf[MSGBUF_SIZE];

   // No description available for this error code?
   if (text.error[code] == NULL) {
      if (details == NULL)
         snprintf(msgbuf, MSGBUF_SIZE,
         "(error code %u)", code);
      else
         snprintf(msgbuf, MSGBUF_SIZE,
         "(error code %u: \"%s\")", code, details);

      msgbuf[MSGBUF_SIZE-1] = '\0';
   }

   // No parameter?
   else if (details == NULL) {
      strncpy(msgbuf, text.error[code], MSGBUF_SIZE);
      msgbuf[MSGBUF_SIZE-1] = '\0';
   }

   // With parameter?
   else {
      // Where the error message string starts
      const char *src = text.error[code];

      // Get where {param} is
      const char *paramptr = strstr(src, "{param}");

      // Copy first part of the string
      char *dest = msgbuf;
      size_t left = MSGBUF_SIZE-1;
      while (left > 0 && *src != '\0' && src != paramptr) {
         *dest++ = *src++;
         left--;
      }

      // Copy parameter
      if (*src != '\0') {
         src += 7;
         const char *paramsrc = details;
         while (left > 0 && *paramsrc != '\0') {
            *dest++ = *paramsrc++;
            left--;
         }
      }

      // Copy second part of string
      while (left > 0 && *src != '\0') {
         *dest++ = *src++;
         left--;
      }

      // Add null terminator
      *dest = '\0';
   }

   // Not needed anymore
#undef MSGBUF_SIZE

   // Show messagebox
   int error = 0;
   error = SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR,
      text.error_title, msgbuf, NULL);

#ifndef DEBUG
   // If we couldn't show a message box for whatever reason (meaning
   // something is seriously wrong), attempt outputting the error to stderr
   // instead. This requires the game to have been run from a console though,
   // so it's kind of limited...
   // To-do: adapt this code to use the "{param}" convention
   if (error) {
#else
   // In the debug build we *always* output the error to stderr. However,
   // this triggers warnings in the compiler since the variable pretty much
   // goes unused (it's set but its value never checked), so work around
   // this by telling the compiler this is done on purpose.
   (void) error;
#endif
#ifndef _WIN32
      fputs("\x1B[1;7;31m ", stderr);
      fprintf(stderr, "%s", text.error_title);
      fputs(" \x1B[0;1;31m", stderr);
#else
      fputc('[', stderr);
      fprintf(stderr, "%s", text.error_title);
      fputc(']', stderr);
#endif
      fputc(' ', stderr);
      fputs(msgbuf, stderr);
      fputc('\n', stderr);
#ifndef _WIN32
      fputs("\x1B[m", stderr);
#endif
#ifndef DEBUG
   }
#endif

   // Shut down program
#if defined(DEBUG) || defined(DUMB_MEM)
   unload_language();
#endif
   exit(EXIT_FAILURE);
}

//***************************************************************************
// parse_arguments [internal]
// Parses the command line arguments.
//---------------------------------------------------------------------------
// param argc: number of command line arguments
// param argv: list of command line argumnets
//***************************************************************************

static void parse_arguments(int argc, char **argv) {
   // If no arguments are passed at all, assume we should use the launcher
   Args *args = NULL;
   if (argc <= 1) {
#if HAS_LAUNCHER
      args = launcher();
      if (args == NULL)
         return;
#else
      return;
#endif
   }

   // Determine which argument list to use
   // This is so the launcher can pass arguments to us
   char **arglist;
   int argoffset;

   if (args != NULL) {
      argc = args->count;
      arglist = args->list;
      argoffset = 0;
   } else {
      arglist = argv;
      argoffset = 1;
   }

   // To see what arguments we're receiving
   // Not on Windows due to missing z modifier (debugging is done on Ubuntu
   // anyway, so not really an issue)
#ifdef DEBUG
#ifndef _WIN32
   /*
   if (args != NULL) {
      for (size_t i = 0; i < args->count; i++)
         printf("Argument #%zu: \"%s\"\n", i+1, args->list[i]);
   } else {
      for (size_t i = 1; i < argc; i++)
         printf("Argument #%zu: \"%s\"\n", i, argv[i]);
   }
   */

   for (int i = argoffset; i < argc; i++)
      printf("Argument #%d: \"%s\"\n", i, arglist[i]);
#endif
#endif

   // These flags are used to keep track of options that show information to
   // the user (more specifically: --help, --version).
   int show_help = 0;         // Show command line help (--help)
   int show_version = 0;      // Show program version (--version)
   int open_config = 0;       // Open configuration dir (--config)

   // This flag remains set as long as we are allowed to check for options.
   // The argument -- tells the program to stop parsing options there and
   // assume everything else is a filename.
   int check_options = 1;

   // If there was an error while parsing this flag will be set
   int error = 0;

   // Check all arguments
   for (int i = argoffset; i < argc; i++) {
      // Get the pointer to this argument. Assigning it to this variable so
      // it's easier to read later in the code >_>
      const char *arg = arglist[i];

      // Is it an option?
      if (check_options && arg[0] == '-') {
         // Enable debug mode?
         if (!strcmp(arg, "-d") || !strcmp(arg, "--debug"))
            settings.debug = 1;

         // Enable recording?
         else if (!strcmp(arg, "-r") || !strcmp(arg, "--record"))
            settings.record = 1;

         // Enable attract mode?
         else if (!strcmp(arg, "-a") || !strcmp(arg, "--attract"))
            settings.attract = 1;

         // Force one-switch?
         else if (!strcmp(arg, "-1") || !strcmp(arg, "--oneswitch")) {
            settings.one_switch = 1;
            settings.override_one_switch = 1;
         } else if (!strcmp(arg, "-n1") || !strcmp(arg, "--nooneswitch")) {
            settings.one_switch = 0;
            settings.override_one_switch = 1;
         }

         // Force mouse-switch?
         else if (!strcmp(arg, "-m") || !strcmp(arg, "--mouseswitch")) {
            settings.mouse_switch = 1;
            settings.override_mouse_switch = 1;
         } else if (!strcmp(arg, "-nm") || !strcmp(arg, "--nomouseswitch")) {
            settings.mouse_switch = 0;
            settings.override_mouse_switch = 1;
         }

         // Set screenreader method?
         else if (!strcmp(arg, "-sr1") || !strcmp(arg, "--screenreader1")) {
            settings.reader = READER_NATIVE;
            settings.override_reader = 1;
         } else if (!strcmp(arg, "-sr2") || !strcmp(arg, "--screenreader2")) {
            settings.reader = READER_CLIPBOARD;
            settings.override_reader = 1;
         } else if (!strcmp(arg, "-sr3") || !strcmp(arg, "--screenreader3")) {
            settings.reader = READER_TITLEBAR;
            settings.override_reader = 1;
         } else if (!strcmp(arg, "-sr4") || !strcmp(arg, "--screenreader4")) {
            settings.reader = READER_STDOUT;
            settings.override_reader = 1;
         } else if (!strcmp(arg, "-sr0") || !strcmp(arg, "--noscreenreader")) {
            settings.reader = READER_NONE;
            settings.override_reader = 1;
         }

         // Force fullscreen/windowed?
         else if (!strcmp(arg, "-f") || !strcmp(arg, "--fullscreen")) {
            settings.fullscreen = 1;
            settings.override_fullscreen = 1;
         } else if (!strcmp(arg, "-w") || !strcmp(arg, "--windowed")) {
            settings.fullscreen = 0;
            settings.override_fullscreen = 1;
         }

         // Disable subsystems?
         else if (!strcmp(arg, "-sw") || !strcmp(arg, "--software"))
            settings.no_hwaccel = 1;
         else if (!strcmp(arg, "-ns") || !strcmp(arg, "--nosound"))
            settings.no_sound = 1;
         else if (!strcmp(arg, "-nj") || !strcmp(arg, "--nojoystick"))
            settings.no_joystick = 1;
         else if (!strcmp(arg, "-njx") || !strcmp(arg, "--nojoystickaxis"))
            settings.no_joystick_axis = 1;

         // Render using audiovideo mode?
         else if (!strcmp(arg, "-av") || !strcmp(arg, "--audiovideo"))
            settings.audiovideo = 1;

         // Safe mode?
         else if (!strcmp(arg, "-s") || !strcmp(arg, "--safe")) {
            settings.fullscreen = 0;
            settings.override_fullscreen = 1;
            settings.no_hwaccel = 1;
            settings.no_sound = 1;
            settings.no_joystick = 1;
         }

         // Enable cheats?
         else if (!strcmp(arg, "-l") || !strcmp(arg, "--levelselect"))
            settings.level_select = 1;
         else if (!strcmp(arg, "-nd") || !strcmp(arg, "--nodamage"))
            settings.no_damage = 1;
         else if (!strcmp(arg, "-nh") || !strcmp(arg, "--nohealth"))
            settings.no_health = 1;
         else if (!strcmp(arg, "-xh") || !strcmp(arg, "--extrahealth"))
            settings.extra_health = 1;
         else if (!strcmp(arg, "-ne") || !strcmp(arg, "--noenemies"))
            settings.no_enemies = 1;
         else if (!strcmp(arg, "-nch") || !strcmp(arg, "--nocheckpoints"))
            settings.no_checkpoints = 1;
         else if (!strcmp(arg, "-nf") || !strcmp(arg, "--nofriction"))
            settings.no_friction = 1;
         else if (!strcmp(arg, "-p") || !strcmp(arg, "--pinball"))
            settings.pinball = 1;
         else if (!strcmp(arg, "-fv") || !strcmp(arg, "--forcevirtual"))
            settings.force_virtual = 1;
         else if (!strcmp(arg, "-cga") || !strcmp(arg, "--cgamode"))
            settings.cga_mode = 1;
         else if (!strcmp(arg, "-b") || !strcmp(arg, "--beeper"))
            settings.beeper = 1;

         // Color filters?
         else if (!strcmp(arg, "-cb1") || !strcmp(arg, "--colorblind1"))
            settings.color_mode = COLOR_TYPE1;
         else if (!strcmp(arg, "-cb2") || !strcmp(arg, "--colorblind2"))
            settings.color_mode = COLOR_TYPE2;
         else if (!strcmp(arg, "-cb3") || !strcmp(arg, "--colorblind3"))
            settings.color_mode = COLOR_MONO;
         /*else if (!strcmp(arg, "-cb4") || !strcmp(arg, "--colorblind4"))
            settings.color_mode = COLOR_DICHRO;*/
         else if (!strcmp(arg, "-n") || !strcmp(arg, "--negative"))
            settings.color_mode = COLOR_NEGATIVE;
         else if (!strcmp(arg, "-rb") || !strcmp(arg, "--rgbswap"))
            settings.color_mode = COLOR_RGBSWAP;
         else if (!strcmp(arg, "-lv1") || !strcmp(arg, "--lowvision1"))
            settings.color_mode = COLOR_LOVISION1;
         else if (!strcmp(arg, "-lv2") || !strcmp(arg, "--lowvision2"))
            settings.color_mode = COLOR_LOVISION2;
         else if (!strcmp(arg, "-tv") || !strcmp(arg, "--tvmode"))
            settings.color_mode = COLOR_TVMODE;

         // Contrast settings?
         else if (!strcmp(arg, "-lc") || !strcmp(arg, "--lowcontrast")) {
            settings.contrast = CONTRAST_LOW;
            settings.override_contrast = 1;
         } else if (!strcmp(arg, "-hc") || !strcmp(arg, "--highcontrast")) {
            settings.contrast = CONTRAST_HIGH;
            settings.override_contrast = 1;
         }

         // Show information to the user?
         else if (!strcmp(arg, "-h") || !strcmp(arg, "--help") ||
         !strcmp(arg, "-?"))
            show_help = 1;
         else if (!strcmp(arg, "-v") || !strcmp(arg, "--version"))
            show_version = 1;
         else if (!strcmp(arg, "-c") || !strcmp(arg, "--config"))
            open_config = 1;

         // Stop parsing options?
         else if (!strcmp(arg, "--"))
            check_options = 0;

         // Invalid option?
         else {
            fprintf(stderr, "Error: unknown option \"%s\"\n", arg);
            error = 1;
         }
      }

      // Is it a filename?
      else {
         // Append it to the mod list
         add_mod(arg);
      }
   }

   // If the launcher passed us an argument list now we should get rid of it
   // <Sik> ...except for the part where it seems PhysicsFS doesn't keep its
   // own copy of the strings. Ugh. Eh, just keep this around and let it
   // leak, it's a tiny amount and it'll be freed when the program quits (and
   // we need to keep it around for practically up to that point anyway)
   /*if (args != NULL)
      free_args(args);*/

   // If there was an error while parsing stop now
   if (error)
      exit(EXIT_FAILURE);

   // Make sure that the configuration file exists
   // Also to work around a bug where the configuration gets reset...
   // We need to load the languages to prevent the language setting from
   // getting reset, btw
   init_languages();
   load_config();
   save_config();

   // Show command line help?
   if (show_help) {
      // Output a list of options
      printf(
         "Usage: %s [options] [filenames]\n"
         "\n"
         "Options:\n"
         "  -d   --debug ........... Enable debug mode\n"
         "\n"
         "  -f   --fullscreen ...... Force fullscreen mode\n"
         "  -w   --windowed ........ Force windowed mode\n"
         "  -1   --oneswitch ....... Force one-switch mode on\n"
         "  -n1  --nooneswitch ..... Force one-switch mode off\n"
         "  -m   --mouseswitch ..... Force mouse-switch mode on\n"
         "  -nm  --nomouseswitch ... Force mouse-switch mode off\n"
         "\n"
         "  -sw  --software ........ Disable hardware acceleration\n"
         "  -ns  --nosound ......... Disable sound support\n"
         "  -nj  --nojoystick ...... Disable joystick support\n"
         "  -s   --safe ............ Safe mode\n"
         "\n"
         "  -lc  --lowcontrast ..... Low contrast colors\n"
         "  -hc  --highcontrast .... High contrast colors\n"
         "  -cb1 --colorblind1 ..... Red-green colorblindness simulation\n"
         "  -cb2 --colorblind2 ..... Blue-yellow colorblindness simulation\n"
         "  -cb3 --colorblind3 ..... Monochrome colorblindness simulation\n"
         "  -lv1 --lowvision1 ...... Mild low vision simulation\n"
         "  -lv2 --lowvision2 ...... Strong low vision simulation\n"
         "\n"
         "  -h   --help ............ Show command line help\n"
         "  -v   --version ......... Show program version\n"
         "  -c   --config .......... Show configuration files\n"
         "\n"
         "  Pass -- to stop parsing options.\n"
         "  See manual for cheats.\n",
         argv[0]);

      // Stop program here
      quit_program();
   }

   // Show program version?
   if (show_version) {
      printf("%d.%02d%s\n", VER_MAJOR, VER_MINOR, VER_SUFFIX);
      quit_program();
   }

   // Open directory with configuration files?
   if (open_config) {
      // Get directory where the configuration files reside
      char *dir = get_config_dir();

      // Allocate memory to generate the command
      char *cmd = (char *) malloc(strlen(dir) + 0x40);
      if (cmd == NULL)
         abort_program(ERR_NOMEMORY, NULL);

      // Open folder!
      // And yes, explorer.exe returns *1* on success, not 0
      // <Sik> Disabling until I can confirm for sure
#ifdef _WIN32
      /*
      if (run_program_with_param("explorer", dir) != 1)
         abort_program(ERR_UNKNOWN, NULL);
      */
      run_program_with_param("explorer", dir);
#else
      if (run_program_with_param("xdg-open", dir) != 0)
         abort_program(ERR_UNKNOWN, NULL);
#endif

      // Done here
      free(cmd);
      free(dir);
      quit_program();
   }
}

//***************************************************************************
// clean_up [internal]
// Calls all clean-up functions when quitting the program (either normally or
// because there was a fatal error).
//---------------------------------------------------------------------------
// THE LANGUAGE TEXT IS NOT UNLOADED. This is because abort_program needs it.
// Make sure to unload the text data manually (then again, in practice the OS
// will do it for you when shutting down the process...).
//***************************************************************************

static void clean_up(void) {
   // Shut down all subsystems
   unload_scene_list();
   unload_font();
   deinit_input();
   deinit_framerate();
   deinit_reader();
   deinit_video();
   deinit_sound();
   deinit_languages();
   deinit_filesystem();
}

//***************************************************************************
// run_program
// Executes a program. It works the same as the system function from the
// standard library, it's just here to work around platform differences.
//---------------------------------------------------------------------------
// param cmd: command to execute (UTF-8)
// return: whatever system would return
//***************************************************************************

int run_program(const char *cmd) {
#ifdef DEBUG
   // Was used to see if it was getting the wrong command line on Windows
   // Turns out that START is broken and can't handle quotes... ("fixed" it
   // by invoking EXPLORER directly instead)
   printf("RUNNING: %s\n", cmd);
#endif

#ifdef _WIN32
   // On Windows system doesn't use UTF-8 (and setlocale can't change this),
   // so we need to resort to _wsystem which uses UTF-16 instead. This means
   // converting the string to UTF-16 too. Ack.
   wchar_t *wcmd = utf8_to_utf16(cmd);
   int result = _wsystem(wcmd);
   free(wcmd);
   return result;
#else
   // Everywhere else we can just assume system is using UTF-8, so pass the
   // command to it directly
   return system(cmd);
#endif
}

//***************************************************************************
// run_program_with_param
// Like the function above, but taking an argument, used to escape quotes and
// such as needed.
//---------------------------------------------------------------------------
// param cmd: command to execute (UTF-8)
// param param: parameter for the command (UTF-8)
// return: whatever system would return
//***************************************************************************

int run_program_with_param(const char *cmd, const char *param) {
#ifdef __linux__
   // On *nix we need to escape the quotes so account for that
   // Also make sure to use quotes!
   size_t len = 0;
   for (const char *ptr = param; *ptr != '\0'; ptr++)
      len += (*ptr == '\"' || *ptr == '\\') ? 2 : 1;
   char *realparam = (char *) malloc(len + 3);
   if (realparam == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   char *dest = realparam;
   *dest++ = '\"';
   for (const char *src = param; *src != '\0'; src++) {
      if (*src == '\"' || *src == '\\')
         *dest++ = '\\';
      *dest++ = *src;
   }
   *dest++ = '\"';
   *dest++ = '\0';
#else
   // On Windows just use double quotes as-is...
   size_t len = strlen(param) + 2;
   char *realparam = (char *) malloc(len + 3);
   if (realparam == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   sprintf(realparam, "\"%s\"", param);
#endif

   // Now we need to generate a command with the argument in it
   len = strlen(cmd) + strlen(realparam) + 1;
   char *realcmd = (char *) malloc(len + 1);
   if (realcmd == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   sprintf(realcmd, "%s %s", cmd, realparam);

   // Execute command
   int retval = run_program(realcmd);

   // Remember to free up this!
   free(realcmd);
   free(realparam);

   // There, done
   return retval;
}
