//***************************************************************************
// "objects.c"
// Object manager. This is the code that controls all objects in-game.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stdint.h>
#include <stdlib.h>
#include "main.h"
#include "bosses.h"
#include "enemies.h"
#include "hazards.h"
#include "ingame.h"
#include "items.h"
#include "objects.h"
#include "particles.h"
#include "physics.h"
#include "platforms.h"
#include "player.h"
#include "projectile.h"
#include "scenery.h"
#include "settings.h"
#include "video.h"

// Object lists
// There's one list for each object group for two reasons: to speed up object
// interaction (e.g. there's no reason an enemy should query all objects when
// they just want the players!) and drawing order (some objects should always
// appear below other objects regardless of when they were created, etc.).
static struct {
   Object *first;          // First object
   Object *last;           // Last object
} lists[NUM_OBJGROUPS];

// Information about each object type
static const struct {
   ObjGroup group;               // To which group this type belongs
   void (*init)(Object *);       // Function: object initialization
   void (*run)(Object *);        // Function: process logic frame
} const obj_info[NUM_OBJTYPES] = {
   { OBJGROUP_ITEM, NULL, run_dummy_obj },

   { OBJGROUP_PLAYER, init_player, run_player },
   { OBJGROUP_PLAYER, init_player, run_player },
   { OBJGROUP_PLAYER, init_player, run_player },
   { OBJGROUP_PLAYER, init_player, run_player },
   { OBJGROUP_PLAYER, init_player, run_player },

   { OBJGROUP_POWERUP, NULL, NULL },

   { OBJGROUP_ENEMY, init_flamer, run_flamer },
   { OBJGROUP_ENEMY, init_sprayer, run_sprayer },
   { OBJGROUP_ENEMY, init_turret, run_turret },
   { OBJGROUP_ENEMY, init_roller, run_roller },
   { OBJGROUP_ENEMY, init_sprayer, run_grabber },
   { OBJGROUP_ENEMY, init_spider, run_spider },
   { OBJGROUP_ENEMY, init_heater, run_heater },
   { OBJGROUP_ENEMY, init_bomb, run_bomb },

   { OBJGROUP_BOSS, NULL, run_boss1 },
   { OBJGROUP_BOSS, NULL, run_boss2 },
   { OBJGROUP_BOSS, NULL, run_boss3 },
   { OBJGROUP_BOSS, NULL, run_boss4 },
   { OBJGROUP_BOSS, NULL, run_boss5 },
   { OBJGROUP_BOSS6_L1, init_boss6, run_boss6 },

   { OBJGROUP_BOSS, init_boss_spawn, run_boss_spawn },
   { OBJGROUP_BOSS, init_boss_spawn, run_boss_spawn },
   { OBJGROUP_BOSS, init_boss_spawn, run_boss_spawn },
   { OBJGROUP_BOSS, init_boss_spawn, run_boss_spawn },
   { OBJGROUP_BOSS, init_boss_spawn, run_boss_spawn },
   { OBJGROUP_BOSS, init_boss_spawn, run_boss_spawn },

   { OBJGROUP_BOSS, init_driver, run_falling_particle },

   { OBJGROUP_PLATFORM, init_ship, run_ship },
   { OBJGROUP_SCENERY, NULL, NULL },
   { OBJGROUP_HIPARTICLE, NULL, NULL },
   { OBJGROUP_PROJECTILE, init_grenade, run_grenade },
   { OBJGROUP_HAZARD, init_boss4_trailer, run_boss4_trailer },

   { OBJGROUP_BOSS6_L1, init_mrevil_base, run_mrevil_base },
   { OBJGROUP_BOSS6_L1, init_mrevil_flame, run_mrevil_flame },
   { OBJGROUP_BOSS6_L1, NULL, run_mrevil_launcher },
   { OBJGROUP_BOSS6_L1, NULL, run_mrevil_hand },
   { OBJGROUP_BOSS6_L1, NULL, run_mrevil_evil },
   { OBJGROUP_BOSS6_L2, init_missile_in, run_missile_in },
   { OBJGROUP_BOSS6_L3, init_missile_out, run_missile_out },

   { OBJGROUP_PROJECTILE, init_mortar, run_mortar },
   { OBJGROUP_PROJECTILE, init_toxic_gas, run_toxic_gas },
   { OBJGROUP_PROJECTILE, init_fireball, run_toxic_gas },
   { OBJGROUP_PROJECTILE, init_explosion, run_explosion },
   { OBJGROUP_PROJECTILE, init_big_explosion, run_explosion },

   { OBJGROUP_PARTICLE, init_scrap_gear, run_falling_particle },
   { OBJGROUP_PARTICLE, init_scrap_spring, run_falling_particle },
   { OBJGROUP_PARTICLE, init_smoke, run_expiring_particle },
   { OBJGROUP_BOSS6_L3, init_explosion, run_expiring_particle },

   { OBJGROUP_ITEM, init_spring, run_spring },
   { OBJGROUP_ITEM, init_spring, run_spring },
   { OBJGROUP_ITEM, init_balloon, run_balloon },
   { OBJGROUP_PLATFORM, init_pushable, run_pushable },
   { OBJGROUP_PLATFORM, init_platform, run_platform },
   { OBJGROUP_PLATFORM, init_platform, run_horizontal_platform },
   { OBJGROUP_PLATFORM, init_platform, run_vertical_platform },
   { OBJGROUP_PLATFORM, NULL, run_breakable_platform },
   { OBJGROUP_PARTICLE, init_platform_leftover, run_falling_particle },
   { OBJGROUP_PLATFORM, init_obstruction, run_dummy_obj },
   { OBJGROUP_PARTICLE, init_obstruction_leftover, run_falling_particle },
   { OBJGROUP_PLATFORM, init_door, run_door },
   { OBJGROUP_PLATFORM, init_door, run_door },
   { OBJGROUP_PLATFORM, init_door, run_door },
   { OBJGROUP_PLATFORM, init_door, run_door },
   { OBJGROUP_PLATFORM, init_door, run_door },
   { OBJGROUP_PLATFORM, init_door, run_door },
   { OBJGROUP_PLATFORM, init_door, run_door },
   { OBJGROUP_PLATFORM, init_door, run_door },
   { OBJGROUP_SCENERY, init_generator, run_generator },
   { OBJGROUP_SCENERY, init_generator, run_generator },

   { OBJGROUP_ITEM, init_heart, run_heart },
   { OBJGROUP_ITEM, init_invincibility, run_invincibility },
   { OBJGROUP_HIPARTICLE, init_sparkle, run_expiring_particle },
   { OBJGROUP_PARTICLE, init_sparkle, run_falling_particle },
   { OBJGROUP_ITEM, init_checkpoint, run_checkpoint },
   { OBJGROUP_ITEM, init_goal, run_goal },
   { OBJGROUP_ITEM, init_blue_star, run_blue_star },
   { OBJGROUP_SHIELD, NULL, run_shield },
   { OBJGROUP_ITEM, init_shield_item, run_shield_item },
   { OBJGROUP_ITEM, init_powerup_item, run_powerup_item },
   { OBJGROUP_ITEM, init_powerup_item, run_powerup_item },
   { OBJGROUP_ITEM, init_powerup_item, run_powerup_item },
   { OBJGROUP_ITEM, init_powerup_item, run_powerup_item },

   { OBJGROUP_ITEM, init_switch, run_switch },
   { OBJGROUP_ITEM, init_switch, run_switch },
   { OBJGROUP_ITEM, init_switch, run_switch },
   { OBJGROUP_ITEM, init_switch, run_switch },
   { OBJGROUP_ITEM, init_switch, run_switch },
   { OBJGROUP_ITEM, init_switch, run_switch },
   { OBJGROUP_ITEM, init_switch, run_switch },
   { OBJGROUP_ITEM, init_switch, run_switch },

   { OBJGROUP_HAZARD, init_spikes, run_spikes },
   { OBJGROUP_HAZARD, init_spikes, run_spikes },
   { OBJGROUP_HAZARD, init_spikes, run_spikes },
   { OBJGROUP_HAZARD, init_spikes, run_spikes },
   { OBJGROUP_PLATFORM, init_crusher, run_crusher },
   { OBJGROUP_HIHAZARD, init_liquid_hazard, run_liquid_hazard },
   { OBJGROUP_HAZARD, init_bouncing_hazard, run_bouncing_hazard },
   { OBJGROUP_HAZARD, init_fire_hazard, run_fire_hazard },
   { OBJGROUP_HAZARD, init_stalactite, run_stalactite },
   { OBJGROUP_HAZARD, init_coil, run_coil },
   { OBJGROUP_HAZARD, init_coil, run_coil },
   { OBJGROUP_HAZARD, init_buzzsaw, run_buzzsaw },
   { OBJGROUP_HAZARD, init_buzzsaw, run_buzzsaw },
   { OBJGROUP_HAZARD, init_buzzsaw, run_buzzsaw },

   { OBJGROUP_SCENERY, NULL, run_small_scenery },
   { OBJGROUP_SCENERY, NULL, run_big_scenery },
   { OBJGROUP_SCENERY, init_dancer, run_dancer },

   { OBJGROUP_SCENERY, init_warning, NULL },
   { OBJGROUP_SONAR, init_hsonar, run_hsonar },
   { OBJGROUP_SONAR, init_vsonar, run_vsonar }
};

// Colors to be used for each group type in audiovideo mode
// Using AV_EMPTY will make the group not be rendered
static const uint32_t group_av[] = {
   AV_DANGER,           // OBJGROUP_BOSS6_L1
   AV_DANGER,           // OBJGROUP_BOSS6_L2
   AV_GOODIE,           // OBJGROUP_SCENERY     // Dancers need this
   AV_GAP,              // OBJGROUP_PLATFORM
   AV_HAZARD,           // OBJGROUP_HAZARD
   AV_GOODIE,           // OBJGROUP_ITEM
   AV_DANGER,           // OBJGROUP_PROJECTILE
   AV_DANGER,           // OBJGROUP_BOSS
   AV_DANGER,           // OBJGROUP_ENEMY
   AV_EMPTY,            // OBJGROUP_PARTICLE
   AV_EMPTY,            // OBJGROUP_POWERUP
   AV_EMPTY,            // OBJGROUP_PLAYER
   AV_EMPTY,            // OBJGROUP_SHIELD
   AV_EMPTY,            // OBJGROUP_HIPARTICLE
   AV_HAZARD,           // OBJGROUP_HIHAZARD
   AV_DANGER            // OBJGROUP_BOSS6_L3
};

// Where level-specific graphics are stored
static GraphicsSet *gfxset_object = NULL;
static AnimFrame *anim_object[NUM_OB_ANIM];

// IDs of each generic object animation
static const char *anim_names[] = {
   "scrap_gear",           // OB_ANIM_SCRAPGEAR
   "scrap_spring",         // OB_ANIM_SCRAPSPRING
   "mortar",               // OB_ANIM_MORTAR
   "toxic_gas",            // OB_ANIM_TOXICGAS
   "fireball",             // OB_ANIM_FIREBALL
   "explosion",            // OB_ANIM_EXPLOSION
   "big_explosion",        // OB_ANIM_BIGEXPLOSION
   "smoke",                // OB_ANIM_SMOKE

   "spring_idle",          // OB_ANIM_SPRINGIDLE
   "spring_work",          // OB_ANIM_SPRINGWORK
   "superspring_idle",     // OB_ANIM_SSPRINGIDLE
   "superspring_work",     // OB_ANIM_SSPRINGWORK

   "heart_idle",           // OB_ANIM_HEARTIDLE
   "heart_taken",          // OB_ANIM_HEARTTAKEN
   "invincibility",        // OB_ANIM_INVINCIBILITY
   "sparkle",              // OB_ANIM_SPARKLE
   "checkpoint_off",       // OB_ANIM_CHECKOFF
   "checkpoint_on",        // OB_ANIM_CHECKON
   "goal_off",             // OB_ANIM_GOALOFF
   "goal_on",              // OB_ANIM_GOALON
   "blue_star",            // OB_ANIM_BLUESTAR
   "shield",               // OB_ANIM_SHIELD
   "shield_item",          // OB_ANIM_SHIELD_ITEM

   "wings",                // OB_ANIM_WINGS
   "wings_powerup",        // OB_ANIM_WINGS_ITEM
   "spider",               // OB_ANIM_SPIDER
   "spider_powerup",       // OB_ANIM_SPIDER_ITEM
   "hammer",               // OB_ANIM_HAMMER
   "hammer_powerup",       // OB_ANIM_HAMMER_ITEM
   "parasol",              // OB_ANIM_PARASOL
   "parasol_powerup",      // OB_ANIM_PARASOL_ITEM

   "switch_off",           // OB_ANIM_SWITCH_OFF
   "switch_on",            // OB_ANIM_SWITCH_ON

   "generator",            // OB_ANIM_GENERATOR
   "main_generator",       // OB_ANIM_MAINGENERATOR

   "spikes_up",            // OB_ANIM_SPIKES_U
   "spikes_down",          // OB_ANIM_SPIKES_D
   "spikes_left",          // OB_ANIM_SPIKES_L
   "spikes_right",         // OB_ANIM_SPIKES_R
   "floor_coil_idle",      // OB_ANIM_COIL_F_IDLE
   "floor_coil_ready",     // OB_ANIM_COIL_F_READY
   "floor_coil_active",    // OB_ANIM_COIL_F_ACTIVE
   "ceiling_coil_idle",    // OB_ANIM_COIL_C_IDLE
   "ceiling_coil_ready",   // OB_ANIM_COIL_C_READY
   "ceiling_coil_active",  // OB_ANIM_COIL_C_ACTIVE
   "buzzsaw",              // OB_ANIM_BUZZSAW
   "ceiling_buzzsaw",      // OB_ANIM_BUZZSAW_C
   "floor_buzzsaw",        // OB_ANIM_BUZZSAW_F

   "dancer_waiting",       // OB_ANIM_DANCERWAIT
   "dancer_dancing",       // OB_ANIM_DANCERDANCE

   "warning"               // OB_ANIM_WARNING
};

//***************************************************************************
// load_objects
// Loads resources used by miscellaneous objects (e.g. sprites).
//***************************************************************************

void load_objects(void) {
   // Load graphics
   gfxset_object = load_graphics_set("graphics/objects");

   // Get a list of all the animations we need
   for (ObjectAnim i = 0; i < NUM_OB_ANIM; i++)
      anim_object[i] = get_anim(gfxset_object, anim_names[i]);

   // Use an alternate animation for shields in CGA mode because otherwise
   // they can turn invisible against some backgrounds
   if (settings.cga_mode)
      anim_object[OB_ANIM_SHIELD] = get_anim(gfxset_object, "shield_cga");
}

//***************************************************************************
// unload_objects
// Unloads resources used by miscellaneous objects (e.g. sprites).
//***************************************************************************

void unload_objects(void) {
   // Unload graphics
   if (gfxset_object) {
      destroy_graphics_set(gfxset_object);
      gfxset_object = NULL;
   }
}

//***************************************************************************
// init_objects
// Resets the object manager.
//***************************************************************************

void init_objects(void) {
   // Mark all object lists as empty
   for (unsigned i = 0; i < NUM_OBJGROUPS; i++) {
      lists[i].first = NULL;
      lists[i].last = NULL;
   }
}

//***************************************************************************
// add_object
// Creates a new object of the specified type at the specified coordinates.
// The object is returned in case you want to do some more manipulation with
// it (e.g. to keep track of it or to set some extra parameters).
//---------------------------------------------------------------------------
// param type: what type of object is it
// param x: initial horizontal coordinate
// param y: initial vertical coordinate
// param dir: initial direction (0 = right, 1 = left)
// return: pointer to new object
//***************************************************************************

Object *add_object(ObjType type, int32_t x, int32_t y, int dir) {
   // Allocate memory for the new object
   Object *obj = (Object *) malloc(sizeof(Object));
   if (obj == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // I don't feel like typing the entire thing all the time
   ObjGroup group = obj_info[type].group;

   // Do NOT spawn an enemy if the "no enemies" cheat is enabled
   if (group == OBJGROUP_ENEMY && settings.no_enemies)
      type = OBJ_DUMMY;

   // Do NOT spawn checkpoints if the "no checkpoint" cheat is enabled
   // Note that this will *not* reset the spawn point (which matters if you
   // resume a level from a savegame).
   if (type == OBJ_CHECKPOINT && settings.no_checkpoints)
      type = OBJ_DUMMY;

   // Initialize object information
   obj->type = type;
   obj->group = group;
   obj->otherobj = NULL;

   obj->x = x;
   obj->y = y;
   obj->speed = 0;
   obj->gravity = 0;
   obj->base_x = x;
   obj->base_y = y;
   obj->timer = 0;
   obj->invincibility = 0;
   obj->invulnerability = 0;
   obj->health = 1;

   obj->dir = dir;
   obj->crouching = 0;
   obj->jumping = 0;
   obj->active = 0;
   obj->hurt = 0;
   obj->dead = 0;
   obj->shield = 0;

   obj->pinball = 0;

   obj->on_ground = 1;
   obj->on_cliff = 0;
   obj->on_cliff_l = 0;
   obj->on_cliff_r = 0;
   obj->on_wall = 0;
   obj->on_ceiling = 0;
   obj->blocked = 0;
   obj->has_hitbox = 0;
   obj->current = 0;

   obj->frame = NULL;
   obj->last_anim = NULL;

   // Append object to the list of its group
   obj->next = NULL;
   obj->prev = lists[group].last;
   if (lists[group].first)
      lists[group].last->next = obj;
   else
      lists[group].first = obj;
   lists[group].last = obj;

   // Do any type-specific initialization if needed
   if (obj_info[type].init != NULL)
      obj_info[type].init(obj);

   // Keep track of the enemy count!
   if (group == OBJGROUP_ENEMY || group == OBJGROUP_BOSS)
      num_enemies++;

   // Return pointer to new object, in case the caller wants to do something
   // more with it beyond the basic initialization (or in case the caller
   // wants to keep track of it)
   return obj;
}

//***************************************************************************
// get_first_object
// Returns the pointer to the first object of a given group.
//---------------------------------------------------------------------------
// param group: object group to look for
// return: pointer to object, or NULL if none exist
//***************************************************************************

Object *get_first_object(ObjGroup group) {
   // Return first object from the group's list
   return lists[group].first;
}

//***************************************************************************
// run_objects
// Processes a logic frame for all objects
//***************************************************************************

void run_objects(void) {
   // Execute the behavior of all objects
   for (unsigned list_id = 0; list_id < NUM_OBJGROUPS; list_id++)
   for (Object *obj = lists[list_id].first; obj != NULL; ) {
      // Mark object as the one being processed right now
      obj->current = 1;

      // Reduce invulnerability time if the object is invulnerable yet
      if (obj->invulnerability)
         obj->invulnerability--;

      // Update object animation if needed
      if (obj->frame && !is_too_far(obj)) {
         obj->frame_time--;
         if (obj->frame_time == 0) {
            obj->frame = obj->frame->next;
            if (obj->frame) {
               obj->frame_time = obj->frame->duration;
               if (obj->frame->sfx != SFX_NONE) {
                  if (!obj->frame->clue || settings.audiovideo) {
                     play_2d_sfx(obj->frame->sfx,
                     obj->x + obj->frame->sfx_x,
                     obj->y + obj->frame->sfx_y);
                  }
               }
            }
         }
      }

      // Run the logic for this object, if any
      if (obj_info[obj->type].run != NULL)
         obj_info[obj->type].run(obj);

      // Did the object attempt to delete itself? If so, skip any further
      // processing and go with the next object, while properly destroying
      // the current one.
      if (!obj->current) {
         Object *next = obj->next;
         delete_object(obj);
         obj = next;
         continue;
      }

      // Update absolute hit box
      refresh_hitbox(obj);

      // Go for next object, if any
      obj->current = 0;
      obj = obj->next;
   }
}

//***************************************************************************
// draw_object_range [internal]
// Helper function used to draw all objects in a range of lists.
//---------------------------------------------------------------------------
// param min: first object list
// param max: last object list
//***************************************************************************

static void draw_object_range(unsigned min, unsigned max) {
   // Go through all requested object lists
   for (unsigned list_id = min; list_id <= max; list_id++) {
      // Go through all objects in this list
      for (Object *obj = lists[list_id].first; obj != NULL;
      obj = obj->next) {
         // Skip dead players, we'll render them elsewhere so they appear
         // above everything (even the level's high layer).
         if (list_id == OBJGROUP_PLAYER && obj->dead)
            continue;

         // Draw object
         if (!is_too_far(obj))
            draw_single_object(obj);
      }
   }
}

//***************************************************************************
// draw_objects
// Draws all objects on screen.
//***************************************************************************

void draw_objects(void) {
   draw_object_range(OBJGROUP_SCENERY, OBJGROUP_HIHAZARD);
}

//***************************************************************************
// draw_far_objects
// Draws all objects below the foreground.
//***************************************************************************

void draw_far_objects(void) {
   draw_object_range(OBJGROUP_BOSS6_L1, OBJGROUP_BOSS6_L2);
}

//***************************************************************************
// draw_near_objects
// Draws all objects above the foreground.
//***************************************************************************

void draw_near_objects(void) {
   // Go through all dead players
   for (Object *obj = lists[OBJGROUP_PLAYER].first; obj != NULL;
   obj = obj->next) {
      // Skip alive players, d'oh
      if (!obj->dead)
         continue;

      // Draw object
      draw_single_object(obj);
   }

   // Go through the rest of the objects in front of the foreground
   draw_object_range(OBJGROUP_BOSS6_L3, OBJGROUP_BOSS6_L3);
}

//***************************************************************************
// draw_single_object
// Draws the specified object. Use this is for whatever reason you want to
// draw an object individually rather than using draw_objects (dead players
// use this).
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void draw_single_object(const Object *obj) {
   // Nothing to show?
   if (obj->frame == NULL)
      return;

   // Flash if invulnerable
   if (obj->invulnerability & 0x04)
      return;

   // In audiovideo mode we don't render the sprite, instead we render the
   // object as a rectangle with a color that identifies what kind of object
   // is it (not even remotely accurate, but enough to get an idea of what's
   // going on by context)
   if (settings.audiovideo) {
      // We don't draw these objects
      if (group_av[obj->group] == AV_EMPTY)
         return;
      if (!obj->has_hitbox)
         return;

#if 0
      // Determine dimensions of the rectangle
      int32_t x1 = obj->abs_hitbox.x1 - camera.x;
      int32_t y1 = obj->abs_hitbox.y1 - camera.y;
      int32_t x2 = obj->abs_hitbox.x2 - camera.x;
      int32_t y2 = obj->abs_hitbox.y2 - camera.y;

      // Account for the resolution loss
      x1 &= 0xFFFFFFF8;
      y1 &= 0xFFFFFFF8;
      x2 &= 0xFFFFFFF8;
      y2 &= 0xFFFFFFF8;
      x2 += 7;
      y2 += 7;

      // Now draw the object :)
      //fill_rectangle(x1, y1, x2, y2, group_av[obj->group]);

      uint32_t color = group_av[obj->group];
      fill_rectangle(x1, y1, x1+7, y1+7, color);
      fill_rectangle(x2-7, y1, x2, y1+7, color);
      fill_rectangle(x1, y2-7, x1+7, y2, color);
      fill_rectangle(x2-7, y2-7, x2, y2, color);
#else
      // Determine object boundaries
      int32_t min_x = obj->abs_hitbox.x1 - camera.x;
      int32_t min_y = obj->abs_hitbox.y1 - camera.y;
      int32_t max_x = obj->abs_hitbox.x2 - camera.x;
      int32_t max_y = obj->abs_hitbox.y2 - camera.y;

      // Determine ideal coordinate for the point
      int32_t x = screen_cx;
      int32_t y = screen_cy + 0x10;

      // Make sure it doesn't go further than the object boundaries
      // This will put the point in a suitable place for the sonar
      if (x < min_x) x = min_x;
      if (x > max_x) x = max_x;
      if (y < min_y) y = min_y;
      if (y > max_y) y = max_y;

      // Draw the point
      fill_rectangle(x - 4, y - 4, x + 3, y + 3, group_av[obj->group]);
#endif

      return;
   }

   // Get sprite to use
   Sprite *spr = obj->frame->sprite;
   if (spr == NULL)
      return;

   // Get sprite flags for this frame
   // Flip horizontally when the object is facing left (if the sprite
   // was already flipped, then just "unflip" it back)
   unsigned flags = obj->frame->flags;
   if (obj->dir) flags ^= SPR_HFLIP;

   // Determine where to draw the sprite
   int x = obj->x - camera.x;
   if (flags & SPR_HFLIP) {
      x += obj->frame->x_offset;
      x -= spr->width - 1;
   } else
      x -= obj->frame->x_offset;

   int y = obj->y - camera.y;
   if (flags & SPR_VFLIP) {
      y += obj->frame->y_offset;
      y -= spr->height - 1;
   } else
      y -= obj->frame->y_offset;

   // Put sprite on screen
   draw_sprite(spr, x, y, flags);
}

//***************************************************************************
// delete_object
// Deletes an object.
//---------------------------------------------------------------------------
// param obj: pointer to object (becomes invalid after call)
//***************************************************************************

void delete_object(Object *obj) {
   // Okaaay...
   if (obj == NULL)
      return;

   // If it's the current object, make it non-current. This will tell
   // run_objects that the object deleted itself and will be able to retrieve
   // any data it needs from it before the object is actually deleted.
   if (obj->current) {
      obj->current = 0;
      return;
   }

   // Link the previous and next objects together
   if (obj->next) obj->next->prev = obj->prev;
   if (obj->prev) obj->prev->next = obj->next;

   // If this was the first object in the list, set the next object as the
   // first one (if there is no other object, then it will get set to NULL,
   // meaning empty list)
   if (lists[obj->group].first == obj)
      lists[obj->group].first = obj->next;
   if (lists[obj->group].last == obj)
      lists[obj->group].last = obj->prev;

   // Deallocate object
   free(obj);
}

//***************************************************************************
// deinit_objects
// Deinitialize the object manager, freeing all resources as needed.
//***************************************************************************

void deinit_objects(void) {
   // Delete all objects in all the lists
   for (unsigned i = 0; i < NUM_OBJGROUPS; i++) {
      Object *obj = lists[i].first;
      while (obj != NULL) {
         Object *next = obj->next;
         delete_object(obj);
         obj = next;
      }
   }
}

//***************************************************************************
// set_object_anim
// Changes the animation of an object.
//---------------------------------------------------------------------------
// param obj: pointer to object
// param anim: pointer to animation
//***************************************************************************

void set_object_anim(Object *obj, const AnimFrame *anim) {
   // Okaaaaay...
   if (obj == NULL)
      return;

   // Don't restart animation if it's the same!
   if (anim == obj->last_anim)
      return;

   // Set the animation of the object, if any
   // Set the time for the first frame for the animation
   obj->last_anim = anim;
   if (anim == NULL)
      obj->frame = NULL;
   else {
      obj->frame = anim;
      obj->frame_time = obj->frame->duration;
   }

   // If the animation played a sound in the initial frame,
   // we need to ensure it plays
   if (obj->frame != NULL && obj->frame->sfx != SFX_NONE) {
      if (!obj->frame->clue || settings.audiovideo) {
         play_2d_sfx(obj->frame->sfx,
         obj->x + obj->frame->sfx_x,
         obj->y + obj->frame->sfx_y);
      }
   }
}

//***************************************************************************
// retrieve_object_anim
// Retrieves an animation from the miscellaneous objects graphics set.
//---------------------------------------------------------------------------
// param id: ID of animation to retrieve
// return: pointer to animation
//***************************************************************************

const AnimFrame *retrieve_object_anim(ObjectAnim id) {
   return anim_object[id];
}

//***************************************************************************
// is_too_far
// Checks if a process is too far from the camera. Used by some objects to
// determine whether they should perform expensive calculations (such as
// physics) or not. Also particle objects that die when going off-screen.
//---------------------------------------------------------------------------
// return: non-zero if too far, zero if still near enough to bother with
//***************************************************************************

int is_too_far(const Object *obj) {
   // Way too far to be worth bothering?
   if (obj->x < camera.x - 0x100 ||
   obj->x > camera.x + screen_w + 0x100 ||
   obj->y < camera.y - 0x100 ||
   obj->y > camera.y + screen_h + 0x100)
      return 1;

   // Still quite near the player
   return 0;
}
