//***************************************************************************
// "options_input.c"
// Code for the options keyboard and joystick menus.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdio.h>
#include "background.h"
#include "menu.h"
#include "input.h"
#include "reader.h"
#include "settings.h"
#include "sound.h"
#include "text.h"
#include "video.h"

// UI positions
#define INPUT_X1 (screen_cx-144)    // Input X coordinate, 1st section
#define INPUT_X2 (screen_cx)        // Input X coordinate, 2nd section
#define INPUT_X3 (screen_cx+72)     // Input X coordinate, 3rd section
#define INPUT_X4 (screen_cx+143)    // Input X coordinate, right limit

#define INPUT_YDIST 0x1C
#define INPUT_Y1 (INPUT_Y2-INPUT_YDIST)
#define INPUT_Y2 (INPUT_Y3-INPUT_YDIST)
#define INPUT_Y3 (screen_cy-0x0C)
#define INPUT_Y4 (INPUT_Y3+INPUT_YDIST)
#define INPUT_Y5 (INPUT_Y4+INPUT_YDIST)
#define INPUT_Y6 (INPUT_Y5+INPUT_YDIST)

#define INPUT_BX (INPUT_X3)
#define INPUT_BY (INPUT_Y1-INPUT_YDIST)

// List of buttons
enum {
   BUTTON_LEFT_1,       // Run left 1st button
   BUTTON_LEFT_2,       // Run left 2nd button
   BUTTON_RIGHT_1,      // Run right 1st button
   BUTTON_RIGHT_2,      // Run right 2nd button
   BUTTON_DOWN_1,       // Crouch 1st button
   BUTTON_DOWN_2,       // Crouch 2nd button
   BUTTON_UP_1,         // Look up 1st button
   BUTTON_UP_2,         // Look up 2nd button
   BUTTON_JUMP_1,       // Jump 1st button
   BUTTON_JUMP_2,       // Jump 2nd button
   BUTTON_PAUSE_1,      // Pause 1st button (joystick only)
   BUTTON_PAUSE_2,      // Pause 2nd button (joystick only)
   //BUTTON_SAVE,         // Quit menu and keep changes
   BUTTON_BACK          // Quit menu without keeping changes
};

// Where graphics are stored
static GraphicsSet *gfxset = NULL;
static Sprite *spr_button_input[2];
static Sprite *spr_button_back[5];
static Sprite *spr_icons[6][5];

// Names for each icon
static const char *icon_names[] = {
   "left",
   "right",
   "down",
   "up",
   "jump",
   "pause"
};

// Variable used to animate the buttons
static unsigned icon_anim;

// Variable used to tell when the user is entering a new key
static unsigned entering;

// Private function prototypes
static void init_options_input_menu(void);
static void button_enter(void);
static void button_back(void);
static void cancel_callback(void);

//***************************************************************************
// load_options_input
// Loads the assets for the options keyboard and joystick menus. Called while
// the main menu is loading (so everything loads in a bulk).
//***************************************************************************

void load_options_input(void) {
   // Load graphics
   gfxset = load_graphics_set("graphics/options_input");

   // Get some sprites
#define SPR(name) get_sprite(gfxset, name)
   spr_button_input[0] = SPR("button_input_dim");
   spr_button_input[1] = SPR("button_input_lit");
   spr_button_back[0] = SPR("button_back_dim");
   spr_button_back[1] = SPR("button_back_lit_1");
   spr_button_back[2] = SPR("button_back_lit_2");
   spr_button_back[3] = SPR("button_back_lit_3");
   spr_button_back[4] = spr_button_back[2];

   for (unsigned i = 0; i < 6; i++) {
      char buffer[0x40];
      sprintf(buffer, "input_%s_lit_1", icon_names[i]);
      spr_icons[i][0] = SPR(buffer);
      sprintf(buffer, "input_%s_lit_2", icon_names[i]);
      spr_icons[i][1] = SPR(buffer);
      sprintf(buffer, "input_%s_lit_3", icon_names[i]);
      spr_icons[i][2] = SPR(buffer);
      sprintf(buffer, "input_%s_dim", icon_names[i]);
      spr_icons[i][3] = spr_icons[i][1];
      spr_icons[i][4] = SPR(buffer);
   }
#undef SPR
}

//***************************************************************************
// init_options_input
// Initializes the options keyboard and joystick menus.
//***************************************************************************

void init_options_input(void) {
   // Reset animation
   icon_anim = 0;

   // Not entering anything yet
   entering = 0;

   // Initialize menu
   init_options_input_menu();

   // Make the cursor visible
   set_cursor(CURSOR_ARROW);

   // Make screen visible
   fade_on();
}

//***************************************************************************
// init_options_input_menu [internal]
// Sets up the menu in this screen.
//***************************************************************************

static void init_options_input_menu(void) {
   init_menu();
   set_reinit_menu_func(init_options_input_menu);

   menu.defoption.up = (game_mode == GAMEMODE_OPT_JOYSTICK) ?
      BUTTON_PAUSE_1 : BUTTON_JUMP_1;
   menu.defoption.down = BUTTON_BACK;
   menu.defoption.left = BUTTON_LEFT_2;
   menu.defoption.right = BUTTON_LEFT_1;
   menu.cancel = cancel_callback;

   menu.options[BUTTON_LEFT_1].box.x1 = INPUT_X2;
   menu.options[BUTTON_LEFT_1].box.y1 = INPUT_Y1;
   menu.options[BUTTON_LEFT_1].box.x2 = INPUT_X3;
   menu.options[BUTTON_LEFT_1].box.y2 = INPUT_Y1+0x17;
   if (game_mode == GAMEMODE_OPT_JOYSTICK)
      menu.options[BUTTON_LEFT_1].move.up = BUTTON_PAUSE_1;
   else
      menu.options[BUTTON_LEFT_1].move.up = BUTTON_JUMP_1;
   menu.options[BUTTON_LEFT_1].move.down = BUTTON_RIGHT_1;
   menu.options[BUTTON_LEFT_1].move.left = BUTTON_LEFT_2;
   menu.options[BUTTON_LEFT_1].move.right = BUTTON_LEFT_2;
   menu.options[BUTTON_LEFT_1].move.oneswitch = BUTTON_LEFT_2;
   menu.options[BUTTON_LEFT_1].action.accept = button_enter;

   menu.options[BUTTON_LEFT_2].box.x1 = INPUT_X3;
   menu.options[BUTTON_LEFT_2].box.y1 = INPUT_Y1;
   menu.options[BUTTON_LEFT_2].box.x2 = INPUT_X4;
   menu.options[BUTTON_LEFT_2].box.y2 = INPUT_Y1+0x17;
   menu.options[BUTTON_LEFT_2].move.up = BUTTON_BACK;
   menu.options[BUTTON_LEFT_2].move.down = BUTTON_RIGHT_2;
   menu.options[BUTTON_LEFT_2].move.left = BUTTON_LEFT_1;
   menu.options[BUTTON_LEFT_2].move.right = BUTTON_LEFT_1;
   menu.options[BUTTON_LEFT_2].move.oneswitch = BUTTON_RIGHT_1;
   menu.options[BUTTON_LEFT_2].action.accept = button_enter;

   menu.options[BUTTON_RIGHT_1].box.x1 = INPUT_X2;
   menu.options[BUTTON_RIGHT_1].box.y1 = INPUT_Y2;
   menu.options[BUTTON_RIGHT_1].box.x2 = INPUT_X3;
   menu.options[BUTTON_RIGHT_1].box.y2 = INPUT_Y2+0x17;
   menu.options[BUTTON_RIGHT_1].move.up = BUTTON_LEFT_1;
   menu.options[BUTTON_RIGHT_1].move.down = BUTTON_DOWN_1;
   menu.options[BUTTON_RIGHT_1].move.left = BUTTON_RIGHT_2;
   menu.options[BUTTON_RIGHT_1].move.right = BUTTON_RIGHT_2;
   menu.options[BUTTON_RIGHT_1].move.oneswitch = BUTTON_RIGHT_2;
   menu.options[BUTTON_RIGHT_1].action.accept = button_enter;

   menu.options[BUTTON_RIGHT_2].box.x1 = INPUT_X3;
   menu.options[BUTTON_RIGHT_2].box.y1 = INPUT_Y2;
   menu.options[BUTTON_RIGHT_2].box.x2 = INPUT_X4;
   menu.options[BUTTON_RIGHT_2].box.y2 = INPUT_Y2+0x17;
   menu.options[BUTTON_RIGHT_2].move.up = BUTTON_LEFT_2;
   menu.options[BUTTON_RIGHT_2].move.down = BUTTON_DOWN_2;
   menu.options[BUTTON_RIGHT_2].move.left = BUTTON_RIGHT_1;
   menu.options[BUTTON_RIGHT_2].move.right = BUTTON_RIGHT_1;
   menu.options[BUTTON_RIGHT_2].move.oneswitch = BUTTON_DOWN_1;
   menu.options[BUTTON_RIGHT_2].action.accept = button_enter;

   menu.options[BUTTON_DOWN_1].box.x1 = INPUT_X2;
   menu.options[BUTTON_DOWN_1].box.y1 = INPUT_Y3;
   menu.options[BUTTON_DOWN_1].box.x2 = INPUT_X3;
   menu.options[BUTTON_DOWN_1].box.y2 = INPUT_Y3+0x17;
   menu.options[BUTTON_DOWN_1].move.up = BUTTON_RIGHT_1;
   menu.options[BUTTON_DOWN_1].move.down = BUTTON_UP_1;
   menu.options[BUTTON_DOWN_1].move.left = BUTTON_DOWN_2;
   menu.options[BUTTON_DOWN_1].move.right = BUTTON_DOWN_2;
   menu.options[BUTTON_DOWN_1].move.oneswitch = BUTTON_DOWN_2;
   menu.options[BUTTON_DOWN_1].action.accept = button_enter;

   menu.options[BUTTON_DOWN_2].box.x1 = INPUT_X3;
   menu.options[BUTTON_DOWN_2].box.y1 = INPUT_Y3;
   menu.options[BUTTON_DOWN_2].box.x2 = INPUT_X4;
   menu.options[BUTTON_DOWN_2].box.y2 = INPUT_Y3+0x17;
   menu.options[BUTTON_DOWN_2].move.up = BUTTON_RIGHT_2;
   menu.options[BUTTON_DOWN_2].move.down = BUTTON_UP_2;
   menu.options[BUTTON_DOWN_2].move.left = BUTTON_DOWN_1;
   menu.options[BUTTON_DOWN_2].move.right = BUTTON_DOWN_1;
   menu.options[BUTTON_DOWN_2].move.oneswitch = BUTTON_UP_1;
   menu.options[BUTTON_DOWN_2].action.accept = button_enter;

   menu.options[BUTTON_UP_1].box.x1 = INPUT_X2;
   menu.options[BUTTON_UP_1].box.y1 = INPUT_Y4;
   menu.options[BUTTON_UP_1].box.x2 = INPUT_X3;
   menu.options[BUTTON_UP_1].box.y2 = INPUT_Y4+0x17;
   menu.options[BUTTON_UP_1].move.up = BUTTON_DOWN_1;
   menu.options[BUTTON_UP_1].move.down = BUTTON_JUMP_1;
   menu.options[BUTTON_UP_1].move.left = BUTTON_UP_2;
   menu.options[BUTTON_UP_1].move.right = BUTTON_UP_2;
   menu.options[BUTTON_UP_1].move.oneswitch = BUTTON_UP_2;
   menu.options[BUTTON_UP_1].action.accept = button_enter;

   menu.options[BUTTON_UP_2].box.x1 = INPUT_X3;
   menu.options[BUTTON_UP_2].box.y1 = INPUT_Y4;
   menu.options[BUTTON_UP_2].box.x2 = INPUT_X4;
   menu.options[BUTTON_UP_2].box.y2 = INPUT_Y4+0x17;
   menu.options[BUTTON_UP_2].move.up = BUTTON_DOWN_2;
   menu.options[BUTTON_UP_2].move.down = BUTTON_JUMP_2;
   menu.options[BUTTON_UP_2].move.left = BUTTON_UP_1;
   menu.options[BUTTON_UP_2].move.right = BUTTON_UP_1;
   menu.options[BUTTON_UP_2].move.oneswitch = BUTTON_JUMP_1;
   menu.options[BUTTON_UP_2].action.accept = button_enter;

   menu.options[BUTTON_JUMP_1].box.x1 = INPUT_X2;
   menu.options[BUTTON_JUMP_1].box.y1 = INPUT_Y5;
   menu.options[BUTTON_JUMP_1].box.x2 = INPUT_X3;
   menu.options[BUTTON_JUMP_1].box.y2 = INPUT_Y5+0x17;
   menu.options[BUTTON_JUMP_1].move.up = BUTTON_UP_1;
   if (game_mode == GAMEMODE_OPT_JOYSTICK)
      menu.options[BUTTON_JUMP_1].move.down = BUTTON_PAUSE_1;
   else
      menu.options[BUTTON_JUMP_1].move.down = BUTTON_LEFT_1;
   menu.options[BUTTON_JUMP_1].move.left = BUTTON_JUMP_2;
   menu.options[BUTTON_JUMP_1].move.right = BUTTON_JUMP_2;
   menu.options[BUTTON_JUMP_1].move.oneswitch = BUTTON_JUMP_2;
   menu.options[BUTTON_JUMP_1].action.accept = button_enter;

   menu.options[BUTTON_JUMP_2].box.x1 = INPUT_X3;
   menu.options[BUTTON_JUMP_2].box.y1 = INPUT_Y5;
   menu.options[BUTTON_JUMP_2].box.x2 = INPUT_X4;
   menu.options[BUTTON_JUMP_2].box.y2 = INPUT_Y5+0x17;
   menu.options[BUTTON_JUMP_2].move.up = BUTTON_UP_2;
   if (game_mode == GAMEMODE_OPT_JOYSTICK)
      menu.options[BUTTON_JUMP_2].move.down = BUTTON_PAUSE_2;
   else
      menu.options[BUTTON_JUMP_2].move.down = BUTTON_BACK;
   menu.options[BUTTON_JUMP_2].move.left = BUTTON_JUMP_1;
   menu.options[BUTTON_JUMP_2].move.right = BUTTON_JUMP_1;
   if (game_mode == GAMEMODE_OPT_JOYSTICK)
      menu.options[BUTTON_JUMP_2].move.oneswitch = BUTTON_PAUSE_1;
   else
      menu.options[BUTTON_JUMP_2].move.oneswitch = BUTTON_BACK;
   menu.options[BUTTON_JUMP_2].action.accept = button_enter;

   if (game_mode == GAMEMODE_OPT_JOYSTICK) {
      menu.options[BUTTON_PAUSE_1].box.x1 = INPUT_X2;
      menu.options[BUTTON_PAUSE_1].box.y1 = INPUT_Y6;
      menu.options[BUTTON_PAUSE_1].box.x2 = INPUT_X3;
      menu.options[BUTTON_PAUSE_1].box.y2 = INPUT_Y6+0x17;
      menu.options[BUTTON_PAUSE_1].move.up = BUTTON_JUMP_1;
      menu.options[BUTTON_PAUSE_1].move.down = BUTTON_LEFT_1;
      menu.options[BUTTON_PAUSE_1].move.left = BUTTON_PAUSE_2;
      menu.options[BUTTON_PAUSE_1].move.right = BUTTON_PAUSE_2;
      menu.options[BUTTON_PAUSE_1].move.oneswitch = BUTTON_PAUSE_2;
      menu.options[BUTTON_PAUSE_1].action.accept = button_enter;

      menu.options[BUTTON_PAUSE_2].box.x1 = INPUT_X3;
      menu.options[BUTTON_PAUSE_2].box.y1 = INPUT_Y6;
      menu.options[BUTTON_PAUSE_2].box.x2 = INPUT_X4;
      menu.options[BUTTON_PAUSE_2].box.y2 = INPUT_Y6+0x17;
      menu.options[BUTTON_PAUSE_2].move.up = BUTTON_JUMP_2;
      menu.options[BUTTON_PAUSE_2].move.down = BUTTON_BACK;
      menu.options[BUTTON_PAUSE_2].move.left = BUTTON_PAUSE_1;
      menu.options[BUTTON_PAUSE_2].move.right = BUTTON_PAUSE_1;
      menu.options[BUTTON_PAUSE_2].move.oneswitch = BUTTON_BACK;
      menu.options[BUTTON_PAUSE_2].action.accept = button_enter;
   }

   menu.options[BUTTON_BACK].box.x1 = INPUT_BX;
   menu.options[BUTTON_BACK].box.y1 = INPUT_BY;
   menu.options[BUTTON_BACK].box.x2 = INPUT_BX + 71;
   menu.options[BUTTON_BACK].box.y2 = INPUT_BY + 24;
   if (game_mode == GAMEMODE_OPT_JOYSTICK)
      menu.options[BUTTON_BACK].move.up = BUTTON_PAUSE_2;
   else
      menu.options[BUTTON_BACK].move.up = BUTTON_JUMP_2;
   menu.options[BUTTON_BACK].move.down = BUTTON_LEFT_2;
   menu.options[BUTTON_BACK].move.left = BUTTON_LEFT_1;
   menu.options[BUTTON_BACK].move.right = BUTTON_LEFT_1;
   menu.options[BUTTON_BACK].move.oneswitch = BUTTON_LEFT_1;
   menu.options[BUTTON_BACK].action.accept = button_back;
   menu.options[BUTTON_BACK].sfx = SFX_CANCEL;
}

//***************************************************************************
// run_options_input
// Processes the logic for the options keyboard and joystick menus.
//***************************************************************************

void run_options_input(void) {
   // Entering new input?
   if (entering) {
      // Press Esc to abort
      if (input.rawpress.keyboard == SDL_SCANCODE_ESCAPE) {
         play_sfx(SFX_CANCEL);
         entering = 0;
      }

      // Keyboard input?
      else if (game_mode == GAMEMODE_OPT_KEYBOARD) {
         // If an input has been entered, accept it
         if (input.rawpress.keyboard != NO_KEY) {
            int i = menu.selected;
            settings.keyboard[i/2][i%2] = input.rawpress.keyboard;
            play_sfx(SFX_OK);
            entering = 0;
            reset_input();
         }
      }

      // Joystick input?
      else {
         // If an input has been entered, accept it
         if (input.rawpress.joystick != JOYIN_NONE) {
            int i = menu.selected;
            settings.joystick[i/2][i%2] = input.rawpress.joystick;
            play_sfx(SFX_OK);
            entering = 0;
            reset_input();
         }
      }
   }

   // Update the menu as usual otherwise
   // To-do: find a better way to do this?
   else {
      update_menu();
   }

   // Update background animation
   update_background();

   // Update icon animation
   icon_anim++;
}

//***************************************************************************
// draw_options_input
// Draws the options keyboard and joystick menus.
//***************************************************************************

void draw_options_input(void) {
   // Used to keep track of the two halves of the string used when showing
   // the current option to the screen reader
   const char *str_action = NULL;      // Action (e.g. "jump")
   const char *str_input = NULL;       // Input (key or button)
   int str_input_id = 0;               // Number of input (e.g. axis number)

   // Draw background
   draw_background();

   // Draw back button
   draw_sprite(spr_button_back[menu.selected == BUTTON_BACK ?
      (icon_anim >> 3 & 3) + 1 : 0], INPUT_BX, INPUT_BY, SPR_NOFLIP);

   // Draw input buttons
   int max_inputs = (game_mode == GAMEMODE_OPT_JOYSTICK) ? 6 : 5;
   for (int i = 0; i < max_inputs; i++) {
      // Draw background
      draw_sprite(spr_button_input[
         (menu.selected >> 1) == i ? 1 : 0],
         INPUT_X1, INPUT_Y1 + i*INPUT_YDIST, SPR_NOFLIP);

      // Draw icon
      if (menu.selected >> 1 == i) {
         draw_sprite(spr_icons[i][icon_anim >> 3 & 3],
         INPUT_X1 + 4, INPUT_Y1 + i*INPUT_YDIST + 4, SPR_NOFLIP);
      } else {
         draw_sprite(spr_icons[i][4],
         INPUT_X1 + 4, INPUT_Y1 + i*INPUT_YDIST + 4, SPR_NOFLIP);
      }
   }

   // Draw input button labels
   draw_text(text.options_input.left,
      INPUT_X1 + 0x16, INPUT_Y1 + 12,
      (menu.selected >> 1) == 0 ? FONT_LIT : FONT_DIM, ALIGN_LEFT);
   draw_text(text.options_input.right,
      INPUT_X1 + 0x16, INPUT_Y2 + 12,
      (menu.selected >> 1) == 1 ? FONT_LIT : FONT_DIM, ALIGN_LEFT);
   draw_text(text.options_input.down,
      INPUT_X1 + 0x16, INPUT_Y3 + 12,
      (menu.selected >> 1) == 2 ? FONT_LIT : FONT_DIM, ALIGN_LEFT);
   draw_text(text.options_input.up,
      INPUT_X1 + 0x16, INPUT_Y4 + 12,
      (menu.selected >> 1) == 3 ? FONT_LIT : FONT_DIM, ALIGN_LEFT);
   draw_text(text.options_input.jump,
      INPUT_X1 + 0x16, INPUT_Y5 + 12,
      (menu.selected >> 1) == 4 ? FONT_LIT : FONT_DIM, ALIGN_LEFT);
   if (game_mode == GAMEMODE_OPT_JOYSTICK)
      draw_text(text.options_input.pause,
      INPUT_X1 + 0x16, INPUT_Y6 + 12,
      (menu.selected >> 1) == 5 ? FONT_LIT : FONT_DIM, ALIGN_LEFT);

   // Keyboard menu?
   if (game_mode == GAMEMODE_OPT_KEYBOARD) {
      // Draw key names
      for (int i = BUTTON_LEFT_1; i <= BUTTON_JUMP_2; i++) {
         // Get which key is it
         int scancode = settings.keyboard[i/2][i%2];

         // Draw key name
         draw_text_int(get_key_name(scancode), scancode,
            (menu.options[i].box.x1 + menu.options[i].box.x2) / 2,
            (menu.options[i].box.y1 + menu.options[i].box.y2) / 2,
            menu.selected == i ? FONT_LIT : FONT_DIM, ALIGN_CENTER);

         // Is this the selected option?
         if (menu.selected == i)
            str_input = get_key_name(scancode);
      }
   }

   // Joystick menu?
   else {
      // Draw button names
      for (int i = BUTTON_LEFT_1; i <= BUTTON_PAUSE_2; i++) {
         // Get which input is it
         uint16_t joyin = settings.joystick[i/2][i%2];
         uint16_t id = joyin & 0x00FF;

         // Get relevant string
         const char *str;
         switch (joyin & 0xFF00) {
            // Nothing
            case JOYIN_NONE:
               str = "N/A";
               break;

            // Button
            case JOYIN_BUTTON:
               str = text.options_input.button;
               break;

            // Axis
            case JOYIN_AXISPOS:
               if (id == 0)
                  str = text.options_input.axis_x_pos;
               else if (id == 1)
                  str = text.options_input.axis_y_pos;
               else
                  str = text.options_input.axis_pos;
               break;
            case JOYIN_AXISNEG:
               if (id == 0)
                  str = text.options_input.axis_x_neg;
               else if (id == 1)
                  str = text.options_input.axis_y_neg;
               else
                  str = text.options_input.axis_neg;
               break;

            // Hat
            case JOYIN_HATUP:
               str = id == 0 ?
                  text.options_input.dpad_up :
                  text.options_input.hat_up;
               break;
            case JOYIN_HATDOWN:
               str = id == 0 ?
                  text.options_input.dpad_down :
                  text.options_input.hat_down;
               break;
            case JOYIN_HATLEFT:
               str = id == 0 ?
                  text.options_input.dpad_left :
                  text.options_input.hat_left;
               break;
            case JOYIN_HATRIGHT:
               str = id == 0 ?
                  text.options_input.dpad_right :
                  text.options_input.hat_right;
               break;

            // Huuuuuh...
            default:
               str = "???";
               break;
         }

         // Draw joystick input
         draw_text_int(str, id,
            (menu.options[i].box.x1 + menu.options[i].box.x2) / 2,
            (menu.options[i].box.y1 + menu.options[i].box.y2) / 2,
            menu.selected == i ? FONT_LIT : FONT_DIM, ALIGN_CENTER);

         // Is this the selected option?
         if (menu.selected == i) {
            str_input = str;
            str_input_id = id;
         }
      }
   }

   // If entering an input, highlight the input being entered
   if (entering) {
      // Determine color
      uint32_t color = 0x6060FF;
      unsigned anim = icon_anim & 0x1F;
      if (anim >= 0x10) anim = 0x1F - anim;
      color += anim << 10;
      color += anim << 18;

      // Draw highlight
      fill_rectangle(menu.options[menu.selected].box.x1+1,
                     menu.options[menu.selected].box.y1+4,
                     menu.options[menu.selected].box.x2-1,
                     menu.options[menu.selected].box.y2-4,
                     color);

      // Tell player to enter something
      set_reader_text(game_mode == GAMEMODE_OPT_KEYBOARD ?
         text.options_input.press_key : text.options_input.press_joy);
   }

   // Draw border around selected option
   if (menu.selected != -1 && (icon_anim & 0x08)) {
      draw_rectangle(menu.options[menu.selected].box.x1-1,
                     menu.options[menu.selected].box.y1-1,
                     menu.options[menu.selected].box.x2+1,
                     menu.options[menu.selected].box.y2+1,
                     settings.box_lit[1]);
      draw_rectangle(menu.options[menu.selected].box.x1,
                     menu.options[menu.selected].box.y1,
                     menu.options[menu.selected].box.x2,
                     menu.options[menu.selected].box.y2,
                     settings.box_lit[0]);
      draw_rectangle(menu.options[menu.selected].box.x1+1,
                     menu.options[menu.selected].box.y1+1,
                     menu.options[menu.selected].box.x2-1,
                     menu.options[menu.selected].box.y2-1,
                     settings.box_lit[1]);
      draw_hline(menu.options[menu.selected].box.x1-1,
                 menu.options[menu.selected].box.y2+2,
                 menu.options[menu.selected].box.x2+1,
                 settings.box_lit[2]);
   }

   // Draw title
   draw_text(game_mode == GAMEMODE_OPT_KEYBOARD ?
      text.options.keyboard : text.options.joystick,
      INPUT_X1, 0x10, FONT_LIT, ALIGN_TOPLEFT);

   // Show selected option over screen reader
   if (!entering) {
      // Back button
      if (menu.selected == BUTTON_BACK) {
         set_reader_text(text.options.back);
      }

      // Mapping button
      else if (menu.selected != -1) {
         // Determine action string
         switch (menu.selected) {
            case BUTTON_UP_1: case BUTTON_UP_2:
               str_action = text.options_input.up; break;
            case BUTTON_DOWN_1: case BUTTON_DOWN_2:
               str_action = text.options_input.down; break;
            case BUTTON_LEFT_1: case BUTTON_LEFT_2:
               str_action = text.options_input.left; break;
            case BUTTON_RIGHT_1: case BUTTON_RIGHT_2:
               str_action = text.options_input.right; break;
            case BUTTON_JUMP_1: case BUTTON_JUMP_2:
               str_action = text.options_input.jump; break;
            case BUTTON_PAUSE_1: case BUTTON_PAUSE_2:
               str_action = text.options_input.pause; break;
            default:
               str_action = "???"; break;
         }

         // Concatenate both action and input so we can display them together
         // to the screen reader
         char *buffer = (char *) malloc(strlen(str_action) +
            strlen(str_input) + 3);
         if (buffer == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         sprintf(buffer, "%s %s", str_action, str_input);

         // Output concatenated string to the screen reader
         set_reader_punctuation(menu.selected >= BUTTON_LEFT_1 &&
            menu.selected <= BUTTON_PAUSE_2);
         set_reader_text_int(buffer, str_input_id);
         free(buffer);
      }
   }
}

//***************************************************************************
// deinit_options_input
// Deinitializes the options keyboard and joystick menus.
//***************************************************************************

void deinit_options_input(void) {
   // Make sure we don't speak punctuation anymore
   set_reader_punctuation(0);
}

//***************************************************************************
// unload_options_input
// Unloads the assets for the options keyboard and joystick menus. Called
// while the main menu is unloading (as everything was loaded in a bulk).
//***************************************************************************

void unload_options_input(void) {
   // Unload graphics
   destroy_graphics_set(gfxset);
}

//***************************************************************************
// button_enter [internal]
// Starts "entering" mode to let the player enter a new input.
//***************************************************************************

static void button_enter(void) {
   // Start entering mode
   play_sfx(SFX_BEEP);
   entering = 1;
}

//***************************************************************************
// button_back [internal]
// Button for going back. Returns to the options main menu.
//***************************************************************************

static void button_back(void) {
   fade_off_and_switch(GAMEMODE_OPTIONS);
}

//***************************************************************************
// cancel_callback [internal]
// Callback for the cancel button. Normally would be the back button, but
// we need a way to erase inputs so we resort to this instead.
//***************************************************************************

static void cancel_callback(void) {
   // Is a control selected?
   if (menu.selected >= BUTTON_LEFT_1 && menu.selected <= BUTTON_PAUSE_2) {
      // Keyboard input?
      if (game_mode == GAMEMODE_OPT_KEYBOARD) {
         int i = menu.selected;
         settings.keyboard[i/2][i%2] = NO_KEY;
      }

      // Joystick input?
      else {
         int i = menu.selected;
         settings.joystick[i/2][i%2] = JOYIN_NONE;
      }
   }

   // Quit otherwise
   else {
      button_back();
   }
}
