//***************************************************************************
// "options_sound.c"
// Code for the options sound menu.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include "background.h"
#include "menu.h"
#include "reader.h"
#include "settings.h"
#include "sound.h"
#include "text.h"
#include "video.h"

// UI positions
#define SOUNDTEST_X1 (screen_cx - 0x98)   // X: Sound test prev
#define SOUNDTEST_X2 (screen_cx - 0x60)   // X: Sound test play
#define SOUNDTEST_X3 (screen_cx + 0x68)   // X: Sound test next
#define SOUNDTEST_TX (screen_cx + 0x08)   // X: Sound test text
#define SOUNDTEST_Y (screen_cy - 0x48)    // Y: Sound test buttons
#define SOUNDTEST_TY (SOUNDTEST_Y + 0x10) // Y: Sound test text

#define VOLUME_X (screen_cx - 272/2)      // X: volume buttons
#define VOLUME_VX (VOLUME_X + 0x38)       // X: volume bars
#define VOLUME_Y1 (SOUNDTEST_Y + 0x28)    // Y: BGM volume button
#define VOLUME_VY1 (VOLUME_Y1 + 8)        // Y: BGM volume bar
#define VOLUME_Y2 (VOLUME_Y1 + 0x24)      // Y: SFX volume button
#define VOLUME_VY2 (VOLUME_Y2 + 8)        // Y: SFX volume bar

#define BACK_X (screen_cx - 40)           // X: back button
#define BACK_Y (VOLUME_Y2 + 0x28)         // Y: back button

#define STEREO_X (BACK_X - 88)            // X: stereo button
#define STEREO_Y BACK_Y                   // Y: stereo button

#define REVERSE_X (BACK_X + 88)           // X: reverse stereo button
#define REVERSE_Y BACK_Y                  // Y: reverse stereo button

// List of buttons
enum {
   BUTTON_PLAY,
   BUTTON_PREV,
   BUTTON_NEXT,
   BUTTON_BGMVOLUME,
   BUTTON_SFXVOLUME,
   BUTTON_STEREO,
   BUTTON_REVERSE,
   BUTTON_BACK,
   NUM_BUTTONS
};

// Where graphics are stored
static GraphicsSet *gfxset = NULL;
static Sprite *spr_button[NUM_BUTTONS][4];
static Sprite *spr_stereo[2][4];
static Sprite *spr_reverse[2][4];

// Used to animate stuff
static unsigned anim;
static unsigned button_anim;

// Current selection in the sound test
static int sound_test = -1;
static int current_bgm;

// Names for each button
static const char *button_names[] = {
   "play",
   "previous",
   "next",
   "bgm_volume",
   "sfx_volume",
   NULL,
   NULL,
   "back"
};

// Used to modify the bars in one-switch mode
static unsigned oneswitch_edit;     // Set if modifying a setting
static unsigned oneswitch_timer;    // Delay before next value

// Private function prototypes
static void init_options_sound_menu(void);
static void draw_volume_bars(int, int32_t, int32_t);
static void button_play(void);
static void button_previous(void);
static void button_next(void);
static void button_dec_bgm_volume(void);
static void button_inc_bgm_volume(void);
static void button_dec_sfx_volume(void);
static void button_inc_sfx_volume(void);
static void button_stereo(void);
static void button_reverse(void);
static void button_back(void);
static void button_one_switch(void);

//***************************************************************************
// load_options_sound
// Loads the assets for the options sound menu. Called while the main menu is
// loading (so everything loads in a bulk).
//***************************************************************************

void load_options_sound(void) {
   // Load graphics
   gfxset = load_graphics_set("graphics/options_sound");

   // Load sprites
#define SPR(name) get_sprite(gfxset, name)
   for (unsigned i = 0; i < NUM_BUTTONS; i++) {
      char buffer[0x40];
      sprintf(buffer, "button_%s_dim", button_names[i]);
      spr_button[i][0] = SPR(buffer);
      sprintf(buffer, "button_%s_lit_1", button_names[i]);
      spr_button[i][1] = SPR(buffer);
      sprintf(buffer, "button_%s_lit_2", button_names[i]);
      spr_button[i][2] = SPR(buffer);
      sprintf(buffer, "button_%s_lit_3", button_names[i]);
      spr_button[i][3] = SPR(buffer);
   }

   spr_stereo[0][0] = SPR("button_stereo_off_dim");
   spr_stereo[0][1] = SPR("button_stereo_off_lit_1");
   spr_stereo[0][2] = SPR("button_stereo_off_lit_2");
   spr_stereo[0][3] = SPR("button_stereo_off_lit_3");
   spr_stereo[1][0] = SPR("button_stereo_on_dim");
   spr_stereo[1][1] = SPR("button_stereo_on_lit_1");
   spr_stereo[1][2] = SPR("button_stereo_on_lit_2");
   spr_stereo[1][3] = SPR("button_stereo_on_lit_3");

   spr_reverse[0][0] = SPR("button_reverse_off_dim");
   spr_reverse[0][1] = SPR("button_reverse_off_lit_1");
   spr_reverse[0][2] = SPR("button_reverse_off_lit_2");
   spr_reverse[0][3] = SPR("button_reverse_off_lit_3");
   spr_reverse[1][0] = SPR("button_reverse_on_dim");
   spr_reverse[1][1] = SPR("button_reverse_on_lit_1");
   spr_reverse[1][2] = SPR("button_reverse_on_lit_2");
   spr_reverse[1][3] = SPR("button_reverse_on_lit_3");
#undef SPR
}

//***************************************************************************
// init_options_sound
// Initializes the options sound menu.
//***************************************************************************

void init_options_sound(void) {
   // When we enter the menu BGM is playing
   current_bgm = BGM_MENU;

   // Reset animations
   anim = 0;

   // Initialize menu
   init_options_sound_menu();

   // Make the cursor visible
   set_cursor(CURSOR_ARROW);

   // Make screen visible
   fade_on();
}

//***************************************************************************
// init_options_sound_menu [internal]
// Sets up the menu in this screen.
//***************************************************************************

static void init_options_sound_menu(void) {
   init_menu();
   set_reinit_menu_func(init_options_sound_menu);

   menu.cancel = button_back;
   menu.defoption.up = BUTTON_BACK;
   menu.defoption.down = BUTTON_PLAY;
   menu.defoption.left = BUTTON_PLAY;
   menu.defoption.right = BUTTON_PLAY;

   menu.options[BUTTON_PLAY].box.x1 = SOUNDTEST_X2;
   menu.options[BUTTON_PLAY].box.y1 = SOUNDTEST_Y;
   menu.options[BUTTON_PLAY].box.x2 = SOUNDTEST_X2 + 0xBF;
   menu.options[BUTTON_PLAY].box.y2 = SOUNDTEST_Y + 0x1F;
   menu.options[BUTTON_PLAY].move.up = BUTTON_BACK;
   menu.options[BUTTON_PLAY].move.down = BUTTON_BGMVOLUME;
   menu.options[BUTTON_PLAY].move.oneswitch = BUTTON_BGMVOLUME;
   menu.options[BUTTON_PLAY].action.left = button_previous;
   menu.options[BUTTON_PLAY].action.right = button_next;
   menu.options[BUTTON_PLAY].action.accept = button_play;
   menu.options[BUTTON_PLAY].action.oneswitch = button_one_switch;
   menu.options[BUTTON_PLAY].caption = text.options_sound.play_sound;
   menu.options[BUTTON_PLAY].sfx = SFX_NONE;

   if (!settings.audiovideo) {
      menu.options[BUTTON_PREV].box.x1 = SOUNDTEST_X1;
      menu.options[BUTTON_PREV].box.y1 = SOUNDTEST_Y;
      menu.options[BUTTON_PREV].box.x2 = SOUNDTEST_X1 + 0x2F;
      menu.options[BUTTON_PREV].box.y2 = SOUNDTEST_Y + 0x1F;
      menu.options[BUTTON_PREV].move.up = BUTTON_SFXVOLUME;
      menu.options[BUTTON_PREV].move.down = BUTTON_BGMVOLUME;
      menu.options[BUTTON_PREV].move.left = BUTTON_PLAY;
      menu.options[BUTTON_PREV].move.right = BUTTON_PLAY;
      menu.options[BUTTON_PREV].action.accept = button_previous;
      menu.options[BUTTON_PREV].caption = text.options_sound.prev_sound;

      menu.options[BUTTON_NEXT].box.x1 = SOUNDTEST_X3;
      menu.options[BUTTON_NEXT].box.y1 = SOUNDTEST_Y;
      menu.options[BUTTON_NEXT].box.x2 = SOUNDTEST_X3 + 0x2F;
      menu.options[BUTTON_NEXT].box.y2 = SOUNDTEST_Y + 0x1F;
      menu.options[BUTTON_NEXT].move.up = BUTTON_SFXVOLUME;
      menu.options[BUTTON_NEXT].move.down = BUTTON_BGMVOLUME;
      menu.options[BUTTON_NEXT].move.left = BUTTON_PLAY;
      menu.options[BUTTON_NEXT].move.right = BUTTON_PLAY;
      menu.options[BUTTON_NEXT].action.accept = button_next;
      menu.options[BUTTON_NEXT].caption = text.options_sound.next_sound;
   }

   menu.options[BUTTON_BGMVOLUME].box.x1 = VOLUME_X;
   menu.options[BUTTON_BGMVOLUME].box.y1 = VOLUME_Y1;
   menu.options[BUTTON_BGMVOLUME].box.x2 = VOLUME_X + 271;
   menu.options[BUTTON_BGMVOLUME].box.y2 = VOLUME_Y1 + 31;
   menu.options[BUTTON_BGMVOLUME].move.up = BUTTON_PLAY;
   menu.options[BUTTON_BGMVOLUME].move.down = BUTTON_SFXVOLUME;
   menu.options[BUTTON_BGMVOLUME].move.oneswitch = BUTTON_SFXVOLUME;
   menu.options[BUTTON_BGMVOLUME].action.left = button_dec_bgm_volume;
   menu.options[BUTTON_BGMVOLUME].action.right = button_inc_bgm_volume;
   menu.options[BUTTON_BGMVOLUME].action.oneswitch = button_one_switch;
   menu.options[BUTTON_BGMVOLUME].caption = text.options_sound.bgm_volume;

   menu.options[BUTTON_SFXVOLUME].box.x1 = VOLUME_X;
   menu.options[BUTTON_SFXVOLUME].box.y1 = VOLUME_Y2;
   menu.options[BUTTON_SFXVOLUME].box.x2 = VOLUME_X + 271;
   menu.options[BUTTON_SFXVOLUME].box.y2 = VOLUME_Y2 + 31;
   menu.options[BUTTON_SFXVOLUME].move.up = BUTTON_BGMVOLUME;
   menu.options[BUTTON_SFXVOLUME].move.down = BUTTON_BACK;
   menu.options[BUTTON_SFXVOLUME].move.oneswitch = BUTTON_STEREO;
   menu.options[BUTTON_SFXVOLUME].action.left = button_dec_sfx_volume;
   menu.options[BUTTON_SFXVOLUME].action.right = button_inc_sfx_volume;
   menu.options[BUTTON_SFXVOLUME].action.oneswitch = button_one_switch;
   menu.options[BUTTON_SFXVOLUME].caption = text.options_sound.sfx_volume;

   menu.options[BUTTON_STEREO].box.x1 = STEREO_X;
   menu.options[BUTTON_STEREO].box.y1 = STEREO_Y;
   menu.options[BUTTON_STEREO].box.x2 = STEREO_X + 79;
   menu.options[BUTTON_STEREO].box.y2 = STEREO_Y + 27;
   menu.options[BUTTON_STEREO].move.up = BUTTON_SFXVOLUME;
   menu.options[BUTTON_STEREO].move.down = BUTTON_PLAY;
   menu.options[BUTTON_STEREO].move.left = BUTTON_REVERSE;
   menu.options[BUTTON_STEREO].move.right = BUTTON_BACK;
   menu.options[BUTTON_STEREO].move.oneswitch = BUTTON_BACK;
   menu.options[BUTTON_STEREO].action.accept = button_stereo;
   menu.options[BUTTON_STEREO].caption = text.options_sound.stereo;

   menu.options[BUTTON_BACK].box.x1 = BACK_X;
   menu.options[BUTTON_BACK].box.y1 = BACK_Y;
   menu.options[BUTTON_BACK].box.x2 = BACK_X + 79;
   menu.options[BUTTON_BACK].box.y2 = BACK_Y + 27;
   menu.options[BUTTON_BACK].move.up = BUTTON_SFXVOLUME;
   menu.options[BUTTON_BACK].move.down = BUTTON_PLAY;
   menu.options[BUTTON_BACK].move.left = BUTTON_STEREO;
   menu.options[BUTTON_BACK].move.right = BUTTON_REVERSE;
   menu.options[BUTTON_BACK].move.oneswitch = BUTTON_PLAY;
   menu.options[BUTTON_BACK].action.accept = button_back;
   menu.options[BUTTON_BACK].caption = text.options.back;
   menu.options[BUTTON_BACK].sfx = SFX_CANCEL;

   menu.options[BUTTON_REVERSE].box.x1 = REVERSE_X;
   menu.options[BUTTON_REVERSE].box.y1 = REVERSE_Y;
   menu.options[BUTTON_REVERSE].box.x2 = REVERSE_X + 79;
   menu.options[BUTTON_REVERSE].box.y2 = REVERSE_Y + 27;
   menu.options[BUTTON_REVERSE].move.up = BUTTON_SFXVOLUME;
   menu.options[BUTTON_REVERSE].move.down = BUTTON_PLAY;
   menu.options[BUTTON_REVERSE].move.left = BUTTON_BACK;
   menu.options[BUTTON_REVERSE].move.right = BUTTON_STEREO;
   menu.options[BUTTON_REVERSE].move.oneswitch = BUTTON_PLAY;
   menu.options[BUTTON_REVERSE].action.accept = button_reverse;
   menu.options[BUTTON_REVERSE].caption = text.options_sound.reverse;
}

//***************************************************************************
// run_options_sound
// Processes the logic for the options sound menu.
//***************************************************************************

void run_options_sound(void) {
   // Changing a value? (in one-switch mode)
   if (oneswitch_edit) {
      // Delay between changes
      oneswitch_timer--;
      if (oneswitch_timer == 0) {
         oneswitch_timer = settings.menu_delay /
            (menu.selected == BUTTON_PLAY ? 1 : 2);

         // Change value as needed
         switch (menu.selected) {
            case BUTTON_PLAY:
               button_next();
               break;

            case BUTTON_BGMVOLUME:
               button_dec_bgm_volume();
               break;

            case BUTTON_SFXVOLUME:
               button_dec_sfx_volume();
               play_sfx(SFX_BEEP);
               break;

            default:
               break;
         }
      }

      // Tap to be done
      if (input.oneswitch.tap || input.oneswitch.tap2) {
         oneswitch_edit = 0;
         play_sfx(SFX_OK);
         if (menu.selected == BUTTON_PLAY)
            button_play();
      }
   }

   // Update the menu
   else {
      update_menu();
   }

   // Update background animation
   update_background();

   // Update animations
   anim++;
   switch (anim >> 3 & 3) {
      case 0: button_anim = 1; break;
      case 1: button_anim = 2; break;
      case 2: button_anim = 3; break;
      case 3: button_anim = 2; break;
   }

   // Click on the volume bars to change the volume
   if (input.cursor.click &&
   input.cursor.x >= VOLUME_VX - 10 && input.cursor.x <= VOLUME_VX + 199 &&
   input.cursor.y >= VOLUME_VY1 && input.cursor.y <= VOLUME_VY1 + 15) {
      settings.bgm_volume = (input.cursor.x - VOLUME_VX + 10) / 10 * 5;
      play_sfx(SFX_OK);
   }
   if (input.cursor.click &&
   input.cursor.x >= VOLUME_VX - 10 && input.cursor.x <= VOLUME_VX + 199 &&
   input.cursor.y >= VOLUME_VY2 && input.cursor.y <= VOLUME_VY2 + 15) {
      settings.sfx_volume = (input.cursor.x - VOLUME_VX + 10) / 10 * 5;
      play_sfx(SFX_OK);
   }
}

//***************************************************************************
// draw_options_sound
// Draws the options sound menu.
//***************************************************************************

void draw_options_sound(void) {
   // Draw background
   draw_background();

   // Draw all buttons
   for (int i = 0; i < NUM_BUTTONS; i++) {
      draw_sprite(menu.selected == i ?
         spr_button[i][button_anim] : spr_button[i][0],
         menu.options[i].box.x1, menu.options[i].box.y1, SPR_NOFLIP);
   }
   draw_sprite(spr_stereo[settings.stereo]
      [menu.selected == BUTTON_STEREO ? button_anim : 0],
      STEREO_X, STEREO_Y, SPR_NOFLIP);
   draw_sprite(spr_reverse[settings.reverse]
      [menu.selected == BUTTON_REVERSE ? button_anim : 0],
      REVERSE_X, REVERSE_Y, SPR_NOFLIP);

   // Draw current sound test selection
   draw_text(text.options_sound.bgm_names[sound_test+1],
      SOUNDTEST_TX, SOUNDTEST_TY, menu.selected == BUTTON_PLAY ?
      FONT_LIT : FONT_DIM, ALIGN_CENTER);

   // Draw volume bars
   draw_volume_bars(settings.bgm_volume, VOLUME_VX, VOLUME_VY1);
   draw_volume_bars(settings.sfx_volume, VOLUME_VX, VOLUME_VY2);

   // If an option is selected, show a border around it as well as its
   // caption above all the buttons
   if (menu.selected != -1) {
      // Where the data for the selected option is stored
      MenuOption *ptr = &menu.options[menu.selected];

      // Draw the border on top of the button
      if ((anim & 0x08) || oneswitch_edit) {
         draw_rectangle(ptr->box.x1+1, ptr->box.y1+1,
            ptr->box.x2-1, ptr->box.y2-1, settings.box_lit[1]);
         draw_rectangle(ptr->box.x1, ptr->box.y1,
            ptr->box.x2, ptr->box.y2, settings.box_lit[0]);
         draw_rectangle(ptr->box.x1-1, ptr->box.y1-1,
            ptr->box.x2+1, ptr->box.y2+1, settings.box_lit[0]);
         draw_rectangle(ptr->box.x1-2, ptr->box.y1-2,
            ptr->box.x2+2, ptr->box.y2+2, settings.box_lit[1]);
         draw_hline(ptr->box.x1-2, ptr->box.y2+3,
            ptr->box.x2+2, settings.box_lit[2]);
      }

      // Determine "param" to use for the caption
      int param; const char *strparam = NULL;

      switch (menu.selected) {
         case BUTTON_BGMVOLUME:
            param = settings.bgm_volume;
            break;
         case BUTTON_SFXVOLUME:
            param = settings.sfx_volume;
            break;

         case BUTTON_STEREO:
            strparam = settings.stereo ?
               text.options_sound.enabled :
               text.options_sound.disabled;
            break;

         case BUTTON_REVERSE:
            strparam = settings.reverse ?
               text.options_sound.reversed :
               text.options_sound.normal;
            break;

         default:
            param = 0;
            break;
      }

      // Draw the caption
      if (ptr->caption) {
         if (strparam) {
            draw_text_str(ptr->caption, strparam,
            screen_cx, screen_h - 0x10, FONT_LIT, ALIGN_BOTTOM);
         } else {
            draw_text_int(ptr->caption, param,
            screen_cx, screen_h - 0x10, FONT_LIT, ALIGN_BOTTOM);
         }
      }

      // Output option to screen reader
      switch (menu.selected) {
         // If any of the sound test buttons is selected show the currently
         // selected tune instead of the caption (so the player can tell what
         // music is selected)
         case BUTTON_PLAY:
         case BUTTON_PREV:
         case BUTTON_NEXT:
            set_reader_text(text.options_sound.bgm_names[sound_test+1]);
            break;

         // Show normal caption otherwise
         default:
            set_reader_text_int(ptr->caption, param);
            break;
      }
   }

   // Draw title
   draw_text(text.options.sound, screen_cx, 0x10, FONT_LIT, ALIGN_TOP);
}

//***************************************************************************
// draw_volume_bars [internal]
// Helper used to draw the BGM and SFX volume bars.
//---------------------------------------------------------------------------
// param volume: current volume
// param x: base X coordinate
// param y: base Y coordinate
//***************************************************************************

static void draw_volume_bars(int volume, int32_t x, int32_t y) {
   // Colors used for the segments
   static const uint32_t colors[] = {
      0x00B800, 0x00C000, 0x20C800, 0x40D000,
      0x60D800, 0x80E000, 0xA0E800, 0xC0F000,
      0xE0F800, 0xFFFF00, 0xFFF000, 0xFFE000,
      0xFFD000, 0xFFC000, 0xFFB000, 0xFFA000,
      0xFF9000, 0xFF8000, 0xFF7000, 0xFF6000
   };

   // Volume is in steps of 5%
   volume /= 5;

   // Draw all segments
   for (int i = 0; i < 20; i++, x += 10) {
      // Draw segment lit or not depending on how much volume this represents
      // and how much is the selected volume
      fill_rectangle(x+2, y+1, x+7, y+14,
         i < volume ? colors[i] : 0x404040);

      // Draw bright spot ^_^
      draw_vline(x+3, y+2, y+4, 0xFFFFFF);

      // Draw outline
      draw_rectangle(x+1, y, x+8, y+15, 0x000000);
   }
}

//***************************************************************************
// deinit_options_sound
// Deinitializes the options sound menu.
//***************************************************************************

void deinit_options_sound(void) {
   // Restore menu BGM if needed
   if (current_bgm != BGM_MENU)
      play_bgm(BGM_MENU);
}

//***************************************************************************
// unload_options_sound
// Unloads the assets for the options sound menu. Called while the main menu
// is unloading (as everything was loaded in a bulk).
//***************************************************************************

void unload_options_sound(void) {
   // Unload graphics
   destroy_graphics_set(gfxset);
}

//***************************************************************************
// button_play [internal]
// Button for the sound test play control. Plays the current sound.
//***************************************************************************

static void button_play(void) {
   // Unload BGM if it isn't the menu one
   if (current_bgm != BGM_MENU && current_bgm != -1)
      unload_bgm(current_bgm);
   current_bgm = sound_test;

   // Stop music?
   if (sound_test == -1) {
      stop_bgm();
      return;
   }

   // Play selected tune
   if (sound_test != BGM_MENU)
      load_bgm(sound_test);
   play_bgm(sound_test);
}

//***************************************************************************
// button_previous [internal]
// Button for the sound test previous control. Selects the previous sound.
//***************************************************************************

static void button_previous(void) {
   // Wraparound?
   if (sound_test == BGM_NONE)
      sound_test = NUM_BGM;

   // Select previous BGM
   sound_test--;

   // Skip it?
   if (!bgm_exists(sound_test))
      button_previous();
}

//***************************************************************************
// button_next [internal]
// Button for the sound test next control. Selects the next sound.
//***************************************************************************

static void button_next(void) {
   // Select next BGM
   sound_test++;

   // Wraparound?
   if (sound_test == NUM_BGM)
      sound_test = BGM_NONE;

   // Skip it?
   if (!bgm_exists(sound_test))
      button_next();
}

//***************************************************************************
// button_dec_bgm_volume [internal]
// Callback when pressing left when BGM volume is selected. Decreases the
// BGM volume by 5%.
//***************************************************************************

static void button_dec_bgm_volume(void) {
   if (settings.bgm_volume == 0)
      settings.bgm_volume = 100;
   else
      settings.bgm_volume -= 5;
}

//***************************************************************************
// button_inc_bgm_volume [internal]
// Callback when pressing right when BGM volume is selected. Increases the
// BGM volume by 5%.
//***************************************************************************

static void button_inc_bgm_volume(void) {
   if (settings.bgm_volume == 100)
      settings.bgm_volume = 0;
   else
      settings.bgm_volume += 5;
}

//***************************************************************************
// button_dec_sfx_volume [internal]
// Callback when pressing left when SFX volume is selected. Decreases the
// SFX volume by 5%.
//***************************************************************************

static void button_dec_sfx_volume(void) {
   if (settings.sfx_volume == 0)
      settings.sfx_volume = 100;
   else
      settings.sfx_volume -= 5;
}

//***************************************************************************
// button_inc_sfx_volume [internal]
// Callback when pressing right when SFX volume is selected. Increases the
// SFX volume by 5%.
//***************************************************************************

static void button_inc_sfx_volume(void) {
   if (settings.sfx_volume == 100)
      settings.sfx_volume = 0;
   else
      settings.sfx_volume += 5;
}

//***************************************************************************
// button_stereo [internal]
// Button for toggling stereo and mono.
//***************************************************************************

static void button_stereo(void) {
   settings.stereo ^= 1;
}

//***************************************************************************
// button_reverse [internal]
// Button for toggling reverse stereo.
//***************************************************************************

static void button_reverse(void) {
   settings.reverse ^= 1;
}

//***************************************************************************
// button_back [internal]
// Button for going back. Returns to the options main menu.
//***************************************************************************

static void button_back(void) {
   fade_off_and_switch(GAMEMODE_OPTIONS);
}

//***************************************************************************
// button_one_switch [internal]
// Starts the one-switch editing mode for the bars.
//***************************************************************************

static void button_one_switch(void) {
   oneswitch_edit = 1;
   oneswitch_timer = settings.menu_delay;
}
