//***************************************************************************
// "options_video.c"
// Code for the options video menu.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdio.h>
#include "background.h"
#include "menu.h"
#include "reader.h"
#include "settings.h"
#include "text.h"
#include "video.h"

// UI positions
#define RATIO_X (screen_cx - 64)       // X: ratio buttons
#define RATIO_SX (RATIO_X - 48)        // X: smaller ratios
#define RATIO_LX (RATIO_X + 128)       // X: larger ratios
#define RATIO_Y1 (screen_cy - 64)      // Y: 4:3 ratio button
#define RATIO_Y2 (RATIO_Y1 + 24)       // Y: 16:9 ratio button
#define RATIO_Y3 (RATIO_Y2 + 24)       // Y: 16:10 ratio button

#define RATIO_TX (RATIO_X + 76)        // X: ratio resolutions
#define RATIO_TY (RATIO_Y1 + 12)       // Y: ratio resolutions

#define FULL_X (screen_cx - 0x6C)      // X: fullscreen
#define FULL_Y (RATIO_Y3 + 0x20)       // Y: fullscreen
#define VSYNC_X (screen_cx - 0x34)     // X: vsync
#define VSYNC_Y (FULL_Y)               // Y: vsync
#define FILTER_X (screen_cx + 0x04)    // X: filtering
#define FILTER_Y (FULL_Y)                // Y: filtering
#define BACK_X (screen_cx + 0x3C)      // X: back
#define BACK_Y (FULL_Y)                // Y: back

#define CONTR_X (screen_cx - 132)      // X: contrast
#define CONTR_Y (screen_cy - 0x38)     // Y: contrast
#define CONTR_BX1 (CONTR_X + 0x38)     // X: contrast "50%"
#define CONTR_BX2 (CONTR_BX1 + 0x28)   // X: contrast "75%"
#define CONTR_BX3 (CONTR_BX2 + 0x28)   // X: contrast "100%"
#define CONTR_BX4 (CONTR_BX3 + 0x28)   // X: contrast "150%"
#define CONTR_BX5 (CONTR_BX4 + 0x28)   // X: contrast "200%"
#define CONTR_BY (CONTR_Y + 0x08)      // Y: contrast values

#define BG_X (screen_cx - 0x6C)        // X: enable background
#define BG_Y (screen_cy + 0x08)        // Y: enable background
#define DIM_X (screen_cx - 0x34)       // X: dim background
#define DIM_Y (BG_Y)                   // Y: dim background
#define ZOOM_X (screen_cx + 0x04)      // X: zoom in
#define ZOOM_Y (BG_Y)                  // Y: zoom in
#define SHAKING_X (screen_cx + 0x3C)   // X: enable shaking
#define SHAKING_Y (BG_Y)               // Y: enable shaking

#define NEXT_X (screen_cx + 0x78)      // X: next page
#define NEXT_Y (screen_cy - 0x18)      // Y: next page
#define PREV_X (screen_cx - 0x98)      // X: previous page
#define PREV_Y (NEXT_Y + 0x10)         // Y: previous page

// Which page is currently visible
static unsigned page;

// List of buttons
enum {
   BUTTON_SMALLER,
   BUTTON_LARGER,
   BUTTON_RATIO1,
   BUTTON_RATIO2,
   BUTTON_RATIO3,
   BUTTON_FULLSCREEN,
   BUTTON_VSYNC,
   BUTTON_FILTER,
   BUTTON_BACK,

   BUTTON_CONTRAST,
   BUTTON_BACKGROUND,
   BUTTON_DIMBG,
   BUTTON_ZOOM,
   BUTTON_SHAKING,

   BUTTON_NEXT,
   BUTTON_PREV,
   NUM_BUTTONS
};

// To keep track of which resolution is selected within the valid list
static unsigned current_resolution;

// If set, the video mode needs to be reset
static unsigned changed = 0;

// Where graphics are stored
static GraphicsSet *gfxset = NULL;
static Sprite *spr_ratio[3][2][2];
static Sprite *spr_smaller[4];
static Sprite *spr_larger[4];
static Sprite *spr_fullscreen[2][4];
static Sprite *spr_vsync[2][4];
static Sprite *spr_filtering[2][4];
static Sprite *spr_contrast[4];
static Sprite *spr_contrast_button[5][2];
static Sprite *spr_background[2][4];
static Sprite *spr_dim[2][4];
static Sprite *spr_zoom[2][4];
static Sprite *spr_shaking[2][4];
static Sprite *spr_back[4];
static Sprite *spr_next[4];
static Sprite *spr_prev[4];

// Used to modify the contrast in one-switch mode
static unsigned oneswitch_edit;     // Set if modifying a setting
static unsigned oneswitch_timer;    // Delay before next value

// Used to animate stuff
static unsigned anim;
static unsigned button_anim;

// Private function prototypes
static void draw_page_1(void);
static void draw_page_2(void);
static void show_page_1(void);
static void show_page_2(void);
static void button_ratio_1(void);
static void button_ratio_2(void);
static void button_ratio_3(void);
static void set_ratio(unsigned);
static void button_smaller(void);
static void button_larger(void);
static void button_fullscreen(void);
static void button_vsync(void);
static void button_filtering(void);
static void button_dec_contrast(void);
static void button_inc_contrast(void);
static void button_background(void);
static void button_dim(void);
static void button_zoom(void);
static void button_shaking(void);
static void button_back(void);
static void button_one_switch(void);

//***************************************************************************
// load_options_video
// Loads the assets for the options video menu. Called while the main menu is
// loading (so everything loads in a bulk).
//***************************************************************************

void load_options_video(void) {
   // Load graphics
   gfxset = load_graphics_set("graphics/options_video");

   // Load sprites
#define SPR(name) get_sprite(gfxset, name)
   spr_ratio[0][0][0] = SPR("button_4_3_off_dim");
   spr_ratio[0][0][1] = SPR("button_4_3_off_lit");
   spr_ratio[0][1][0] = SPR("button_4_3_on_dim");
   spr_ratio[0][1][1] = SPR("button_4_3_on_lit");
   spr_ratio[1][0][0] = SPR("button_16_9_off_dim");
   spr_ratio[1][0][1] = SPR("button_16_9_off_lit");
   spr_ratio[1][1][0] = SPR("button_16_9_on_dim");
   spr_ratio[1][1][1] = SPR("button_16_9_on_lit");
   spr_ratio[2][0][0] = SPR("button_16_10_off_dim");
   spr_ratio[2][0][1] = SPR("button_16_10_off_lit");
   spr_ratio[2][1][0] = SPR("button_16_10_on_dim");
   spr_ratio[2][1][1] = SPR("button_16_10_on_lit");

   spr_smaller[0] = SPR("button_smaller_dim");
   spr_smaller[1] = SPR("button_smaller_lit_1");
   spr_smaller[2] = SPR("button_smaller_lit_2");
   spr_smaller[3] = SPR("button_smaller_lit_3");

   spr_larger[0] = SPR("button_larger_dim");
   spr_larger[1] = SPR("button_larger_lit_1");
   spr_larger[2] = SPR("button_larger_lit_2");
   spr_larger[3] = SPR("button_larger_lit_3");

   spr_fullscreen[0][0] = SPR("button_fullscreen_off_dim");
   spr_fullscreen[0][1] = SPR("button_fullscreen_off_lit_1");
   spr_fullscreen[0][2] = SPR("button_fullscreen_off_lit_2");
   spr_fullscreen[0][3] = SPR("button_fullscreen_off_lit_3");
   spr_fullscreen[1][0] = SPR("button_fullscreen_on_dim");
   spr_fullscreen[1][1] = SPR("button_fullscreen_on_lit_1");
   spr_fullscreen[1][2] = SPR("button_fullscreen_on_lit_2");
   spr_fullscreen[1][3] = SPR("button_fullscreen_on_lit_3");

   spr_vsync[0][0] = SPR("button_vsync_off_dim");
   spr_vsync[0][1] = SPR("button_vsync_off_lit_1");
   spr_vsync[0][2] = SPR("button_vsync_off_lit_2");
   spr_vsync[0][3] = SPR("button_vsync_off_lit_3");
   spr_vsync[1][0] = SPR("button_vsync_on_dim");
   spr_vsync[1][1] = SPR("button_vsync_on_lit_1");
   spr_vsync[1][2] = SPR("button_vsync_on_lit_2");
   spr_vsync[1][3] = SPR("button_vsync_on_lit_3");

   spr_filtering[0][0] = SPR("button_filtering_off_dim");
   spr_filtering[0][1] = SPR("button_filtering_off_lit_1");
   spr_filtering[0][2] = SPR("button_filtering_off_lit_2");
   spr_filtering[0][3] = SPR("button_filtering_off_lit_3");
   spr_filtering[1][0] = SPR("button_filtering_on_dim");
   spr_filtering[1][1] = SPR("button_filtering_on_lit_1");
   spr_filtering[1][2] = SPR("button_filtering_on_lit_2");
   spr_filtering[1][3] = SPR("button_filtering_on_lit_3");

   spr_contrast[0] = SPR("button_contrast_dim");
   spr_contrast[1] = SPR("button_contrast_lit_1");
   spr_contrast[2] = SPR("button_contrast_lit_2");
   spr_contrast[3] = SPR("button_contrast_lit_3");

   spr_contrast_button[CONTRAST_LOW][0] = SPR("button_contrast_low_off");
   spr_contrast_button[CONTRAST_LOW][1] = SPR("button_contrast_low_on");
   spr_contrast_button[CONTRAST_MIDLOW][0] = SPR("button_contrast_midlow_off");
   spr_contrast_button[CONTRAST_MIDLOW][1] = SPR("button_contrast_midlow_on");
   spr_contrast_button[CONTRAST_NORMAL][0] = SPR("button_contrast_medium_off");
   spr_contrast_button[CONTRAST_NORMAL][1] = SPR("button_contrast_medium_on");
   spr_contrast_button[CONTRAST_MIDHIGH][0] = SPR("button_contrast_midhigh_off");
   spr_contrast_button[CONTRAST_MIDHIGH][1] = SPR("button_contrast_midhigh_on");
   spr_contrast_button[CONTRAST_HIGH][0] = SPR("button_contrast_high_off");
   spr_contrast_button[CONTRAST_HIGH][1] = SPR("button_contrast_high_on");

   spr_background[0][0] = SPR("button_background_off_dim");
   spr_background[0][1] = SPR("button_background_off_lit_1");
   spr_background[0][2] = SPR("button_background_off_lit_2");
   spr_background[0][3] = SPR("button_background_off_lit_3");
   spr_background[1][0] = SPR("button_background_on_dim");
   spr_background[1][1] = SPR("button_background_on_lit_1");
   spr_background[1][2] = SPR("button_background_on_lit_2");
   spr_background[1][3] = SPR("button_background_on_lit_3");

   spr_dim[0][0] = SPR("button_dim_off_dim");
   spr_dim[0][1] = SPR("button_dim_off_lit_1");
   spr_dim[0][2] = SPR("button_dim_off_lit_2");
   spr_dim[0][3] = SPR("button_dim_off_lit_3");
   spr_dim[1][0] = SPR("button_dim_on_dim");
   spr_dim[1][1] = SPR("button_dim_on_lit_1");
   spr_dim[1][2] = SPR("button_dim_on_lit_2");
   spr_dim[1][3] = SPR("button_dim_on_lit_3");

   spr_zoom[0][0] = SPR("button_zoom_off_dim");
   spr_zoom[0][1] = SPR("button_zoom_off_lit_1");
   spr_zoom[0][2] = SPR("button_zoom_off_lit_2");
   spr_zoom[0][3] = SPR("button_zoom_off_lit_3");
   spr_zoom[1][0] = SPR("button_zoom_on_dim");
   spr_zoom[1][1] = SPR("button_zoom_on_lit_1");
   spr_zoom[1][2] = SPR("button_zoom_on_lit_2");
   spr_zoom[1][3] = SPR("button_zoom_on_lit_3");

   spr_shaking[0][0] = SPR("button_shaking_off_dim");
   spr_shaking[0][1] = SPR("button_shaking_off_lit_1");
   spr_shaking[0][2] = SPR("button_shaking_off_lit_2");
   spr_shaking[0][3] = spr_shaking[0][1];
   spr_shaking[1][0] = SPR("button_shaking_on_dim");
   spr_shaking[1][1] = SPR("button_shaking_on_lit_1");
   spr_shaking[1][2] = SPR("button_shaking_on_lit_2");
   spr_shaking[1][3] = spr_shaking[1][1];

   spr_back[0] = SPR("button_back_dim");
   spr_back[1] = SPR("button_back_lit_1");
   spr_back[2] = SPR("button_back_lit_2");
   spr_back[3] = SPR("button_back_lit_3");

   spr_next[0] = SPR("next_page_dim");
   spr_next[1] = SPR("next_page_lit_1");
   spr_next[2] = SPR("next_page_lit_2");
   spr_next[3] = SPR("next_page_lit_3");

   spr_prev[0] = SPR("previous_page_dim");
   spr_prev[1] = SPR("previous_page_lit_1");
   spr_prev[2] = SPR("previous_page_lit_2");
   spr_prev[3] = SPR("previous_page_lit_3");
#undef SPR
}

//***************************************************************************
// init_options_video
// Initializes the options video menu.
//***************************************************************************

void init_options_video(void) {
   // Check what's the current resolution
   for (unsigned i = 0; ; i++) {
      if (resolutions[i].width == settings.windowed_width &&
      resolutions[i].height == settings.windowed_height) {
         current_resolution = i;
         break;
      }
   }

   // Don't reset video mode yet!
   changed = 0;

   // Reset animations
   anim = 0;

   // Default to the first page
   show_page_1();
   oneswitch_edit = 0;
   oneswitch_timer = 0;

   // Make the cursor visible
   set_cursor(CURSOR_ARROW);

   // Make screen visible
   fade_on();
}

//***************************************************************************
// run_options_video
// Processes the logic for the options video menu.
//***************************************************************************

void run_options_video(void) {
   // Changing contrast? (in one-switch mode)
   if (oneswitch_edit) {
      // Delay between changes
      oneswitch_timer--;
      if (oneswitch_timer == 0) {
         oneswitch_timer = settings.menu_delay;

         // Change contrast as needed
         button_inc_contrast();
      }

      // Tap to be done
      if (input.oneswitch.tap || input.oneswitch.tap2) {
         oneswitch_edit = 0;
         play_sfx(SFX_OK);
      }
   }

   // Update the menu
   else {
      update_menu();
   }

   // Click on the contrast buttons to change the contrast
   // (this is in page 2 if you wonder)
   if (page == 2)
   for (int i = CONTRAST_LOW; i <= CONTRAST_HIGH; i++) {
      if (input.cursor.click &&
      input.cursor.x >= CONTR_BX1+(i*0x28) &&
      input.cursor.x <= CONTR_BX1+(i*0x28) + 0x1F &&
      input.cursor.y >= CONTR_BY &&
      input.cursor.y <= CONTR_BY + 0x1F) {
         settings.contrast = i;
         play_sfx(SFX_OK);
         break;
      }
   }

   // Update background animation
   update_background();

   // Update animations
   anim++;
   switch (anim >> 3 & 3) {
      case 0: button_anim = 1; break;
      case 1: button_anim = 2; break;
      case 2: button_anim = 3; break;
      case 3: button_anim = 2; break;
   }
}

//***************************************************************************
// draw_options_video
// Draws the options video menu.
//***************************************************************************

void draw_options_video(void) {
   // Draw background
   draw_background();
   if (settings.zoom)
      zoom_screen();

   // Draw buttons
   switch (page) {
      case 1: draw_page_1(); break;
      case 2: draw_page_2(); break;
      default: break;
   }

   // If an option is selected, show a border around it as well as its
   // caption above all the buttons
   if (menu.selected != -1) {
      // Where the data for the selected option is stored
      MenuOption *ptr = &menu.options[menu.selected];

      // Draw the border on top of the button
      if (anim & 0x08 && menu.selected < BUTTON_NEXT) {
         draw_rectangle(ptr->box.x1+1, ptr->box.y1+1,
            ptr->box.x2-1, ptr->box.y2-1, settings.box_lit[1]);
         draw_rectangle(ptr->box.x1, ptr->box.y1,
            ptr->box.x2, ptr->box.y2, settings.box_lit[0]);
         draw_rectangle(ptr->box.x1-1, ptr->box.y1-1,
            ptr->box.x2+1, ptr->box.y2+1, settings.box_lit[0]);
         draw_rectangle(ptr->box.x1-2, ptr->box.y1-2,
            ptr->box.x2+2, ptr->box.y2+2, settings.box_lit[1]);
         draw_hline(ptr->box.x1-2, ptr->box.y2+3,
            ptr->box.x2+2, settings.box_lit[2]);
      }

      // Determine "param" to use for the caption
      const char *param;
      switch (menu.selected) {
         case BUTTON_RATIO1: param = "4:3"; break;
         case BUTTON_RATIO2: param = "16:9"; break;
         case BUTTON_RATIO3: param = "16:10"; break;

         case BUTTON_FULLSCREEN:
            param = settings.fullscreen ?
               text.options_video.enabled :
               text.options_video.disabled;
            break;

         case BUTTON_VSYNC:
            param = settings.vsync ?
               text.options_video.enabled :
               text.options_video.disabled;
            break;

         case BUTTON_FILTER:
            param = settings.filter ?
               text.options_video.enabled :
               text.options_video.disabled;
            break;

         case BUTTON_CONTRAST:
            switch (settings.contrast) {
               /*
               case CONTRAST_LOW: param = text.options_video.low; break;
               case CONTRAST_NORMAL: param = text.options_video.medium; break;
               case CONTRAST_HIGH: param = text.options_video.high; break;
               */
               case CONTRAST_LOW: param = "50%"; break;
               case CONTRAST_MIDLOW: param = "75%"; break;
               case CONTRAST_NORMAL: param = "100%"; break;
               case CONTRAST_MIDHIGH: param = "150%"; break;
               case CONTRAST_HIGH: param = "200%"; break;
               default: param = "???"; break;
            }
            break;

         case BUTTON_BACKGROUND:
            param = settings.background ?
               text.options_video.show :
               text.options_video.hide;
            break;

         case BUTTON_DIMBG:
            param = settings.dark_bg ?
               text.options_video.dim :
               text.options_video.no_dim;
            break;

         case BUTTON_ZOOM:
            param = settings.zoom ?
               text.options_video.zoomed :
               text.options_video.no_zoom;
            break;

         case BUTTON_SHAKING:
            param = settings.shaking ?
               text.options_video.enabled :
               text.options_video.disabled;
            break;

         default:
            param = "???";
            break;
      }

      // Draw the caption
      draw_text_str(ptr->caption, param,
         screen_cx, screen_h - 0x10, FONT_LIT, ALIGN_BOTTOM);

      // Output option to the screen reader
      switch (menu.selected) {
         // For the resolutions, show the currently selected one
         case BUTTON_RATIO1:
         case BUTTON_RATIO2:
         case BUTTON_RATIO3: {
            unsigned which = current_resolution / 3 * 3 +
               (menu.selected - BUTTON_RATIO1);
            char buffer[0x40];
            sprintf(buffer, "%u×%u", resolutions[which].width,
               resolutions[which].height);
            set_reader_text(buffer);
         } break;

         // Show caption as-is for other options
         default:
            set_reader_text_str(ptr->caption, param);
            break;
      }
   }

   // Draw title
   draw_text(text.options.video, screen_cx, 0x10, FONT_LIT, ALIGN_TOP);
}

//***************************************************************************
// draw_page_1 [internal]
// Draws the buttons from the first page.
//***************************************************************************

static void draw_page_1(void) {
   // Resolution list
   draw_sprite(spr_smaller
      [menu.selected == BUTTON_SMALLER ? button_anim : 0],
      RATIO_SX, RATIO_Y1, SPR_NOFLIP);
   draw_sprite(spr_ratio[0][current_resolution % 3 == 0 ? 1 : 0]
      [menu.selected == BUTTON_RATIO1 ? 1 : 0],
      RATIO_X, RATIO_Y1, SPR_NOFLIP);
   draw_sprite(spr_ratio[1][current_resolution % 3 == 1 ? 1 : 0]
      [menu.selected == BUTTON_RATIO2 ? 1 : 0],
      RATIO_X, RATIO_Y2, SPR_NOFLIP);
   draw_sprite(spr_ratio[2][current_resolution % 3 == 2 ? 1 : 0]
      [menu.selected == BUTTON_RATIO3 ? 1 : 0],
      RATIO_X, RATIO_Y3, SPR_NOFLIP);
   draw_sprite(spr_larger
      [menu.selected == BUTTON_LARGER ? button_anim : 0],
      RATIO_LX, RATIO_Y1, SPR_NOFLIP);

   // Draw resolutions
   for (unsigned i = 0; i < 3; i++) {
      // Get ID of this resolution
      unsigned which = current_resolution / 3 * 3 + i;

      // Generate label
      char buffer[0x40];
      sprintf(buffer, "%u×%u", resolutions[which].width,
         resolutions[which].height);

      // Draw resolution
      draw_text(buffer, RATIO_TX, RATIO_TY + i*24,
         FONT_LIT, ALIGN_CENTER);
   }

   // Other buttons
   draw_sprite(spr_fullscreen[settings.fullscreen]
      [menu.selected == BUTTON_FULLSCREEN ? button_anim : 0],
      FULL_X, FULL_Y, SPR_NOFLIP);
   draw_sprite(spr_vsync[settings.vsync]
      [menu.selected == BUTTON_VSYNC ? button_anim : 0],
      VSYNC_X, VSYNC_Y, SPR_NOFLIP);
   draw_sprite(spr_filtering[settings.filter]
      [menu.selected == BUTTON_FILTER ? button_anim : 0],
      FILTER_X, FILTER_Y, SPR_NOFLIP);
   draw_sprite(spr_back
      [menu.selected == BUTTON_BACK ? button_anim : 0],
      BACK_X, BACK_Y, SPR_NOFLIP);

   // Next page
   draw_sprite(spr_next
      [menu.selected == BUTTON_NEXT ? button_anim : 0],
      NEXT_X, NEXT_Y, SPR_NOFLIP);
}

//***************************************************************************
// draw_page_2 [internal]
// Draws the buttons from the sceond page.
//***************************************************************************

static void draw_page_2(void) {
   // Draw contrast
   draw_sprite(spr_contrast
      [menu.selected == BUTTON_CONTRAST ? button_anim : 0],
      CONTR_X, CONTR_Y, SPR_NOFLIP);
   draw_sprite(spr_contrast_button[CONTRAST_LOW]
      [settings.contrast == CONTRAST_LOW ? 1 : 0],
      CONTR_BX1, CONTR_BY, SPR_NOFLIP);
   draw_sprite(spr_contrast_button[CONTRAST_MIDLOW]
      [settings.contrast == CONTRAST_MIDLOW ? 1 : 0],
      CONTR_BX2, CONTR_BY, SPR_NOFLIP);
   draw_sprite(spr_contrast_button[CONTRAST_NORMAL]
      [settings.contrast == CONTRAST_NORMAL ? 1 : 0],
      CONTR_BX3, CONTR_BY, SPR_NOFLIP);
   draw_sprite(spr_contrast_button[CONTRAST_MIDHIGH]
      [settings.contrast == CONTRAST_MIDHIGH ? 1 : 0],
      CONTR_BX4, CONTR_BY, SPR_NOFLIP);
   draw_sprite(spr_contrast_button[CONTRAST_HIGH]
      [settings.contrast == CONTRAST_HIGH ? 1 : 0],
      CONTR_BX5, CONTR_BY, SPR_NOFLIP);

   // Other buttons
   draw_sprite(spr_background[settings.background]
      [menu.selected == BUTTON_BACKGROUND ? button_anim : 0],
      BG_X, BG_Y, SPR_NOFLIP);
   draw_sprite(spr_dim[settings.dark_bg]
      [menu.selected == BUTTON_DIMBG ? button_anim : 0],
      DIM_X, DIM_Y, SPR_NOFLIP);
   draw_sprite(spr_zoom[settings.zoom]
      [menu.selected == BUTTON_ZOOM ? button_anim : 0],
      ZOOM_X, ZOOM_Y, SPR_NOFLIP);
   draw_sprite(spr_shaking[settings.shaking]
      [menu.selected == BUTTON_SHAKING ? button_anim : 0],
      SHAKING_X, SHAKING_Y, SPR_NOFLIP);

   // Previous page
   draw_sprite(spr_prev
      [menu.selected == BUTTON_PREV ? button_anim : 0],
      PREV_X, PREV_Y, SPR_NOFLIP);
}

//***************************************************************************
// deinit_options_video
// Deinitializes the options video menu.
//***************************************************************************

void deinit_options_video(void) {
   // Reinitialize video mode if needed
   if (changed) {
      // Update window size
      settings.windowed_width = resolutions[current_resolution].width;
      settings.windowed_height = resolutions[current_resolution].height;

      // Reinitialize video subsystem
      deinit_video();
      reset_input();
      init_video();
   }
}

//***************************************************************************
// unload_options_video
// Unloads the assets for the options video menu menu. Called while the main
// menu is unloading (as everything was loaded in a bulk).
//***************************************************************************

void unload_options_video(void) {
   // Unload graphics
   destroy_graphics_set(gfxset);
}

//***************************************************************************
// show_page_1 [internal]
// Shows the first page of the menu.
//***************************************************************************

static void show_page_1(void) {
   // Show first page
   page = 1;

   // Initialize menu
   init_menu();
   set_reinit_menu_func(show_page_1);

   menu.cancel = button_back;
   menu.defoption.up = BUTTON_FULLSCREEN;
   menu.defoption.down = BUTTON_RATIO1;
   menu.defoption.left = BUTTON_NEXT;
   menu.defoption.right = BUTTON_FULLSCREEN;
   menu.selected = settings.one_switch ?
      BUTTON_SMALLER : BUTTON_FULLSCREEN;

   // Don't include the resolution settings in audiovideo mode
   // The whole section is (mostly?) useless, but we're still figuring out
   // what to do and this particular setting can get really annoying in case
   // somebody attempts to get in here
   if (!settings.audiovideo) {
      // Decrease resolution
      menu.options[BUTTON_SMALLER].box.x1 = RATIO_SX;
      menu.options[BUTTON_SMALLER].box.y1 = RATIO_Y1;
      menu.options[BUTTON_SMALLER].box.x2 = RATIO_SX + 47;
      menu.options[BUTTON_SMALLER].box.y2 = RATIO_Y3 + 23;
      menu.options[BUTTON_SMALLER].move.up = BUTTON_FULLSCREEN;
      menu.options[BUTTON_SMALLER].move.down = BUTTON_FULLSCREEN;
      menu.options[BUTTON_SMALLER].move.left = BUTTON_RATIO2;
      menu.options[BUTTON_SMALLER].move.right = BUTTON_RATIO2;
      menu.options[BUTTON_SMALLER].move.oneswitch = BUTTON_RATIO1;
      menu.options[BUTTON_SMALLER].action.accept = button_smaller;
      menu.options[BUTTON_SMALLER].caption = text.options_video.smaller;

      // Increase resolution
      menu.options[BUTTON_LARGER].box.x1 = RATIO_LX;
      menu.options[BUTTON_LARGER].box.y1 = RATIO_Y1;
      menu.options[BUTTON_LARGER].box.x2 = RATIO_LX + 47;
      menu.options[BUTTON_LARGER].box.y2 = RATIO_Y3 + 23;
      menu.options[BUTTON_LARGER].move.up = BUTTON_BACK;
      menu.options[BUTTON_LARGER].move.down = BUTTON_BACK;
      menu.options[BUTTON_LARGER].move.left = BUTTON_RATIO2;
      menu.options[BUTTON_LARGER].move.right = BUTTON_RATIO2;
      menu.options[BUTTON_LARGER].move.oneswitch = BUTTON_FULLSCREEN;
      menu.options[BUTTON_LARGER].action.accept = button_larger;
      menu.options[BUTTON_LARGER].caption = text.options_video.larger;

      // 4:3 proportion
      menu.options[BUTTON_RATIO1].box.x1 = RATIO_X;
      menu.options[BUTTON_RATIO1].box.y1 = RATIO_Y1;
      menu.options[BUTTON_RATIO1].box.x2 = RATIO_X + 127;
      menu.options[BUTTON_RATIO1].box.y2 = RATIO_Y1 + 23;
      menu.options[BUTTON_RATIO1].move.up = BUTTON_VSYNC;
      menu.options[BUTTON_RATIO1].move.down = BUTTON_RATIO2;
      menu.options[BUTTON_RATIO1].move.oneswitch = BUTTON_RATIO2;
      menu.options[BUTTON_RATIO1].action.left = button_smaller;
      menu.options[BUTTON_RATIO1].action.right = button_larger;
      menu.options[BUTTON_RATIO1].action.accept = button_ratio_1;
      menu.options[BUTTON_RATIO1].caption = text.options_video.ratio;

      // 16:9 proportion
      menu.options[BUTTON_RATIO2].box.x1 = RATIO_X;
      menu.options[BUTTON_RATIO2].box.y1 = RATIO_Y2;
      menu.options[BUTTON_RATIO2].box.x2 = RATIO_X + 127;
      menu.options[BUTTON_RATIO2].box.y2 = RATIO_Y2 + 23;
      menu.options[BUTTON_RATIO2].move.up = BUTTON_RATIO1;
      menu.options[BUTTON_RATIO2].move.down = BUTTON_RATIO3;
      menu.options[BUTTON_RATIO2].move.oneswitch = BUTTON_RATIO3;
      menu.options[BUTTON_RATIO2].action.left = button_smaller;
      menu.options[BUTTON_RATIO2].action.right = button_larger;
      menu.options[BUTTON_RATIO2].action.accept = button_ratio_2;
      menu.options[BUTTON_RATIO2].caption = text.options_video.ratio;

      // 16:10 proportion
      menu.options[BUTTON_RATIO3].box.x1 = RATIO_X;
      menu.options[BUTTON_RATIO3].box.y1 = RATIO_Y3;
      menu.options[BUTTON_RATIO3].box.x2 = RATIO_X + 127;
      menu.options[BUTTON_RATIO3].box.y2 = RATIO_Y3 + 23;
      menu.options[BUTTON_RATIO3].move.up = BUTTON_RATIO2;
      menu.options[BUTTON_RATIO3].move.down = BUTTON_VSYNC;
      menu.options[BUTTON_RATIO3].move.oneswitch = BUTTON_LARGER;
      menu.options[BUTTON_RATIO3].action.left = button_smaller;
      menu.options[BUTTON_RATIO3].action.right = button_larger;
      menu.options[BUTTON_RATIO3].action.accept = button_ratio_3;
      menu.options[BUTTON_RATIO3].caption = text.options_video.ratio;
   }

   // Toggle fullscreen
   menu.options[BUTTON_FULLSCREEN].box.x1 = FULL_X;
   menu.options[BUTTON_FULLSCREEN].box.y1 = FULL_Y;
   menu.options[BUTTON_FULLSCREEN].box.x2 = FULL_X + 0x2F;
   menu.options[BUTTON_FULLSCREEN].box.y2 = FULL_Y + 0x2F;
   menu.options[BUTTON_FULLSCREEN].move.up = BUTTON_RATIO3;
   menu.options[BUTTON_FULLSCREEN].move.down = BUTTON_RATIO1;
   menu.options[BUTTON_FULLSCREEN].move.left = BUTTON_NEXT;
   menu.options[BUTTON_FULLSCREEN].move.right = BUTTON_VSYNC;
   menu.options[BUTTON_FULLSCREEN].move.oneswitch = BUTTON_VSYNC;
   menu.options[BUTTON_FULLSCREEN].action.accept = button_fullscreen;
   menu.options[BUTTON_FULLSCREEN].caption = text.options_video.fullscreen;

   // Toggle vsync
   menu.options[BUTTON_VSYNC].box.x1 = VSYNC_X;
   menu.options[BUTTON_VSYNC].box.y1 = VSYNC_Y;
   menu.options[BUTTON_VSYNC].box.x2 = VSYNC_X + 0x2F;
   menu.options[BUTTON_VSYNC].box.y2 = VSYNC_Y + 0x2F;
   menu.options[BUTTON_VSYNC].move.up = BUTTON_RATIO3;
   menu.options[BUTTON_VSYNC].move.down = BUTTON_RATIO1;
   menu.options[BUTTON_VSYNC].move.left = BUTTON_FULLSCREEN;
   menu.options[BUTTON_VSYNC].move.right = BUTTON_FILTER;
   menu.options[BUTTON_VSYNC].move.oneswitch = BUTTON_FILTER;
   menu.options[BUTTON_VSYNC].action.accept = button_vsync;
   menu.options[BUTTON_VSYNC].caption = text.options_video.vsync;

   // Toggle filtering
   menu.options[BUTTON_FILTER].box.x1 = FILTER_X;
   menu.options[BUTTON_FILTER].box.y1 = FILTER_Y;
   menu.options[BUTTON_FILTER].box.x2 = FILTER_X + 0x2F;
   menu.options[BUTTON_FILTER].box.y2 = FILTER_Y + 0x2F;
   menu.options[BUTTON_FILTER].move.up = BUTTON_RATIO3;
   menu.options[BUTTON_FILTER].move.down = BUTTON_RATIO1;
   menu.options[BUTTON_FILTER].move.left = BUTTON_VSYNC;
   menu.options[BUTTON_FILTER].move.right = BUTTON_BACK;
   menu.options[BUTTON_FILTER].move.oneswitch = BUTTON_BACK;
   menu.options[BUTTON_FILTER].action.accept = button_filtering;
   menu.options[BUTTON_FILTER].caption = text.options_video.filtering;

   // Back button
   menu.options[BUTTON_BACK].box.x1 = BACK_X;
   menu.options[BUTTON_BACK].box.y1 = BACK_Y;
   menu.options[BUTTON_BACK].box.x2 = BACK_X + 0x2F;
   menu.options[BUTTON_BACK].box.y2 = BACK_Y + 0x2F;
   menu.options[BUTTON_BACK].move.up = BUTTON_RATIO3;
   menu.options[BUTTON_BACK].move.down = BUTTON_RATIO1;
   menu.options[BUTTON_BACK].move.left = BUTTON_FILTER;
   menu.options[BUTTON_BACK].move.right = BUTTON_NEXT;
   menu.options[BUTTON_BACK].move.oneswitch = BUTTON_NEXT;
   menu.options[BUTTON_BACK].action.accept = button_back;
   menu.options[BUTTON_BACK].caption = text.options.back;
   menu.options[BUTTON_BACK].sfx = SFX_CANCEL;

   // Next page
   menu.options[BUTTON_NEXT].box.x1 = NEXT_X;
   menu.options[BUTTON_NEXT].box.y1 = NEXT_Y;
   menu.options[BUTTON_NEXT].box.x2 = NEXT_X + 0x1F;
   menu.options[BUTTON_NEXT].box.y2 = NEXT_Y + 0x2F;
   menu.options[BUTTON_NEXT].move.up = BUTTON_RATIO1;
   menu.options[BUTTON_NEXT].move.down = BUTTON_BACK;
   menu.options[BUTTON_NEXT].move.left = BUTTON_BACK;
   menu.options[BUTTON_NEXT].move.right = BUTTON_FULLSCREEN;
   menu.options[BUTTON_NEXT].move.oneswitch = BUTTON_SMALLER;
   menu.options[BUTTON_NEXT].action.accept = show_page_2;
   menu.options[BUTTON_NEXT].caption = text.options_video.next;
}

//***************************************************************************
// show_page_2 [internal]
// Shows the second page of the menu.
//***************************************************************************

static void show_page_2(void) {
   // Show second page
   page = 2;

   // Initialize menu
   init_menu();
   set_reinit_menu_func(show_page_2);

   menu.cancel = button_back;
   menu.defoption.up = BUTTON_BACKGROUND;
   menu.defoption.down = BUTTON_CONTRAST;
   menu.defoption.left = BUTTON_BACK;
   menu.defoption.right = BUTTON_PREV;
   menu.selected = settings.one_switch ?
      BUTTON_CONTRAST : BUTTON_BACKGROUND;

   // Contrast
   menu.options[BUTTON_CONTRAST].box.x1 = CONTR_X;
   menu.options[BUTTON_CONTRAST].box.y1 = CONTR_Y;
   menu.options[BUTTON_CONTRAST].box.x2 = CONTR_X + 263;
   menu.options[BUTTON_CONTRAST].box.y2 = CONTR_Y + 0x2F;
   menu.options[BUTTON_CONTRAST].move.up = BUTTON_DIMBG;
   menu.options[BUTTON_CONTRAST].move.down = BUTTON_DIMBG;
   menu.options[BUTTON_CONTRAST].move.oneswitch = BUTTON_BACKGROUND;
   menu.options[BUTTON_CONTRAST].action.left = button_dec_contrast;
   menu.options[BUTTON_CONTRAST].action.right = button_inc_contrast;
   menu.options[BUTTON_CONTRAST].action.oneswitch = button_one_switch;
   menu.options[BUTTON_CONTRAST].caption = text.options_video.contrast;

   // Toggle background
   menu.options[BUTTON_BACKGROUND].box.x1 = BG_X;
   menu.options[BUTTON_BACKGROUND].box.y1 = BG_Y;
   menu.options[BUTTON_BACKGROUND].box.x2 = BG_X + 0x2F;
   menu.options[BUTTON_BACKGROUND].box.y2 = BG_Y + 0x2F;
   menu.options[BUTTON_BACKGROUND].move.up = BUTTON_CONTRAST;
   menu.options[BUTTON_BACKGROUND].move.down = BUTTON_CONTRAST;
   menu.options[BUTTON_BACKGROUND].move.left = BUTTON_PREV;
   menu.options[BUTTON_BACKGROUND].move.right = BUTTON_DIMBG;
   menu.options[BUTTON_BACKGROUND].move.oneswitch = BUTTON_DIMBG;
   menu.options[BUTTON_BACKGROUND].action.accept = button_background;
   menu.options[BUTTON_BACKGROUND].caption = text.options_video.background;

   // Dim background
   menu.options[BUTTON_DIMBG].box.x1 = DIM_X;
   menu.options[BUTTON_DIMBG].box.y1 = DIM_Y;
   menu.options[BUTTON_DIMBG].box.x2 = DIM_X + 0x2F;
   menu.options[BUTTON_DIMBG].box.y2 = DIM_Y + 0x2F;
   menu.options[BUTTON_DIMBG].move.up = BUTTON_CONTRAST;
   menu.options[BUTTON_DIMBG].move.down = BUTTON_CONTRAST;
   menu.options[BUTTON_DIMBG].move.left = BUTTON_BACKGROUND;
   menu.options[BUTTON_DIMBG].move.right = BUTTON_ZOOM;
   menu.options[BUTTON_DIMBG].move.oneswitch = BUTTON_ZOOM;
   menu.options[BUTTON_DIMBG].action.accept = button_dim;
   menu.options[BUTTON_DIMBG].caption = text.options_video.dim_bg;

   // Zoom in screen
   menu.options[BUTTON_ZOOM].box.x1 = ZOOM_X;
   menu.options[BUTTON_ZOOM].box.y1 = ZOOM_Y;
   menu.options[BUTTON_ZOOM].box.x2 = ZOOM_X + 0x2F;
   menu.options[BUTTON_ZOOM].box.y2 = ZOOM_Y + 0x2F;
   menu.options[BUTTON_ZOOM].move.up = BUTTON_CONTRAST;
   menu.options[BUTTON_ZOOM].move.down = BUTTON_CONTRAST;
   menu.options[BUTTON_ZOOM].move.left = BUTTON_DIMBG;
   menu.options[BUTTON_ZOOM].move.right = BUTTON_SHAKING;
   menu.options[BUTTON_ZOOM].move.oneswitch = BUTTON_SHAKING;
   menu.options[BUTTON_ZOOM].action.accept = button_zoom;
   menu.options[BUTTON_ZOOM].caption = text.options_video.zoom;

   // Back button
   menu.options[BUTTON_SHAKING].box.x1 = SHAKING_X;
   menu.options[BUTTON_SHAKING].box.y1 = SHAKING_Y;
   menu.options[BUTTON_SHAKING].box.x2 = SHAKING_X + 0x2F;
   menu.options[BUTTON_SHAKING].box.y2 = SHAKING_Y + 0x2F;
   menu.options[BUTTON_SHAKING].move.up = BUTTON_CONTRAST;
   menu.options[BUTTON_SHAKING].move.down = BUTTON_CONTRAST;
   menu.options[BUTTON_SHAKING].move.left = BUTTON_ZOOM;
   menu.options[BUTTON_SHAKING].move.right = BUTTON_PREV;
   menu.options[BUTTON_SHAKING].move.oneswitch = BUTTON_PREV;
   menu.options[BUTTON_SHAKING].action.accept = button_shaking;
   menu.options[BUTTON_SHAKING].caption = text.options_video.shaking;

   // Previous page
   menu.options[BUTTON_PREV].box.x1 = PREV_X;
   menu.options[BUTTON_PREV].box.y1 = PREV_Y;
   menu.options[BUTTON_PREV].box.x2 = PREV_X + 0x1F;
   menu.options[BUTTON_PREV].box.y2 = PREV_Y + 0x2F;
   menu.options[BUTTON_PREV].move.up = BUTTON_CONTRAST;
   menu.options[BUTTON_PREV].move.down = BUTTON_BACKGROUND;
   menu.options[BUTTON_PREV].move.left = BUTTON_SHAKING;
   menu.options[BUTTON_PREV].move.right = BUTTON_BACKGROUND;
   menu.options[BUTTON_PREV].move.oneswitch = BUTTON_CONTRAST;
   menu.options[BUTTON_PREV].action.accept = show_page_1;
   menu.options[BUTTON_PREV].caption = text.options_video.prev;
   menu.options[BUTTON_PREV].sfx = SFX_CANCEL;
}

//***************************************************************************
// button_ratio_* [internal]
// Set a different resolution ratio.
//***************************************************************************

static void button_ratio_1(void) { set_ratio(0); }
static void button_ratio_2(void) { set_ratio(1); }
static void button_ratio_3(void) { set_ratio(2); }

//***************************************************************************
// set_ratio [internal]
// Sets the resolution ratio to the specified one.
//***************************************************************************

static void set_ratio(unsigned which) {
   current_resolution /= 3;
   current_resolution *= 3;
   current_resolution += which;
   changed = 1;
}

//***************************************************************************
// button_smaller [internal]
// Chooses smaller resolutions.
//***************************************************************************

static void button_smaller(void) {
   if (current_resolution < 3)
      current_resolution += NUM_RESOLUTIONS;
   current_resolution -= 3;
   changed = 1;
}

//***************************************************************************
// button_larger [internal]
// Chooses larger resolutions.
//***************************************************************************

static void button_larger(void) {
   current_resolution += 3;
   if (current_resolution >= NUM_RESOLUTIONS)
      current_resolution -= NUM_RESOLUTIONS;
   changed = 1;
}

//***************************************************************************
// button_fullscreen [internal]
// Toggles fullscreen mode.
//***************************************************************************

static void button_fullscreen(void) {
   settings.fullscreen ^= 1;
   changed = 1;
}

//***************************************************************************
// button_vsync [internal]
// Toggles use of vsync.
//***************************************************************************

static void button_vsync(void) {
   settings.vsync ^= 1;
   changed = 1;
}

//***************************************************************************
// button_filtering [internal]
// Toggles bilinear filtering.
//***************************************************************************

static void button_filtering(void) {
   settings.filter ^= 1;
   changed = 1;
}

//***************************************************************************
// button_dec_contrast [internal]
// Callback when pressing left when contrast is selected. Decreases the
// contrast.
//***************************************************************************

static void button_dec_contrast(void) {
   if (settings.contrast == CONTRAST_LOW)
      settings.contrast = CONTRAST_HIGH;
   else
      settings.contrast--;
}

//***************************************************************************
// button_inc_contrast [internal]
// Callback when pressing right when contrast is selected. Increases the
// contrast.
//***************************************************************************

static void button_inc_contrast(void) {
   if (settings.contrast == CONTRAST_HIGH)
      settings.contrast = CONTRAST_LOW;
   else
      settings.contrast++;
}

//***************************************************************************
// button_background [internal]
// Toggles the background.
//***************************************************************************

static void button_background(void) {
   settings.background ^= 1;
}

//***************************************************************************
// button_dim [internal]
// Toggles dimmig of the background.
//***************************************************************************

static void button_dim(void) {
   settings.dark_bg ^= 1;
}

//***************************************************************************
// button_zoom [internal]
// Toggles zooming in.
//***************************************************************************

static void button_zoom(void) {
   settings.zoom ^= 1;
}

//***************************************************************************
// button_shaking [internal]
// Toggles shaking (quakes, etc.).
//***************************************************************************

static void button_shaking(void) {
   settings.shaking ^= 1;
}

//***************************************************************************
// button_back [internal]
// Button for going back. Returns to the options main menu.
//***************************************************************************

static void button_back(void) {
   fade_off_and_switch(GAMEMODE_OPTIONS);
}

//***************************************************************************
// button_one_switch [internal]
// Starts the one-switch editing mode for the contrast.
//***************************************************************************

static void button_one_switch(void) {
   oneswitch_edit = 1;
   oneswitch_timer = settings.menu_delay;
}
