//***************************************************************************
// "particles.c"
// Code for particle objects.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include "ingame.h"
#include "objects.h"

// Some behavior parameters...
#define PARTICLE_WEIGHT 0x20        // Weight of a generic particle

// How long do some particles last. Note that this should match their
// animation!
#define SMOKE_LEN          24       // Smoke
#define SPARKLE_LEN        20       // Invincibility sparkle

//***************************************************************************
// init_scrap_gear
// Initializes a gear scrap object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_scrap_gear(Object *obj) {
   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_SCRAPGEAR));
}

//***************************************************************************
// init_scrap_spring
// Initializes a spring scrap object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_scrap_spring(Object *obj) {
   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_SCRAPSPRING));
}

//***************************************************************************
// init_smoke
// Initializes a smoke object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_smoke(Object *obj) {
   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_SMOKE));

   // Set how long this particle lasts
   obj->timer = SMOKE_LEN;
}

//***************************************************************************
// init_sparkle
// Initializes an invincibility sparkle object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_sparkle(Object *obj) {
   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_SPARKLE));

   // Set how long this particle lasts
   obj->timer = SPARKLE_LEN;
}

//***************************************************************************
// run_falling_particle
// Logic for generic falling particles.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_falling_particle(Object *obj) {
   // Apply momentum
   obj->x += speed_to_int(obj->speed);
   obj->y += speed_to_int(obj->gravity);
   obj->gravity += PARTICLE_WEIGHT;

   // Are we too far from the screen to bother?
   if (is_too_far(obj)) {
      delete_object(obj);
      return;
   }
}

//***************************************************************************
// run_expiring_particle
// Logic for generic expiring particles.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_expiring_particle(Object *obj) {
   // Apply momentum (if any)
   obj->x += speed_to_int(obj->speed);
   obj->y += speed_to_int(obj->gravity);

   // Expire after a while
   obj->timer--;
   if (obj->timer == 0) {
      delete_object(obj);
      return;
   }
}
