//***************************************************************************
// "reader.c"
// Screen reader code. Handles the output to the screen reader.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// This version of Sol was built on a MinGW toolchain that didn't have SAPI
// yet. I managed to get sapi.h from a newer version but not the relevant
// libraries yet so this hack works around it to get a missing part. Get rid
// of it if you're using an up-to-date MinGW!
#define MINGW_SAPI_HACK

// Set which platform-specific APIs are supported in each OS
#ifdef _WIN32
#define HAS_SAPI 1            // SAPI
#endif
#ifdef __linux__
#define HAS_SPEECHD 1         // Speech-dispatcher
#endif

// Required headers
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <SDL2/SDL.h>
#include "main.h"
#include "reader.h"
#include "settings.h"
#include "text.h"

#ifdef HAS_SAPI
#ifdef MINGW_SAPI_HACK
#ifdef __MINGW32__
#warning "MinGW SAPI hack active, if you have an up-to-date version disable it!"
#warning "Look at the top of reader.c to see how to disable this warning"
#define INITGUID
#endif
#endif
#include <windows.h>
#include <sapi.h>
#endif

#ifdef HAS_SPEECHD
#include <libspeechd.h>
#endif

// Buffer containing the string that will be shown to the screen reader
static char *textbuffer = NULL;     // Back buffer
static char *oldbuffer = NULL;      // Front buffer

// Output settings
static int punctuation = 0;         // Spell out punctuation

// SAPI stuff
#ifdef HAS_SAPI
static ISpVoice *sapi_voice = NULL;
#endif

// Speech-dispatcher stuff
#ifdef HAS_SPEECHD
static SPDConnection *speechd_connection = NULL;
#endif

//***************************************************************************
// init_reader
// Initializes the screen reader.
//***************************************************************************

void init_reader(void) {
#ifdef DEBUG
   fputs("Initializing screen reader\n", stderr);
#endif

#ifdef HAS_SAPI
   // Initialize SAPI if using that
   if (settings.reader == READER_NATIVE) {
      // Initialize COM
      if (FAILED(CoInitialize(NULL)))
         abort_program(ERR_INITREADER, NULL);

      // Get a SAPI voice
      if (FAILED(CoCreateInstance(&CLSID_SpVoice, NULL, CLSCTX_ALL,
      &IID_ISpVoice, (void **) &sapi_voice)))
         abort_program(ERR_INITREADER, NULL);

      // We do this so we can cut text immediately (this happens when the
      // textbuffer changes and we want to output the new data)
      sapi_voice->lpVtbl->SetAlertBoundary(sapi_voice, SPEI_PHONEME);
   }
#endif

#ifdef HAS_SPEECHD
   // Initialize speech-dispatcher if using that
   if (settings.reader == READER_NATIVE) {
      // Hack to ensure speech-dispatcher is initialized properly
      // Also used to work around a crash-level bug in speech-dispatcher
      // Ugh having to "handle" the return value to shut up GCC
      if (system("spd-say ' '")) { }

      // Now set up speech-dispatcher for real
      speechd_connection = spd_open("soledad", NULL, NULL, SPD_MODE_THREADED);
      if (speechd_connection == NULL)
         abort_program(ERR_INITREADER, NULL);
      spd_set_data_mode(speechd_connection, SPD_DATA_TEXT);
   }
#endif

   // Reset buffers
   if (textbuffer) free(textbuffer);
   if (oldbuffer) free(oldbuffer);
   textbuffer = NULL;
   oldbuffer = NULL;

   // Initialize settings
   punctuation = 0;

   // Refresh screen reader output
   update_reader();
}

//***************************************************************************
// reset_reader
// Resets the screen reader text.
//***************************************************************************

void reset_reader(void) {
   // Clear textbuffer
   if (textbuffer) free(textbuffer);
   textbuffer = NULL;
}

//***************************************************************************
// set_reader_text
// Sets the text shown to the screen reader.
//---------------------------------------------------------------------------
// param str: text to show
//***************************************************************************

// List of codepoints that get replaced by strings
// The first parameter is the UTF-8 codepoint
// The second parameter is the length in bytes
// The third parameter is text.reader.<param>
#define REPLACELIST \
   REPLACE("↑", 3, up) \
   REPLACE("↓", 3, down) \
   REPLACE("←", 3, left) \
   REPLACE("→", 3, right) \
   REPLACE("", 3, lclick) \
   REPLACE("", 3, mclick) \
   REPLACE("", 3, rclick)

void set_reader_text(const char *str) {
   // Hmmm...
   if (str == NULL)
      str = "";

   // Reset textbuffer
   reset_reader();

   /*
   // Make a copy of the string inside the textbuffer
   textbuffer = (char *) malloc(strlen(str)+1);
   if (textbuffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(textbuffer, str);

   // Replace all newlines with spaces, because screen readers by general
   // rule are line-oriented so we don't want issues
   for (char *ptr = textbuffer; *ptr != '\0'; ptr++) {
      if (*ptr == '\r') *ptr = ' ';
      if (*ptr == '\n') *ptr = ' ';
   }
   */

   // Check for final string length
   size_t len = 0;
   for (const char *ptr = str; *ptr != '\0';) {
#define REPLACE(unicode, size, with) \
   if (strncmp(ptr, unicode, size) == 0) { \
      len += strlen(text.reader. with); ptr += size; \
   } else
      REPLACELIST
#undef REPLACE
      {
         len++; ptr++;
      }
   }

   // Allocate memory for the copy to store in the textbuffer
   textbuffer = (char *) malloc(len + 1);
   if (textbuffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Store copy in the textbuffer
   // Replace some characters as needed
   char *dest = textbuffer;
   for (const char *src = str; *src != '\0';) {
#define REPLACE(unicode, size, with) \
   if (strncmp(src, unicode, size) == 0) { \
      strcpy(dest, text.reader. with); \
      dest += strlen(text.reader. with); \
      src += size; \
   } else
      REPLACELIST
#undef REPLACE
      {
         *dest++ = *src++;
      }
   }
   *dest = '\0';
}

//***************************************************************************
// set_reader_text_int
// Sets the text shown to the screen reader. It takes an integer parameter
// that will replace {param} in the source text.
//---------------------------------------------------------------------------
// param str: text to show
// param param: integer parameter
//***************************************************************************

void set_reader_text_int(const char *str, int param) {
   // Hmmm...
   if (str == NULL)
      set_reader_text("");

   // Generate string from integer
   char paramtext[0x20];
   sprintf(paramtext, "%d", param);

   // Render it
   set_reader_text_str(str, paramtext);
}

//***************************************************************************
// set_reader_text_str
// Sets the text shown to the screen reader. It takes a string parameter
// that will replace {param} in the source text.
//---------------------------------------------------------------------------
// param str: text to show
// param param: string parameter
//***************************************************************************

void set_reader_text_str(const char *str, const char *param) {
   // Hmmm...
   if (str == NULL)
      set_reader_text("");

   // Check where {param} is in the source text
   // If it's nowhere then just render it as-is...
   const char *where = strstr(str, "{param}");
   if (where == NULL) {
      set_reader_text(str);
      return;
   }

   // Get lengths of each section
   size_t startlen = (size_t)(where - str);
   size_t paramlen = strlen(param);
   size_t endlen = strlen(where + 7);

   // Allocate buffer for modified string
   char *buffer = (char *) malloc(startlen + paramlen + endlen + 1);
   if (buffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Generate string to show
   char *ptr = buffer;
   memcpy(ptr, str, startlen);
   ptr += startlen;
   memcpy(ptr, param, paramlen);
   str += startlen + 7;
   ptr += paramlen;
   memcpy(ptr, str, endlen);
   ptr += endlen;
   ptr[0] = '\0';

   // Render it
   set_reader_text(buffer);
   free(buffer);
}

//***************************************************************************
// update_reader
// Updates the displayed text in the screen reader.
//***************************************************************************

void update_reader(void) {
   // Get string to show to the screen reader
   // If there's nothing to show then output an empty string
   const char *str = textbuffer;
   if (str == NULL) str = "";

   // Check if the textbuffer even changed in the first place
   // Don't update the output if that hasn't happened
   if (oldbuffer == NULL && *str == '\0')
      return;
   if (oldbuffer != NULL && strcmp(str, oldbuffer) == 0)
      return;

   // Output string depending on what screen reader method we're using
   switch (settings.reader) {
#ifdef HAS_SAPI
      // SAPI output
      case READER_NATIVE: {
         // Determine flags for Speak
         uint32_t flags = 0;
         flags |= SPF_PURGEBEFORESPEAK;
         flags |= SPF_IS_NOT_XML;
         flags |= SPF_ASYNC;
         flags |= punctuation ? SPF_NLP_SPEAK_PUNC : 0;

         // Output the text... ugh UTF-16
         // Note that we output the text even when it's the empty string,
         // because doing so is effectively the way to make SAPI shut up
         // To-do: set output language (does anybody know how, short of
         // resorting to SSML?)
         wchar_t *wstr = utf8_to_utf16((const char *) str);
         sapi_voice->lpVtbl->Speak(sapi_voice, wstr, flags, NULL);
         free(wstr);
      }
#endif

#ifdef HAS_SPEECHD
      // Speech-dispatcher output
      case READER_NATIVE: {
         // Make sure nothing else is getting in the way
         spd_stop_all(speechd_connection);
         spd_cancel_all(speechd_connection);

         // Saying empty strings crashes speech-dispatcher...
         // Since the empty string is used to make the speech reader shut
         // up, and we just achieved that above, we're done here
         if (*str == '\0')
            break;

         // Output the text
         spd_set_language(speechd_connection, get_language_id());
         spd_set_punctuation(speechd_connection, punctuation ?
            SPD_PUNCT_ALL : SPD_PUNCT_NONE);
         spd_say(speechd_connection, SPD_TEXT, str);
      } break;
#endif

      // Clipboard output
      case READER_CLIPBOARD: {
         /*
         char *now = SDL_GetClipboardText();
         if (now != NULL) {
            if (strcmp(now, str) != 0)
               SDL_SetClipboardText(str);
            SDL_free(now);
         } else if (str[0] != '\0')
            SDL_SetClipboardText(str);
         */
         SDL_SetClipboardText(str);
      } break;

      // Titlebar output
      case READER_TITLEBAR: {
         /*
         const char *now = get_window_title();
         if (now != NULL) {
            if (strcmp(now, str) != 0)
               set_window_title(str);
         } else //if (str[0] != '\0')
            set_window_title(str);
         */
         set_window_title(str);
      } break;

      // Stdout output
      case READER_STDOUT:
         puts(str);
         break;

      // Unknown, assume disabled
      default:
         break;
   }

   // Swap buffers! ...or something like that
   if (oldbuffer) free(oldbuffer);
   oldbuffer = textbuffer;
   textbuffer = NULL;
}

//***************************************************************************
// repeat_reader_text
// Forces the screen reader to output the textbuffer again next time it's
// updated, even if the textbuffer hasn't changed.
//***************************************************************************

void repeat_reader_text(void) {
   if (oldbuffer) free(oldbuffer);
   oldbuffer = NULL;
}

//***************************************************************************
// set_reader_punctuation
// Sets whether punctuation should be spelled out or not.
//---------------------------------------------------------------------------
// param spell: zero to speak normall, non-zero to spell out symbols
//***************************************************************************

void set_reader_punctuation(int spell) {
   punctuation = spell ? 1 : 0;
}

//***************************************************************************
// deinit_reader
// Deinitializes the screen reader.
//***************************************************************************

void deinit_reader(void) {
   // Clear screen reader output
   reset_reader();
   update_reader();

   // Deallocate buffers
   if (textbuffer) free(textbuffer);
   if (oldbuffer) free(oldbuffer);
   textbuffer = NULL;
   oldbuffer = NULL;

#ifdef HAS_SAPI
   // Deinitialize SAPI
   if (settings.reader == READER_NATIVE) {
      if (sapi_voice != NULL) {
         sapi_voice->lpVtbl->Speak(sapi_voice, L"",
            SPF_IS_NOT_XML | SPF_PURGEBEFORESPEAK | SPF_ASYNC, NULL);
         sapi_voice->lpVtbl->WaitUntilDone(sapi_voice, INFINITE);
         sapi_voice->lpVtbl->Release(sapi_voice);
         sapi_voice = NULL;
      }
      CoUninitialize();
   }
#endif

#ifdef HAS_SPEECHD
   // Deinitialize speech-dispatcher
   if (speechd_connection != NULL) {
      spd_close(speechd_connection);
      speechd_connection = NULL;
   }
#endif
}
