//***************************************************************************
// "recording.c"
// Movie recording code.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stdint.h>
#include <stdio.h>
#include <time.h>
#include <SDL2/SDL.h>
#include "audiovideo.h"
#include "file.h"
#include "settings.h"
#include "sound.h"
#include "text.h"
#include "video.h"

// Recording mode
// 0 = record for WebM (60FPS, with sound)
// 1 = record for GIF (20FPS, without sound)
#define RECORD_MODE 0

// Keeps track of whether we're recording or not
static int recording = 0;

// Keeps track of sounds as they play
typedef struct {
   uint32_t frame;
   BgmID bgm;
   SfxID sfx;
   uint64_t delay;
   uint32_t flags;
   int32_t x;
   int32_t y;
   uint16_t vol;
} SoundLog;
static SoundLog *sound_log = NULL;
static size_t logged_sounds = 0;
static BgmID start_bgm;

// Used to store the audiovideo log while recording
// This is only stored when it's enabled since it eats up a considerable
// amount of memory (3.43MB per minute! ouch... OK could be much worse)
static uint8_t *avlog = NULL;

// Directory name used for the recording
// This buffer should be large enough for it
static char dirname[0x80];

// Which frame we're recording right now
static uint32_t frame;

//***************************************************************************
// start_recording
// Starts recording
//***************************************************************************

void start_recording(void) {
   // Get current date and time (we'll use them to generate the directory
   // name)
   time_t nowstamp = time(NULL);
   struct tm *now = localtime(&nowstamp);
   if (now == NULL)
      abort_program(ERR_UNKNOWN, NULL);

   // Determine directory name
   sprintf(dirname, "videos/soledad_%04d%02d%02d%02d%02d%02d/frame%%d.png",
      (now->tm_year + 1900) % 10000, (now->tm_mon + 1) % 100,
      now->tm_mday % 100, now->tm_hour % 100, now->tm_min % 100,
      now->tm_sec % 100);
   dirname[sizeof(dirname)-1] = '\0';

   // Reset recording status
   recording = 1;
   logged_sounds = 0;
   frame = 0;

   // Get starting BGM so we can restore it when rerendering the audio
   start_bgm = get_current_bgm();
}

//***************************************************************************
// update_recording
// Pushes a frame into the recording
//***************************************************************************

void update_recording(void) {
   // First of all: don't record if the game is paused
   if (settings.pause)
      return;

   // Generate PNG file for this frame
#if RECORD_MODE
   if (frame % 3 == 0) {
      char filename[0x100];
      sprintf(filename, "videos/%04u.png", frame / 3);
      save_png(filename);
   }
#else
   if (frame % 2 == 0) {
      char filename[0x100];
      sprintf(filename, "videos/%04u.png", frame / 2);
      save_png(filename);
   }
#endif

   // Advance frame
   frame++;
}

//***************************************************************************
// update_audiovideo_recording
// Logs the audiovideo buffer while recording.
//---------------------------------------------------------------------------
// param buffer: audiovideo framebuffer
//***************************************************************************

void update_audiovideo_recording(const uint8_t *buffer) {
   // Allocate memory for the new frame
   avlog = (uint8_t *) realloc(avlog, AV_BUFSIZE * (frame+1));
   if (avlog == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Copy the frame into the log buffer
   memcpy(avlog + AV_BUFSIZE * frame, buffer, AV_BUFSIZE);
}

//***************************************************************************
// stop_recording
// Stops recording
//***************************************************************************

void stop_recording(void) {
   // Done recording
   recording = 0;

#if !RECORD_MODE
   // Open WAV file
   File *wav = open_save_file("videos/audio.wav", FILE_WRITE);
   if (wav == NULL)
      abort_program(ERR_UNKNOWN, NULL);

   // Generate WAV header (ugh)
   write_file(wav, "RIFF", 4);
   write_uint32_le(wav, frame*735*2*2 + 36);
   write_file(wav, "WAVE", 4);
   write_file(wav, "fmt ", 4);
   write_uint32_le(wav, 16);
   write_uint16_le(wav, 1);
   write_uint16_le(wav, 2);
   write_uint32_le(wav, 44100);
   write_uint32_le(wav, 44100*2*2);
   write_uint16_le(wav, 2*2);
   write_uint16_le(wav, 16);
   write_file(wav, "data", 4);
   write_uint32_le(wav, frame*735*2*2);

   // Proceed to render the sound into a WAV file
   SDL_PauseAudio(1);
   stop_bgm();
   stop_sfx();
   play_bgm(start_bgm);

   size_t sound_pos = 0;
   for (uint32_t curr_frame = 0; curr_frame < frame; curr_frame++) {
#ifdef DEBUG
      printf("\rRendering WAV: frame #%u", curr_frame);
      fflush(stdout);
#endif
      // Play sounds as needed
      // This tricks the renderer into playing them
      while (sound_pos < logged_sounds &&
      sound_log[sound_pos].frame == curr_frame) {
         if (sound_log[sound_pos].bgm != BGM_NONE)
            play_bgm(sound_log[sound_pos].bgm);
         if (sound_log[sound_pos].sfx != SFX_NONE)
            play_sfx_ex(sound_log[sound_pos].sfx,
                        sound_log[sound_pos].x,
                        sound_log[sound_pos].y,
                        sound_log[sound_pos].vol,
                        sound_log[sound_pos].delay,
                        sound_log[sound_pos].flags |
                        SFLAG_ABSOLUTE | SFLAG_NOADJUST);
         sound_pos++;
      }

      // Get the renderer to play sounds
      // There are exactly 735 samples in a frame (yep, 44100 is a multiple
      // of 60, which is rather convenient, especially for sync)
      uint8_t buffer[735*2*2];
      sound_render_callback(NULL, buffer, sizeof(buffer));

      // Signedness shenanigans
      for (unsigned i = 1; i < sizeof(buffer); i += 2)
         buffer[i] ^= 0x80;

      // Output them!
      if (write_file(wav, buffer, sizeof(buffer)))
         abort_program(ERR_UNKNOWN, NULL);
   }
   SDL_PauseAudio(0);

   // We're done with sound
   close_file(wav);
#endif
   free(sound_log);
   logged_sounds = 0;

   // We can get rid of this too
   if (avlog != NULL) {
      free(avlog);
      avlog = NULL;
   }
}

//***************************************************************************
// is_recording
// Returns whether we're recording a movie or not
//---------------------------------------------------------------------------
// return: non-zero if recording, zero otherwise
//***************************************************************************

int is_recording(void) {
   return recording;
}

//***************************************************************************
// draw_recording_hud
// Draws the HUD that shows that we're recording
//***************************************************************************

void draw_recording_hud(void) {
   // Draw "REC" message
   draw_text("REC", screen_w - 16, 8, FONT_LIT, ALIGN_TOPRIGHT);

   // Get timer values
   uint32_t minutes = frame / 3600;
   uint32_t seconds = frame / 60 % 60;
   uint32_t centiseconds = frame % 60 * 100 / 60;

   // Draw timer on screen
   char buffer[0x100];
   sprintf(buffer, "%u:%02u.%02u", minutes, seconds, centiseconds);
   draw_text(buffer, screen_w - 16, 24, FONT_LIT, ALIGN_TOPRIGHT);
}

//***************************************************************************
// record_bgm
// Adds a BGM into the recording log.
//---------------------------------------------------------------------------
// param bgm: BGM ID
//***************************************************************************

void record_bgm(BgmID bgm) {
   // Allocate room for new entry
   logged_sounds++;
   sound_log = (SoundLog *) realloc(sound_log,
      sizeof(SoundLog) * logged_sounds);
   if (sound_log == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Store entry
   SoundLog *ptr = &sound_log[logged_sounds - 1];
   ptr->frame = frame;
   ptr->bgm = bgm;
   ptr->sfx = SFX_NONE;
}

//***************************************************************************
// record_sfx
// Adds a SFX into the recording log.
//---------------------------------------------------------------------------
// param bgm: SFX ID
// param x: X coordinate
// param y: Y coordinate
// param vol: volume (0..0x100)
// param delay: how many samples to delay
// param flags: see SFLAG_*
//***************************************************************************

void record_sfx(SfxID sfx, int32_t x, int32_t y, uint16_t vol,
uint64_t delay, uint32_t flags) {
   // Allocate room for new entry
   logged_sounds++;
   sound_log = (SoundLog *) realloc(sound_log,
      sizeof(SoundLog) * logged_sounds);
   if (sound_log == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Store entry
   SoundLog *ptr = &sound_log[logged_sounds - 1];
   ptr->frame = frame;
   ptr->bgm = BGM_NONE;
   ptr->sfx = sfx;
   ptr->x = x;
   ptr->y = y;
   ptr->vol = vol;
   ptr->delay = delay;
   ptr->flags = flags;
}
