//***************************************************************************
// "savegame.c"
// Code for handling savegames
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include "main.h"
#include "file.h"
#include "parser.h"
#include "savegame.h"
#include "scene.h"
#include "settings.h"

// Where savegame data is stored
static SaveGame savegame;

// Magic identifier
static const uint8_t magic[] = { 0x1A, 0x73, 0x4F, 0x4C };

// Used to store the savegame filename
// If this is empty, the savegame is disabled
static char filename[0x40] = "";

//***************************************************************************
// invalidate_savegame
// Invalidates the savegame file.
//***************************************************************************

void invalidate_savegame(void) {
   filename[0] = '\0';
}

//***************************************************************************
// set_savegame_name
// Sets the filename to use for the savegame. Do NOT include the extension!
//---------------------------------------------------------------------------
// param name: new savegame name
//***************************************************************************

void set_savegame_name(const char *name) {
   if (strlen(name) > 0x1F) {
      invalidate_savegame();
      return;
   }
   sprintf(filename, "savedata/%s", name);
   if (!is_valid_id(&filename[9])) {
      invalidate_savegame();
      return;
   }
   strcat(filename, ".sav");

#ifdef DEBUG
   fprintf(stderr, "Using savegame \"%s.sav\"\n", name);
#endif
}

//***************************************************************************
// load_savegame
// Loads the savegame data. Note that this handles the cases where the
// savegame is void, and also the fake savegame used if the level select is
// enabled.
//***************************************************************************

void load_savegame(void) {
   // Recording voids the savegame
   if (settings.record)
      settings.level_select = 1;

   // Invalidated savegame?
   if (*filename == '\0')
      abort_program(ERR_BADSAVEGAME, NULL);

   // Level select enabled?
   if (settings.level_select) {
      // Load faked savegame which emulates a complete game
      savegame.last_scene = get_num_scenes()-1;
      savegame.curr_scene = 0;
      savegame.checkpoint_x = 0xFFFF;
      savegame.checkpoint_y = 0xFFFF;

      // Abort here, don't actually load anything
      return;
   }

   // Load empty savegame
   savegame.curr_scene = 0;
   savegame.last_scene = 0;
   savegame.checkpoint_x = 0xFFFF;
   savegame.checkpoint_y = 0xFFFF;

   // Load data from disk
   // To-do: handle back-up system
   uint8_t blob[14];
   File *file = open_save_file(filename, FILE_READ);
   if (file == NULL)
      return;
   if (read_file(file, blob, sizeof(blob))) {
      close_file(file);
      return;
   }
   close_file(file);

   // Check header
   if (memcmp(&blob[0], magic, 4) != 0)
      return;

   // Check version (only 1.0 supported so far)
   if (blob[4] != 0x01 && blob[5] != 0x00)
      return;

   // Verify checksum
   uint8_t checksum_hi = blob[8] ^ blob[10] ^ blob[12];
   uint8_t checksum_lo = blob[9] ^ blob[11] ^ blob[13];
   if (blob[6] != checksum_hi || blob[7] != checksum_lo)
      return;

   // Retrieve data
   savegame.last_scene = blob[8];
   savegame.curr_scene = blob[9];
   savegame.checkpoint_x = blob[10] << 8 | blob[11];
   savegame.checkpoint_y = blob[12] << 8 | blob[13];
}

//***************************************************************************
// save_savegame
// Saves the savegame data. This handles not storing the data to disk when
// the savegame is void or the level select cheat is enabled.
//***************************************************************************

void save_savegame(const SaveGame *new_data) {
   // Replaying? (ignore anything that happens here)
   if (settings.replay)
      return;

   // Invalidated savegame?
   if (*filename == '\0')
      abort_program(ERR_BADSAVEGAME, NULL);

   // Update savegame
   if (new_data != NULL)
      savegame = *new_data;

   // Some sanity checks
   if (savegame.last_scene < savegame.curr_scene)
      savegame.last_scene = savegame.curr_scene;

   // Don't save the data if the level select is active!
   if (settings.level_select)
      return;

   // Where we'll store the file contents
   // Yeah, it's that short
   uint8_t blob[14];

   // Identifier
   memcpy(&blob[0], magic, 4);

   // Version (1.0)
   blob[4] = 0x01;   // Major number
   blob[5] = 0x00;   // Minor number

   // Savegame data
   blob[8] = savegame.last_scene;
   blob[9] = savegame.curr_scene;
   blob[10] = savegame.checkpoint_x >> 8;
   blob[11] = savegame.checkpoint_x;
   blob[12] = savegame.checkpoint_y >> 8;
   blob[13] = savegame.checkpoint_y;

   // Checksum
   blob[6] = blob[8] ^ blob[10] ^ blob[12];
   blob[7] = blob[9] ^ blob[11] ^ blob[13];

   // Store data in disk
   // To-do: handle back up system
   File *file = open_save_file(filename, FILE_WRITE);
   if (file == NULL)
      return;
   if (write_file(file, blob, sizeof(blob))) {
      close_file(file);
      return;
   }
   close_file(file);
}

//***************************************************************************
// get_savegame
// Gets the data currently stored in the savegame.
//---------------------------------------------------------------------------
// param where: where to copy the data
//***************************************************************************

void get_savegame(SaveGame *where) {
   // Replaying? (load a bogus savegame)
   if (settings.replay) {
      where->checkpoint_x = 0xFFFF;
      where->checkpoint_y = 0xFFFF;
      where->curr_scene = 0;
      where->last_scene = 0;
      return;
   }

   // Just copy the savegame data
   // <Sik> No, I don't remember why the code doesn't just access the
   // savegame data directly. Doesn't matter though, this is probably safer
   // (less risk of corrupting data in case of a mistake).
   *where = savegame;
}

//***************************************************************************
// set_savegame_scene
// Changes the current scene in the savegame.
//---------------------------------------------------------------------------
// param scene: new current scene
//***************************************************************************

void set_savegame_scene(unsigned scene) {
   // Replaying? (ignore anything that happens here)
   if (settings.replay)
      return;

   // Unlock every non-level scene from this point on
   // Sorry for the somewhat dirty-looking loop, but hey, this works :P
   size_t last_scene = scene;
   size_t num_scenes = get_num_scenes();
   for (last_scene = scene; ; last_scene++) {
      // Last scene? (can't go further)
      if (last_scene == num_scenes - 1)
         break;

      // Is it a level? (if so, stop here)
      SceneType type = get_scene_type_by_id(last_scene);
      if (type == SCENE_LEVEL || type == SCENE_BONUS)
         break;
   }
   if (last_scene > savegame.last_scene)
      savegame.last_scene = last_scene;

   // Set savegame to the specified scene
   // This nullifies any checkpoint save if it's switching
   if (savegame.curr_scene != scene) {
      savegame.curr_scene = scene;
      savegame.checkpoint_x = 0xFFFF;
      savegame.checkpoint_y = 0xFFFF;
   }

   // Update file
   save_savegame(NULL);
}

//***************************************************************************
// advance_savegame_scene
// Advances the current savegame scene.
//***************************************************************************

void advance_savegame_scene(void) {
   set_savegame_scene(savegame.curr_scene + 1);
}
