//***************************************************************************
// "settings.c"
// Configuration code. Responsible for initializing the game configuration
// and saving it. Also where the configuration structure is stored.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stdlib.h>
#include <string.h>
#include <SDL2/SDL.h>
#include "main.h"
#include "file.h"
#include "savegame.h"
#include "settings.h"
#include "text.h"

// Characters considered to be whitespace
#define WHITESPACE " \n\r\t\f\v"

// Somewhere the configuration structure should be stored, right?
// Storing it here where the rest of the configuration code is :P
Settings settings;

// Type used to hold list of possible values for each setting
typedef struct {
   const char *name;
   const int value;
} ValidValues;

// Possible values for boolean settings
static const ValidValues values_boolean[] = {
   { "1", 1 }, { "0", 0 },
   { "yes", 1 }, { "no", 0 },
   { "enable", 1 }, { "disable", 0 },
   { "enabled", 1 }, { "disabled", 0 },
   { "true", 1 }, { "false", 0 },
   // { "file_not_found", 2 },
   { NULL, 0 }
};

// Possible values for the difficulty
static const ValidValues values_difficulty[] = {
   { "easy", DIFF_EASY },
   { "normal", DIFF_NORMAL },
   { "hard", DIFF_HARD },
   { NULL, 0 }
};

// Possible values for the game speed
static const ValidValues values_speed[] = {
   { "25%", 0 },
   { "37.5%", 1 },
   { "50%", 2 },
   { "75%", 3 },
   { "100%", 4 },
   { "125%", 5 },
   { "150%", 6 },
   { "175%", 7 },
   { "200%", 8 },

   { NULL, 0 }
};

// Possible values for the contrast
static const ValidValues values_contrast[] = {
   { "50%", CONTRAST_LOW },
   { "75%", CONTRAST_MIDLOW },
   { "100%", CONTRAST_NORMAL },
   { "150%", CONTRAST_MIDHIGH },
   { "200%", CONTRAST_HIGH },
   { "low", CONTRAST_LOW },
   { "midlow", CONTRAST_MIDLOW },
   { "normal", CONTRAST_NORMAL },
   { "midhigh", CONTRAST_MIDHIGH },
   { "high", CONTRAST_HIGH },
   { NULL, 0 }
};

// Possible values for the screen reader type
static const ValidValues values_reader[] = {
   { "none", READER_NONE },
   { "native", READER_NATIVE },
   { "clipboard", READER_CLIPBOARD },
   { "titlebar", READER_TITLEBAR },
   { "stdout", READER_STDOUT },
   { NULL, 0 }
};

// Possible values for the timing method
static const ValidValues values_timing[] = {
   { "auto", TIMING_AUTO },
   { "speed", TIMING_SPEED },
   { "power", TIMING_POWER },
   { NULL, 0 }
};

// Possible values for enforcing power-ups
static const ValidValues values_power[] = {
   { "none", POWER_NONE },
   { "wings", POWER_WINGS },
   { "spider", POWER_SPIDER },
   { "hammer", POWER_HAMMER },
   { "parasol", POWER_PARASOL },
   { NULL, 0 }
};

// Possible values for enemy AI
static const ValidValues values_ai[] = {
   { "pacifist", 0 },
   { "normal", 1 },
   { "aggressive", 2 },
   { NULL, 0 }
};

// Private function prototypes
static int get_setting_value(const Ini *, const char *, const char *,
   const ValidValues *, int);
static int get_language_value(const Ini *, const char *, const char *, int);
static int get_keyboard_value(const Ini *, const char *, const char *, int);
static uint16_t get_joystick_value(const Ini *, const char *, const char *);
static uint32_t get_color_value(const Ini *, const char *, const char *,
   uint32_t);
static void set_setting_value(Ini *, const char *, const char *,
   const ValidValues *, int);
static void set_keyboard_value(Ini *, const char *, const char *, int);
static void set_joystick_value(Ini *, const char *, const char *, uint16_t);

//***************************************************************************
// init_config
// Initializes the entire configuration to reasonable defaults.
//***************************************************************************

void init_config(void) {
#ifdef DEBUG
   fputs("Initializing configuration\n", stderr);
#endif

   // Default game settings
   settings.difficulty = DIFF_NORMAL;
   settings.language = -1;
   settings.game_speed = 4;

   // Default one-switch settings
   settings.one_switch = 0;
   settings.tap_delay = 9;
   settings.fivetap_delay = 30;
   settings.menu_delay = 60;

   // Default mouse-switch settings
   settings.mouse_switch = 0;
   settings.mousesw_grid = 1;
   settings.mousesw_x = 30;
   settings.mousesw_y = 30;

   // Default video settings
   settings.fullscreen = 0;
   settings.windowed_width = 0;
   settings.windowed_height = 0;
   settings.vsync = 1;
   settings.filter = 0;
   settings.background = 1;
   settings.zoom = 0;
   settings.dark_bg = 0;
   settings.shaking = 1;
   settings.lcd_fix = 0;
   settings.contrast = CONTRAST_NORMAL;
   settings.monitor = 0;

   // Default screen reader settings
   settings.reader = READER_NONE;

   // Default sound settings
   settings.bgm_volume = 70;
   settings.sfx_volume = 70;
   settings.enable_bgm = 1;
   settings.enable_sfx = 1;
   settings.stereo = 1;
   settings.reverse = 1;

   // Default keyboard controls
   settings.keyboard[PL_INPUT_UP][0] = SDL_SCANCODE_UP;
   settings.keyboard[PL_INPUT_UP][1] = NO_KEY;
   settings.keyboard[PL_INPUT_DOWN][0] = SDL_SCANCODE_DOWN;
   settings.keyboard[PL_INPUT_DOWN][1] = NO_KEY;
   settings.keyboard[PL_INPUT_LEFT][0] = SDL_SCANCODE_LEFT;
   settings.keyboard[PL_INPUT_LEFT][1] = NO_KEY;
   settings.keyboard[PL_INPUT_RIGHT][0] = SDL_SCANCODE_RIGHT;
   settings.keyboard[PL_INPUT_RIGHT][1] = NO_KEY;
   settings.keyboard[PL_INPUT_ACTION][0] = SDL_SCANCODE_SPACE;
   settings.keyboard[PL_INPUT_ACTION][1] = NO_KEY;
   settings.keyboard[PL_INPUT_PAUSE][0] = SDL_SCANCODE_ESCAPE;
   settings.keyboard[PL_INPUT_PAUSE][1] = NO_KEY;

   // Default joystick controls
   settings.joystick[PL_INPUT_UP][0] = JOYIN_NONE;
   settings.joystick[PL_INPUT_UP][1] = JOYIN_NONE;
   settings.joystick[PL_INPUT_DOWN][0] = JOYIN_NONE;
   settings.joystick[PL_INPUT_DOWN][1] = JOYIN_NONE;
   settings.joystick[PL_INPUT_LEFT][0] = JOYIN_NONE;
   settings.joystick[PL_INPUT_LEFT][1] = JOYIN_NONE;
   settings.joystick[PL_INPUT_RIGHT][0] = JOYIN_NONE;
   settings.joystick[PL_INPUT_RIGHT][1] = JOYIN_NONE;
   settings.joystick[PL_INPUT_ACTION][0] = JOYIN_NONE;
   settings.joystick[PL_INPUT_ACTION][1] = JOYIN_NONE;
   settings.joystick[PL_INPUT_PAUSE][0] = JOYIN_NONE;
   settings.joystick[PL_INPUT_PAUSE][1] = JOYIN_NONE;

   // Disable debug mode
   settings.debug = 0;

   // Initialize debug settings (in case debug mode is enabled later)
   settings.pause = 0;
   settings.show_coord = 0;
   settings.free_move = 0;
   settings.show_hitbox = 0;
   settings.show_time = 0;

   // Disable recording
   settings.record = 0;
   settings.replay = 0;

   // Miscellaneous
   settings.attract = 0;

   // Advanced settings
   settings.timing = TIMING_AUTO;

   // Tweaks
   settings.score_tally = 1;
   settings.goal_vanish = 0;
   settings.spin_jump = 0;
   settings.refill_health = 1;
   settings.power_shield = 1;
   settings.collect_health = 0;
   settings.clear_enemies = 0;
   settings.blue_limit = 99999999;
   settings.invincibility_time = 1224;

   settings.player_speed = 0x280;
   settings.player_accel = 0x20;
   settings.player_fast = 99999999;
   settings.player_climb = 0x100;
   settings.player_friction = 0x20;
   settings.player_jump = 0x640;
   settings.player_weight = 0x30;
   settings.player_height = 38;
   settings.player_crouch = 22;
   settings.force_power = POWER_NONE;
   settings.show_power = 1;

   settings.flamer_ai = 1;
   settings.sprayer_ai = 1;
   settings.turret_ai = 1;
   settings.grabber_ai = 1;
   settings.spider_ai = 1;
   settings.heater_ai = 1;

   settings.font_lit[0] = 0xFFFFFF00;
   settings.font_lit[1] = 0xFFFF0000;
   settings.font_lit[2] = 0xFF000000;
   settings.font_dim[0] = 0xFFFF0000;
   settings.font_dim[1] = 0xFF000000;
   settings.font_dim[2] = 0xFF000000;
   settings.box_lit[0] = 0xFFFFFF00;
   settings.box_lit[1] = 0xFFFF0000;
   settings.box_lit[2] = 0xFF000000;
   settings.box_dim[0] = 0xFF000000;
   settings.box_dim[1] = 0xFFFF0000;
   settings.box_dim[2] = 0xFF000000;

   // Disable cheats
   settings.level_select = 0;
   settings.no_damage = 0;
   settings.no_health = 0;
   settings.extra_health = 0;
   settings.no_enemies = 0;
   settings.no_checkpoints = 0;
   settings.no_friction = 0;
   settings.pinball = 0;
   settings.force_virtual = 0;
   settings.cga_mode = 0;
   settings.beeper = 0;
   settings.color_mode = COLOR_NORMAL;

   // No overrides by default
   settings.override_fullscreen = 0;
   settings.override_contrast = 0;
   settings.override_one_switch = 0;
   settings.override_mouse_switch = 0;
   settings.override_reader = 0;

   // Enable all subsystems by default
   settings.no_hwaccel = 0;
   settings.no_sound = 0;
   settings.no_joystick = 0;
   settings.no_joystick_axis = 0;
   settings.audiovideo = 0;
}

//***************************************************************************
// load_tweaks
// Loads the tweaks file (system/tweaks.ini). It allows mods to make several
// tweaks to the game.
//***************************************************************************

void load_tweaks(void) {
#ifdef DEBUG
   fputs("Loading tweaks\n", stderr);
#endif

   // Try to load INI file with tweaks if possible
   Ini *ini = load_ini("system/tweaks.ini");
   if (ini == NULL)
      return;

   // To keep pointers to each setting value
   const char *str;

   // Check if the savegame name has been changed (this is important!)
   const char *savename = get_ini_var(ini, "system", "savename", "");
   if (*savename != '\0')
      set_savegame_name(savename);

   // Get gameplay tweaks
   settings.no_health |= get_setting_value(ini, "gameplay", "nohealth",
      values_boolean, 0);
   settings.no_damage |= get_setting_value(ini, "gameplay", "nodamage",
      values_boolean, 0);
   settings.refill_health = get_setting_value(ini, "gameplay", "refillhealth",
      values_boolean, settings.refill_health);
   settings.power_shield = get_setting_value(ini, "gameplay", "powershield",
      values_boolean, settings.power_shield);
   settings.score_tally = get_setting_value(ini, "gameplay", "scoretally",
      values_boolean, settings.score_tally);
   settings.goal_vanish = get_setting_value(ini, "gameplay", "goalvanish",
      values_boolean, settings.goal_vanish);
   settings.collect_health = get_setting_value(ini, "gameplay", "collecthealth",
      values_boolean, settings.collect_health);
   settings.clear_enemies = get_setting_value(ini, "gameplay", "clearenemies",
      values_boolean, settings.clear_enemies);
   settings.spin_jump = get_setting_value(ini, "gameplay", "spinjump",
      values_boolean, settings.spin_jump);
   settings.no_friction |= get_setting_value(ini, "gameplay", "nofriction",
      values_boolean, 0);
   settings.pinball |= get_setting_value(ini, "gameplay", "pinball",
      values_boolean, 0);
   str = get_ini_var(ini, "gameplay", "invincibilitytime", NULL);
   if (str[0]) settings.invincibility_time = atoi(str);
   str = get_ini_var(ini, "gameplay", "bluelimit", NULL);
   if (str[0]) settings.blue_limit = atoi(str);

   // Get player physics tweaks
   str = get_ini_var(ini, "player", "speed", NULL);
   if (str[0]) settings.player_speed = atoi(str);
   str = get_ini_var(ini, "player", "accel", NULL);
   if (str[0]) settings.player_accel = atoi(str);
   str = get_ini_var(ini, "player", "friction", NULL);
   if (str[0]) settings.player_friction = atoi(str);
   str = get_ini_var(ini, "player", "fast", NULL);
   if (str[0]) settings.player_fast = atoi(str);
   str = get_ini_var(ini, "player", "jump", NULL);
   if (str[0]) settings.player_jump = atoi(str);
   str = get_ini_var(ini, "player", "weight", NULL);
   if (str[0]) settings.player_weight = atoi(str);
   str = get_ini_var(ini, "player", "height", NULL);
   if (str[0]) settings.player_height = atoi(str);
   str = get_ini_var(ini, "player", "crouch", NULL);
   if (str[0]) settings.player_crouch = atoi(str);
   settings.force_power = get_setting_value(ini, "player", "forcepower",
      values_power, settings.force_power);
   settings.show_power = get_setting_value(ini, "player", "showpower",
      values_boolean, settings.show_power);

   settings.flamer_ai = get_setting_value(ini, "enemies", "ai_flamer",
      values_ai, settings.flamer_ai);
   settings.sprayer_ai = get_setting_value(ini, "enemies", "ai_sprayer",
      values_ai, settings.sprayer_ai);
   settings.turret_ai = get_setting_value(ini, "enemies", "ai_turret",
      values_ai, settings.turret_ai);
   settings.grabber_ai = get_setting_value(ini, "enemies", "ai_grabber",
      values_ai, settings.grabber_ai);
   settings.spider_ai = get_setting_value(ini, "enemies", "ai_spider",
      values_ai, settings.spider_ai);
   settings.heater_ai = get_setting_value(ini, "enemies", "ai_heater",
      values_ai, settings.heater_ai);

   // Get font palettes
#define GET(x, y) x = get_color_value(ini, "font", y, x);
   GET(settings.font_lit[0], "lit_1");
   GET(settings.font_lit[1], "lit_2");
   GET(settings.font_lit[2], "lit_3");
   GET(settings.font_dim[0], "dim_1");
   GET(settings.font_dim[1], "dim_2");
   GET(settings.font_dim[2], "dim_3");
#undef GET

   // Get box palettes
#define GET(x, y) x = get_color_value(ini, "boxes", y, x);
   GET(settings.box_lit[0], "lit_1");
   GET(settings.box_lit[1], "lit_2");
   GET(settings.box_lit[2], "lit_3");
   GET(settings.box_dim[0], "dim_1");
   GET(settings.box_dim[1], "dim_2");
   GET(settings.box_dim[2], "dim_3");
#undef GET

   // Done with the INI file
   destroy_ini(ini);
}

//***************************************************************************
// load_config
// Tries to load the configuration on this system. If it can't determine some
// setting it'll resort to the defaults.
//***************************************************************************

void load_config(void) {
#ifdef DEBUG
   fputs("Loading configuration\n", stderr);
#endif

   // Try to load INI file with settings if possible
   Ini *ini = load_ini("savedata/options.ini");
   if (ini == NULL)
      return;

   // To keep pointers to each setting value
   const char *str;

   // Determine default language
   settings.language = get_language_value(ini, "gameplay", "language", -1);

   // Get difficulty setting
   settings.difficulty = get_setting_value(ini, "gameplay", "difficulty",
      values_difficulty, DIFF_NORMAL);

   // Get one-switch mode
   if (!settings.override_one_switch)
      settings.one_switch = get_setting_value(ini, "one_switch", "enabled",
      values_boolean, 0);

   // Get one-switch delays
   str = get_ini_var(ini, "one_switch", "tap_delay", "9");
   settings.tap_delay = atoi(str);
   if (settings.tap_delay < 1) settings.tap_delay = 1;

   str = get_ini_var(ini, "one_switch", "fivetap_delay", "30");
   settings.fivetap_delay = atoi(str);
   if (settings.fivetap_delay < 1) settings.fivetap_delay = 1;

   str = get_ini_var(ini, "one_switch", "menu_delay", "60");
   settings.menu_delay = atoi(str);
   if (settings.menu_delay < 1) settings.menu_delay = 1;

   // Get mouse-switch mode
   if (!settings.override_mouse_switch)
      settings.mouse_switch = get_setting_value(ini, "mouse_switch",
      "enabled", values_boolean, 0);

   // Get mouse-switch properties
   settings.mousesw_grid = get_setting_value(ini, "mouse_switch", "grid",
      values_boolean, 1);

   str = get_ini_var(ini, "mouse_switch", "x", "60");
   settings.mousesw_x = atoi(str) / 2;
   if (settings.mousesw_x < 1) settings.mousesw_x = 1;

   str = get_ini_var(ini, "mouse_switch", "y", "60");
   settings.mousesw_y = atoi(str) / 2;
   if (settings.mousesw_y < 1) settings.mousesw_y = 1;

   // Get game speed
   settings.game_speed = get_setting_value(ini, "gameplay", "speed",
      values_speed, 4);

   // Get if fullscreen is enabled
   if (!settings.override_fullscreen)
      settings.fullscreen = get_setting_value(ini, "video", "fullscreen",
      values_boolean, 0);

   // Get windowed resolution
   // To-do: check that it's a number
   str = get_ini_var(ini, "video", "width", "0");
   settings.windowed_width = atoi(str);
   str = get_ini_var(ini, "video", "height", "0");
   settings.windowed_height = atoi(str);

   // Get other video settings
   settings.vsync = get_setting_value(ini, "video", "vsync",
      values_boolean, 0);
   settings.filter = get_setting_value(ini, "video", "filter",
      values_boolean, 0);
   settings.background = get_setting_value(ini, "video", "background",
      values_boolean, 1);
   settings.dark_bg = get_setting_value(ini, "video", "darkbg",
      values_boolean, 0);
   settings.zoom = get_setting_value(ini, "video", "zoom",
      values_boolean, 0);
   settings.shaking = get_setting_value(ini, "video", "shaking",
      values_boolean, 1);
   str = get_ini_var(ini, "video", "monitor", "0");
   settings.monitor = atoi(str);
   settings.lcd_fix = get_setting_value(ini, "advanced", "lcdfix",
      values_boolean, 0);

   // Get contrast level
   if (!settings.override_contrast)
      settings.contrast = get_setting_value(ini, "video", "contrast",
      values_contrast, CONTRAST_NORMAL);

   // Get screen reader settings
   if (!settings.override_reader)
      settings.reader = get_setting_value(ini, "reader", "type",
      values_reader, READER_NONE);

   // Get background music volume
   // To-do: check that it's a number
   str = get_ini_var(ini, "sound", "bgm_volume", "70");
   settings.bgm_volume = atoi(str);
   if (settings.bgm_volume < 0) settings.bgm_volume = 0;
   if (settings.bgm_volume > 100) settings.bgm_volume = 100;
   settings.stereo = get_setting_value(ini, "sound", "stereo",
      values_boolean, 1);
   settings.reverse = get_setting_value(ini, "sound", "reverse",
      values_boolean, 0);

   // Get sound effect volume
   // To-do: check that it's a number
   str = get_ini_var(ini, "sound", "sfx_volume", "70");
   settings.sfx_volume = atoi(str);
   if (settings.sfx_volume < 0) settings.sfx_volume = 0;
   if (settings.sfx_volume > 100) settings.sfx_volume = 100;

   // Get default keyboard mappings
   settings.keyboard[PL_INPUT_UP][0] = get_keyboard_value(ini,
      "keyboard", "up_1", settings.keyboard[PL_INPUT_UP][0]);
   settings.keyboard[PL_INPUT_UP][1] = get_keyboard_value(ini,
      "keyboard", "up_2", settings.keyboard[PL_INPUT_UP][1]);
   settings.keyboard[PL_INPUT_DOWN][0] = get_keyboard_value(ini,
      "keyboard", "down_1", settings.keyboard[PL_INPUT_DOWN][0]);
   settings.keyboard[PL_INPUT_DOWN][1] = get_keyboard_value(ini,
      "keyboard", "down_2", settings.keyboard[PL_INPUT_DOWN][1]);
   settings.keyboard[PL_INPUT_LEFT][0] = get_keyboard_value(ini,
      "keyboard", "left_1", settings.keyboard[PL_INPUT_LEFT][0]);
   settings.keyboard[PL_INPUT_LEFT][1] = get_keyboard_value(ini,
      "keyboard", "left_2", settings.keyboard[PL_INPUT_LEFT][1]);
   settings.keyboard[PL_INPUT_RIGHT][0] = get_keyboard_value(ini,
      "keyboard", "right_1", settings.keyboard[PL_INPUT_RIGHT][0]);
   settings.keyboard[PL_INPUT_RIGHT][1] = get_keyboard_value(ini,
      "keyboard", "right_2", settings.keyboard[PL_INPUT_RIGHT][1]);
   settings.keyboard[PL_INPUT_ACTION][0] = get_keyboard_value(ini,
      "keyboard", "jump_1", settings.keyboard[PL_INPUT_ACTION][0]);
   settings.keyboard[PL_INPUT_ACTION][1] = get_keyboard_value(ini,
      "keyboard", "jump_2", settings.keyboard[PL_INPUT_ACTION][1]);

   // Get default joystick mappings
   settings.joystick[PL_INPUT_UP][0] = get_joystick_value(ini,
      "joystick", "up_1");
   settings.joystick[PL_INPUT_UP][1] = get_joystick_value(ini,
      "joystick", "up_2");
   settings.joystick[PL_INPUT_DOWN][0] = get_joystick_value(ini,
      "joystick", "down_1");
   settings.joystick[PL_INPUT_DOWN][1] = get_joystick_value(ini,
      "joystick", "down_2");
   settings.joystick[PL_INPUT_LEFT][0] = get_joystick_value(ini,
      "joystick", "left_1");
   settings.joystick[PL_INPUT_LEFT][1] = get_joystick_value(ini,
      "joystick", "left_2");
   settings.joystick[PL_INPUT_RIGHT][0] = get_joystick_value(ini,
      "joystick", "right_1");
   settings.joystick[PL_INPUT_RIGHT][1] = get_joystick_value(ini,
      "joystick", "right_2");
   settings.joystick[PL_INPUT_ACTION][0] = get_joystick_value(ini,
      "joystick", "jump_1");
   settings.joystick[PL_INPUT_ACTION][1] = get_joystick_value(ini,
      "joystick", "jump_2");
   settings.joystick[PL_INPUT_PAUSE][0] = get_joystick_value(ini,
      "joystick", "pause_1");
   settings.joystick[PL_INPUT_PAUSE][1] = get_joystick_value(ini,
      "joystick", "pause_2");

   // Get advanced settings
   settings.timing = get_setting_value(ini, "advanced", "timing",
      values_timing, TIMING_AUTO);

   // Done with the INI file
   destroy_ini(ini);
}

//***************************************************************************
// load_launcher_config
// Loads some additional settings only used with the launcher.
//***************************************************************************

void load_launcher_config(void) {
#ifdef DEBUG
   fputs("Loading launcher-only configuration\n", stderr);
#endif

   // Try to load INI file with settings if possible
   Ini *ini = load_ini("savedata/options.ini");
   if (ini == NULL)
      return;

   // Load these the launcher-only stuff
   // This will override some settings but doesn't matter since if the
   // launcher is used then the command line didn't get any arguments
   settings.no_hwaccel = get_setting_value(ini, "launcher", "software",
      values_boolean, settings.no_hwaccel);
   settings.no_sound = get_setting_value(ini, "launcher", "nosound",
      values_boolean, settings.no_sound);
   settings.no_joystick = get_setting_value(ini, "launcher", "nojoystick",
      values_boolean, settings.no_joystick);
   settings.no_joystick_axis = get_setting_value(ini, "launcher", "nojoyaxis",
      values_boolean, settings.no_joystick_axis);

   // Done with the INI file
   destroy_ini(ini);
}

//***************************************************************************
// get_setting_value [internal]
// Parses a setting stored as a string to determine its value as an integer.
//---------------------------------------------------------------------------
// param ini: pointer to INI object
// param section: name of section
// param variable: name of variable
// param values: list of valid values
// param defvalue: default value
// return: value of setting
//***************************************************************************

static int get_setting_value(const Ini *ini, const char *section,
const char *variable, const ValidValues *values, int defvalue) {
   // Get value from INI file
   const char *str = get_ini_var(ini, section, variable, NULL);
   if (str[0] == '\0') return defvalue;

   // Look all possible values for a match
   // To-do: do a case-insensitive comparison
   for (size_t i = 0; values[i].name != NULL; i++) {
      if (strcmp(values[i].name, str) == 0)
         return values[i].value;
   }

   // No match?
   return defvalue;
}

//***************************************************************************
// get_language_value
// Parses a language stored as a string to determine its language ID
//---------------------------------------------------------------------------
// param ini: pointer to INI object
// param section: name of section
// param variable: name of variable
// param defvalue: default value
// return: value of setting
//***************************************************************************

static int get_language_value(const Ini *ini, const char *section,
const char *variable, int defvalue) {
   // Get value from INI file
   const char *str = get_ini_var(ini, section, variable, NULL);
   if (str == NULL) return defvalue;
   if (str[0] == '\0') return defvalue;

   // Scan all languages for a match
   unsigned num_languages = get_num_languages();
   for (unsigned i = 0; i < num_languages; i++) {
      if (strcmp(str, get_language_longid(i)) == 0)
         return i;
   }

   // Unrecognized language
   return defvalue;
}

//***************************************************************************
// get_keyboard_value [internal]
// Parses a setting stored as a string to determine its value as a key
// scancode
//---------------------------------------------------------------------------
// param ini: pointer to INI object
// param section: name of section
// param variable: name of variable
// param defvalue: default value
// return: value of setting
//***************************************************************************

static int get_keyboard_value(const Ini *ini, const char *section,
const char *variable, int defvalue) {
   // Get value from INI file
   const char *str = get_ini_var(ini, section, variable, NULL);
   if (str == NULL) return defvalue;
   if (str[0] == '\0') return defvalue;

   // Determine scancode for this key name
   return id_to_scancode(str);
}

//***************************************************************************
// get_joystick_value [internal]
// Parses a setting stored as a string to determine its value as a joystick
// input ("button ##", "axis ## pos", "hat ## up", etc.)
//---------------------------------------------------------------------------
// param ini: pointer to INI object
// param section: name of section
// param variable: name of variable
// return: value of setting
//***************************************************************************

static uint16_t get_joystick_value(const Ini *ini, const char *section,
const char *variable) {
   // Get value from INI file
   const char *str = get_ini_var(ini, section, variable, NULL);
   if (str == NULL) return JOYIN_UNDEF;
   if (str[0] == '\0') return JOYIN_UNDEF;

   // Should have thought of this before...
   char *buffer = (char *) malloc(strlen(str)+1);
   if (buffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(buffer, str);

   // Get first token
   char *token = strtok(buffer, WHITESPACE);
   if (token == NULL)
      goto bad;

   // Button?
   if (!strcmp(token, "button")) {
      // Get button ID
      token = strtok(NULL, WHITESPACE);
      if (token == NULL)
         goto bad;
      int button = atoi(token);

      // Sanity check
      if (button < 0 || button > 0xFF)
         goto bad;

      // Return button ID
      free(buffer);
      return JOYIN_BUTTON|button;
   }

   // Axis?
   else if (!strcmp(token, "axis")) {
      // Get axis ID
      token = strtok(NULL, WHITESPACE);
      if (token == NULL)
         goto bad;
      int axis = atoi(token);

      // Sanity check
      if (axis < 0 || axis > 0xFF)
         goto bad;

      // Get orientation
      token = strtok(NULL, WHITESPACE);
      if (token == NULL)
         goto bad;
      if (!strcmp(token, "pos")) {
         free(buffer);
         return JOYIN_AXISPOS|axis;
      } else if (!strcmp(token, "neg")) {
         free(buffer);
         return JOYIN_AXISNEG|axis;
      } else
         goto bad;
   }

   // Hat?
   else if (!strcmp(token, "hat")) {
      // Get hat ID
      token = strtok(NULL, WHITESPACE);
      if (token == NULL)
         goto bad;
      int hat = atoi(token);

      // Sanity check
      if (hat < 0 || hat > 0xFF)
         goto bad;

      // Get orientation
      token = strtok(NULL, WHITESPACE);
      if (token == NULL)
         goto bad;
      if (!strcmp(token, "up")) {
         free(buffer);
         return JOYIN_HATUP|hat;
      } else if (!strcmp(token, "down")) {
         free(buffer);
         return JOYIN_HATDOWN|hat;
      } else if (!strcmp(token, "left")) {
         free(buffer);
         return JOYIN_HATLEFT|hat;
      } else if (!strcmp(token, "right")) {
         free(buffer);
         return JOYIN_HATRIGHT|hat;
      } else
         goto bad;
   }

   // No match?
bad:
   free(buffer);
   return JOYIN_UNDEF;
}

//***************************************************************************
// get_color_value [internal]
// Parses a setting stored as a string to determine its value as a color
//---------------------------------------------------------------------------
// param ini: pointer to INI object
// param section: name of section
// param variable: name of variable
// param defvalue: default value
// return: value of setting
//***************************************************************************

static uint32_t get_color_value(const Ini *ini, const char *section,
const char *variable, uint32_t defvalue) {
   // Get value from INI file
   const char *str = get_ini_var(ini, section, variable, NULL);

   // See if it's valid
   if (str == NULL)
      return defvalue;
   if (str[0] == '\0')
      return defvalue;
   if (strlen(str) != 7)
      return defvalue;
   if (str[0] != '#')
      return defvalue;
   for (unsigned i = 1; i <= 6; i++)
      if (!isxdigit(str[i]))
         return defvalue;

   // Return color
   return 0xFF000000 | strtol(&str[1], NULL, 0x10);
}

//***************************************************************************
// save_config
// Tries to save the configuration on this system.
//***************************************************************************

void save_config(void) {
#ifdef DEBUG
   fputs("Saving configuration\n", stderr);
#endif

   // Create empty INI file
   Ini *ini = create_ini();

   // Used to create some strings...
   char buffer[0x40];

   // Gameplay settings
   if (settings.language != -1)
      set_ini_var(ini, "gameplay", "language",
      get_language_longid(settings.language));
   set_setting_value(ini, "gameplay", "difficulty", values_difficulty,
      settings.difficulty);
   set_setting_value(ini, "gameplay", "speed", values_speed,
      settings.game_speed);

   // One-switch settings
   set_ini_var(ini, "one_switch", "enabled",
      settings.one_switch ? "yes" : "no");
   sprintf(buffer, "%d", settings.tap_delay);
   set_ini_var(ini, "one_switch", "tap_delay", buffer);
   sprintf(buffer, "%d", settings.fivetap_delay);
   set_ini_var(ini, "one_switch", "fivetap_delay", buffer);
   sprintf(buffer, "%d", settings.menu_delay);
   set_ini_var(ini, "one_switch", "menu_delay", buffer);

   // Mouse-switch settings
   set_ini_var(ini, "mouse_switch", "enabled",
      settings.mouse_switch ? "yes" : "no");
   set_ini_var(ini, "mouse_switch", "grid",
      settings.mousesw_grid ? "yes" : "no");
   sprintf(buffer, "%d", settings.mousesw_x * 2);
   set_ini_var(ini, "mouse_switch", "x", buffer);
   sprintf(buffer, "%d", settings.mousesw_y * 2);
   set_ini_var(ini, "mouse_switch", "y", buffer);

   // Video settings
   set_ini_var(ini, "video", "fullscreen",
      settings.fullscreen ? "yes" : "no");
   sprintf(buffer, "%d", settings.windowed_width);
   set_ini_var(ini, "video", "width", buffer);
   sprintf(buffer, "%d", settings.windowed_height);
   set_ini_var(ini, "video", "height", buffer);
   set_ini_var(ini, "video", "vsync",
      settings.vsync ? "yes" : "no");
   set_ini_var(ini, "video", "filter",
      settings.filter ? "yes" : "no");
   set_ini_var(ini, "video", "background",
      settings.background ? "yes" : "no");
   set_ini_var(ini, "video", "zoom",
      settings.zoom ? "yes" : "no");
   set_ini_var(ini, "video", "darkbg",
      settings.dark_bg ? "yes" : "no");
   set_ini_var(ini, "video", "shaking",
      settings.shaking ? "yes" : "no");
   set_setting_value(ini, "video", "contrast", values_contrast,
      settings.contrast);
   sprintf(buffer, "%d", settings.monitor);
   set_ini_var(ini, "video", "monitor", buffer);
   set_ini_var(ini, "advanced", "lcdfix",
      settings.lcd_fix ? "yes" : "no");

   // Screen reader settings
   set_setting_value(ini, "reader", "type", values_reader,
      settings.reader);

   // Sound settings
   sprintf(buffer, "%d", settings.bgm_volume);
   set_ini_var(ini, "sound", "bgm_volume", buffer);
   sprintf(buffer, "%d", settings.sfx_volume);
   set_ini_var(ini, "sound", "sfx_volume", buffer);
   set_ini_var(ini, "sound", "stereo",
      settings.stereo ? "yes" : "no");
   set_ini_var(ini, "sound", "reverse",
      settings.reverse ? "yes" : "no");

   // Keyboard settings
   set_keyboard_value(ini, "keyboard", "up_1",
      settings.keyboard[PL_INPUT_UP][0]);
   set_keyboard_value(ini, "keyboard", "up_2",
      settings.keyboard[PL_INPUT_UP][1]);
   set_keyboard_value(ini, "keyboard", "down_1",
      settings.keyboard[PL_INPUT_DOWN][0]);
   set_keyboard_value(ini, "keyboard", "down_2",
      settings.keyboard[PL_INPUT_DOWN][1]);
   set_keyboard_value(ini, "keyboard", "left_1",
      settings.keyboard[PL_INPUT_LEFT][0]);
   set_keyboard_value(ini, "keyboard", "left_2",
      settings.keyboard[PL_INPUT_LEFT][1]);
   set_keyboard_value(ini, "keyboard", "right_1",
      settings.keyboard[PL_INPUT_RIGHT][0]);
   set_keyboard_value(ini, "keyboard", "right_2",
      settings.keyboard[PL_INPUT_RIGHT][1]);
   set_keyboard_value(ini, "keyboard", "jump_1",
      settings.keyboard[PL_INPUT_ACTION][0]);
   set_keyboard_value(ini, "keyboard", "jump_2",
      settings.keyboard[PL_INPUT_ACTION][1]);

   // Joystick settings
   set_joystick_value(ini, "joystick", "up_1",
      settings.joystick[PL_INPUT_UP][0]);
   set_joystick_value(ini, "joystick", "up_2",
      settings.joystick[PL_INPUT_UP][1]);
   set_joystick_value(ini, "joystick", "down_1",
      settings.joystick[PL_INPUT_DOWN][0]);
   set_joystick_value(ini, "joystick", "down_2",
      settings.joystick[PL_INPUT_DOWN][1]);
   set_joystick_value(ini, "joystick", "left_1",
      settings.joystick[PL_INPUT_LEFT][0]);
   set_joystick_value(ini, "joystick", "left_2",
      settings.joystick[PL_INPUT_LEFT][1]);
   set_joystick_value(ini, "joystick", "right_1",
      settings.joystick[PL_INPUT_RIGHT][0]);
   set_joystick_value(ini, "joystick", "right_2",
      settings.joystick[PL_INPUT_RIGHT][1]);
   set_joystick_value(ini, "joystick", "jump_1",
      settings.joystick[PL_INPUT_ACTION][0]);
   set_joystick_value(ini, "joystick", "jump_2",
      settings.joystick[PL_INPUT_ACTION][1]);
   set_joystick_value(ini, "joystick", "pause_1",
      settings.joystick[PL_INPUT_PAUSE][0]);
   set_joystick_value(ini, "joystick", "pause_2",
      settings.joystick[PL_INPUT_PAUSE][1]);

   // Advanced settings
   set_setting_value(ini, "advanced", "timing", values_timing,
      settings.timing);

   // Launcher settings
   set_ini_var(ini, "launcher", "software",
      settings.no_hwaccel ? "yes" : "no");
   set_ini_var(ini, "launcher", "nosound",
      settings.no_sound ? "yes" : "no");
   set_ini_var(ini, "launcher", "nojoystick",
      settings.no_joystick ? "yes" : "no");
   set_ini_var(ini, "launcher", "nojoyaxis",
      settings.no_joystick_axis ? "yes" : "no");

   // Save INI file
   // To-do: safety checks
   save_ini("savedata/options.ini", ini);
   destroy_ini(ini);
}

//***************************************************************************
// set_setting_value [internal]
// Takes an integer and stores its value from a list of possible strings...
// er, it's just the set counterpart to get_setting_value.
//---------------------------------------------------------------------------
// param ini: pointer to INI object
// param section: name of section
// param variable: name of variable
// param values: list of valid values
// param value: value to store
//***************************************************************************

static void set_setting_value(Ini *ini, const char *section,
const char *variable, const ValidValues *values, int value) {
   // Look all possible values for a match
   for (size_t i = 0; values[i].name != NULL; i++) {
      if (values[i].value == value) {
         set_ini_var(ini, section, variable, values[i].name);
         return;
      }
   }

   // No match?
   return;
}

//***************************************************************************
// set_keyboard_value [internal]
// Stores a keyboard setting in an INI object
//---------------------------------------------------------------------------
// param ini: pointer to INI object
// param section: name of section
// param variable: name of variable
// param scancode: key scancode
//***************************************************************************

static void set_keyboard_value(Ini *ini, const char *section,
const char *variable, int scancode) {
   set_ini_var(ini, section, variable, scancode_to_id(scancode));
}

//***************************************************************************
// set_joystick_value [internal]
// Converts a joystick input value into a string and stores it into the INI
// object ("button ##", "axis ## pos", "hat ## up", etc.)
//---------------------------------------------------------------------------
// param ini: pointer to INI object
// param section: name of section
// param variable: name of variable
// param value: value to store
//***************************************************************************

static void set_joystick_value(Ini *ini, const char *section,
const char *variable, uint16_t value) {
   // Where the generated string will be stored
   char buffer[0x40];

   // Determine type of value
   switch (value & 0xFF00) {
      // Nothing?
      case JOYIN_NONE:
         set_ini_var(ini, section, variable, "none");
         break;

      // Button?
      case JOYIN_BUTTON:
         sprintf(buffer, "button %d", value & 0xFF);
         set_ini_var(ini, section, variable, buffer);
         break;

      // Axis?
      case JOYIN_AXISPOS:
         sprintf(buffer, "axis %d pos", value & 0xFF);
         set_ini_var(ini, section, variable, buffer);
         break;
      case JOYIN_AXISNEG:
         sprintf(buffer, "axis %d neg", value & 0xFF);
         set_ini_var(ini, section, variable, buffer);
         break;

      // Hat?
      case JOYIN_HATUP:
         sprintf(buffer, "hat %d up", value & 0xFF);
         set_ini_var(ini, section, variable, buffer);
         break;
      case JOYIN_HATDOWN:
         sprintf(buffer, "hat %d down", value & 0xFF);
         set_ini_var(ini, section, variable, buffer);
         break;
      case JOYIN_HATLEFT:
         sprintf(buffer, "hat %d left", value & 0xFF);
         set_ini_var(ini, section, variable, buffer);
         break;
      case JOYIN_HATRIGHT:
         sprintf(buffer, "hat %d right", value & 0xFF);
         set_ini_var(ini, section, variable, buffer);
         break;

      // Undefined, don't save it
      case JOYIN_UNDEF:
         return;

      // Not set
      default:
         break;
   }
}
