//***************************************************************************
// "settings.h"
// Header file for the configuration code. Contains the definition for the
// configuration structure, accessed by the rest of the program to determine
// the game settings.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

#ifndef SETTINGS_H
#define SETTINGS_H

// Required headers
#include "input.h"

// Number of keys that can be set per action
#define KEYS_PER_ACTION 2

// Available difficulty levels
typedef enum {
   DIFF_EASY,           // Easy
   DIFF_NORMAL,         // Normal
   DIFF_HARD            // Hard
} Difficulty;

// Available contrast levels
typedef enum {
   CONTRAST_LOW,        // Low (half contrast)
   CONTRAST_MIDLOW,     // Between low and normal
   CONTRAST_NORMAL,     // Normal (unaltered)
   CONTRAST_MIDHIGH,    // Between high and normal
   CONTRAST_HIGH        // High (double contrast)
} Contrast;

// Available color filter modes
typedef enum {
   COLOR_NORMAL,              // Normal colors
   COLOR_TYPE1,               // Red-green deficiency
   COLOR_TYPE2,               // Blue deficiency
   COLOR_MONO,                // Monochrome
   COLOR_DICHRO,              // Dichrome
   COLOR_NEGATIVE,            // Negative
   COLOR_RGBSWAP,             // Swap RGB to BGR
   COLOR_LOVISION1,           // Low vision (mild)
   COLOR_LOVISION2,           // Low vision (heavy)
   COLOR_TVMODE               // TV output (composite?)
} ColorMode;

// Available screen reader output methods
typedef enum {
   READER_NONE,               // No output
   READER_NATIVE,             // Output through native screen reader
   READER_CLIPBOARD,          // Output through clipboard
   READER_TITLEBAR,           // Output through titlebar
   READER_STDOUT,             // Output through stdout
} Reader;

// Possible types of timing modes
typedef enum {
   TIMING_AUTO,               // Let the game decide
   TIMING_SPEED,              // Optimize performance
   TIMING_POWER,              // Optimize power usage
} Timing;

// Possible power-ups that can be forced
typedef enum {
   POWER_NONE,                // Don't enforce a given power
   POWER_SPIDER,              // Force spider power-up
   POWER_HAMMER,              // Force hammer power-up
   POWER_WINGS,               // Force wings power-up
   POWER_PARASOL              // Force parasol power-up
} PowerType;

// Structure where settings are stored
typedef struct {
   // Game settings
   Difficulty difficulty;           // Game difficulty
   int language;                    // Game language (-1 if not set)
   uint8_t game_speed;              // Game speed setting

   // One-switch settings
   unsigned one_switch: 1;          // Enable one-switch mode
   unsigned tap_delay;              // Maximum delay for tap
   unsigned fivetap_delay;          // Maximum delay for five-tap
   unsigned menu_delay;             // Delay between menu options

   // Mouse-switch settings
   unsigned mouse_switch: 1;        // Enable mouse-switch mode
   unsigned mousesw_grid: 1;        // Show mouse-switch regions grid
   unsigned mousesw_x;              // Horizontal division distance
   unsigned mousesw_y;              // Vertical division distance

   // Video settings
   int fullscreen_width;            // Horizontal resolution for fullscreen
   int fullscreen_height;           // Vertical resolution for fullscreen
   int windowed_width;              // Horizontal resolution for windowed
   int windowed_height;             // Vertical resolution for windowed
   unsigned fullscreen: 1;          // Fullscreen = 1, windowed = 0
   unsigned vsync: 1;               // Set to enable vsync
   unsigned filter: 1;              // Set to enable bilinear filtering
   unsigned background: 1;          // Set to enable background
   unsigned zoom: 1;                // Set to zoom in-game
   unsigned dark_bg: 1;             // Set to darken the background
   unsigned shaking: 1;             // Set to enable quakes and such
   unsigned lcd_fix: 1;             // Set to work around LCD contrast ugliness
   Contrast contrast;               // Contrast level
   int monitor;                     // Which monitor to go fullscreen on

   // Screen reader settings
   Reader reader;                   // Output method

   // Sound settings
   int8_t bgm_volume;               // Background music volume (0..100)
   int8_t sfx_volume;               // Sound effects volume (0..100)
   unsigned enable_bgm : 1;         // Background music is enabled
   unsigned enable_sfx : 1;         // Sound effects are enabled
   unsigned stereo: 1;              // Toggle between stereo and mono
   unsigned reverse: 1;             // Toggle speaker positions

   // Controls
   KeyCode keyboard[NUM_PL_INPUT][KEYS_PER_ACTION];
   uint16_t joystick[NUM_PL_INPUT][KEYS_PER_ACTION];

   // Debugging
   unsigned debug: 1;               // Enable debug mode
   unsigned pause: 1;               // Pause everything
   unsigned show_coord: 1;          // Show coordinates
   unsigned free_move: 1;           // Free movemenet
   unsigned show_hitbox: 1;         // Show object hitboxes
   unsigned show_time: 1;           // Show time spent in the level

   // Replay recording
   unsigned record: 1;              // Set if recording replay
   unsigned replay: 1;              // Set if playing replay

   // Miscellaneous
   unsigned attract: 1;             // Set to enable attract mode

   // Advanced settings
   Timing timing;                   // How to optimize timing

   // Tweaks
   unsigned score_tally: 1;         // Set if there's a score tally
   unsigned goal_vanish: 1;         // Set to vanish when reaching the goal
   unsigned spin_jump: 1;           // Set for Sol to spin when jumping
   unsigned refill_health: 1;       // Set to allow health refilling
   unsigned power_shield: 1;        // Set to allow power-ups shield you
   unsigned collect_health: 1;      // Set to make health a collectible
   unsigned clear_enemies: 1;       // Set to allow winning by destroying enemies
   unsigned invincibility_time;     // How long does invincibility last?
   unsigned blue_limit;             // Blue stars limit

   int32_t player_speed;            // Maximum player speed
   int32_t player_accel;            // Player acceleration speed
   int32_t player_climb;            // Player climbing speed
   int32_t player_friction;         // Player decceleration speed
   int32_t player_fast;             // When the player is going too fast
   int32_t player_jump;             // Jump force (good luck measuring this)
   int32_t player_weight;           // Player falling speed
   int32_t player_height;           // Player's usual height
   int32_t player_crouch;           // Player health when crouching
   uint8_t force_power;             // Force this specific power-up
   unsigned show_power: 1;          // Show power-up behind player's back

   unsigned flamer_ai: 2;           // Flamer AI type (0..2)
   unsigned sprayer_ai: 2;          // Sprayer AI type (0..2)
   unsigned turret_ai: 2;           // Turret AI type (0..2)
   unsigned grabber_ai: 2;          // Grabber AI type (0..2)
   unsigned spider_ai: 2;           // Spider AI type (0..2)
   unsigned heater_ai: 2;           // Heater AI type (0..2)

   uint32_t font_lit[3];            // Lit font colors
   uint32_t font_dim[3];            // Dim font colors
   uint32_t box_lit[3];             // Lit box colors
   uint32_t box_dim[3];             // Dim box colors

   // Cheats
   unsigned level_select: 1;        // Enable level select
   unsigned no_damage: 1;           // Player can't lose health
   unsigned no_health: 1;           // Player behaves as if it had 1HP
   unsigned extra_health: 1;        // Player can have up to 5HP
   unsigned no_enemies: 1;          // Remove all enemies
   unsigned no_checkpoints: 1;      // Remove all checkpoints
   unsigned no_friction: 1;         // No friction for player
   unsigned pinball: 1;             // Pinball mode!
   unsigned force_virtual: 1;       // Force virtual theme
   unsigned cga_mode: 1;            // Use CGA-like colors
   unsigned beeper: 1;              // Use beeper-like sounds
   ColorMode color_mode;            // Color mode

   // Overriden settings
   unsigned override_fullscreen: 1; // Override fullscreen setting
   unsigned override_contrast: 1;   // Override contrast setting
   unsigned override_one_switch: 1; // Override one switch mode
   unsigned override_mouse_switch:1;// Override mouse switch mode
   unsigned override_reader: 1;     // Override screen reader setting

   // Subsystem control
   unsigned no_hwaccel: 1;          // Disable hardware acceleration
   unsigned no_sound: 1;            // Disable sound subsystem
   unsigned no_joystick: 1;         // Disable joystick subsystem
   unsigned no_joystick_axis: 1;    // Disable joystick axes only
   unsigned audiovideo: 1;          // Render in audiovideo mode

   // Used to tell if we're the demo or the full version
   // Clear: full version - Set: demo version
   unsigned demo: 1;
} Settings;
extern Settings settings;

// Function prototypes
void init_config(void);
void load_tweaks(void);
void load_config(void);
void load_launcher_config(void);
void save_config(void);

#endif
