//***************************************************************************
// "sound.h"
// Sound subsystem definitions.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

#ifndef SOUND_H
#define SOUND_H

// Required headers
#include <stdint.h>
#include "audiovideo.h"

// List of BGM IDs
typedef enum {
   BGM_NONE = -1,       // ID that means "no BGM"s

   BGM_VIRTUAL,         // Virtual theme
   BGM_PARK,            // Park theme
   BGM_SEWER,           // Sewer theme
   BGM_HARBOR,          // Harbor theme
   BGM_DESERT,          // Desert theme
   BGM_CAVE,            // Cave theme
   BGM_FACTORY,         // Factory theme
   BGM_CARNIVAL,        // Carnival theme

   BGM_BOSS,            // Boss theme
   BGM_FINALBOSS,       // Final boss theme

   BGM_CUTSCENE,        // Cutscene theme
   BGM_CUTSCENE2,       // Cutscene theme (alt)
   BGM_MREVIL,          // Mr.Evil cutscene

   BGM_TITLE,           // Title screen
   BGM_MENU,            // Menu screens
   BGM_ENDING,          // Ending
   BGM_INVINCIBILITY,   // Invincibility
   BGM_FINISH,          // Finish level

   BGM_TRAILER,         // Trailer theme
   NUM_BGM              // Number of BGMs
} BgmID;

// List of sound effect IDs
typedef enum {
   SFX_NONE = -1,       // ID that means "no SFX"

   SFX_JUMP,            // Player jumps
   SFX_FALL,            // Player falls (audiovideo only)
   SFX_SKID,            // Player quickly changes direction
   SFX_HURT,            // Player gets hurt
   SFX_FLY,             // Player flies
   SFX_HAMMER,          // Player uses hammer
   SFX_FOOTSTEP,        // Player footstep (audiovideo only)
   SFX_LAND,            // Player lands on the floor (audiovideo only)
   SFX_WALL,            // Player bumps into wall (audiovideo only)
   SFX_CEILING,         // Player bumps into ceiling (audiovideo only)
   SFX_PINBALL,         // Pinball mode!
   SFX_DESTROY,         // Destroy enemy
   SFX_WARNING,         // Enemy warning
   SFX_WARNINGALT,      // Hazard warning
   SFX_ROBOTFOOTSTEP,   // Robot footstep
   SFX_ROBOTCLIMBUP,    // Robot climbing (upwards)
   SFX_ROBOTCLIMBDOWN,  // Robot climbing (downwards)
   SFX_ROBOTDRIVE,      // Wheeled robot driving
   SFX_ROBOTROLL,       // Robot is rolling
   SFX_ROBOTIDLE,       // Robot is idling
   SFX_ROBOTHELI,       // Robot helicopter sound
   SFX_EXPLOSION,       // Normal explosion
   SFX_BIGEXPLOSION,    // Big explosion
   SFX_CRASH,           // Something crashed (normal)
   SFX_CRUSH,           // Something crashed (heavy)
   SFX_BREAK,           // Something broke
   SFX_CRACK,           // Platform breaks
   SFX_SPRING,          // Spring
   SFX_CLUESPRING,      // Spring (audiovideo clue)
   SFX_BALLOON,         // Balloon
   SFX_CLUEBALLOON,     // Balloon (audiovideo clue)
   SFX_HEART,           // Heart
   SFX_CLUEHEART,       // Heart (audiovideo clue)
   SFX_SHIELD,          // Shield
   SFX_CLUESHIELD,      // Shield (audiovideo clue)
   SFX_CLUEINVINCIBILITY,// Invincibility (audiovideo clue)
   SFX_POWERUP,         // Power-up
   SFX_CLUEPOWERUP,     // Power-up (audiovideo clue)
   SFX_YUPEE,           // Yupee!
   SFX_CHECKPOINT,      // Checkpoint
   SFX_CLUECHECKPOINT,  // Checkpoint (audiovideo clue)
   SFX_GOAL,            // Goal
   SFX_CLUEGOAL,        // Goal (audiovideo clue)
   SFX_BLUESTAR,        // Blue star
   SFX_CLUEBLUESTAR,    // Blue star (audiovideo clue)
   SFX_SWITCH,          // Switch
   SFX_CLUESWITCH,      // Switch (audiovideo clue)
   SFX_DROP,            // Something drops
   SFX_SHOOT,           // Shoot projectile
   SFX_MISSILE,         // Shoot missile
   SFX_FIRE,            // Shooting fire
   SFX_SHOCK,           // Electric shock
   SFX_CLUESTALACTITE,  // Stalactite (audiovideo clue)
   SFX_CLUEPROJECTILE,  // Projectile (audiovideo clue)
   SFX_CLUEBOUNCING,    // Bouncing hazard (audiovideo clue)
   SFX_CLUELIQUID,      // Liquid hazard (audiovideo clue)
   SFX_CLUECOIL,        // Coil (audiovideo clue)
   SFX_CLUEBUZZSAW,     // Buzzsaw (audiovideo clue)
   SFX_STATIC,          // Static noise
   SFX_SONARLEFT,       // Sonar signal left (audiovideo clue)
   SFX_SONARRIGHT,      // Sonar signal right (audiovideo clue)
   SFX_SONARUP,         // Sonar signal up (audiovideo clue)
   SFX_SONARDOWN,       // Sonar signal down (audiovideo clue)
   SFX_BEEP,            // Menu beep
   SFX_OK,              // Menu OK
   SFX_CANCEL,          // Menu cancel
   SFX_SPEAKROBOT,      // Text speech (robotic voice)
   SFX_SPEAKSOL,        // Text speech (Sol voice)
   SFX_SPEAKRUBY,       // Text speech (Ruby voice)
   SFX_SPEAKEVIL,       // Text speech (Mr. Evil voice)
   SFX_LOGO,            // Logo screen
   SFX_CHEAT,           // Entering a cheat

   NUM_SFX              // Number of SFXs
} SfxID;

// Sound effect group IDs
// Groups are used to prevent more than one kind of sound effect playing at
// the same time, helps reducing noise clutter and such if the game is too
// active at any given moment.
typedef enum {
   SFXGROUP_PLAYER,     // Player-related sounds
   SFXGROUP_ITEM,       // Generic channel for items
   SFXGROUP_ALERT,      // Something to be alert of
   SFXGROUP_NOISE,      // Loud noises
   SFXGROUP_QUIET,      // Quiet noises
   SFXGROUP_MISC,       // Miscellaneous effects
   SFXGROUP_UI,         // UI sound effects
   NUM_SFXGROUP         // Number of SFX groups
} SfxGroupID;

// Flags for sfx_play_ex
#define SFLAG_ABSOLUTE     0x00000001     // Not relative to player
#define SFLAG_HIPITCH      0x00000002     // Use a higher pitched version
#define SFLAG_LOPITCH      0x00000004     // Use a lower pitched version
#define SFLAG_REVERBOK     0x00000008     // Allow ceiling reverberation
#define SFLAG_NOADJUST     0x00000010     // Adjust position in audio mode
#define SFLAG_NOWALLS      0x00000020     // Don't let walls block sound

// Function prototypes
void init_sound(void);
void deinit_sound(void);
void sound_render_callback(void *, uint8_t *, int);
void load_bgm(BgmID);
void play_bgm(BgmID);
void pause_bgm(void);
void resume_bgm(void);
void stop_bgm(void);
BgmID get_current_bgm(void);
void unload_bgm(BgmID);
void unload_all_bgm(void);
void play_sfx(SfxID);
void play_2d_sfx(SfxID, int32_t, int32_t);
void play_sfx_ex(SfxID id, int32_t, int32_t, uint16_t, uint64_t, uint32_t);
void stop_sfx(void);
BgmID get_bgm_by_name(const char *);
SfxID get_sfx_by_name(const char *);
int bgm_exists(BgmID);

#endif
