//***************************************************************************
// "text.c"
// Font loading and text rendering
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stdint.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <SDL2/SDL.h>
#include "main.h"
#include "file.h"
#include "level.h"
#include "parser.h"
#include "settings.h"
#include "text.h"
#include "video.h"

// Where the list of supported languages is stored
typedef struct {
   char *id;            // Two-letter ID (for most stuff)
   char *longid;        // Full name ID (for language filenames)
   char *name;          // What appears in language select menu
} LangList;
static size_t num_languages = 0;
static LangList *language_list = NULL;

// Where the game text is stored
static Ini *lang_ini = NULL;
Text text;

// Definition of a character
typedef struct {
   const Sprite *lit;         // Sprite used for the lit font
   const Sprite *dim;         // Sprite used for the dim font
   unsigned width;            // Width of character in pixels
} Character;

// Where we store the characters
// The organization of this is somewhat wacky. This is an array to arrays.
// Unicode is way too big and most characters won't be stored at all, so
// instead what we do is split it into groups of 256 characters each.
// ...in other words, it's a freaking hash table >_>
#define NUM_BLOCKS 0x1100
static Character *charset[NUM_BLOCKS] = { 0 };

// A list to keep track of all the sprites we create for the font (we can't
// store them in the characters since some sprites may be shared among
// several of them - e.g. uppercase and lowercase).
static Sprite *sprite_list = NULL;

// Default language file to try if we can't load the current language
#define DEF_LANG "languages/english.ini"

// Private function prototypes
static const Character *get_char_data(uint32_t);

//***************************************************************************
// set_default_err_msg
// Sets default error messages for when the language file isn't loaded.
//---------------------------------------------------------------------------
// NO, THIS FUNCTION CANNOT FAIL, IF IT *SOMEHOW* MANAGES TO CRASH THEN YOUR
// SYSTEM IS SERIOUSLY SCREWED AND YOU HAVE MUCH BIGGER PROBLEMS.
//***************************************************************************

void set_default_err_msg(void) {
   // Reset all error messages to NULL
   // This is used to mean they aren't loaded
   for (unsigned i = 0; i < NUM_ERRCODES; i++)
      text.error[i] = NULL;

   // These errors can happen when the language file isn't loaded, so make
   // sure to have something to show if that happens.
   text.error_title = "Error";
   text.error[ERR_NOMEMORY] = "Ran out of memory.";
   text.error[ERR_NOASSETS] = "Could not load game assets.";
   text.error[ERR_LOADMOD] = "Could not load mod \"{param}\".";
   text.error[ERR_INITFS] = "Could not initialize PhysFS.";
   text.error[ERR_NOLANG] = "No languages listed.";
   text.error[ERR_LOADLANG] = "Could not load language file \"{param}\".";
   text.error[ERR_UNKNOWN] = "Unknown error.";
}

//***************************************************************************
// init_languages
// Initializes the language list.
//***************************************************************************

void init_languages(void) {
   // Just to be sure
   deinit_languages();

   // Try to open the language list file
   File *file = open_file("system/languages", FILE_READ);
   if (file != NULL) {
      // Scan all the lines in the file
      while (!end_of_file(file)) {
         // Get next line
         char *line = read_line(file);
         if (line == NULL) break;

         // Retrieve arguments
         Args *args = parse_args(line);
         if (args == NULL) {
            free(line);
            continue;
         }

         // Check that the argument count is right (there are three: short
         // ID, long ID and name that appears in the language select)
         if (args->count != 3) {
            free_args(args);
            free(line);
            continue;
         }

         // These two should be valid IDs
         if (!is_valid_id(args->list[0]) && !is_valid_id(args->list[1])) {
            free_args(args);
            free(line);
            continue;
         }

         // Make room for new language
         num_languages++;
         language_list = (LangList *) realloc(language_list,
            sizeof(LangList) * num_languages);
         if (language_list == NULL)
            abort_program(ERR_NOMEMORY, NULL);

         LangList *lang = &language_list[num_languages-1];
         lang->id = NULL;
         lang->longid = NULL;
         lang->name = NULL;

         // Store short ID
         lang->id = (char *) malloc(strlen(args->list[0]) + 1);
         if (lang->id == NULL) abort_program(ERR_NOMEMORY, NULL);
         strcpy(lang->id, args->list[0]);

         // Store long ID
         lang->longid = (char *) malloc(strlen(args->list[1]) + 1);
         if (lang->longid == NULL) abort_program(ERR_NOMEMORY, NULL);
         strcpy(lang->longid, args->list[1]);

         // Store language name
         lang->name = (char *) malloc(strlen(args->list[2]) + 1);
         if (lang->name == NULL) abort_program(ERR_NOMEMORY, NULL);
         strcpy(lang->name, args->list[2]);

         // Done with line
         free_args(args);
         free(line);
      }

      // Done with reading the list
      close_file(file);
   }

   // No languages found?
   if (num_languages == 0)
      abort_program(ERR_NOLANG, NULL);

#ifdef DEBUG
   // Show every listed language
   for (unsigned i = 0; i < num_languages; i++) {
      const LangList *lang = &language_list[i];
      printf("Language #%u: short ID \"%s\", long ID \"%s\", name \"%s\"\n",
         i+1, lang->id, lang->longid, lang->name);
   }
#endif
}

//***************************************************************************
// deinit_languages
// Deinitializes the language list.
//***************************************************************************

void deinit_languages(void) {
   // Get rid of language list if needed
   if (language_list != NULL) {
      for (size_t i = 0; i < num_languages; i++) {
         free(language_list[i].id);
         free(language_list[i].longid);
         free(language_list[i].name);
      }
      free(language_list);
   }

   // Reset language list
   num_languages = 0;
   language_list = NULL;
}

//***************************************************************************
// load_language
// Loads the text for the current language.
//***************************************************************************

void load_language(void) {
#ifdef DEBUG
   fputs("Loading language\n", stderr);
#endif

   // Just to be sure
   unload_language();

   // Can't load language if none was selected yet
   /*if (settings.language == -1)
      return;*/

   // Load INI file with the translations
   char filename[0x80];
   if (settings.language != -1) {
      sprintf(filename, "languages/%s.ini",
         language_list[settings.language].longid);
      lang_ini = load_ini(filename);
   }
   if (lang_ini == NULL) {
      sprintf(filename, "languages/%s.ini", language_list[0].longid);
      lang_ini = load_ini(filename);
      if (lang_ini == NULL)
         abort_program(ERR_LOADLANG,
         language_list[settings.language].longid);
   }

   // To make the following code easier to read
#define INI(sect, var) get_ini_var(lang_ini, sect, var, NULL)

   // Launcher text
   text.launcher.play = INI("launcher", "play");
   text.launcher.settings = INI("launcher", "settings");
   text.launcher.accessibility = INI("launcher", "accessibility");
   text.launcher.modding = INI("launcher", "modding");

   text.launcher.play_sol = INI("launcher", "playsol");
   text.launcher.play_minisol = INI("launcher", "playminisol");
   text.launcher.manual = INI("launcher", "manual");
   text.launcher.saves = INI("launcher", "saves");
   text.launcher.quit = INI("launcher", "quit");

   text.launcher.fullscreen = INI("launcher", "set_fullscreen");
   text.launcher.software = INI("launcher", "set_software");
   text.launcher.no_sound = INI("launcher", "set_nosound");
   text.launcher.no_joystick = INI("launcher", "set_nojoystick");

   text.launcher.reader.title = INI("launcher", "reader");
   text.launcher.reader.disabled = INI("launcher", "reader_disabled");
#ifdef _WIN32
   text.launcher.reader.native = INI("launcher", "reader_sapi");
#else
   text.launcher.reader.native = INI("launcher", "reader_speechd");
#endif
   text.launcher.reader.clipboard = INI("launcher", "reader_clipboard");
   text.launcher.reader.titlebar = INI("launcher", "reader_titlebar");

   text.launcher.audiovideo = INI("launcher", "audiovideo");

   // Ugly hack used only for beta testing
   if (text.launcher.audiovideo == NULL ||
   text.launcher.audiovideo[0] == '\0')
      text.launcher.audiovideo = "Enable audio mode";

   text.launcher.onesw.title = INI("launcher", "onesw");
   text.launcher.onesw.enable = INI("launcher", "onesw_enable");
   text.launcher.onesw.tap = INI("launcher", "onesw_tap");
   text.launcher.onesw.menu = INI("launcher", "onesw_menu");
   text.launcher.onesw.fivetap = INI("launcher", "onesw_fivetap");
   text.launcher.onesw.comment = INI("launcher", "onesw_comment");

   text.launcher.mod.list = INI("launcher", "mod_list");
   text.launcher.mod.add = INI("launcher", "mod_add");
   text.launcher.mod.rem = INI("launcher", "mod_remove");
   text.launcher.mod.args = INI("launcher", "mod_args");

   // In-game text
   text.ingame.demo_mode = INI("ingame", "demo");
   text.ingame.pause = INI("ingame", "pause");
   text.ingame.pause_title = INI("ingame", "paused");
   text.ingame.opt_continue = INI("ingame", "continue");
   text.ingame.opt_restart = INI("ingame", "restart");
   text.ingame.opt_quit = INI("ingame", "quit");

   // Score tally
   text.tally.title = INI("tally", "title");
   text.tally.enemy = INI("tally", "enemy");
   text.tally.items = INI("tally", "items");
   text.tally.special = INI("tally", "special");
   text.tally.total = INI("tally", "total");
   text.tally.time = INI("tally", "time");

   // Logo screen
   text.logo.name = INI("logo", "name");

   // Title screen
   text.title.copyright = INI("title", "copyright");
   text.title.start = INI("title", "start");
   text.title.editor = INI("title", "editor");
   text.title.options = INI("title", "options");
   text.title.quit = INI("title", "quit");

   // Save slot screen
   text.saveselect.title = INI("saveselect", "select");
   text.saveselect.cancel = INI("saveselect", "cancel");
   text.saveselect.level = INI("saveselect", "level");
   text.saveselect.bonus = INI("saveselect", "bonuslevel");
   text.saveselect.cutscene = INI("saveselect", "cutscene");

   // Options menu
   text.options.title = INI("options", "options");
   text.options.gameplay = INI("options", "gameplay");
   text.options.video = INI("options", "video");
   text.options.sound = INI("options", "sound");
   text.options.keyboard = INI("options", "keyboard");
   text.options.joystick = INI("options", "joystick");
   text.options.mouse = INI("options", "mouse");
   text.options.back = INI("options", "back");

   // Options gameplay menu
   text.options_gameplay.difficulty = INI("options_gameplay", "difficulty");
   text.options_gameplay.game_speed = INI("options_gameplay", "game_speed");
   text.options_gameplay.one_switch = INI("options_gameplay", "one_switch");
   text.options_gameplay.language = INI("options_gameplay", "language");

   // Difficulty names
   text.options_gameplay.diff_names[DIFF_EASY] =
      INI("options_gameplay", "easy");
   text.options_gameplay.diff_names[DIFF_NORMAL] =
      INI("options_gameplay", "normal");
   text.options_gameplay.diff_names[DIFF_HARD] =
      INI("options_gameplay", "hard");

   // One-switch statuses
   text.options_gameplay.enabled = INI("options_gameplay", "enabled");
   text.options_gameplay.disabled = INI("options_gameplay", "disabled");

   // One-switch warning screen
   text.options_gameplay.warning = INI("oneswitch", "warning");

   text.options_gameplay.onesw_on[0] = INI("oneswitch", "text_1a");
   text.options_gameplay.onesw_on[1] = INI("oneswitch", "text_2a");
   text.options_gameplay.onesw_on[2] = INI("oneswitch", "text_3a");
   text.options_gameplay.onesw_on[3] = INI("oneswitch", "text_4a");
   text.options_gameplay.onesw_on[4] = INI("oneswitch", "text_5a");
   text.options_gameplay.onesw_on[5] = INI("oneswitch", "text_6a");
   text.options_gameplay.onesw_on[6] = INI("oneswitch", "text_7a");
   text.options_gameplay.onesw_on[7] = INI("oneswitch", "text_8a");

   text.options_gameplay.onesw_off[0] = INI("oneswitch", "text_1b");
   text.options_gameplay.onesw_off[1] = INI("oneswitch", "text_2b");
   text.options_gameplay.onesw_off[2] = INI("oneswitch", "text_3b");
   text.options_gameplay.onesw_off[3] = INI("oneswitch", "text_4b");
   text.options_gameplay.onesw_off[4] = INI("oneswitch", "text_5b");
   text.options_gameplay.onesw_off[5] = INI("oneswitch", "text_6b");
   text.options_gameplay.onesw_off[6] = INI("oneswitch", "text_7b");
   text.options_gameplay.onesw_off[7] = INI("oneswitch", "text_8b");

   text.options_gameplay.ok = INI("oneswitch", "ok");
   text.options_gameplay.cancel = INI("oneswitch", "cancel");

   // Options video menu
   text.options_video.ratio = INI("options_video","ratio");
   text.options_video.smaller = INI("options_video","smaller");
   text.options_video.larger = INI("options_video","larger");
   text.options_video.fullscreen = INI("options_video", "fullscreen");
   text.options_video.vsync = INI("options_video", "vsync");
   text.options_video.filtering = INI("options_video", "filtering");

   text.options_video.contrast = INI("options_video", "contrast");
   text.options_video.background = INI("options_video", "background");
   text.options_video.dim_bg = INI("options_video", "dimbg");
   text.options_video.zoom = INI("options_video", "zoom");
   text.options_video.shaking = INI("options_video", "shaking");

   text.options_video.next = INI("options_video", "next");
   text.options_video.prev = INI("options_video", "prev");

   text.options_video.enabled = INI("options_video", "enabled");
   text.options_video.disabled = INI("options_video", "disabled");
   text.options_video.low = INI("options_video", "low");
   text.options_video.medium = INI("options_video", "medium");
   text.options_video.high = INI("options_video", "high");
   text.options_video.show = INI("options_video", "show");
   text.options_video.hide = INI("options_video", "hide");
   text.options_video.dim = INI("options_video", "dim");
   text.options_video.no_dim = INI("options_video", "nodim");
   text.options_video.zoomed = INI("options_video", "zoomed");
   text.options_video.no_zoom = INI("options_video", "nozoom");

   // Options sound menu
   text.options_sound.play_sound = INI("options_sound", "play_sound");
   text.options_sound.prev_sound = INI("options_sound", "prev_sound");
   text.options_sound.next_sound = INI("options_sound", "next_sound");
   text.options_sound.bgm_volume = INI("options_sound", "bgm_volume");
   text.options_sound.sfx_volume = INI("options_sound", "sfx_volume");
   text.options_sound.stereo = INI("options_sound", "stereo");
   text.options_sound.enabled = INI("options_sound", "enabled");
   text.options_sound.disabled = INI("options_sound", "disabled");
   text.options_sound.reverse = INI("options_sound", "reverse");
   text.options_sound.normal = INI("options_sound", "normal");
   text.options_sound.reversed = INI("options_sound", "reversed");

   // Sound test BGM names
   text.options_sound.bgm_names[0] =
      INI("sound_test", "silence");
   text.options_sound.bgm_names[BGM_VIRTUAL+1] =
      INI("sound_test", "virtual");
   text.options_sound.bgm_names[BGM_PARK+1] =
      INI("sound_test", "park");
   text.options_sound.bgm_names[BGM_SEWER+1] =
      INI("sound_test", "sewer");
   text.options_sound.bgm_names[BGM_HARBOR+1] =
      INI("sound_test", "harbor");
   text.options_sound.bgm_names[BGM_DESERT+1] =
      INI("sound_test", "desert");
   text.options_sound.bgm_names[BGM_CAVE+1] =
      INI("sound_test", "cave");
   text.options_sound.bgm_names[BGM_FACTORY+1] =
      INI("sound_test", "factory");
   text.options_sound.bgm_names[BGM_CARNIVAL+1] =
      INI("sound_test", "carnival");
   text.options_sound.bgm_names[BGM_BOSS+1] =
      INI("sound_test", "boss");
   text.options_sound.bgm_names[BGM_FINALBOSS+1] =
      INI("sound_test", "final_boss");
   text.options_sound.bgm_names[BGM_CUTSCENE+1] =
      INI("sound_test", "cutscene");
   text.options_sound.bgm_names[BGM_CUTSCENE2+1] =
      INI("sound_test", "cutscene_2");
   text.options_sound.bgm_names[BGM_TITLE+1] =
      INI("sound_test", "title");
   text.options_sound.bgm_names[BGM_MENU+1] =
      INI("sound_test", "menu");
   text.options_sound.bgm_names[BGM_ENDING+1] =
      INI("sound_test", "ending");
   text.options_sound.bgm_names[BGM_INVINCIBILITY+1] =
      INI("sound_test", "invincibility");
   text.options_sound.bgm_names[BGM_FINISH+1] =
      INI("sound_test", "finish");
   text.options_sound.bgm_names[BGM_TRAILER+1] =
      INI("sound_test", "trailer");

   // Options keyboard and joystick menus
   text.options_input.left = INI("options_input", "run_left");
   text.options_input.right = INI("options_input", "run_right");
   text.options_input.down = INI("options_input", "crouch");
   text.options_input.up = INI("options_input", "look_up");
   text.options_input.jump = INI("options_input", "jump");
   text.options_input.pause = INI("options_input", "pause");

   text.options_input.button = INI("options_input", "button");
   text.options_input.axis_pos = INI("options_input", "axis_pos");
   text.options_input.axis_neg = INI("options_input", "axis_neg");
   text.options_input.axis_x_pos = INI("options_input", "axis_x_pos");
   text.options_input.axis_x_neg = INI("options_input", "axis_x_neg");
   text.options_input.axis_y_pos = INI("options_input", "axis_y_pos");
   text.options_input.axis_y_neg = INI("options_input", "axis_y_neg");
   text.options_input.hat_up = INI("options_input", "hat_up");
   text.options_input.hat_down = INI("options_input", "hat_down");
   text.options_input.hat_left = INI("options_input", "hat_left");
   text.options_input.hat_right = INI("options_input", "hat_right");
   text.options_input.dpad_up = INI("options_input", "dpad_up");
   text.options_input.dpad_down = INI("options_input", "dpad_down");
   text.options_input.dpad_left = INI("options_input", "dpad_left");
   text.options_input.dpad_right = INI("options_input", "dpad_right");

   text.options_input.press_key = INI("options_input", "press_key");
   text.options_input.press_joy = INI("options_input", "press_joy");

   // Options mouse menu
   text.options_mouse.enable = INI("options_mouse", "enable");
   text.options_mouse.yes = INI("options_mouse", "yes");
   text.options_mouse.no = INI("options_mouse", "no");
   text.options_mouse.grid = INI("options_mouse", "grid");
   text.options_mouse.show = INI("options_mouse", "show");
   text.options_mouse.hide = INI("options_mouse", "hide");
   text.options_mouse.test = INI("options_mouse", "test");
   text.options_mouse.inc_x = INI("options_mouse", "inc_x");
   text.options_mouse.dec_x = INI("options_mouse", "dec_x");
   text.options_mouse.inc_y = INI("options_mouse", "inc_y");
   text.options_mouse.dec_y = INI("options_mouse", "dec_y");

   // Editor button descriptions
   text.editor.b_new = INI("editor", "new");
   text.editor.b_info = INI("editor", "info");
   text.editor.b_load = INI("editor", "load");
   text.editor.b_save = INI("editor", "save");
   text.editor.b_play = INI("editor", "play");
   text.editor.b_tilemap = INI("editor", "tilemap");
   text.editor.b_objects = INI("editor", "objects");
   text.editor.b_select = INI("editor", "select");
   text.editor.b_undo = INI("editor", "undo");
   text.editor.b_redo = INI("editor", "redo");
   text.editor.b_help = INI("editor", "help");
   text.editor.b_quit = INI("editor", "quit");

   // Editor level info dialog text
   text.level_info.level_theme = INI("level_info", "level_theme");
   text.level_info.map_size = INI("level_info", "map_size");
   text.level_info.top = INI("level_info", "top");
   text.level_info.bottom = INI("level_info", "bottom");
   text.level_info.left = INI("level_info", "left");
   text.level_info.right = INI("level_info", "right");
   text.level_info.current_size = INI("level_info", "current_size");
   text.level_info.new_size = INI("level_info", "new_size");
   text.level_info.b_apply = INI("level_info", "apply");
   text.level_info.b_cancel = INI("level_info", "cancel");

   text.level_info.themes[THEME_VIRTUAL] = INI("level_themes", "virtual");
   text.level_info.themes[THEME_PARK] = INI("level_themes", "park");
   text.level_info.themes[THEME_SEWER] = INI("level_themes", "sewer");
   text.level_info.themes[THEME_HARBOR] = INI("level_themes", "harbor");
   text.level_info.themes[THEME_DESERT] = INI("level_themes", "desert");
   text.level_info.themes[THEME_CAVE] = INI("level_themes", "cave");
   text.level_info.themes[THEME_FACTORY] = INI("level_themes", "factory");
   text.level_info.themes[THEME_CARNIVAL] = INI("level_themes", "carnival");

   // Editor file select dialog text
   text.file_select.load_title = INI("file_select", "load_title");
   text.file_select.save_title = INI("file_select", "save_title");
   text.file_select.b_load = INI("file_select", "load");
   text.file_select.b_save = INI("file_select", "save");
   text.file_select.b_cancel = INI("file_select", "cancel");

   // The name of the home directory is different for Windows ("My
   // Documents") than for everything else (...just "home"), so this string
   // needs to change depending on the platform the game is being built for.
#ifdef _WIN32
   text.file_select.home = INI("file_select", "mydoc");
#else
   text.file_select.home = INI("file_select", "home");
   text.file_select.root = INI("file_select", "root");
#endif

   // Get fixed strings
   text.tile_select.title_coll = INI("tile_select", "tile");
   text.tile_select.title_obj = INI("tile_select", "object");

   // Get collision tile type names
   for (unsigned i = 0; i < NUM_TILETYPES; i++) {
      char name[0x10];
      sprintf(name, "type_%02x", i);
      text.tile_select.coll_names[i] = INI("collision_names", name);
   }

   // Get object tile type names
   for (unsigned i = 0; i < NUM_LVOBJIDS; i++) {
      char name[0x10];
      sprintf(name, "type_%02x", i);
      text.tile_select.obj_names[i] = INI("object_names", name);
   }

   // Get object group names
   text.tile_select.items = INI("object_groups", "items");
   text.tile_select.scenery = INI("object_groups", "scenery");
   text.tile_select.enemies = INI("object_groups", "enemies");
   text.tile_select.hazards = INI("object_groups", "hazards");
   text.tile_select.switches = INI("object_groups", "switches");

   // Editor help text
   text.editor_help.title = INI("editor_help", "title");
   text.editor_help.page = INI("editor_help", "page");

   text.editor_help.scroll[0] = INI("editor_help", "scroll_1");
   text.editor_help.scroll[1] = INI("editor_help", "scroll_2");
   text.editor_help.scroll_fast[0] = INI("editor_help", "fast_1");
   text.editor_help.scroll_fast[1] = INI("editor_help", "fast_2");
   text.editor_help.draw_tiles[0] = INI("editor_help", "draw_1");
   text.editor_help.draw_tiles[1] = INI("editor_help", "draw_2");
   text.editor_help.erase_tiles[0] = INI("editor_help", "erase_1");
   text.editor_help.erase_tiles[1] = INI("editor_help", "erase_2");
   text.editor_help.copy_tile[0] = INI("editor_help", "copy_1");
   text.editor_help.copy_tile[1] = INI("editor_help", "copy_2");
   text.editor_help.start_point[0] = INI("editor_help", "start_1");
   text.editor_help.start_point[1] = INI("editor_help", "start_2");
   text.editor_help.flip_object[0] = INI("editor_help", "flip_1");
   text.editor_help.flip_object[1] = INI("editor_help", "flip_2");
   text.editor_help.select_tile[0] = INI("editor_help", "select_1a");
   text.editor_help.select_tile[1] = INI("editor_help", "select_2");
   text.editor_help.select_tile_2[0] = INI("editor_help", "select_1b");
   text.editor_help.select_tile_2[1] = INI("editor_help", "select_2");

   text.editor_help.new_level[0] = INI("editor_help", "new_1");
   text.editor_help.new_level[1] = INI("editor_help", "new_2");
   text.editor_help.load_level[0] = INI("editor_help", "load_1");
   text.editor_help.load_level[1] = INI("editor_help", "load_2");
   text.editor_help.save_level[0] = INI("editor_help", "save_1");
   text.editor_help.save_level[1] = INI("editor_help", "save_2");
   text.editor_help.edit_info[0] = INI("editor_help", "info_1");
   text.editor_help.edit_info[1] = INI("editor_help", "info_2");
   text.editor_help.play_level[0] = INI("editor_help", "play_1");
   text.editor_help.play_level[1] = INI("editor_help", "play_2");
   text.editor_help.tilemap_mode[0] = INI("editor_help", "tilemap_1");
   text.editor_help.tilemap_mode[1] = INI("editor_help", "tilemap_2");
   text.editor_help.objects_mode[0] = INI("editor_help", "objects_1");
   text.editor_help.objects_mode[1] = INI("editor_help", "objects_2");
   text.editor_help.show_help[0] = INI("editor_help", "help_1");
   text.editor_help.show_help[1] = INI("editor_help", "help_2");
   text.editor_help.quit_editor[0] = INI("editor_help", "quit_1");
   text.editor_help.quit_editor[1] = INI("editor_help", "quit_2");

   // Editor error messages
   text.editor_error.load_level = INI("editor_error", "load_level");
   text.editor_error.save_level = INI("editor_error", "save_level");
   text.editor_error.demo_error = INI("editor_error", "demo_error");
   text.editor_error.press_key = INI("editor_error", "press_key");

   // Screen reader-only text
   text.reader.up = INI("reader", "up");
   text.reader.down = INI("reader", "down");
   text.reader.left = INI("reader", "left");
   text.reader.right = INI("reader", "right");
   text.reader.lclick = INI("reader", "lclick");
   text.reader.mclick = INI("reader", "mclick");
   text.reader.rclick = INI("reader", "rclick");

   // Error messages
   text.error_title = INI("error", "title");
   text.error[ERR_NOMEMORY] = INI("error", "nomemory");
   text.error[ERR_INITSDL] = INI("error", "initsdl");
   text.error[ERR_INITVIDEO] = INI("error", "initvideo");
   text.error[ERR_UPDATEVIDEO] = INI("error", "updatevideo");
   text.error[ERR_INITREADER] = INI("error", "initreader");
   text.error[ERR_BADSAVEGAME] = INI("error", "badsavegame");
   text.error[ERR_LOADFONT] = INI("error", "loadfont");
   text.error[ERR_LOADSCENES] = INI("error", "loadscenes");
   text.error[ERR_NUMSCENES] = INI("error", "numscenes");
   text.error[ERR_LOADLEVEL] = INI("error", "loadlevel");
   text.error[ERR_LOADLEVELED] = INI("error", "loadleveled");
   text.error[ERR_LOADCUTSCENE] = INI("error", "loadcutscene");
   text.error[ERR_LOADBG] = INI("error", "loadbg");
   text.error[ERR_BADCOMMAND] = INI("error", "badcommand");
   text.error[ERR_NUMARGS] = INI("error", "numargs");
   text.error[ERR_GFXSETID] = INI("error", "gfxsetid");
   text.error[ERR_ANIMID] = INI("error", "animid");
   text.error[ERR_ELEMENTID] = INI("error", "elementid");
   text.error[ERR_BGMID] = INI("error", "bgmid");
   text.error[ERR_SFXID] = INI("error", "sfxid");
   text.error[ERR_COORDVAL] = INI("error", "coordval");
   text.error[ERR_SPEEDVAL] = INI("error", "speedval");
   text.error[ERR_TIMERVAL] = INI("error", "timerval");
   text.error[ERR_VOICEID] = INI("error", "voiceid");
   text.error[ERR_FILTERTYPE] = INI("error", "filtertype");
   text.error[ERR_FILELIST] = INI("error", "filelist");
   text.error[ERR_MEDIALIST] = INI("error", "medialist");
   text.error[ERR_UNKNOWN] = INI("error", "unknown");

   // Done with this
#undef INI
}

//***************************************************************************
// unload_language
// Unloads the language data (freeing up the resources it was using).
//***************************************************************************

void unload_language(void) {
   // Unload INI file
   if (lang_ini != NULL) {
      destroy_ini(lang_ini);
      lang_ini = NULL;
   }

   // Make sure no dangling pointers are left around
   memset(&text, 0, sizeof(text));

   // Restore some default error messages in case they were overriden by the
   // language file, since they may still happen and we don't want any issues
   // if they do.
   set_default_err_msg();
}

//***************************************************************************
// get_num_languages
// Returns how many languages are available.
//---------------------------------------------------------------------------
// return: number of languages
//***************************************************************************

size_t get_num_languages(void) {
   return num_languages;
}

//***************************************************************************
// get_language_id
// Gets the two-letter ID for the current language.
//---------------------------------------------------------------------------
// return: two-letter ID
//***************************************************************************

const char *get_language_id(void) {
   if (settings.language == -1)
      return language_list[0].id;
   else
      return language_list[settings.language].id;
}

//***************************************************************************
// get_language_longid
// Gets the full-name ID for a language.
//---------------------------------------------------------------------------
// param which: which language
// return: full-name ID
//***************************************************************************

const char *get_language_longid(size_t which) {
   return language_list[which].longid;
}

//***************************************************************************
// get_language_name
// Gets the name of a language.
//---------------------------------------------------------------------------
// param which: which language
// return: language name
//***************************************************************************

const char *get_language_name(size_t which) {
   return language_list[which].name;
}

//***************************************************************************
// load_font
// Loads the game font.
//---------------------------------------------------------------------------
// Yes, this function is more complex than you'd expect...
//***************************************************************************

void load_font(void) {
#ifdef DEBUG
   fputs("Loading font\n", stderr);
#endif
   // Table that we'll use to convert the colors from the PNG file into the
   // colors used for the lit and dim fonts
   struct {
      uint32_t src;        // Source color
      uint32_t lit;        // Converted color in lit font
      uint32_t dim;        // Converted color in dim font
   } table[3] = {
      { 0xFFFFFF00, settings.font_lit[0], settings.font_dim[0] },
      { 0xFFFF0000, settings.font_lit[1], settings.font_dim[1] },
      { 0xFF000000, settings.font_lit[2], settings.font_dim[2] }
   };

   // We need to fix the look-up table in CGA mode
   if (settings.cga_mode) {
      for (unsigned id = 0; id < 3; id++) {
         table[id].src = get_cga_color(table[id].src, 0);
         table[id].lit = get_cga_color(table[id].lit, 0);
         table[id].dim = get_cga_color(table[id].dim, 0);
      }
   }

   // Open definition file
   File *file = open_file("graphics/font/charset", FILE_READ);
   if (file == NULL)
      abort_program(ERR_LOADFONT, NULL);

   // Where we keep a pointer to the source bitmap (loaded as a sprite for
   // the sake of simplicitly). We'll read characters off this bitmap. The
   // source is specified in the definition file.
   Sprite *src = NULL;
   unsigned src_stride = 0;

   // Go through all the file
   for (;;) {
      // Get next line
      char *line = read_line(file);
      if (line == NULL) break;

      // Get first token
      // If the line is blank don't bother with it...
      char *token = strtok(line, SEPARATORS);
      if (token == NULL)
         goto skip;

      // New character?
      if (!strcmp(token, "-")) {
         // Eh, we can't read characters out of nowhere...
         // Ignore any characters before a source file has been specified
         if (!src)
            goto skip;

         // Get where this character is located and its width
         token = strtok(NULL, SEPARATORS);
         if (token == NULL) goto skip;
         unsigned x = atoi(token) << 4;
         token = strtok(NULL, SEPARATORS);
         if (token == NULL) goto skip;
         unsigned y = atoi(token) << 4;
         token = strtok(NULL, SEPARATORS);
         if (token == NULL) goto skip;
         unsigned width = atoi(token);

         // Um... outbounds?
         if (x + 0x10 > src->width || y + 0x10 > src->height)
            goto skip;

         // Here should be the ">" marker
         // Make sure the marker is here
         token = strtok(NULL, SEPARATORS);
         if (token == NULL) goto skip;
         if (strcmp(token, ">") != 0) goto skip;

         // Create sprites needed for this character
         // One is for the "lit" font (the standard colors)
         // Other is for the "dim" font (non-selected options)
         Sprite *lit = create_sprite(0x10, 0x10);
         Sprite *dim = create_sprite(0x10, 0x10);

         // We aren't allowing partial translucency in the font. This should
         // make text rendering faster
         lit->translucent = 0;
         dim->translucent = 0;

         // Add sprites to the sprite list
         // Yes, this looks sloppy, but I want to make it look clear. The
         // compiler will know what to do anyways when optimizing...
         lit->link = sprite_list;
         sprite_list = lit;
         dim->link = sprite_list;
         sprite_list = dim;

         // Get the position of the first pixel in each sprite
         uint32_t *ptr_src = &src->data[x + y * src->width];
         uint32_t *ptr_lit = lit->data;
         uint32_t *ptr_dim = dim->data;

         // Go through all the rows for this character
         for (unsigned yr = 0; yr < 0x10; yr++) {
            // Copy all pixels in this row for this character into the
            // relevant sprites. This converts the colors into the respective
            // palettes used by the lit and dim fonts.
            for (unsigned xr = 0; xr < 0x10; xr++) {
               uint32_t color = *ptr_src++;

               if (color == table[0].src) {
                  *ptr_lit++ = table[0].lit;
                  *ptr_dim++ = table[0].dim;
               } else if (color == table[1].src) {
                  *ptr_lit++ = table[1].lit;
                  *ptr_dim++ = table[1].dim;
               } else if (color == table[2].src) {
                  *ptr_lit++ = table[2].lit;
                  *ptr_dim++ = table[2].dim;
               } else {
                  *ptr_lit++ = 0;
                  *ptr_dim++ = 0;
               }
            }

            // Go for next line
            ptr_src += src_stride;
         }

         // Determine to which codepoints this character belongs
         for (;;) {
            // Get next codepoint, if any
            token = strtok(NULL, SEPARATORS);
            if (token == NULL) break;
            if (*token == '#') break;
            if (*token == '$') token++;
            unsigned codepoint = strtoul(token, NULL, 16);

            // Determine to which character block does this codepoint belong
            // If the codepoint is outside the set we support then just drop
            // it and don't bother anymore
            unsigned block = codepoint >> 8;
            if (block >= NUM_BLOCKS) continue;

            // Does the block where this codepoint belongs exist yet?
            // If it doesn't then we have to allocate it
            if (charset[block] == NULL) {
               // Get memory for it
               charset[block] = (Character *)
                  malloc(sizeof(Character) * 0x100);
               if (charset[block] == NULL)
                  abort_program(ERR_NOMEMORY, NULL);

               // Initialize all characters in it
               for (unsigned i = 0; i < 0x100; i++) {
                  charset[block][i].lit = NULL;
                  charset[block][i].dim = NULL;
                  charset[block][i].width = UINT_MAX;
               }
            }

            // Store information for this codepoint
            charset[block][codepoint & 0xFF].lit = lit;
            charset[block][codepoint & 0xFF].dim = dim;
            charset[block][codepoint & 0xFF].width = width;
         }
      }

      // New source file?
      else if (is_valid_id(token)) {
         // Get rid of current source file, if any
         if (src) destroy_sprite(src);

         // Load new source file as a sprite (I'm not going to bother with
         // libpng again, we just want the bitmap anyways)
         char filename[0x100];
         sprintf(filename, "graphics/font/%s.png", token);
         src = load_sprite(filename);
         if (src == NULL)
            abort_program(ERR_UNKNOWN, filename);

         // Get information about this sprite
         // Some of it may be invalid if the sprite is too small, but if that
         // happens then characters will be always outbounds and they won't
         // be processed anyways.
         src_stride = src->width - 0x10;
      }

      // Done with the line
      skip: free(line);
   }

   // Done, get rid of everything
   if (src) destroy_sprite(src);
   close_file(file);
}

//***************************************************************************
// draw_text
// Draws a string of text on screen.
//---------------------------------------------------------------------------
// param str: string to draw (must be UTF-8!)
// param x: X coordinate
// param y: Y coordinate
// param font: which font to use
// param align: how to align the text
//---------------------------------------------------------------------------
// FIXME: horizontal alignment is broken for multiline text, it'll set the
// alignment of the *entire* block instead of each line separately. So far
// it's only an issue in the message boxes in the level editor (where
// admittedly it looks fine anyway), but if you plan to modify the game you
// may want to take this into account (or even fix it).
//***************************************************************************

void draw_text(const char *str, int x, int y, FontID font, Align align) {
   // We can't show text in this mode
   if (settings.audiovideo)
      return;

   // Don't bother if nothing is passed...
   if (str == NULL)
      return;

   // Adjust vertical position based on alignment
   switch (align) {
      // Top alignment
      case ALIGN_TOPLEFT:
      case ALIGN_TOP:
      case ALIGN_TOPRIGHT:
         break;

      // Middle alignment
      case ALIGN_LEFT:
      case ALIGN_CENTER:
      case ALIGN_RIGHT:
         y -= calc_text_height(str) >> 1;
         break;

      // Bottom alignment
      case ALIGN_BOTTOMLEFT:
      case ALIGN_BOTTOM:
      case ALIGN_BOTTOMRIGHT:
         y -= calc_text_height(str);
         break;
   }

   // Adjust horizontal position based on alignment
   switch (align) {
      // Left alignment
      case ALIGN_TOPLEFT:
      case ALIGN_LEFT:
      case ALIGN_BOTTOMLEFT:
         break;

      // Center alignment
      case ALIGN_TOP:
      case ALIGN_CENTER:
      case ALIGN_BOTTOM:
         x -= calc_text_len(str) >> 1;
         break;

      // Right alignment
      case ALIGN_TOPRIGHT:
      case ALIGN_RIGHT:
      case ALIGN_BOTTOMRIGHT:
         x -= calc_text_len(str);
         break;
   }

   // Keep track of the original X coordinate
   // We need to restore it for newlines
   int32_t original_x = x;

   // Go through all characters and draw each of them individually
   while (*str) {
      // Get next character
      uint32_t codepoint = get_utf8_char(str);
      str += get_utf8_charlen(str);

      // Newline?
      if (codepoint == '\n') {
         x = original_x;
         y += 16;
         continue;
      }

      // Get character for this codepoint
      // If the character is unrenderable, show it as a space...
      const Character *ch = get_char_data(codepoint);
      if (ch == NULL) {
         x += 4;
         continue;
      }

      // Determine which sprite to use to draw this character
      // If no sprite is available we won't draw this character
      const Sprite *spr;
      switch (font) {
         case FONT_LIT: spr = ch->lit; break;
         case FONT_DIM: spr = ch->dim; break;
         default: spr = NULL; break;
      }
      if (spr == NULL) {
         x += 4;
         continue;
      }

      // Draw character!
      draw_sprite(spr, x, y, SPR_NOFLIP);
      x += ch->width;
   }
}

//***************************************************************************
// draw_text_int
// Like draw_text, but will replace the first instance of "{param}" in the
// text with the number passed as the second argument.
//---------------------------------------------------------------------------
// param str: string to draw (must be UTF-8!)
// param param: integer to use in place of "{param}"
// param x: X coordinate
// param y: Y coordinate
// param font: which font to use
// param align: how to align the text
//***************************************************************************

void draw_text_int(const char *str, int param, int x, int y, FontID font,
Align align) {
   // Don't bother if nothing is passed...
   if (str == NULL)
      return;

   // Check if the string even has a parameter field in it
   // Otherwise just render it as-is
   const char *where = strstr(str, "{param}");
   if (where == NULL) {
      draw_text(str, x, y, font, align);
      return;
   }

   // Turn the integer into a string
   size_t param_len = snprintf(NULL, 0, "%d", param);
   char *param_buf = malloc(param_len + 1);
   if (param_buf == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   sprintf(param_buf, "%d", param);

   // Allocate memory for the string to output
   // The 7 is pretty much strlen("{param}")
   size_t output_len = strlen(str) + param_len - 7;
   char *output_buf = malloc(output_len + 1);
   if (output_buf == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Generate string to draw
   // Place the integer where {param} was
   char *ptr = output_buf;
   memcpy(ptr, str, (size_t)(where - str));
   ptr += (size_t)(where - str);
   strcpy(ptr, param_buf);
   ptr += strlen(param_buf);
   strcpy(ptr, where + 7);

   // Draw generated text
   draw_text(output_buf, x, y, font, align);

   // Get rid of temporary buffers
   free(output_buf);
   free(param_buf);
}

//***************************************************************************
// draw_text_str
// Like draw_text, but will replace the first instance of "{param}" in the
// text with the string passed as the second argument.
//---------------------------------------------------------------------------
// param str: string to draw (must be UTF-8!)
// param param: string to use in place of "{param}"
// param x: X coordinate
// param y: Y coordinate
// param font: which font to use
// param align: how to align the text
//***************************************************************************

void draw_text_str(const char *str, const char *param, int x, int y,
FontID font, Align align) {
   // Don't bother if nothing is passed...
   if (str == NULL)
      return;

   // Check if the string even has a parameter field in it
   // Otherwise just render it as-is
   const char *where = strstr(str, "{param}");
   if (where == NULL) {
      draw_text(str, x, y, font, align);
      return;
   }

   // If the passed string is empty, replace it with an empty string
   if (param == NULL)
      param = "";

   // Allocate memory for the string to output
   // The 7 is pretty much strlen("{param}")
   size_t param_len = strlen(param);
   size_t output_len = strlen(str) + param_len - 7;
   char *output_buf = malloc(output_len + 1);
   if (output_buf == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Generate string to draw
   // Place the string where {param} was
   char *ptr = output_buf;
   memcpy(ptr, str, (size_t)(where - str));
   ptr += (size_t)(where - str);
   strcpy(ptr, param);
   ptr += param_len;
   strcpy(ptr, where + 7);

   // Draw generated text
   draw_text(output_buf, x, y, font, align);

   // Get rid of temporary buffers
   free(output_buf);
}

//***************************************************************************
// calc_text_len
// Calculates the width a string would take up in pixels when drawn.
//---------------------------------------------------------------------------
// param str: pointer to string (must be UTF-8!)
//***************************************************************************

int32_t calc_text_len(const char *str) {
   // If the string is null for whatever reason behave like it was empty
   if (str == NULL)
      return 0;

   // Where we store the maximum line length we found so far
   int32_t max_length = 0;

   // Where we store the length of the current line
   // The length is the width of the string in pixels
   int32_t length = 0;

   // Go through all characters
   while (*str != '\0') {
      // Get next character
      uint32_t codepoint = get_utf8_char(str);
      str += get_utf8_charlen(str);

      // New line?
      if (codepoint == '\n') {
         length = 0;
         continue;
      }

      // Add the width of this character
      const Character *ch = get_char_data(codepoint);
      if (ch == NULL) length += 4; else length += ch->width;

      // Is this line wider than the rest?
      if (max_length < length)
         max_length = length;
   }

   // Done, return the maximum length of the string in pixels
   return max_length;
}

//***************************************************************************
// calc_text_height
// Calculates the height a string would take up in pixels when drawn.
//---------------------------------------------------------------------------
// param str: pointer to string (must be UTF-8!)
//***************************************************************************

int32_t calc_text_height(const char *str) {
   // where the current height is stored
   int32_t height = 16;

   // Calculate height from the amount of lines in the string
   while (*str != '\0') {
      // Get next character
      uint32_t codepoint = get_utf8_char(str);
      str += get_utf8_charlen(str);

      // New line?
      if (codepoint == '\n')
         height += 16;
   }

   // Done, return height of the string in pixels
   return height;
}

//***************************************************************************
// calc_char_pos
// Calculates the position in pixels of a given characters. Passing the
// length of the string is the same as calling calc_text_len. Meant for
// helping drawing the cursor in the level editor.
//---------------------------------------------------------------------------
// param str: pointer to string (must be UTF-8!)
// param pos: position of the character to go for
//---------------------------------------------------------------------------
// TO-DO: this function is extremely similar to calc_text_len (differing only
// in where it stops), maybe find a way so they use the same code?
//***************************************************************************

int32_t calc_char_pos(const char *str, size_t pos) {
   // If the string is null for whatever reason behave like it was empty
   if (str == NULL)
      return 0;

   // Where we store the amount of pixels counted so far
   // Beginning of the string = 0 pixels
   int32_t pixels = 0;

   // Count pixels until we find the position we're looking for
   while (pos-- && *str != '\0') {
      // Get next character
      uint32_t codepoint = get_utf8_char(str);
      str += get_utf8_charlen(str);

      // Add the width of this character
      const Character *ch = get_char_data(codepoint);
      if (ch == NULL) pixels += 4; else pixels += ch->width;
   }

   // Done, return the position of this character in pixels
   return pixels;
}

//***************************************************************************
// get_char_at_pixel
// Figures out which character is at the given horizontal pixel (assumes that
// the string is at x == 0). Meant for helping figure out which character the
// user clicked in the level editor.
//---------------------------------------------------------------------------
// param str: pointer to string (must be UTF-8!)
// param x: X coordinate of character to go for
// return: position of character in the string
//---------------------------------------------------------------------------
// TO-DO: this function always rounds down, make it round to nearest
//***************************************************************************

size_t get_char_at_pixel(const char *str, int32_t x) {
   // Sanity checks
   if (str == NULL)
      return 0;
   if (x <= 0)
      return 0;

   // Current character
   size_t pos = 0;

   // Where we store the amount of pixels counted so far
   // Beginning of the string = 0 pixels
   int32_t pixels = 0;

   // Scan all characters
   while (*str != '\0') {
      // Get next character
      uint32_t codepoint = get_utf8_char(str);
      str += get_utf8_charlen(str);

      // Get width of this character
      const Character *ch = get_char_data(codepoint);
      if (ch == NULL) pixels += 4; else pixels += ch->width;

      // Went too far?
      if (pixels > x)
         return pos;

      // Go for next character
      pos++;
   }

   // Done, return position
   return pos;
}

//***************************************************************************
// get_char_at_pos
// Gets the requested character in the given string.
//---------------------------------------------------------------------------
// param str: pointer to string (must be UTF-8!)
// param pos: position of requested character
// return: pointer to requested character
//***************************************************************************

const char *get_char_at_pos(const char *str, size_t pos) {
   // Er... (any better way to handle this?)
   if (str == NULL)
      return NULL;

   // Skip characters
   while (pos > 0 && *str != 0) {
      str += get_utf8_charlen(str);
      pos--;
   }

   // Return requested characters
   return str;
}

//***************************************************************************
// get_char_data
// Gets the data for the character with the specified codepoint
//---------------------------------------------------------------------------
// param codepoint: Unicode codepoint of character
// return: pointer to sprite, or NULL if we can't do anything...
//***************************************************************************

static const Character *get_char_data(uint32_t codepoint) {
   // Always draw spaces as gaps
   if (codepoint == ' ')
      return NULL;

   // Determine block for this character
   unsigned block = codepoint >> 8;
   if (block >= NUM_BLOCKS || charset[block] == NULL) {
      if (codepoint == '?') return NULL;
      return get_char_data('?');
   }

   // Get data for this character
   const Character *ch = &charset[block][codepoint & 0xFF];
   if (ch->width == UINT_MAX) {
      if (codepoint == '?') return NULL;
      return get_char_data('?');
   }

   // Return data for this character
   return ch;
}

//***************************************************************************
// unload_font
// Unloads the game font.
//***************************************************************************

void unload_font(void) {
#if defined(DEBUG) || defined(DUMB_MEM)
   // Destroy all sprites we created to hold the character graphics
   Sprite *spr = sprite_list;
   while (spr != NULL) {
      Sprite *next = spr->link;
      destroy_sprite(spr);
      spr = next;
   }
   sprite_list = NULL;

   // Deallocate the character blocks
   for (unsigned i = 0; i < NUM_BLOCKS; i++)
   if (charset[i] != NULL) {
      free(charset[i]);
      charset[i] = NULL;
   }
#endif
}

//***************************************************************************
// get_utf8_char
// Gets the codepoint at the beginning of the given string.
//---------------------------------------------------------------------------
// param str: pointer to string
// return: codepoint (replacement character on error)
//***************************************************************************

uint32_t get_utf8_char(const char *str) {
   // Single byte codepoint?
   if (!(str[0] & 0x80))
      return *str;

   // Two bytes codepoint?
   else if ((str[0] & 0xE0) == 0xC0) {
      if ((str[1] & 0xC0) != 0x80) return 0xFFFD;
      uint32_t codepoint = (str[0] & 0x1F) << 6 |
                           (str[1] & 0x3F);
      if (codepoint < 0x80) return 0xFFFD;
      return codepoint;
   }

   // Three bytes codepoint?
   else if ((str[0] & 0xF0) == 0xE0) {
      if ((str[1] & 0xC0) != 0x80) return 0xFFFD;
      if ((str[2] & 0xC0) != 0x80) return 0xFFFD;
      uint32_t codepoint = (str[0] & 0x0F) << 12 |
                           (str[1] & 0x3F) << 6 |
                           (str[2] & 0x3F);
      if (codepoint < 0x800) return 0xFFFD;
      return codepoint;
   }

   // Four bytes codepoint?
   else if ((str[0] & 0xF8) == 0xF0) {
      if ((str[1] & 0xC0) != 0x80) return 0xFFFD;
      if ((str[2] & 0xC0) != 0x80) return 0xFFFD;
      if ((str[3] & 0xC0) != 0x80) return 0xFFFD;
      uint32_t codepoint = (str[0] & 0x07) << 18 |
                           (str[1] & 0x3F) << 12 |
                           (str[2] & 0x3F) << 6 |
                           (str[3] & 0x3F);
      if (codepoint < 0x10000) return 0xFFFD;
      return codepoint;
   }

   // Error while parsing the codepoint, return replacement character as
   // a fallback
   else
      return 0xFFFD;
}

//***************************************************************************
// get_utf8_charlen
// Calculates how many bytes the codepoint at the beginning of the given
// string takes up. Yes, this takes into account malformed byte sequences.
//---------------------------------------------------------------------------
// param str: pointer to string
// return: size in bytes
//***************************************************************************

unsigned get_utf8_charlen(const char *str) {
   // Single character?
   if (!(str[0] & 0x80))
      return 1;

   // To count how many bytes does this character take up
   // Yes, this may seem slow... blame UTF-8 for that >_>
   unsigned length = 0;

   // Skip initial byte of a multibyte codepoint
   // Yes, we're accounting for malformed byte sequences too...
   if ((*str & 0xC0) != 0x80) {
      length++;
      str++;
   }

   // Skip all trailing bytes
   while ((*str & 0xC0) == 0x80) {
      length++;
      str++;
   }

   // Return how many bytes does this character take up
   return length;
}

//***************************************************************************
// get_utf8_strlen
// Calculates how many characters are in an UTF-8 string.
//---------------------------------------------------------------------------
// param str: pointer to string
// return: size in characters
//***************************************************************************

size_t get_utf8_strlen(const char *str) {
   // Where we'll store the length
   size_t len = 0;

   // Count how many characters are in the string
   // Yup, this is more trivial than it sounds XD
   while (*str != '\0') {
      str += get_utf8_charlen(str);
      len++;
   }

   // Return length
   return len;
}

#ifdef _WIN32
//***************************************************************************
// utf16_to_utf8
// Converts an UTF-16 string to UTF-8 (I'll let you figure out why do we
// even need this in the first place). Make sure to call free() to release
// the returned string when done with it.
//---------------------------------------------------------------------------
// param wstr: UTF-16 string
// return: UTF-8 string
//***************************************************************************

char *utf16_to_utf8(const wchar_t *wstr) {
   // Used for pointers later on
   // Because we can't declare both within a for loop...
   const wchar_t *wptr;
   char *ptr;

   // Determine how much do we need to allocate for the UTF-8 string
   // Initialized at 1 to account for the nul character
   size_t size = 1;
   for (wptr = wstr; *wptr != '\0';) {
      if (*wptr < 0x0080) {
         size++;
         wptr++;
      } else if (*wptr < 0x0800) {
         size += 2;
         wptr++;
      } else if (wptr[0] >= 0xD800 && wptr[0] <= 0xDBFF &&
      wptr[1] >= 0xDC00 && wptr[1] <= 0xDFFF) {
         size += 4;
         wptr += 2;
      } else {
         size += 3;
         wptr++;
      }
   }

   // Allocate string
   char *str = (char *) malloc(size);
   if (str == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Now convert string to UTF-8
   for (wptr = wstr, ptr = str; *wptr != '\0';) {
      // Decode codepoint
      uint32_t codepoint;
      if (wptr[0] >= 0xD800 && wptr[0] <= 0xDBFF &&
      wptr[1] >= 0xDC00 && wptr[1] <= 0xDFFF) {
         codepoint = 0x10000 +
            ((wptr[0] & 0x03FF) << 10 | (wptr[1] & 0x03FF));
         wptr += 2;
      } else if (wptr[0] >= 0xDB00 && wptr[0] <= 0xDFFF) {
         codepoint = 0xFFFD;
         wptr++;
      } else {
         codepoint = wptr[0];
         wptr++;
      }

      // Now reencode codepoint
      if (codepoint < 0x0080) {
         ptr[0] = codepoint;
         ptr++;
      } else if (codepoint < 0x0800) {
         ptr[0] = 0xC0 | (codepoint >> 6 & 0x1F);
         ptr[1] = 0x80 | (codepoint & 0x3F);
         ptr += 2;
      } else if (codepoint < 0x10000) {
         ptr[0] = 0xE0 | (codepoint >> 12 & 0x0F);
         ptr[1] = 0x80 | (codepoint >> 6 & 0x3F);
         ptr[2] = 0x80 | (codepoint & 0x3F);
         ptr += 3;
      } else {
         ptr[0] = 0xF0 | (codepoint >> 18 & 0x07);
         ptr[1] = 0x80 | (codepoint >> 12 & 0x3F);
         ptr[2] = 0x80 | (codepoint >> 6 & 0x3F);
         ptr[3] = 0x80 | (codepoint & 0x3F);
         ptr += 4;
      }
   }

   // Done
   *ptr = '\0';
   return str;
}

//***************************************************************************
// utf8_to_utf16
// Converts an UTF-8 string to UTF-16 (I'll let you figure out why do we
// even need this in the first place). Make sure to call free() to release
// the returned string when done with it.
//---------------------------------------------------------------------------
// param str: UTF-8 string
// return: UTF-16 string
//***************************************************************************

wchar_t *utf8_to_utf16(const char *str) {
   // Allocate memory for the UTF-16 string
   // We do it the dumb way and just allocate enough to fit every possible
   // codepoint (non-BMP characters take up two words), but I guess that in
   // practice that's a waste...
   size_t len = get_utf8_strlen(str);
   wchar_t *wstr = (wchar_t *) malloc(sizeof(wchar_t) * (len*2 + 1));
   if (wstr == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Reencode UTF-8 into UTF-16
   wchar_t *ptr = wstr;
   while (*str != '\0') {
      // Get next codepoint
      uint32_t codepoint = get_utf8_char(str);
      str += get_utf8_charlen(str);

      // Inside the BMP? (encode as-is)
      if (codepoint < 0x10000) {
         *ptr++ = codepoint;
      }

      // Outside the BMP? (encode as surrogate points)
      else {
         codepoint -= 0x10000;
         *ptr++ = 0xD800 | (codepoint >> 10);
         *ptr++ = 0xDC00 | (codepoint & 0x3FF);
      }
   }

   // Done
   *ptr = 0;
   return wstr;
}
#endif
