//***************************************************************************
// "text.h"
// Font loading and text rendering
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

#ifndef TEXT_H
#define TEXT_H

// Required headers
#include <stddef.h>
#include <stdint.h>
#include "main.h"
#include "level.h"
#include "sound.h"

// Where all the localizable text is stored
// Whenever you want to show something to the user, you should use the
// localization table and load text from the files in the language directory
// instead of passing strings directly! This way it'll be shown in whatever
// language the user chooses.
typedef struct {
   // Launcher text
   struct {
      // Tabs
      const char *play;             // Play!
      const char *settings;         // Settings
      const char *accessibility;    // Accessibility
      const char *modding;          // Modding

      // Play! tab
      const char *play_sol;         // Play Sol!
      const char *play_minisol;     // Play Minisol!
      const char *manual;           // Read manual
      const char *saves;            // Saves folder
      const char *quit;             // Quit

      // Subsystem control
      const char *fullscreen;       // Fullscreen
      const char *software;         // No hardware acceleration
      const char *no_sound;         // Disable sound
      const char *no_joystick;      // Disable joystick

      // Screen reader
      struct {
         const char *title;         // Frame title
         const char *disabled;      // Disabled
         const char *native;        // Native
         const char *clipboard;     // Clipboard
         const char *titlebar;      // Titlebar
      } reader;

      // Audiovideo mode
      const char *audiovideo;       // Audiovideo mode

      // One-switch mode
      struct {
         const char *title;         // Frame title
         const char *enable;        // Enable one-switch mode
         const char *tap;           // Duration of a long press
         const char *menu;          // Delay between menu options
         const char *fivetap;       // Maximum delay for pause gesture
         const char *comment;       // 60 = 1 second
      } onesw;

      // Modding
      struct {
         const char *list;          // Mod list
         const char *add;           // Add mod
         const char *rem;           // Remove mod
         const char *args;          // Command line arguments
      } mod;
   } launcher;

   // In-game menu
   struct {
      const char *demo_mode;     // "★ Demo mode ★"
      const char *pause;         // Pause button
      const char *pause_title;   // Pause menu title
      const char *opt_continue;  // "Continue" option
      const char *opt_restart;   // "Restart" option
      const char *opt_quit;      // "Quit" option
   } ingame;

   // Score tally
   struct {
      const char *title;         // "Level complete!"
      const char *enemy;         // "Enemy bonus"
      const char *items;         // "Items bonus"
      const char *special;       // "Special bonus"
      const char *total;         // "Total"
      const char *time;          // "Time"
   } tally;

   // Logo screen strings
   struct {
      const char *name;          // Company name
   } logo;

   // Title screen strings
   struct {
      const char *copyright;     // Copyright notice

      const char *start;         // Button: "Start game"
      const char *editor;        // Button: "Level editor"
      const char *options;       // Button: "Options"
      const char *quit;          // Button: "Quit"
   } title;

   // Save slot screen strings
   struct {
      const char *title;         // Menu title
      const char *cancel;        // "Cancel" button
      const char *level;         // "Level" label
      const char *bonus;         // "Bonus level" label
      const char *cutscene;      // "Cutscene" label
   } saveselect;

   // Options menu strings
   struct {
      const char *title;         // Menu title

      const char *gameplay;      // Button: "Gameplay settings"
      const char *video;         // Button: "Video settings"
      const char *sound;         // Button: "Sound settings"
      const char *keyboard;      // Button: "Keyboard controls"
      const char *joystick;      // Button: "Joystick controls"
      const char *mouse;         // Button: "Mouse controls"
      const char *back;          // Button: "Back"
   } options;

   // Options gameplay menu strings
   struct {
      const char *difficulty;    // Setting: difficulty
      const char *game_speed;    // Setting: game speed
      const char *one_switch;    // Setting: one-switch
      const char *language;      // Setting: language

      // Difficulty names
      const char *diff_names[3];

      // One-switch statuses
      const char *enabled;       // Enabled
      const char *disabled;      // Disabled

      // One-switch warning screen
      const char *warning;       // "Warning!"
      const char *onesw_on[8];   // Text when turning on
      const char *onesw_off[8];  // Text when turning off
      const char *ok;            // OK button
      const char *cancel;        // Cancel button
   } options_gameplay;

   // Options video menu strings
   struct {
      // Page 1
      const char *ratio;         // %d:%d ratio
      const char *smaller;       // Smaller resolutions
      const char *larger;        // Larger resolutions
      const char *fullscreen;    // Fullscreen button
      const char *vsync;         // VSync button
      const char *filtering;     // Enable filtering button

      // Page 2
      const char *contrast;      // Contrast buttons
      const char *background;    // Enable background button
      const char *dim_bg;        // Dim background button
      const char *zoom;          // Zoom screen button
      const char *shaking;       // Enable shaking button

      // Page toggles
      const char *next;          // Next page
      const char *prev;          // Previous page

      // Setting values
      const char *enabled;       // Enabled
      const char *disabled;      // Disabled
      const char *low;           // Low
      const char *medium;        // Medium
      const char *high;          // High
      const char *show;          // Show
      const char *hide;          // Hide
      const char *dim;           // Dim
      const char *no_dim;        // Don't dim
      const char *zoomed;        // Zoom in
      const char *no_zoom;       // Don't zoom
   } options_video;

   // Options sound menu strings
   struct {
      const char *play_sound;    // Sound test: play
      const char *prev_sound;    // Sound test: previous
      const char *next_sound;    // Sound test: next
      const char *bgm_volume;    // Setting: BGM volume
      const char *sfx_volume;    // Setting: SFX volume
      const char *stereo;        // Setting: stereo
      const char *reverse;       // Setting: reverse stereo

      // Setting values
      const char *enabled;       // Enabled
      const char *disabled;      // Disabled
      const char *normal;        // Normal
      const char *reversed;      // Reversed

      // BGM names
      const char *bgm_names[NUM_BGM+1];
   } options_sound;

   // Options keyboard and joystick menu strings
   struct {
      const char *left;          // Action: "Run left"
      const char *right;         // Action: "Run right"
      const char *down;          // Action: "Crouch"
      const char *up;            // Action: "Look up"
      const char *jump;          // Action: "Jump"
      const char *pause;         // Action: "Pause"

      const char *button;        // Joystick: "Button %d"
      const char *axis_pos;      // Joystick: "Axis %d +"
      const char *axis_neg;      // Joystick: "Axis %d -"
      const char *axis_x_pos;    // Joystick: "Axis X ←"
      const char *axis_x_neg;    // Joystick: "Axis X →"
      const char *axis_y_pos;    // Joystick: "Axis Y ↑"
      const char *axis_y_neg;    // Joystick: "Axis Y ↓"
      const char *hat_up;        // Joystick: "Hat %d ↑"
      const char *hat_down;      // Joystick: "Hat %d ↓"
      const char *hat_left;      // Joystick: "Hat %d ←"
      const char *hat_right;     // Joystick: "Hat %d →"
      const char *dpad_up;       // Joystick: "D-pad ↑"
      const char *dpad_down;     // Joystick: "D-pad ↓"
      const char *dpad_left;     // Joystick: "D-pad ←"
      const char *dpad_right;    // Joystick: "D-pad →"

      const char *press_key;     // Reader: "press a key"
      const char *press_joy;     // Reader: "press button or axis"
   } options_input;

   // Options mouse menu strings
   struct {
      const char *enable;        // Enable mouse mode
      const char *yes;           // Enable: "yes"
      const char *no;            // Enable: "no"
      const char *grid;          // Show grid
      const char *show;          // Grid: "show"
      const char *hide;          // Grid: "hide"
      const char *test;          // Test mode
      const char *inc_x;         // Increment X distance
      const char *dec_x;         // Decrement X distance
      const char *inc_y;         // Increment Y distance
      const char *dec_y;         // Decrement Y distance
   } options_mouse;

   // Editor strings
   struct {
      const char *b_new;         // Button: "New"
      const char *b_info;        // Button: "Info"
      const char *b_load;        // Button: "Load"
      const char *b_save;        // Button: "Save"
      const char *b_play;        // Button: "Play"
      const char *b_tilemap;     // Button: "Tilemap"
      const char *b_objects;     // Button: "Objects"
      const char *b_select;      // Button: "Select"
      const char *b_undo;        // Button: "Undo"
      const char *b_redo;        // Button: "Redo"
      const char *b_help;        // Button: "Help"
      const char *b_quit;        // Button: "Quit"
   } editor;

   // Level info dialog strings
   struct {
      const char *level_theme;   // Label: "level theme"
      const char *themes[NUM_THEMES]; // Level theme names

      const char *map_size;      // Label: "map size"
      const char *top;           // Top border
      const char *bottom;        // Bottom border
      const char *left;          // Left border
      const char *right;         // Right border
      const char *current_size;  // Label: "current size"
      const char *new_size;      // Label: "new size"

      const char *b_apply;       // Button: "Apply"
      const char *b_cancel;      // Button: "Cancel"
   } level_info;

   // File selector strings
   struct {
      const char *load_title;    // Load dialog title
      const char *save_title;    // Save dialog title
      const char *b_load;        // Button: "Load"
      const char *b_save;        // Button: "Save"
      const char *b_cancel;      // Button: "Cancel"
      const char *home;          // "Home" directory
      const char *root;          // "Root" directory
   } file_select;

   // Tile select strings
   struct {
      // Titles for each screen
      const char *title_coll;
      const char *title_obj;

      // Names for the tile types
      const char *coll_names[NUM_TILETYPES];
      const char *obj_names[NUM_LVOBJIDS];

      // Names for the object groups
      const char *items;
      const char *scenery;
      const char *enemies;
      const char *hazards;
      const char *switches;
   } tile_select;

   // Editor help strings
   struct {
      const char *title;            // Editor help
      const char *page;             // Page #

      const char *scroll[2];        // Scroll
      const char *scroll_fast[2];   // Scroll faster
      const char *draw_tiles[2];    // Draw tiles
      const char *erase_tiles[2];   // Erase tiles
      const char *copy_tile[2];     // Copy tile
      const char *start_point[2];   // Set start point
      const char *flip_object[2];   // Flip object
      const char *select_tile[2];   // Select tile (wheel)
      const char *select_tile_2[2]; // Select tile (ctrl)

      const char *new_level[2];     // New level
      const char *load_level[2];    // Load level
      const char *save_level[2];    // Save level
      const char *edit_info[2];     // Theme and size
      const char *play_level[2];    // Play level
      const char *tilemap_mode[2];  // Tilemap mode
      const char *objects_mode[2];  // Objects mode
      const char *show_help[2];     // Show help
      const char *quit_editor[2];   // Quit editor
   } editor_help;

   // Editor error messages
   struct {
      const char *load_level;       // Error: could not load level
      const char *save_level;       // Error: could not save level
      const char *demo_error;       // Error: level needs full version
      const char *press_key;        // Press or click to continue
   } editor_error;

   // Screen reader-only strings
   struct {
      const char *up;               // Replaces up arrow
      const char *down;             // Replaces down arrow
      const char *left;             // Replaces left arrow
      const char *right;            // Replaces right arrow

      const char *lclick;           // Replaces left mouse button
      const char *mclick;           // Replaces middle mouse button
      const char *rclick;           // Replaces right mouse button
   } reader;

   // Error codes
   const char *error_title;
   const char *error[NUM_ERRCODES];
} Text;
extern Text text;

// Available fonts
typedef enum {
   FONT_LIT,         // Lit font
   FONT_DIM,         // Dim font
   NUM_FONTS         // Number of available fonts
} FontID;

// Possible alignments when drawing text
typedef enum {
   ALIGN_TOPLEFT,       // Horizontal: left   | Vertical: top
   ALIGN_TOP,           // Horizontal: center | Vertical: top
   ALIGN_TOPRIGHT,      // Horizontal: right  | Vertical: top
   ALIGN_LEFT,          // Horizontal: left   | Vertical: middle
   ALIGN_CENTER,        // Horizontal: center | Vertical: middle
   ALIGN_RIGHT,         // Horizontal: right  | Vertical: middle
   ALIGN_BOTTOMLEFT,    // Horizontal: left   | Vertical: bottom
   ALIGN_BOTTOM,        // Horizontal: center | Vertical: bottom
   ALIGN_BOTTOMRIGHT    // Horizontal: right  | Vertical: bottom
} Align;

// Function prototypes
void set_default_err_msg(void);
void init_languages(void);
void deinit_languages(void);
void load_language(void);
void unload_language(void);
size_t get_num_languages(void);
const char *get_language_id(void);
const char *get_language_longid(size_t);
const char *get_language_name(size_t);
void load_font(void);
void draw_text(const char *, int, int, FontID, Align);
void draw_text_int(const char *, int, int, int, FontID, Align);
void draw_text_str(const char *, const char *, int, int, FontID, Align);
int32_t calc_text_len(const char *);
int32_t calc_text_height(const char *);
int32_t calc_char_pos(const char *, size_t);
size_t get_char_at_pixel(const char *, int32_t);
const char *get_char_at_pos(const char *, size_t);
void unload_font(void);
uint32_t get_utf8_char(const char *);
unsigned get_utf8_charlen(const char *);
size_t get_utf8_strlen(const char *);

// Functions needed only for Windows
#ifdef _WIN32
char *utf16_to_utf8(const wchar_t *);
wchar_t *utf8_to_utf16(const char *);
#endif

#endif
