//***************************************************************************
// "title.c"
// Code for the title screen.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include "main.h"
#include "background.h"
#include "input.h"
#include "loading.h"
#include "menu.h"
#include "reader.h"
#include "replay.h"
#include "savegame.h"
#include "scene.h"
#include "settings.h"
#include "sound.h"
#include "tables.h"
#include "text.h"
#include "video.h"

// Values for controlling Sol's animation
#define SOL_X (screen_cx)     // X coordinate when she stops moving
#define SOL_Y (0x10)          // Y coordinate when she stops moving
#define SOL_XSPEED 6          // How fast she moves in the X axis
#define MAX_SOL_ANIM 0x20     // How long does the animation last
static unsigned sol_anim;

// Values for controlling the title animation
#define TITLE_X (screen_cx-0x80)    // X coordinate of first letter
#define TITLE_Y (0x20+44)           // Y coordinate of title
#define TITLE_WIDTH 0x20            // How wide is a letter
#define TITLE_NUM_LETTERS 3         // Number of letters
#define TITLE_SPEED 0x0C            // How fast letters move initially
#define TITLE_START (TITLE_SPEED*0x40) // Where are letters placed initially
#define TITLE_DIST (TITLE_SPEED*2)  // Distance between letters initially
static int title_pos[TITLE_NUM_LETTERS];
static int title_offset[TITLE_NUM_LETTERS];
static int title_speed[TITLE_NUM_LETTERS];

// Values for the menu positions
#define MENU_X1 (screen_cx-0x78)    // X coordinate of first button
#define MENU_X2 (screen_cx-0x38)    // X coordinate of second button
#define MENU_X3 (screen_cx+0x08)    // X coordinate of third button
#define MENU_X4 (screen_cx+0x48)    // X coordinate of fourth button
#define MENU_Y (screen_h-0x38)      // Y coordinate of buttons
#define MENU_YTEXT (MENU_Y-8)       // Y coordinate of caption
#define MENU_BWIDTH 0x30            // Width of a button
#define MENU_BHEIGHT 0x20           // Height of a button

// IDs for each option in the menu
enum {
   OPT_START,        // Start game
   OPT_EDITOR,       // Level editor
   OPT_OPTIONS,      // Options
   OPT_QUIT,         // Quit
   NUM_OPT           // Number of options
};

// Where graphics are stored
static GraphicsSet *gfxset_title = NULL;
static const Sprite *spr_sol;
static const Sprite *spr_title[TITLE_NUM_LETTERS];
static const Sprite *spr_menu[NUM_OPT][4];

// Used to animate the buttons
static unsigned menu_anim;

// Buffer to enter the cheat codes
enum {
   CHEAT_X,    // Nothing
   CHEAT_U,    // Up button
   CHEAT_D,    // Down button
   CHEAT_L,    // Left button
   CHEAT_R     // Right button
};
#define CHEAT_BUFSIZE 8
static uint8_t cheat_buffer[CHEAT_BUFSIZE];

// List of cheats
#define CHEAT(name, a, b, c, d, e, f, g, h) \
   static const uint8_t cheat_ ## name [CHEAT_BUFSIZE] = { \
      CHEAT_ ## a, CHEAT_ ## b, CHEAT_ ## c, CHEAT_ ## d, \
      CHEAT_ ## e, CHEAT_ ## f, CHEAT_ ## g, CHEAT_ ## h \
   };

CHEAT(levelselect,      U,U,D,D,L,R,L,R);    // Enable level select
CHEAT(debugmode,        D,D,U,U,R,L,R,L);    // Enable debug mode
CHEAT(nodamage,         U,R,D,R,U,R,D,R);    // No damage
CHEAT(nohealth,         U,L,D,R,U,L,D,R);    // No health (1 hit = death)
CHEAT(extrahealth,      D,L,U,R,D,L,U,R);    // Extra health (5 hearts max)
CHEAT(noenemies,        R,D,R,U,L,D,L,U);    // No enemies
CHEAT(nocheckpoints,    L,U,L,D,R,U,R,D);    // No checkpoints
CHEAT(nofriction,       L,R,L,R,U,D,U,D);    // No friction
CHEAT(forcevirtual,     R,R,D,D,R,R,U,U);    // Force virtual mode
CHEAT(pinball,          U,U,D,D,U,D,U,D);    // Pinball mode

#undef CHEAT

// Timer used to trigger demo mode
#define DEMO_DELAY (1800)        // Delay without attract mode (30 sec)
#define ATTRACT_DELAY (450)      // Delay with attract mode (7.5 sec)
static unsigned demo_timer;
static unsigned demo_timer_max;

// Private function prototypes
static void init_title_menu(void);
static void load_title(void);
static void button_start(void);
static void button_editor(void);
static void button_options(void);
static void button_quit(void);

//***************************************************************************
// init_title
// Initializes the title screen.
//***************************************************************************

void init_title(void) {
   // Load graphics
   loading_screen(load_title);
#define SPR(x) get_sprite(gfxset_title, x)
   spr_sol = SPR("sol");

   // To-do: do this in a less hardcoded way
   if (strcmp(get_language_id(), "ja") == 0) {
      spr_title[0] = NULL;
      spr_title[1] = SPR("title_so");
      spr_title[2] = SPR("title_ru");
   } else {
      spr_title[0] = SPR("title_s");
      spr_title[1] = SPR("title_o");
      spr_title[2] = SPR("title_l");
   }

   spr_menu[OPT_START][0] = SPR("button_start_dim");
   spr_menu[OPT_START][1] = SPR("button_start_lit_1");
   spr_menu[OPT_START][2] = SPR("button_start_lit_2");
   spr_menu[OPT_START][3] = SPR("button_start_lit_3");
   spr_menu[OPT_EDITOR][0] = SPR("button_editor_dim");
   spr_menu[OPT_EDITOR][1] = SPR("button_editor_lit_1");
   spr_menu[OPT_EDITOR][2] = SPR("button_editor_lit_2");
   spr_menu[OPT_EDITOR][3] = SPR("button_editor_lit_3");
   spr_menu[OPT_OPTIONS][0] = SPR("button_options_dim");
   spr_menu[OPT_OPTIONS][1] = SPR("button_options_lit_1");
   spr_menu[OPT_OPTIONS][2] = SPR("button_options_lit_2");
   spr_menu[OPT_OPTIONS][3] = SPR("button_options_lit_3");
   spr_menu[OPT_QUIT][0] = SPR("button_quit_dim");
   spr_menu[OPT_QUIT][1] = SPR("button_quit_lit_1");
   spr_menu[OPT_QUIT][2] = SPR("button_quit_lit_2");
   spr_menu[OPT_QUIT][3] = SPR("button_quit_lit_3");
#undef SPR

   // Initialize animations
   sol_anim = 0;
   for (unsigned i = 0; i < TITLE_NUM_LETTERS; i++) {
      title_offset[i] = i * TITLE_WIDTH;
      title_pos[i] = TITLE_START + i * TITLE_SPEED * 2;
      title_speed[i] = -TITLE_SPEED;
   }

   // No menu in attract mode
   if (!settings.attract) {
      // Initialize menu data
      init_title_menu();
   }

   // Empty cheat buffer
   memset(cheat_buffer, CHEAT_X, CHEAT_BUFSIZE);

   // Reset demo mode timer
   demo_timer = 0;
   demo_timer_max = settings.attract ? ATTRACT_DELAY : DEMO_DELAY;

   // Make the cursor visible
   set_cursor(settings.attract ? CURSOR_NONE : CURSOR_ARROW);

   // Start playing music
   if (!settings.attract)
      play_bgm(BGM_TITLE);

   // Make screen visible
   fade_on();
}

//***************************************************************************
// init_title_menu [internal]
// Sets up the title screen menu.
//***************************************************************************

static void init_title_menu(void) {
   menu_anim = 0;
   init_menu();
   set_reinit_menu_func(init_title_menu);

   menu.cancel = button_quit;
   menu.defoption.right = OPT_START;
   menu.defoption.left = OPT_QUIT;

   // "Start game" button
   menu.options[OPT_START].box.x1 = MENU_X1;
   menu.options[OPT_START].box.x2 = MENU_X1 + MENU_BWIDTH-1;
   menu.options[OPT_START].box.y1 = MENU_Y;
   menu.options[OPT_START].box.y2 = MENU_Y + MENU_BHEIGHT-1;
   menu.options[OPT_START].move.left = OPT_QUIT;
   menu.options[OPT_START].move.right = OPT_EDITOR;
   menu.options[OPT_START].move.oneswitch = OPT_EDITOR;
   menu.options[OPT_START].action.accept = button_start;
   menu.options[OPT_START].caption = text.title.start;

   // "Level editor" button
   // To-do: make level editor usable in audiovideo mode
   // Enabling it in debug mode *just in case*
   if (!settings.audiovideo || settings.debug) {
      menu.options[OPT_EDITOR].box.x1 = MENU_X2;
      menu.options[OPT_EDITOR].box.x2 = MENU_X2 + MENU_BWIDTH-1;
      menu.options[OPT_EDITOR].box.y1 = MENU_Y;
      menu.options[OPT_EDITOR].box.y2 = MENU_Y + MENU_BHEIGHT-1;
      menu.options[OPT_EDITOR].move.left = OPT_START;
      menu.options[OPT_EDITOR].move.right = OPT_OPTIONS;
      menu.options[OPT_EDITOR].move.oneswitch = OPT_OPTIONS;
      menu.options[OPT_EDITOR].action.accept = button_editor;
      menu.options[OPT_EDITOR].caption = text.title.editor;
   }

   // "Options" button
   menu.options[OPT_OPTIONS].box.x1 = MENU_X3;
   menu.options[OPT_OPTIONS].box.x2 = MENU_X3 + MENU_BWIDTH-1;
   menu.options[OPT_OPTIONS].box.y1 = MENU_Y;
   menu.options[OPT_OPTIONS].box.y2 = MENU_Y + MENU_BHEIGHT-1;
   menu.options[OPT_OPTIONS].move.left = OPT_EDITOR;
   menu.options[OPT_OPTIONS].move.right = OPT_QUIT;
   menu.options[OPT_OPTIONS].move.oneswitch = OPT_QUIT;
   menu.options[OPT_OPTIONS].action.accept = button_options;
   menu.options[OPT_OPTIONS].caption = text.title.options;

   // "Quit" button
   menu.options[OPT_QUIT].box.x1 = MENU_X4;
   menu.options[OPT_QUIT].box.x2 = MENU_X4 + MENU_BWIDTH-1;
   menu.options[OPT_QUIT].box.y1 = MENU_Y;
   menu.options[OPT_QUIT].box.y2 = MENU_Y + MENU_BHEIGHT-1;
   menu.options[OPT_QUIT].move.left = OPT_OPTIONS;
   menu.options[OPT_QUIT].move.right = OPT_START;
   menu.options[OPT_QUIT].move.oneswitch = OPT_START;
   menu.options[OPT_QUIT].action.accept = button_quit;
   menu.options[OPT_QUIT].caption = text.title.quit;
   menu.options[OPT_QUIT].sfx = SFX_CANCEL;
}

//***************************************************************************
// load_title [internal]
// Loads the assets for the title screen. Run during the loading screen.
//***************************************************************************

static void load_title(void) {
   gfxset_title = load_graphics_set("graphics/title");
   set_loading_total(1, 4);
   load_background("graphics/title/background", gfxset_title);
   set_loading_total(2, 4);
   load_bgm(BGM_TITLE);
   set_loading_total(3, 4);
   load_replay();
   set_loading_total(4, 4);
}

//***************************************************************************
// run_title
// Processes the logic for the title screen.
//***************************************************************************

void run_title(void) {
   // No user input while in attract mode!
   if (!settings.attract) {
      // Update the menu
      update_menu();

      // Check if we got cheat input
      uint8_t cheat_input = CHEAT_X;
      if (input.menu.up) cheat_input = CHEAT_U;
      if (input.menu.down) cheat_input = CHEAT_D;
      if (input.menu.left) cheat_input = CHEAT_L;
      if (input.menu.right) cheat_input = CHEAT_R;

      // New input?
      if (cheat_input != CHEAT_X) {
         // Add input to buffer
         for (unsigned i = 0; i < CHEAT_BUFSIZE-1; i++)
            cheat_buffer[i] = cheat_buffer[i+1];
         cheat_buffer[CHEAT_BUFSIZE-1] = cheat_input;

         // To know if a cheat has been successfully entered
         unsigned cheated = 0;

         // Check for different cheats
         if (!memcmp(cheat_buffer, cheat_levelselect, CHEAT_BUFSIZE)) {
            settings.level_select = 1;
            load_savegame();
            cheated = 1;
         } else if (!memcmp(cheat_buffer, cheat_debugmode, CHEAT_BUFSIZE)) {
            settings.debug = 1;
            cheated = 1;
         } else if (!memcmp(cheat_buffer, cheat_nodamage, CHEAT_BUFSIZE)) {
            settings.no_damage = 1;
            cheated = 1;
         } else if (!memcmp(cheat_buffer, cheat_nohealth, CHEAT_BUFSIZE)) {
            settings.no_health = 1;
            cheated = 1;
         } else if (!memcmp(cheat_buffer, cheat_extrahealth, CHEAT_BUFSIZE)) {
            settings.extra_health = 1;
            cheated = 1;
         } else if (!memcmp(cheat_buffer, cheat_noenemies, CHEAT_BUFSIZE)) {
            settings.no_enemies = 1;
            cheated = 1;
         } else if (!memcmp(cheat_buffer, cheat_nocheckpoints, CHEAT_BUFSIZE)) {
            settings.no_checkpoints = 1;
            cheated = 1;
         } else if (!memcmp(cheat_buffer, cheat_nofriction, CHEAT_BUFSIZE)) {
            settings.no_friction = 1;
            cheated = 1;
         } else if (!memcmp(cheat_buffer, cheat_forcevirtual, CHEAT_BUFSIZE)) {
            settings.force_virtual = 1;
            cheated = 1;
         } else if (!memcmp(cheat_buffer, cheat_pinball, CHEAT_BUFSIZE)) {
            settings.pinball = 1;
            cheated = 1;
         }

         // Entered a cheat? Clear the buffer then (so new cheats can be
         // entered from the start) Also play a sound so the player knows it
         // worked :)
         if (cheated) {
            memset(cheat_buffer, CHEAT_X, CHEAT_BUFSIZE);
            play_sfx(SFX_CHEAT);
         }
      }
   }

   // Animate background
   update_background();

   // Animate foreground, but only when the screen is fully visible already
   if (!is_fading()) {
      // Animate Sol jumping
      if (sol_anim < MAX_SOL_ANIM)
         sol_anim++;

      // Animate title entering
      for (unsigned i = 0; i < TITLE_NUM_LETTERS; i++) {
         // Done moving?
         if (title_pos[i] >= 0 && title_speed[i] > 0)
            continue;

         // Went past our position? (brake and try to correct our speed so we
         // end up in the correct position)
         if (title_pos[i] < 0)
            title_speed[i] += 2;

         // Move letter!
         title_pos[i] += title_speed[i];
      }
   }

   // Animate menu
   menu_anim++;

   // Go into demo mode if inactive for long enough
   demo_timer++;
   if (!input.idle && !settings.attract)
      demo_timer = 0;
   if (demo_timer == demo_timer_max)
      start_replay_playback();

   // Debug: immediately go into demo mode when pressing F1
   if (input.debug.wings)
      start_replay_playback();
}

//***************************************************************************
// draw_title
// Draws the title screen.
//***************************************************************************

void draw_title(void) {
   // Draw background
   draw_background();

   // Draw jumping Sol
   draw_sprite(spr_sol,
      SOL_X + (MAX_SOL_ANIM - sol_anim) * SOL_XSPEED,
      SOL_Y + 0x80 - (sines[sol_anim << 1] >> 1),
      SPR_NOFLIP);

   // Draw title
   for (unsigned i = 0; i < TITLE_NUM_LETTERS; i++) {
      if (spr_title[i] == NULL)
         continue;
      draw_sprite(spr_title[i],
         TITLE_X + title_pos[i] + title_offset[i],
         TITLE_Y - spr_title[i]->height, SPR_NOFLIP);
   }

   // Don't draw menu in attract mode
   if (!settings.attract) {
      // Draw menu buttons
      for (unsigned i = 0; i < NUM_OPT; i++) {
         // Determine sprite to use
         const Sprite *sprite;
         if (menu.selected != (signed) i)
            sprite = spr_menu[i][0];
         else switch (menu_anim >> 3 & 0x03) {
            case 0x00: sprite = spr_menu[i][1]; break;
            case 0x01: sprite = spr_menu[i][2]; break;
            case 0x02: sprite = spr_menu[i][3]; break;
            case 0x03: sprite = spr_menu[i][2]; break;
         }

         // Draw sprite
         draw_sprite(sprite,
            menu.options[i].box.x1, menu.options[i].box.y1, SPR_NOFLIP);
      }

      // If an option is selected, show a border around it as well as its
      // caption above all the buttons
      if (menu.selected != -1) {
         // Where the data for the selected option is stored
         MenuOption *ptr = &menu.options[menu.selected];

         // Draw the border on top of the button
         if (menu_anim & 0x08) {
            draw_rectangle(ptr->box.x1+1, ptr->box.y1+1,
               ptr->box.x2-1, ptr->box.y2-1, settings.box_lit[1]);
            draw_rectangle(ptr->box.x1, ptr->box.y1,
               ptr->box.x2, ptr->box.y2, settings.box_lit[0]);
            draw_rectangle(ptr->box.x1-1, ptr->box.y1-1,
               ptr->box.x2+1, ptr->box.y2+1, settings.box_lit[0]);
            draw_rectangle(ptr->box.x1-2, ptr->box.y1-2,
               ptr->box.x2+2, ptr->box.y2+2, settings.box_lit[1]);
            draw_hline(ptr->box.x1-2, ptr->box.y2+3,
               ptr->box.x2+2, settings.box_lit[2]);
         }

         // Show the caption
         draw_text(ptr->caption, screen_cx, MENU_YTEXT,
            FONT_LIT, ALIGN_BOTTOM);
         set_reader_text(ptr->caption);
      }
   }

   // Draw trailer text
   // Not needed anymore, leaving this here for historical purposes
   // Yes, the trailer was pure unedited game footage :P
   /*
   draw_text("Coming soon to Windows and Linux!", screen_cx, screen_cy+0x28,
      FONT_LIT, ALIGN_TOP);
   draw_text("http://sol.azurasun.com/", screen_cx, screen_cy+0x38,
      FONT_LIT, ALIGN_TOP);
   draw_text("Twitter: @AzuraSun", screen_cx, screen_cy+0x48,
      FONT_LIT, ALIGN_TOP);
   */

   // Draw copyright information
   draw_text(text.title.copyright, screen_w - 4, screen_h,
      FONT_LIT, ALIGN_BOTTOMRIGHT);

#if DEBUG
   // Temp
   // This was used to test kanji support, I guess it isn't needed anymore...
   // (text says "background music, sound effects.")
   draw_text("背景音楽、効果音。", screen_w - 4, screen_h - 0x10,
      FONT_LIT, ALIGN_BOTTOMRIGHT);
#endif
}

//***************************************************************************
// deinit_title
// Deinitializes the title screen.
//***************************************************************************

void deinit_title(void) {
   // Unload music
   unload_all_bgm();

   // Unload graphics
   unload_background();
   if (gfxset_title) {
      destroy_graphics_set(gfxset_title);
      gfxset_title = NULL;
   }
}

//***************************************************************************
// button_start [internal]
// Function for the "Start game" button. Starts the main game.
//***************************************************************************

static void button_start(void) {
   // Go to the load game screen
   fade_off_and_switch(GAMEMODE_SAVESLOT);
   //switch_to_scene(0);
}

//***************************************************************************
// button_editor [internal]
// Function for the "Level editor" button. Goes into the level editor.
//***************************************************************************

static void button_editor(void) {
   // Go into level editor
   fade_off_and_switch(GAMEMODE_EDITOR);
}

//***************************************************************************
// button_options [internal]
// Function for the "Options" button. Goes into the options menu.
//***************************************************************************

static void button_options(void) {
   // Go into options menu
   fade_off_and_switch(GAMEMODE_OPTIONS);
}

//***************************************************************************
// button_quit [internal]
// Function for the "Quit" button. Quits the program.
//***************************************************************************

static void button_quit(void) {
   // Quit the game
   fade_off_and_switch(GAMEMODE_QUIT);
}
