//***************************************************************************
// "video.c"
// Video code. Handles the video output as well as rendering stuff on screen.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Define this to show some GPU information
// Normally you don't want this...
//#define DEBUG_GL

// Required headers
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <time.h>
#include <setjmp.h>
#include <png.h>
#include <SDL2/SDL.h>
#include <physfs.h>
#include "main.h"
#include "audiovideo.h"
#include "file.h"
#include "ingame.h"
#include "menu.h"
#include "parser.h"
#include "settings.h"
#include "sound.h"
#include "recording.h"
#include "video.h"

#ifdef _WIN32
#include <windows.h>
#endif

#ifdef DEBUG_GL
#include <GL/gl.h>
#endif

// Maximum dimensions the framebuffer can have. These are measured in pixels
// from the virtual resolution (i.e. before scaling up), not the actual
// resolution on screen.
#define MAX_SCREEN_W 384
#define MAX_SCREEN_H 240

// Size of the texture used to copy the framebuffer contents to screen. Must
// be at least as large as the framebuffer itself. Using here a power of two
// since that's guaranteed to have the best compatibility *and* performance,
// even on old hardware (early NPOT implementations were slow too so better
// avoid that even if supported... not like the texture is large anyway).
#define MAX_TEXTURE_W 512
#define MAX_TEXTURE_H 256

// Where we keep a list of possible resolutions
// To-do: come up with something smarter?
Resolution resolutions[NUM_RESOLUTIONS];

// Look-up table used to convert colors in the CGA mode
static const uint32_t cga_table[2][0x40] = {
   // Checkerboard color 1
   // Dark half goes here
   {
      0x000000, 0x000000, 0x000000, 0xFF55FF,
      0x000000, 0x000000, 0x000000, 0xFF55FF,
      0x000000, 0x000000, 0xFF55FF, 0xFF55FF,
      0x55FFFF, 0x55FFFF, 0x55FFFF, 0x55FFFF,

      0x000000, 0x000000, 0x000000, 0xFF55FF,
      0x000000, 0x000000, 0x000000, 0xFF55FF,
      0x000000, 0x000000, 0xFF55FF, 0xFF55FF,
      0x55FFFF, 0x55FFFF, 0x55FFFF, 0x55FFFF,

      0x000000, 0x000000, 0xFF55FF, 0xFF55FF,
      0xFF55FF, 0xFF55FF, 0xFF55FF, 0xFF55FF,
      0xFF55FF, 0xFF55FF, 0xFF55FF, 0xFF55FF,
      0x55FFFF, 0x55FFFF, 0x55FFFF, 0x55FFFF,

      0xFF55FF, 0xFF55FF, 0xFF55FF, 0xFF55FF,
      0xFF55FF, 0xFF55FF, 0xFF55FF, 0xFF55FF,
      0xFF55FF, 0xFF55FF, 0xFF55FF, 0xFF55FF,
      0xFFFFFF, 0xFFFFFF, 0xFFFFFF, 0xFFFFFF
   },

   // Checkerboard color 2
   // Bright half goes here
   {
      0x000000, 0xFF55FF, 0x55FFFF, 0x55FFFF,
      0xFF55FF, 0xFF55FF, 0x55FFFF, 0x55FFFF,
      0x55FFFF, 0x55FFFF, 0x55FFFF, 0x55FFFF,
      0x55FFFF, 0x55FFFF, 0x55FFFF, 0x55FFFF,

      0x000000, 0x000000, 0xFF55FF, 0x55FFFF,
      0xFF55FF, 0xFF55FF, 0xFF55FF, 0x55FFFF,
      0x55FFFF, 0x55FFFF, 0x55FFFF, 0x55FFFF,
      0x55FFFF, 0x55FFFF, 0x55FFFF, 0x55FFFF,

      0xFF55FF, 0xFF55FF, 0xFF55FF, 0xFF55FF,
      0xFF55FF, 0xFF55FF, 0xFF55FF, 0xFF55FF,
      0x55FFFF, 0x55FFFF, 0x55FFFF, 0x55FFFF,
      0xFFFFFF, 0xFFFFFF, 0xFFFFFF, 0xFFFFFF,

      0xFF55FF, 0xFF55FF, 0xFF55FF, 0xFF55FF,
      0xFF55FF, 0xFF55FF, 0xFF55FF, 0xFF55FF,
      0xFFFFFF, 0xFFFFFF, 0xFFFFFF, 0xFFFFFF,
      0xFFFFFF, 0xFFFFFF, 0xFFFFFF, 0xFFFFFF
   }
};

// The framebuffer. This is where the game renders everything.
static uint32_t framebuffer[MAX_SCREEN_W * MAX_SCREEN_H];
static uint32_t *lines[MAX_SCREEN_H];
int screen_w = 384;
int screen_h = 240;
int screen_scale = 1;
int screen_cx;
int screen_cy;

// Window that is displayed on screen
static SDL_Window *window = NULL;
static SDL_Renderer *renderer = NULL;
static SDL_Texture *texture = NULL;

#ifdef __linux__
// Ugh, hack to work around fullscreen being broken on X11
// If this is set, assume the hack is in full force
static int fullscreen_hack = 0;
#endif

// Maximum lengths for names
#define MAX_ID_LEN 0x1F

// Possible fading states
typedef enum {
   FADE_IDLE,     // Not fading
   FADE_ON,       // Fading on
   FADE_OFF       // Fading off
} FadingMode;

// Fading information
static struct {
   FadingMode mode;     // What's going on right now
   GameMode switch_to;  // Game mode to switch to (NUM_GAMEMODES if none)
   int pos;             // Current position (0x00 to 0xFF)
} fade = {
   FADE_IDLE,
   NUM_GAMEMODES,
   0x00
};

// How fast to update the fading position when fading is active
#define FADING_SPEED 0x10

// Private function prototypes
static int is_valid_window_size(int, int);
static int validate_sprite_name(char *);
static uint8_t compute_quickhash(const char *);
static void read_png_callback(png_structp, png_bytep, png_size_t);
static void write_png_callback(png_structp, png_bytep, png_size_t);
static void flush_png_callback(png_structp png_ptr);

//***************************************************************************
// init_video
// Initializes the video subsystem.
//***************************************************************************

void init_video(void) {
#ifdef DEBUG
   fputs("Initializing video\n", stderr);
#endif
   // Figure out if the monitor we'll aim for even exists
   // This is for safety in case the monitor count changes
   int num_monitors = SDL_GetNumVideoDisplays();
   if (settings.monitor >= num_monitors)
      settings.monitor = 0;

   // Generate a list of all valid resolutions
   for (unsigned i = 0, scale = 1; i < NUM_RESOLUTIONS; i += 3, scale++) {
      resolutions[i+0].width  = 320 * scale;
      resolutions[i+0].height = 240 * scale;

      resolutions[i+1].width  = 384 * scale;
      resolutions[i+1].height = 216 * scale;

      resolutions[i+2].width  = 384 * scale;
      resolutions[i+2].height = 240 * scale;
   }

   // Get current video mode
   SDL_DisplayMode mode;
   SDL_GetCurrentDisplayMode(settings.monitor, &mode);

   // Validate windowed size first
   if (!is_valid_window_size(settings.windowed_width,
   settings.windowed_height)) {
      settings.windowed_width = 0;
      settings.windowed_height = 0;

      // Determine the maximum size for the window
      // We reduce the limit by 10% to account for taskbars and such
      int max_x = mode.w * 9 / 10;
      int max_y = mode.h * 9 / 10;

      // Find the largest 16:10 resolution that fits
      int test_x = 384;
      int test_y = 240;
      while (test_x + 384 < max_x && test_y + 240 < max_y) {
         test_x += 384;
         test_y += 240;
      }

      // Store the resolution we found
      settings.windowed_width = test_x;
      settings.windowed_height = test_y;
   }

   // Determine resolution to use
   int res_x, res_y;
   if (settings.fullscreen) {
      // In fullscreen just use the desktop resolution
      res_x = mode.w;
      res_y = mode.h;

      // Hack needed for now to get the game not crash when remapping the
      // cursor coordinates (since it uses a division), will be removed once
      // the transition is complete
      settings.fullscreen_width = res_x;
      settings.fullscreen_height = res_y;
   } else {
      // Use the windowed resolution from the settings
      res_x = settings.windowed_width;
      res_y = settings.windowed_height;
   }

   // Calculate how many pixels are at the requested resolution
   unsigned long pixel_count = res_x * res_y;

   // Try to find the best fit for this resolution
   for (screen_scale = 1; ; screen_scale++) {
      // Try all virtual resolutions at this scale
      // To-do: don't use macros...
      unsigned long minimum_border = pixel_count;
#define CHECK(w,h) \
      if ((w) * screen_scale <= res_x && (h) * screen_scale <= res_y && \
      pixel_count - (w) * (h) * screen_scale < minimum_border) { \
         minimum_border = pixel_count - (w) * (h) * screen_scale; \
         screen_w = (w); \
         screen_h = (h); \
      }

      CHECK(320, 240);
      CHECK(384, 240);
      CHECK(384, 216);
#undef CHECK

      // No size fits?
      if (minimum_border == pixel_count) {
         screen_scale--;
         break;
      }
   }

#ifdef __linux__
   // Determine if the fullscreen hack is going to be needed here
   // <Sik> Is this thing even used anymore? I thought it was gone alongside
   // mode-switching fullscreen? Don't have time to figure out right now
   fullscreen_hack = settings.fullscreen &&
      (res_x < mode.w || res_y < mode.h);
#endif

   // Control focus loss behavior in fullscreen. In a single monitor setup
   // we want the game to minimize (since otherwise everything else would be
   // obscured), but in a multimonitor setup we don't want it to minimize
   // (since other windows may be visible in other monitors).
   // We aren't setting this hint with override in case the behavior we
   // define here actually turns out to be a bad idea for somebody.
   SDL_SetHint(SDL_HINT_VIDEO_MINIMIZE_ON_FOCUS_LOSS,
      SDL_GetNumVideoDisplays() == 1 ? "1" : "0");

   // Create window
   window = SDL_CreateWindow(
      settings.reader == READER_TITLEBAR ? "" : GAME_NAME,
      SDL_WINDOWPOS_CENTERED_DISPLAY(settings.monitor),
      SDL_WINDOWPOS_CENTERED_DISPLAY(settings.monitor), res_x, res_y,
      (settings.fullscreen ? SDL_WINDOW_FULLSCREEN_DESKTOP : 0) |
      SDL_WINDOW_ALLOW_HIGHDPI);
   if (window == NULL)
      abort_program(ERR_INITVIDEO, NULL);

   // Create rendering context
   // In theory the vsync flag should force the renderer to use vsync, but
   // we're going to use both the flag and the hint to make sure (apparently
   // using the flag alone fails on some systems)
   SDL_SetHint(SDL_HINT_RENDER_VSYNC, settings.vsync ? "1" : "0");
   renderer = SDL_CreateRenderer(window, -1,
      (settings.vsync ? SDL_RENDERER_PRESENTVSYNC : 0) |
      (settings.no_hwaccel ? SDL_RENDERER_SOFTWARE : 0));
   if (renderer == NULL)
      abort_program(ERR_INITVIDEO, NULL);

#ifdef DEBUG
   // What renderer did we get?
   SDL_RendererInfo renderer_info;
   SDL_GetRendererInfo(renderer, &renderer_info);
   printf("Renderer in use: \"%s\"\n", renderer_info.name);
#endif

   // Enable filtering as needed (we force it off when using software
   // rendering because otherwise it'd be too slow... no idea if SDL respects
   // the filtering setting there anyway)
   if (settings.filter && !settings.no_hwaccel)
      SDL_SetHint(SDL_HINT_RENDER_SCALE_QUALITY, "linear");
   else
      SDL_SetHint(SDL_HINT_RENDER_SCALE_QUALITY, "nearest");

   // Create texture used to copy the framebuffer to screen
   texture = SDL_CreateTexture(renderer,
      SDL_PIXELFORMAT_ARGB8888, SDL_TEXTUREACCESS_STREAMING,
      MAX_TEXTURE_W, MAX_TEXTURE_H);
   if (texture == NULL)
      abort_program(ERR_INITVIDEO, NULL);
   SDL_SetTextureBlendMode(texture, SDL_BLENDMODE_NONE);
   SDL_SetTextureAlphaMod(texture, 0xFF);
   SDL_SetTextureColorMod(texture, 0xFF, 0xFF, 0xFF);

   // Determine where each line starts in the framebuffer
   // Done for speed up purposes when rendering sprites
   for (int i = 0; i < screen_h; i++)
      lines[i] = &framebuffer[i * screen_w];

   // Show window!
   SDL_ShowWindow(window);
   handle_gain_focus();

   // We render the cursor on our own, so disable the OS one for our window
   // When no cursor is meant to be shown, in fullscreen it simply isn't
   // drawn (you would want no cursor there anyway!), while in windowed mode
   // the arrow is drawn instead.
   SDL_ShowCursor(SDL_FALSE);

   // Calculate the center of the screen
   // Doing this once here since we need these values often and recalculating
   // them every time would be a waste of time... (also makes the code a bit
   // cleaner by having less text around)
   screen_cx = screen_w / 2;
   screen_cy = screen_h / 2;
}

//***************************************************************************
// is_valid_window_size [internal]
// Checks if the passed resolution is valid in windowed mode
//---------------------------------------------------------------------------
// param w: desired width
// param h: desired height
// return: non-zero if valid, zero if not
//***************************************************************************

static int is_valid_window_size(int w, int h) {
   /*
   // Check against valid ratios
   if (w % 320 == 0 && h % 240 == 0 && w / 320 == h / 240)
      return 1;
   if (w % 384 == 0 && h % 240 == 0 && w / 384 == h / 240)
      return 1;
   if (w % 384 == 0 && h % 216 == 0 && w / 384 == h / 216)
      return 1;
   */

   // Compare against all valid resolutions
   for (unsigned i = 0; i < NUM_RESOLUTIONS; i++) {
      if (w == resolutions[i].width && h == resolutions[i].height)
         return 1;
   }

   // Nope
   return 0;
}

//***************************************************************************
// update_video
// Updates the screen contents, i.e. it shows the stuff in the framebuffer
// on screen (scaling up as needed).
//***************************************************************************

void update_video(void) {
   // Recording a movie?
   if (is_recording()) {
      update_recording();
      draw_recording_hud();
   }

   // Clear border
   SDL_RenderClear(renderer);

   // Apply a colorblind filter?
   unsigned num_pixels = screen_w * screen_h;
   switch (settings.color_mode) {
      // Normal?
      case COLOR_NORMAL:
         break;

      // Red-green deficiency?
      case COLOR_TYPE1:
         for (unsigned i = 0; i < num_pixels; i++) {
            // Get color components
            uint32_t color = framebuffer[i];
            unsigned r = color >> 16 & 0xFF;
            unsigned g = color >> 8 & 0xFF;
            unsigned b = color & 0xFF;

            // Mix red and green into a single component
            unsigned x = (r + g) >> 1;

            // Store filtered color
            framebuffer[i] = x << 16 | x << 8 | b;
         }
         break;

      // Blue deficiency?
      case COLOR_TYPE2:
         for (unsigned i = 0; i < num_pixels; i++) {
            // Get color components
            uint32_t color = framebuffer[i];
            unsigned r = color >> 16 & 0xFF;
            unsigned g = color >> 8 & 0xFF;
            unsigned b = color & 0xFF;

            // Mix green and blue into a single component
            unsigned x = (g + b) >> 1;

            // Store filtered color
            framebuffer[i] = r << 16 | x << 8 | x;
         }
         break;

      // Monochrome?
      case COLOR_MONO:
         grayscale_filter();
         break;

      // Dichrome?
      case COLOR_DICHRO:
         for (unsigned i = 0; i < num_pixels; i++) {
            // Get color components
            uint32_t color = framebuffer[i];
            unsigned r = color >> 16 & 0xFF;
            unsigned g = color >> 8 & 0xFF;
            unsigned b = color & 0xFF;

            // Distribute green evenly among the other two channels, then
            // blend those back into green. The idea is that this reduces
            // the three channels into two... not sure how it works really :P
            r = (r + g) >> 1;
            b = (b + g) >> 1;
            g = (r + b) >> 1;

            // Store filtered color
            framebuffer[i] = r << 16 | g << 8 | b;
         }
         break;

      // Negative?
      case COLOR_NEGATIVE:
         for (unsigned i = 0; i < num_pixels; i++) {
            // Reverse all components!
            framebuffer[i] = ~framebuffer[i];
         }
         break;

      // Swap RGB to BGR?
      case COLOR_RGBSWAP:
         for (unsigned i = 0; i < num_pixels; i++) {
            // Get color components
            uint32_t color = framebuffer[i];
            unsigned r = color >> 16 & 0xFF;
            unsigned g = color >> 8 & 0xFF;
            unsigned b = color & 0xFF;

            // Write color back swapping red and blue
            framebuffer[i] = r | g << 8 | b << 16;
         }
         break;

      // Mild low vision mode?
      case COLOR_LOVISION1:
         for (unsigned y = 0, i = 0; y < (unsigned) screen_h;
         y += 2, i += screen_w) {
            for (unsigned x = 0; x < (unsigned) screen_w;
            x += 2, i += 2) {
               // Get all colors
               uint32_t color1 = framebuffer[i];
               uint32_t color2 = framebuffer[i + 1];
               uint32_t color3 = framebuffer[i + screen_w];
               uint32_t color4 = framebuffer[i + screen_w + 1];

               // Calculate the average
               color1 >>= 2; color1 &= 0x3F3F3F;
               color2 >>= 2; color2 &= 0x3F3F3F;
               color3 >>= 2; color3 &= 0x3F3F3F;
               color4 >>= 2; color4 &= 0x3F3F3F;
               uint32_t result = color1 + color2 + color3 + color4;

               // Write pixels
               framebuffer[i] = result;
               framebuffer[i + 1] = result;
               framebuffer[i + screen_w] = result;
               framebuffer[i + screen_w + 1] = result;
            }
         }
         break;

      // Heavy low vision mode?
      case COLOR_LOVISION2: {
         // Calculate the different strides (1 line, 2 lines, 3 lines)
         int32_t stride1 = screen_w;
         int32_t stride2 = screen_w * 2;
         int32_t stride3 = screen_w * 3;

         // Scan all pixels
         for (unsigned y = 0, i = 0; y < (unsigned) screen_h;
         y += 4, i += stride3) {
            for (unsigned x = 0; x < (unsigned) screen_w;
            x += 4, i += 4) {
               // Where colors are stored temporarily
               uint32_t colors[16];

               // Get original colors
               colors[0]  = framebuffer[i];
               colors[1]  = framebuffer[i + 1];
               colors[2]  = framebuffer[i + 2];
               colors[3]  = framebuffer[i + 3];
               colors[4]  = framebuffer[i + stride1];
               colors[5]  = framebuffer[i + stride1 + 1];
               colors[6]  = framebuffer[i + stride1 + 2];
               colors[7]  = framebuffer[i + stride1 + 3];
               colors[8]  = framebuffer[i + stride2];
               colors[9]  = framebuffer[i + stride2 + 1];
               colors[10] = framebuffer[i + stride2 + 2];
               colors[11] = framebuffer[i + stride2 + 3];
               colors[12] = framebuffer[i + stride3];
               colors[13] = framebuffer[i + stride3 + 1];
               colors[14] = framebuffer[i + stride3 + 2];
               colors[15] = framebuffer[i + stride3 + 3];

               // Calculate the average
               uint32_t result = 0;
               for (unsigned j = 0; j < 16; j++) {
                  uint32_t temp = colors[j];
                  temp >>= 4;
                  temp &= 0x0F0F0F;
                  result += temp;
               }

               // Write pixels
               framebuffer[i]               = result;
               framebuffer[i + 1]           = result;
               framebuffer[i + 2]           = result;
               framebuffer[i + 3]           = result;
               framebuffer[i + stride1]     = result;
               framebuffer[i + stride1 + 1] = result;
               framebuffer[i + stride1 + 2] = result;
               framebuffer[i + stride1 + 3] = result;
               framebuffer[i + stride2]     = result;
               framebuffer[i + stride2 + 1] = result;
               framebuffer[i + stride2 + 2] = result;
               framebuffer[i + stride2 + 3] = result;
               framebuffer[i + stride3]     = result;
               framebuffer[i + stride3 + 1] = result;
               framebuffer[i + stride3 + 2] = result;
               framebuffer[i + stride3 + 3] = result;
            }
         }
      } break;

      // TV mode?
      case COLOR_TVMODE:
         // Scan all lines
         for (int32_t y = 0; y < screen_h; y++) {
            // Temporary buffers to hold the entire line
            uint32_t buffer1[MAX_SCREEN_W + 2];
            uint32_t buffer2[MAX_SCREEN_W + 2];
            buffer1[0] = 0; buffer1[MAX_SCREEN_W + 1] = 0;
            buffer2[0] = 0; buffer2[MAX_SCREEN_W + 1] = 0;

            // Read original line
            memcpy(&buffer1[1], lines[y], screen_w * sizeof(uint32_t));

            // Apply color bleeding filter
            for (int32_t x = 0; x < screen_w; ) {
               uint32_t curr, other;

               curr = buffer1[x+1] >> 1 & 0x7F0000;
               other = buffer1[x] >> 1 & 0x7F0000;
               curr |= buffer1[x+1] & 0x00FFFF;
               buffer2[x+1] = curr + other;
               x++;

               curr = buffer1[x+1] >> 1 & 0x00007F;
               other = buffer1[x+2] >> 1 & 0x00007F;
               curr |= buffer1[x+1] & 0xFFFF00;
               buffer2[x+1] = curr + other;
               x++;
            }

            // Apply blurring filter
            for (int32_t x = 0; x < screen_w; x++) {
               uint32_t prev = buffer2[x] >> 3 & 0x1F1F1F;
               uint32_t next = buffer2[x+2] >> 3 & 0x1F1F1F;
               uint32_t curr = buffer2[x+1] >> 1 & 0x7F7F7F;
               curr += curr >> 1 & 0x3F3F3F;

               buffer1[x+1] = prev + curr + next;
            }

            // Store filtered line
            memcpy(lines[y], &buffer1[1], screen_w * sizeof(uint32_t));
         }
         break;
   }

   // Apply a contrast filter?
   switch (settings.contrast) {
      // Normal contrast
      case CONTRAST_NORMAL:
         break;

      // Low contrast?
      case CONTRAST_LOW:
         for (unsigned i = 0; i < num_pixels; i++) {
            // Get original color
            uint32_t color = framebuffer[i];

            // Halve the intensity of each component, then recenter them
            // (this reduces the range from 0%-100% to 25%-75%)
            color &= 0xFEFEFE;
            color >>= 1;
            color += 0x404040;

            // Store new color
            framebuffer[i] = color;
         }
         break;

      // Between low and normal?
      case CONTRAST_MIDLOW:
         for (unsigned i = 0; i < num_pixels; i++) {
            // Get original color
            uint32_t color = framebuffer[i];

            // Multiply color by 75%, then recenter it
            // Ugly mess going on here!
            uint32_t half = (color & 0xFEFEFE) >> 1;
            uint32_t quarter = (color & 0xFCFCFC) >> 2;
            color = half + quarter;
            color += 0x202020;

            // Store new color
            framebuffer[i] = color;
         }
         break;

      // High contrast?
      case CONTRAST_HIGH:
         for (unsigned i = 0; i < num_pixels; i++) {
            // Get color components
            uint32_t color = framebuffer[i];
            unsigned r = color >> 16 & 0xFF;
            unsigned g = color >> 8 & 0xFF;
            unsigned b = color & 0xFF;

            // Clamp values so they're only in the 25%-75% range
            // <Sik> Isn't there any more efficient way to do this clamping?
            // (though I'd have to see if the compiler can come up with a
            // more clever way to do it first)
            if (r < 0x40) r = 0x40; if (r > 0xBF) r = 0xBF; r -= 0x40;
            if (g < 0x40) g = 0x40; if (g > 0xBF) g = 0xBF; g -= 0x40;
            if (b < 0x40) b = 0x40; if (b > 0xBF) b = 0xBF; b -= 0x40;

            // Double their intensity
            r <<= 1;
            g <<= 1;
            b <<= 1;

            // Store new color
            framebuffer[i] = r << 16 | g << 8 | b;
         }
         break;

      // Between high and normal?
      case CONTRAST_MIDHIGH:
         for (unsigned i = 0; i < num_pixels; i++) {
            // Get color components
            uint32_t color = framebuffer[i];
            unsigned r = color >> 16 & 0xFF;
            unsigned g = color >> 8 & 0xFF;
            unsigned b = color & 0xFF;

            // Increase values by 50%
            r += r >> 1;
            g += g >> 1;
            b += b >> 1;

            // Clamp values
            if (r < 0x40) r = 0x40; if (r > 0x13F) r = 0x13F; r -= 0x40;
            if (g < 0x40) g = 0x40; if (g > 0x13F) g = 0x13F; g -= 0x40;
            if (b < 0x40) b = 0x40; if (b > 0x13F) b = 0x13F; b -= 0x40;

            // Store new color
            framebuffer[i] = r << 16 | g << 8 | b;
         }
         break;
   }

   // Separate contrast parameter, LCD fix
   // If this is enabled, we slightly mess with the colors to make up for the
   // fact those screens suck at their extreme values. This is indepdendent
   // of the other contrast settings (so you can have 200% contrast and then
   // apply the shift)
   if (settings.lcd_fix) {
      for (unsigned i = 0; i < num_pixels; i++) {
         uint32_t color = framebuffer[i];
         uint32_t bias = (~color & 0xF8F8F8) >> 3;
         framebuffer[i] = color + bias;
      }
   }

   // Copy the framebuffer contents into the texture
   // To-do: check if there's actually some gotcha here. Technically SDL has
   // been told to use native endianess, and supposedly the addresses should
   // be aligned (src definitely is), but we'll want to make sure. This needs
   // to be as fast as possible.
   void *dest; int pitch;
   if (SDL_LockTexture(texture, NULL, &dest, &pitch))
      abort_program(ERR_UPDATEVIDEO, NULL);
   const uint32_t *src = framebuffer;
   for (int y = 0; y < screen_h; y++) {
      memcpy(dest, src, sizeof(uint32_t) * screen_w);
      src += screen_w;
      dest += pitch;
   }
   SDL_UnlockTexture(texture);

   // Determine actual size of the framebuffer in pixels on the screen
   int real_w = screen_w * screen_scale;
   int real_h = screen_h * screen_scale;

   // Send contents to screen!
   int full_w, full_h;
   SDL_GetWindowSize(window, &full_w, &full_h);

   SDL_Rect tex_rect;
   tex_rect.x = 0;
   tex_rect.y = 0;
   tex_rect.w = screen_w;
   tex_rect.h = screen_h;

   SDL_Rect dest_rect;
   dest_rect.w = real_w;
   dest_rect.h = real_h;
   dest_rect.x = (full_w - real_w) >> 1;
   dest_rect.y = (full_h - real_h) >> 1;

   if (SDL_RenderCopy(renderer, texture, &tex_rect, &dest_rect))
   /*if (SDL_RenderCopyEx(renderer, texture, &tex_rect, &dest_rect,
   0, NULL, 0))*/
      abort_program(ERR_UPDATEVIDEO, NULL);
   SDL_RenderPresent(renderer);

   // Prevent the cursor from going off-screen if the relative mouse mode
   // hack is in place
   if (SDL_GetRelativeMouseMode()) {
      int window_w, window_h;
      SDL_GetWindowSize(window, &window_w, &window_h);
      SDL_WarpMouseInWindow(window, 0, 0);
   }
}

//***************************************************************************
// deinit_video
// Deinitializes the video subsystem.
//***************************************************************************

void deinit_video(void) {
   // Determine the monitor the window is currently in
   // Makes it possible to initialize on the same monitor next time
   // Also allows switching monitors just by dragging the window
   int window_x, window_y;
   int width, height;
   SDL_GetWindowPosition(window, &window_x, &window_y);
   SDL_GetWindowSize(window, &width, &height);

   window_x += width / 2;
   window_y += height / 2;

   int num_monitors = SDL_GetNumVideoDisplays();
   for (int i = 0; i < num_monitors; i++) {
      SDL_Rect area;
      SDL_GetDisplayBounds(i, &area);

      if (window_x < area.x) continue;
      if (window_y < area.y) continue;
      if (window_x >= area.x + area.w) continue;
      if (window_y >= area.y + area.h) continue;

      settings.monitor = i;
      break;
   }

   // Not doing this actually breaks message boxes on X11. Yep.
   SDL_ShowCursor(SDL_TRUE);

   // Destroy window and associated objects
   if (texture != NULL) {
      SDL_DestroyTexture(texture);
      texture = NULL;
   }
   if (renderer != NULL) {
      SDL_DestroyRenderer(renderer);
      renderer = NULL;
   }
   if (window != NULL) {
      SDL_DestroyWindow(window);
      window = NULL;
   }
}

//***************************************************************************
// handle_gain_focus
// Sets up the video subsystem to tell it that we just gained focus
//***************************************************************************

void handle_gain_focus(void) {
#ifdef __linux__
   // Horrible hack for X11: resort to relative mouse mode on fullscreen to
   // prevent the cursor from going outbounds, since SDL changes the video
   // mode but not the desktop size.
   if (fullscreen_hack) {
      // Enable relative mode. If this fails we don't care (and just hope SDL
      // doesn't break up), we check if relative mode is enabled by calling
      // SDL_GetRelativeMouseMode
      SDL_SetRelativeMouseMode(SDL_TRUE);
      SDL_SetWindowGrab(window, SDL_TRUE);
   }
#endif
}

//***************************************************************************
// handle_lose_focus
// Sets up the video subsystem to tell it that we just lost focus
//***************************************************************************

void handle_lose_focus(void) {
   // Reset all user input
   reset_input();

#ifdef __linux__
   // Undo the relative mouse mode hack
   if (fullscreen_hack) {
      SDL_SetWindowGrab(window, SDL_FALSE);
      SDL_SetRelativeMouseMode(SDL_FALSE);
   }
#endif
}

//***************************************************************************
// is_fullscreen
// Checks if the window is currently fullscreen. Note that this checks the
// *actual* status, not the current selected setting (which you can easily
// retrieve from settings.fullscreen). Needed for the rare cases where they
// can mismatch (i.e. the video menu).
//---------------------------------------------------------------------------
// return: zero if windowed, non-zero if fullscreen
//***************************************************************************

int is_fullscreen(void) {
   if (window == NULL) return 0;
   return (SDL_GetWindowFlags(window) & SDL_WINDOW_FULLSCREEN) ? 1 : 0;
}

//***************************************************************************
// toggle_fullscreen
// Toggles fullscreen mode.
//***************************************************************************

void toggle_fullscreen(void) {
   // Toggle setting
   settings.fullscreen ^= 1;

   // Recreate window
   deinit_video();
   init_video();

   // Menus cache the positions, so they have to be reinitialized in order
   // to let options appear in the correct location. We preserve the
   // currently selected option since for the sake of simpleness we just
   // remake the entire menu from scratch, and that resets the selected
   // option (and not preserving it can be annoying)
   unsigned selected = menu.selected;
   reinit_menu();
   menu.selected = selected;
}

//***************************************************************************
// get_window_title
// Gets the current title of the winow.
//---------------------------------------------------------------------------
// return: window title
//***************************************************************************

const char *get_window_title(void) {
   return SDL_GetWindowTitle(window);
}

//***************************************************************************
// set_window_title
// Changes the title of the window. Only used with the screen reader output
// set to titlebar, but still (remember the window pointer is private here)
//---------------------------------------------------------------------------
// param title: new title
//***************************************************************************

void set_window_title(const char *title) {
#ifdef __linux__
   // To work around a stupid bug in the Linux implementation
   // This is fixed in SDL 2.0.4 by the way (broken in 2.0.3, though)
   if (title == NULL || title[0] == '\0')
      title = " ";
#endif
   SDL_SetWindowTitle(window, title);
}

//***************************************************************************
// virtual_to_real_coord
// Converts virtual pixel coordinates (i.e. what's used in the framebuffer)
// to real pixel coordinates (i.e. what's actually on screen)
//***************************************************************************

void virtual_to_real_coord(int *x, int *y) {
   // Get virtual values
   int tx = *x;
   int ty = *y;

   // Convert virtual pixels to real pixels
   tx *= screen_scale;
   ty *= screen_scale;

   // Get window size
   int full_w, full_h;
   SDL_GetWindowSize(window, &full_w, &full_h);

   // Apply offset (since display is centered if there's a border)
   tx += (full_w - screen_w * screen_scale) >> 1;
   ty += (full_h - screen_h * screen_scale) >> 1;

   // Store real values
   *x = tx;
   *y = ty;
}

//***************************************************************************
// clear_screen
// Fills the framebuffer with the specified color
//---------------------------------------------------------------------------
// param color: color to fill with
//***************************************************************************

void clear_screen(uint32_t color) {
   // Are we rendering in CGA mode?
   if (settings.cga_mode) {
      uint32_t color1 = get_cga_color(color, 0);
      uint32_t color2 = get_cga_color(color, 1);

      // If both are identical just fill the framebuffer as-is
      if (color1 == color2) {
         uint32_t *ptr = framebuffer;
         for (unsigned i = screen_w * screen_h; i > 0; i--)
            *ptr++ = color1;
      }

      // Otherwise use a checkerboard partner
      // WARNING: this assumes the virtual resolution is a multiple of two
      // TO-DO: find a way to fix that? (not like we're ever going to support
      // uneven resolutions, but still)
      else {
         uint32_t *ptr = framebuffer;
         for (int32_t y = 0; y < screen_h; y++) {
            // Draw the next line
            // Alternate between both colors to do dithering
            for (int32_t x = 0; x < screen_w; x += 2) {
               *ptr++ = color1;
               *ptr++ = color2;
            }

            // Swap the colors for the next line
            // This way we get a checkerboard pattern instead of vertical
            // lines
            color = color1;
            color1 = color2;
            color2 = color;
         }
      }

      // There, rendered, done here
      return;
   }

   // Fill entire screen with the specified color
   uint32_t *ptr = framebuffer;
   for (unsigned i = screen_w * screen_h; i > 0; i--)
      *ptr++ = color;

   // Fun fact: originally this function had a somewhat unrolled loop drawing
   // 16 pixels each iteration instead of one. Turns out that doing that will
   // interfere with the compiler's ability to optimize the code. To give you
   // an idea, in the system where this was tested the old code took 0.20ms
   // to execute, while this new code takes 0.12ms.
}

//***************************************************************************
// draw_pixel
// Draws a single pixel on screen
//---------------------------------------------------------------------------
// param x: horizontal coordinate
// param y: vertical coordinate
// param color: color to draw with
//---------------------------------------------------------------------------
// THIS FUNCTION IS SLOW, as in, you shouldn't attempt to make drawing code
// off this (write into the framebuffer directly!). Use this function only
// when you really need to draw a single pixel and nothing else (e.g. the
// brightest/darkest spots in the sunken frames in the level editor).
//***************************************************************************

void draw_pixel(int x, int y, uint32_t color) {
   // Don't attempt to draw off-screen!
   if (x < 0 || x >= screen_w || y < 0 || y >= screen_h)
      return;

   // Just change the relevant pixel :P
   if (settings.cga_mode) {
      lines[y][x] = get_cga_color(color, (x ^ y) & 1);
   } else
      lines[y][x] = color;
}

//***************************************************************************
// draw_hline
// Draws a horizontal line on screen with the specified color
//---------------------------------------------------------------------------
// param x1: left coordinate (inclusive)
// param y: vertical coordinate
// param x2: right coordinate (inclusive)
// param color: color to draw with
//***************************************************************************

void draw_hline(int x1, int y, int x2, uint32_t color) {
   // Clamp coordinates so they lie within screen boundaries
   if (x1 < 0) x1 = 0;
   if (x2 >= screen_w) x2 = screen_w - 1;

   // Not visible?
   if (x1 > x2 || y < 0 || y >= screen_h)
      return;

   // Determine position of the first pixel in the framebuffer
   uint32_t *ptr = lines[y] + x1;

   // CGA mode? (use a different way to render the colors here)
   if (settings.cga_mode) {
      // Determine colors to use
      int32_t color1 = get_cga_color(color, y & 1);
      int32_t color2 = get_cga_color(color, ~y & 1);

      // Paint line
      for (int xr = x1; xr <= x2; xr++)
         *ptr++ = xr & 1 ? color2 : color1;

      // Done
      return;
   }

   // Draw all pixels inside the line
   for (int xr = x1; xr <= x2; xr++)
      *ptr++ = color;
}

//***************************************************************************
// draw_vline
// Draws a vertical line on screen with the specified color
//---------------------------------------------------------------------------
// param x: horizontal coordinate
// param y1: top coordinate (inclusive)
// param y2: bottom coordinate (inclusive)
// param color: color to draw with
//***************************************************************************

void draw_vline(int x, int y1, int y2, uint32_t color) {
   // Clamp coordinates so they lie within screen boundaries
   if (y1 < 0) y1 = 0;
   if (y2 >= screen_h) y2 = screen_h - 1;

   // Not visible?
   if (y1 > y2 || x < 0 || x >= screen_w)
      return;

   // Determine position of the first pixel in the framebuffer
   uint32_t *ptr = lines[y1] + x;

   // CGA mode? (use a different way to render the colors here)
   if (settings.cga_mode) {
      // Determine colors to use
      int32_t color1 = get_cga_color(color, x & 1);
      int32_t color2 = get_cga_color(color, ~x & 1);

      // Paint line
      for (int yr = y1; yr <= y2; yr++, ptr += screen_w)
         *ptr = yr & 1 ? color2 : color1;

      // Done
      return;
   }

   // Draw all pixels inside the line
   for (int yr = y1; yr <= y2; yr++, ptr += screen_w)
      *ptr = color;
}

//***************************************************************************
// draw_rectangle
// Draws an empty rectangle on screen with the specified color
//---------------------------------------------------------------------------
// param x1: left coordinate (inclusive)
// param y1: top coordinate (inclusive)
// param x2: right coordinate (inclusive)
// param y2: bottom coordinate (inclusive)
// param color: color to draw with
//***************************************************************************

void draw_rectangle(int x1, int y1, int x2, int y2, uint32_t color) {
   // There's not much else to it...
   draw_hline(x1, y1, x2, color);
   draw_hline(x1, y2, x2, color);
   draw_vline(x1, y1, y2, color);
   draw_vline(x2, y1, y2, color);
}

//***************************************************************************
// fill_rectangle
// Draws a filled rectangle on screen with the specified color
//---------------------------------------------------------------------------
// param x1: left coordinate (inclusive)
// param y1: top coordinate (inclusive)
// param x2: right coordinate (inclusive)
// param y2: bottom coordinate (inclusive)
// param color: color to draw with
//***************************************************************************

void fill_rectangle(int x1, int y1, int x2, int y2, uint32_t color) {
   // Clamp coordinates so they lie within screen boundaries
   if (x1 < 0) x1 = 0;
   if (y1 < 0) y1 = 0;
   if (x2 >= screen_w) x2 = screen_w - 1;
   if (y2 >= screen_h) y2 = screen_h - 1;

   // Not visible?
   if (x1 > x2 || y1 > y2)
      return;

   // Determine position of the first pixel in the framebuffer
   // Also determine distance between lines
   uint32_t *ptr = lines[y1] + x1;
   unsigned stride = screen_w - (x2 - x1 + 1);

   // CGA mode? (make sure to use the proper colors then)
   if (settings.cga_mode) {
      // Determine the two colors making up the pattern
      uint32_t color1 = get_cga_color(color, 0);
      uint32_t color2 = get_cga_color(color, 1);

      // Fill the rectangle using a checkerboard pattern with those colors
      for (int yr = y1; yr <= y2; yr++, ptr += stride)
      for (int xr = x1; xr <= x2; xr++)
         *ptr++ = ((xr ^ yr) & 1) ? color2 : color1;

      // Done
      return;
   }

   // Go through all pixels inside the rectangle and overdraw them with the
   // specified color (how else do you want me to comment this without it
   // looking cluttered with the comments?)
   for (int yr = y1; yr <= y2; yr++, ptr += stride)
   for (int xr = x1; xr <= x2; xr++)
      *ptr++ = color;
}

//***************************************************************************
// draw_sprite
// Draws a sprite in the framebuffer
//---------------------------------------------------------------------------
// param spr: pointer to sprite
// param x: X coordinate of sprite
// param y: Y coordinate of sprite
// param flags: FXs, etc.
//***************************************************************************

void draw_sprite(const Sprite *spr, int x, int y, unsigned flags) {
   // Don't draw things this way in audiovideo mode
   if (settings.audiovideo)
      return;

   // Gotcha...
   if (spr == NULL)
      return;

   // Retrieve sprite position and dimensions
   int start_x = x;
   int start_y = y;
   int width = spr->width;
   int height = spr->height;

   // Check that the sprite is visible for starters...
   if (start_x > screen_w) return;
   if (start_y > screen_h) return;
   if (start_x + width < 0) return;
   if (start_y + height < 0) return;

   // Retrieve where the sprite bitmap starts usually
   const uint32_t *src = spr->data;
   int src_stride = 0;
   int pixel_offset = 1;

   // Clamp against the left border?
   if (start_x < 0) {
      width += start_x;
      src -= flags & SPR_HFLIP ? -start_x : start_x;
      src_stride += flags & SPR_HFLIP ? start_x : -start_x;
      start_x = 0;
   }

   // Clamp against the right border?
   int right_side = start_x + width;
   if (right_side > screen_w) {
      unsigned diff = right_side - screen_w;
      src_stride += flags & SPR_HFLIP ? -diff : diff;
      width = screen_w - start_x;
   }

   // Clamp against the top border?
   if (start_y < 0) {
      height += start_y;
      src += spr->width * (flags & SPR_VFLIP ? start_y : -start_y);
      start_y = 0;
   }

   // Clamp against the bottom border?
   int bottom_side = start_y + height;
   if (bottom_side > screen_h)
      height = screen_h - start_y;

   // Flip vertically?
   if (flags & SPR_VFLIP) {
      src += spr->width * (spr->height - 1);
      src_stride -= spr->width * 2;
   }

   // Flip horizontally?
   if (flags & SPR_HFLIP) {
      src += spr->width - 1;
      src_stride += spr->width * 2;
      pixel_offset = -1;
   }

   // Determine where to start drawing in the framebuffer
   // Also determine the distance between each line
   uint32_t *dest = lines[start_y] + start_x;
   unsigned dest_stride = screen_w - width;

   // If sprite is fully opaque, use a different loop which is much faster
   // since it just copies the pixels directly without any sort of check.
   if (spr->opaque) {
      for (int yr = 0; yr < height; yr++) {
         // Draw all pixels in this line
         for (int xr = 0; xr < width; xr++) {
            *dest++ = *src;
            src += pixel_offset;
         }

         // Go for next line
         dest += dest_stride;
         src += src_stride;
      }
   }

   // Even if the sprite has transparency, if it's full translucency (i.e.
   // invisible pixels, not partial translucency), we could make the checks
   // simpler and speed up things a bit. Also note that the check we do in
   // this loop could be computed as a sign check, optimizing more.
   else if (!spr->translucent) {
      for (int yr = 0; yr < height; yr++) {
         // Draw all pixels in this line
         for (int xr = 0; xr < width; xr++) {
            uint32_t color = *src;
            src += pixel_offset;
            if (color >= 0x80000000)
               *dest++ = color;
            else
               dest++;
         }

         // Go for next line
         dest += dest_stride;
         src += src_stride;
      }
   }

   // Sprite with some transparency inside, go with the normal loop
   else {
      for (int yr = 0; yr < height; yr++) {
         // Draw all pixels in this line
         for (int xr = 0; xr < width; xr++) {
            uint32_t color = *src;
            src += pixel_offset;

            // Opaque?
            if (color >= 0xFF000000)
               *dest++ = color;

            // Transparent?
            else if (color <= 0x00FFFFFF)
               dest++;

            // Translucent? (to-do: profile and optimize?)
            // To-do: find out why it isn't turning into SSE code... (other
            // pieces of code are being handled as SSE, yes, even some
            // handling blending such as dim_screen)
            else {
               // Get color to blend width
               uint32_t other = *dest;

               // Calculate the alpha values for both the source (sprite) and
               // destination (framebuffer). The destination alpha is the
               // exact opposite of the source alpha.
               uint8_t as = color >> 25;
               uint8_t ad = 0x7F - as;

               // Calculate the blending values for each RGB component of
               // both source and destination.
               uint32_t rs = (color & 0xFF0000) * as & 0xFF000000;
               uint32_t gs = (color & 0x00FF00) * as & 0x00FF0000;
               uint32_t bs = (color & 0x0000FF) * as & 0x0000FF00;
               uint32_t rd = (other & 0xFF0000) * ad & 0xFF000000;
               uint32_t gd = (other & 0x00FF00) * ad & 0x00FF0000;
               uint32_t bd = (other & 0x0000FF) * ad & 0x0000FF00;

               // Put everything together and write it to the framebuffer
               *dest++ = (rs + gs + bs + rd + gd + bd) >> 7;
            }
         }

         // Go for next line
         dest += dest_stride;
         src += src_stride;
      }
   }
}

//***************************************************************************
// create_sprite
// Creates a new sprite object with the specified dimensions. The contents of
// the sprite bitmap are undefined.
//---------------------------------------------------------------------------
// param width: width in pixels
// param height: height in pixels
// return: pointer to new sprite
//***************************************************************************

Sprite *create_sprite(uint16_t width, uint16_t height) {
   // Allocate memory for the structure
   Sprite *spr = (Sprite *) malloc(sizeof(Sprite));
   if (spr == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Initialize structure
   spr->width = width;
   spr->height = height;
   spr->data = NULL;
   spr->opaque = 0;
   spr->translucent = 1;
   spr->name = NULL;
   spr->link = NULL;

   // Determine how many bytes the bitmap will take up
   // The check is to ensure there is no overflow. This is only an issue on
   // platforms where size_t is 32-bit (and we don't support platforms with
   // smaller sizes of size_t), so we slap a sizeof check for it too, so
   // compilers can happily remove the check when size_t is 64-bit.
   size_t size = spr->width * spr->height;
   if (sizeof(size_t) <= 4 && size >= SIZE_MAX / sizeof(uint32_t))
      abort_program(ERR_NOMEMORY, NULL);
   size *= sizeof(uint32_t);

   // Allocate data for the bitmap
   spr->data = (uint32_t *) malloc(size);
   if (spr->data == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Return new sprite object
   return spr;
}

//***************************************************************************
// load_sprite
// Creates a new sprite and fills its contents with data from a PNG file.
//---------------------------------------------------------------------------
// param filename: name of file to load from
// return: pointer to sprite, or NULL if not found or invalid
//---------------------------------------------------------------------------
// Note: if you don't understand some of the PNG comments, you probably may
// want to take a look at the libpng documentation. Yeah, it's all libpng
// specific stuff.
//***************************************************************************

Sprite *load_sprite(const char *filename) {
   // Open sprite file
   File *file = open_file(filename, FILE_READ);
   if (file == NULL)
      return NULL;

   // Create PNG reading structure
   png_structp png_ptr = png_create_read_struct
      (PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
   if (png_ptr == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Create PNG information structure
   png_infop info_ptr = png_create_info_struct(png_ptr);
   if (info_ptr == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Since libpng uses setjmp for error handling, set up the error point
   // here. To-do: find out exactly what happened so we can just use
   // abort_program with the appropriate error code instead.
   if (setjmp(png_jmpbuf(png_ptr))) {
      png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
      close_file(file);
      return NULL;
   }

   // Set up I/O functions
   png_set_read_fn(png_ptr, file, read_png_callback);

   // Tell libpng to reject files larger than the sprites we support
   png_set_user_limits(png_ptr, 0xFFFF, 0xFFFF);

   // Read bitmap data!
   png_read_png(png_ptr, info_ptr,
      PNG_TRANSFORM_STRIP_16 | PNG_TRANSFORM_PACKING |
      PNG_TRANSFORM_EXPAND | PNG_TRANSFORM_SHIFT |
      PNG_TRANSFORM_GRAY_TO_RGB,
      NULL);

   // Get pointers to each line in the bitmap
   png_bytepp rows = png_get_rows(png_ptr, info_ptr);

   // Get the sprite dimensions. We have told libpng to reject sprites larger
   // than 65535×65535, so yes, the dimensions should fit in 16-bit (this is
   // the limit in the Sprite structure anyway).
   uint16_t width = png_get_image_width(png_ptr, info_ptr);
   uint16_t height = png_get_image_height(png_ptr, info_ptr);
   int type = png_get_color_type(png_ptr, info_ptr);

   // Create a sprite with the specified size
   Sprite *sprite = create_sprite(width, height);

   // Read all pixels from the PNG image and put them into the sprite
   uint32_t *dest = sprite->data;
   switch (type) {
      // Without alpha channel?
      case PNG_COLOR_TYPE_RGB:
         sprite->opaque = 1;
         sprite->translucent = 0;
         for (unsigned y = 0; y < height; y++) {
            const uint8_t *src = rows[y];
            for (unsigned x = 0; x < width; x++) {
               *dest++ = 0xFF000000 | src[0] << 16 |
                         src[1] << 8 | src[2];
               src += 3;
            }
         }
         break;

      // With alpha channel?
      case PNG_COLOR_TYPE_RGBA:
         sprite->opaque = 1;
         sprite->translucent = 0;
         for (unsigned y = 0; y < height; y++) {
            const uint8_t *src = rows[y];
            for (unsigned x = 0; x < width; x++) {
               if (src[3] != 0xFF) sprite->opaque = 0;
               if (src[3] != 0x00) sprite->translucent = 1;
               *dest++ = src[3] << 24 | src[0] << 16 |
                         src[1] << 8 | src[2];
               src += 4;
            }
         }
         if (sprite->opaque)
            sprite->translucent = 0;
         break;

      // Huh...
      default:
         break;
   }

   // CGA mode? (convert sprite)
   if (settings.cga_mode) {
      // Change all colors to use the black-cyan-magenta-white palette
      uint32_t *dest = sprite->data;
      for (unsigned y = 0; y < height; y++)
      for (unsigned x = 0; x < width; x++) {
         *dest = get_cga_color(*dest, (x^y) & 1);
         dest++;
      }

      // No translucency supported, ever
      if (sprite->translucent) {
         sprite->translucent = 0;
         sprite->opaque = 0;
      }
   }

   // Return new sprite
   png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
   close_file(file);
   return sprite;
}

//***************************************************************************
// destroy_sprite
// Destroys a sprite object.
//---------------------------------------------------------------------------
// param spr: pointer to sprite (becomes invalid after call)
//***************************************************************************

void destroy_sprite(Sprite *spr) {
   // Okaaay...
   if (spr == NULL)
      return;

   // Deallocate all memory taken up by this sprite
   if (spr->data)
      free(spr->data);
   free(spr);
}

//***************************************************************************
// load_graphics_set
// Loads an entire graphics set into memory.
//---------------------------------------------------------------------------
// param dir: name of directory to load from
// return: pointer to graphics set
//---------------------------------------------------------------------------
// To-do: MASSIVE error checking of the animation scripts. What's there now
// will handle malformed scripts, but not in a proper way so expect wacky
// stuff to happen (though it shouldn't crash... just misbehave).
//***************************************************************************

GraphicsSet *load_graphics_set(const char *dir) {
   // A quick buffer to store filenames...
   // To-do: make it possible to use arbitrary lengths
   static char filename[0x100];

   // Allocate memory for the graphics set structure
   GraphicsSet *gfxset = (GraphicsSet *) malloc(sizeof(GraphicsSet));
   if (gfxset == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Initialize structure
   gfxset->sprites = NULL;
   gfxset->frames = NULL;
   gfxset->spr_names = NULL;

   // Get a list of the files in the directory
   // To-do: CLEAN UP THE API SO WE DON'T MAKE THIS MESS -_-'
   char *dirtemp = (char*) malloc(strlen(dir) + 7 + 1);
   if (dirtemp == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   sprintf(dirtemp, "gamefs/%s", dir);
   gfxset->spr_names = PHYSFS_enumerateFiles(dirtemp);
   free(dirtemp);

   // Go through all files to see which ones are valid sprites
   for (char **curr_file = gfxset->spr_names; *curr_file != NULL;
   curr_file++) {
      // To make the code more clear...
      char *name = *curr_file;

      // Check if it's possible to make a valid sprite name out of this
      // filename, and don't even bother with this file if we can't.
      if (validate_sprite_name(name))
         continue;

      // Try to load the sprite
      sprintf(filename, "%s/%s.png", dir, name);
      Sprite *spr = load_sprite(filename);
      if (spr == NULL)
         continue;

      // Put sprite into the list
      spr->name = name;
      spr->qhash = compute_quickhash(name);
      spr->link = gfxset->sprites;
      gfxset->sprites = spr;
   }

   // Check the animations file
   sprintf(filename, "%s/animation", dir);
   File *animfile = open_file(filename, FILE_READ);
   if (animfile != NULL) {
      // Name of the current animation
      // Only set for the first frame, then becomes NULL so the following
      // frames in the animation are stored nameless.
      char *anim_name = NULL;

      //
      AnimFrame *last_frame = NULL;    // Last frame we processed
      AnimFrame *loop_point = NULL;    // Where the animation will loop

      // Go through all lines in the file
      for (;;) {
         // Get next line
         char *line = read_line(animfile);
         if (line == NULL) break;

         // Get arguments
         // To-do: bail out in case of error
         Args *args = parse_args(line);
         if (args == NULL) {
            free(line);
            continue;
         }

         // Blank line?
         if (args->count == 0) {
            free_args(args);
            free(line);
            continue;
         }

         // Bullet? (animation frame)
         if (!strcmp(args->list[0], "-") || !strcmp(args->list[0], ">")) {
            // Make sure there are enough parameters
            // To-do: bail out in case of error
            if (args->count < 5) {
               free_args(args);
               free(line);
               continue;
            }

            // Allocate memory for frame
            AnimFrame *frame = (AnimFrame *) malloc(sizeof(AnimFrame));
            if (frame == NULL)
               abort_program(ERR_NOMEMORY, NULL);

            // Loop point?
            if (*args->list[0] == '>' || loop_point == NULL)
               loop_point = frame;

            // Update the last frame
            if (last_frame != NULL)
               last_frame->next = frame;

            // Initialize structure
            frame->sprite = NULL;
            frame->x_offset = 0;
            frame->y_offset = 0;
            frame->duration = 0;
            frame->flags = SPR_NOFLIP;
            frame->next = loop_point;
            frame->name = anim_name;
            frame->link = NULL;

            frame->sfx = SFX_NONE;
            frame->sfx_x = 0;
            frame->sfx_y = 0;
            frame->clue = 0;

            // Get name of the sprite
            // To-do: check that the sprite exists
            frame->sprite = get_sprite(gfxset, args->list[1]);

            // Get X and Y offsets
            // To-do: check that the parameters are valid
            frame->x_offset = atoi(args->list[2]);
            frame->y_offset = atoi(args->list[3]);
            frame->x_attachment = 0;
            frame->y_attachment = 0;

            // Get duration in frames
            // To-do: check that the parameter is valid
            if (!strcmp(args->list[4], "forever")) {
               loop_point = frame;
               frame->duration = UINT_MAX;
               frame->next = frame;
            } else
               frame->duration = atoi(args->list[4]);

            // Check if there are any more parameters
            for (size_t i = 5; i < args->count; i++) {
               // Flipping?
               if (!strcmp(args->list[i], "hflip"))
                  frame->flags = SPR_HFLIP;
               else if (!strcmp(args->list[i], "vflip"))
                  frame->flags = SPR_VFLIP;
               else if (!strcmp(args->list[i], "hvflip"))
                  frame->flags = SPR_HVFLIP;

               // Attachment offset?
               else if (!strcmp(args->list[i], "attach")) {
                  // Make sure we have two more parameters
                  if (i + 2 >= args->count)
                     break;

                  // Retrieve attachment offset
                  // To-do: check that the parameters are valid
                  frame->x_attachment = atoi(args->list[i+1]);
                  frame->y_attachment = atoi(args->list[i+2]);

                  // Skip over parameters
                  i += 2;
               }

               // Play sound effect?
               else if (!strcmp(args->list[i], "play") ||
               !strcmp(args->list[i], "clue")) {
                  // Make sure we have one more parameter
                  if (i + 1 >= args->count)
                     break;

                  // Retrieve sound effect info
                  frame->sfx = get_sfx_by_name(args->list[i+1]);
                  frame->clue = (args->list[i][0] == 'c') ? 1 : 0;

                  // Skip over parameters
                  i += 1;
               }

               // Play sound effect with offset?
               else if (!strcmp(args->list[i], "playex") ||
               !strcmp(args->list[i], "clueex")) {
                  // Make sure we have three more parameters
                  if (i + 3 >= args->count)
                     break;

                  // Retrieve sound effect info
                  frame->sfx = get_sfx_by_name(args->list[i+1]);
                  frame->sfx_x = atoi(args->list[i+2]);
                  frame->sfx_y = atoi(args->list[i+2]);
                  frame->clue = (args->list[i][0] == 'c') ? 1 : 0;

                  // Skip over parameters
                  i += 1;
               }

               // Unknown keyword, may want to break here as any further
               // parameters may be unparseable
               else break;
            }

            // Put animation frame into the list
            frame->link = gfxset->frames;
            gfxset->frames = frame;

            // OK, name assigned and pointer stored, let's make the following
            // frames nameless since they aren't the start of an animation.
            anim_name = NULL;

            // Now *this* is the last frame
            last_frame = frame;
         }

         // Valid name? (animation start)
         else if (is_valid_id(args->list[0])) {
            // Ugh, unused animation name
            if (anim_name != NULL)
               free(anim_name);

            // Make a permanent copy of the name
            size_t len = strlen(args->list[0]);
            anim_name = (char *) malloc(len+1);
            if (anim_name == NULL)
               abort_program(ERR_NOMEMORY, NULL);
            strcpy(anim_name, args->list[0]);

            // Reset animation progression
            last_frame = NULL;
            loop_point = NULL;
         }

         // Done with the line
         free(line);
      }

      // Ugh, unused resources
      if (anim_name) free(anim_name);

      // Done with the file
      close_file(animfile);
   }

   // Return graphics set information
   return gfxset;
}

//***************************************************************************
// validate_sprite_name [internal]
// Attempts to make a valid sprite name out of a filename. The string may
// be modified in the process. On success, the original string will be turned
// into the sprite name; on failure, the original string is undefined.
//---------------------------------------------------------------------------
// param name: pointer to string to check
// return: zero on success, non-zero on failure
//***************************************************************************

static int validate_sprite_name(char *name) {
   // Make entire string lowercase
   for (char *ptr = name; *ptr; ptr++)
      *ptr = tolower(*ptr);

   // Check if the ".png" extension is present (otherwise don't even bother
   // with the file). Strip the extension if found.
   size_t length = strlen(name);
   if (length < 5)
      return -1;
   length -= 4;
   if (strcmp(&name[length], ".png") != 0)
      return -1;
   name[length] = '\0';

   // Name too long?
   if (length > MAX_ID_LEN)
      return -1;

   // Check that all the characters are valid
   for (; *name; name++) {
      if (isalnum(*name)) continue;
      if (*name == '_') continue;
      return -1;
   }

   // String is valid!
   return 0;
}

//***************************************************************************
// get_sprite
// Attemps to retrieve a sprite by its name from a graphics set.
//---------------------------------------------------------------------------
// param gfxset: pointer to graphics set
// param name: name of the sprite to look for
// return: pointer to sprite if found, NULL if not found
//***************************************************************************

Sprite *get_sprite(const GraphicsSet *gfxset, const char *name) {
   // Okaaay...
   if (gfxset == NULL || name == NULL || name[0] == '\0')
      return NULL;

   // Calculate quickhash of desired name
   uint8_t qhash = compute_quickhash(name);

   // Look through all sprites to see if there's a match
   Sprite *spr = gfxset->sprites;
   for (; spr != NULL; spr = spr->link) {
      if (qhash != spr->qhash)
         continue;
      if (strcmp(name, spr->name) == 0)
         return spr;
   }

   // Not found! Return a dummy sprite that's 0×0
   static Sprite dummy_sprite = { NULL, 0, 0, 1, 0, NULL, NULL, 0 };
   return &dummy_sprite;
   //return NULL;
}

//***************************************************************************
// get_anim
// Attemps to retrieve the first frame of an animation by its name from a
// graphics set.
//---------------------------------------------------------------------------
// param gfxset: pointer to graphics set
// param name: name of the animation to look for
// return: pointer to first frame if found, NULL if not found
//***************************************************************************

AnimFrame *get_anim(const GraphicsSet *gfxset, const char *name) {
   // Okaaay...
   if (gfxset == NULL)
      return NULL;

   // Look through all frames to see if there's a match
   AnimFrame *frame = gfxset->frames;
   while (frame != NULL) {
      if (frame->name != NULL)
         if (!strcmp(name, frame->name))
            return frame;
      frame = frame->link;
   }

   // Not found!
   return NULL;
}

//***************************************************************************
// destroy_graphics_set
// Destroys a graphics set (deallocates it from memory).
//---------------------------------------------------------------------------
// param gfxset: pointer to graphics set (becomes invalid after call)
//***************************************************************************

void destroy_graphics_set(GraphicsSet *gfxset) {
   // Okaaay...
   if (gfxset == NULL)
      return;

   // Destroy all loaded sprites
   Sprite *spr = gfxset->sprites;
   while (spr != NULL) {
      Sprite *next = spr->link;
      destroy_sprite(spr);
      spr = next;
   }

   // Destroy all animation frames
   AnimFrame *frame = gfxset->frames;
   while (frame != NULL) {
      AnimFrame *next = frame->link;
      if (frame->name) free(frame->name);
      free(frame);
      frame = next;
   }

   // Get rid of names list
   if (gfxset->spr_names != NULL)
      PHYSFS_freeList(gfxset->spr_names);

   // Deallocate structure
   free(gfxset);
}

//***************************************************************************
// compute_quickhash [internal]
// Computers the quickhash of a string.
//---------------------------------------------------------------------------
// param str: pointer to string
// return: quickhash of string
//***************************************************************************

static uint8_t compute_quickhash(const char *str) {
   // Just add every character...
   uint8_t qhash = 0;
   while (*str != '\0') {
      qhash += *str;
      str++;
   }

   // Done
   return qhash;
}

//***************************************************************************
// dim_screen
// Darkens the screen based on the given alpha value (0x00 = fully black,
// 0xFF = fully visible). Mainly meant for fading, though it can be used with
// other purposes if desired.
//---------------------------------------------------------------------------
// param alpha: how much to show
//***************************************************************************

void dim_screen(uint8_t alpha) {
   // For the extreme values (fully visible and fully black) don't bother
   // doing any blending, just either return or fill the screen with black
   // (respectively). Also ensures 100% correct colors.
   if (alpha == 0xFF)
      return;
   if (alpha == 0x00) {
      clear_screen(0x000000);
      return;
   }

   // Don't mess with audiovideo output!
   if (settings.audiovideo)
      return;

   // CGA can't do fancy blending, so let's fake up something quickly :P
   if (settings.cga_mode) {
      // Determine what pattern to use depending on the alpha "blending"
      uint8_t pattern = 0x00;
      if (alpha < 0x20) pattern |= 0x01;
      if (alpha < 0x60) pattern |= 0x02;
      if (alpha < 0xA0) pattern |= 0x04;
      if (alpha < 0xE0) pattern |= 0x08;

      // Apply pattern on the entire framebuffer
      // We need to count X and Y separately because of the pattern
      // WARNING: this assumes the virtual resolution is a multiple of two
      // TO-DO: find a way to fix that? (not like we're ever going to support
      // uneven resolutions, but still)
      uint32_t *ptr = framebuffer;
      for (int32_t y = 0; y < screen_h; y++) {
         // Determine this line's pattern
         unsigned black1, black2;
         if (y & 1) {
            black1 = pattern & 0x08;
            black2 = pattern & 0x02;
         } else {
            black1 = pattern & 0x01;
            black2 = pattern & 0x04;
         }

         // Apply the pattern over the entire line
         for (int32_t x = 0; x < screen_w; x += 2) {
            *ptr = black1 ? 0xFF000000 : *ptr; ptr++;
            *ptr = black2 ? 0xFF000000 : *ptr; ptr++;
         }
      }

      // Done
      return;
   }

   // Perform blending with black through all the bitmap
   uint32_t *ptr = framebuffer;
   for (unsigned i = screen_w * screen_h; i > 0; i--) {
      uint32_t color = *ptr;
      uint32_t r = (color & 0xFF0000) * alpha & 0xFF000000;
      uint32_t g = (color & 0x00FF00) * alpha & 0x00FF0000;
      uint32_t b = (color & 0x0000FF) * alpha & 0x0000FF00;
      *ptr++ = (r | g | b) >> 8;
   }
}

//***************************************************************************
// grayscale_filter
// Turns the entire framebuffer into grayscale. (obviously whatever is drawn
// after this is not affected, e.g. the text globes in cutscenes)
//***************************************************************************

void grayscale_filter(void) {
   // Scan every pixel
   size_t num_pixels = screen_w * screen_h;
   for (unsigned i = 0; i < num_pixels; i++) {
      // Get color components
      uint32_t color = framebuffer[i];
      unsigned r = color >> 16 & 0xFF;
      unsigned g = color >> 8 & 0xFF;
      unsigned b = color & 0xFF;

      // Multiply each component by its weight in luminosity
      // Note that this is measured in 1000ths (i.e. 1000 = 1.0)
      // <Sik> Are these the correct values? Blue looks too dark for me
      r *= 299;
      g *= 587;
      b *= 114;

      // Get grayscale value
      uint8_t lum = (r + g + b) / 1000;

      /*
      // Huuuuuuuuh
      float rf = r / 255.0;
      float gf = g / 255.0;
      float bf = b / 255.0;
      float yf = sqrt(rf * rf * 0.25 +
                      gf * gf * 0.60 +
                      bf * bf * 0.15);

      uint8_t lum = yf * 0xFF + 0.5;
      */

      // Store grayscale color
      framebuffer[i] = lum << 16 | lum << 8 | lum;
   }
}

//***************************************************************************
// zoom_screen
// Zooms into the screen. Scales up 2x the center of the screen (everything
// drawn after this is still rendered at normal resolution).
//***************************************************************************

void zoom_screen(void) {
   for (int32_t y1 = 0, y2 = screen_cy >> 1, y3 = screen_h - 1, y4 = y3 - y2;
   y1 < screen_cy; y1 += 2, y2++, y3 -= 2, y4--)
   for (int32_t x1 = 0, x2 = screen_cx >> 1, x3 = screen_w - 1, x4 = x3 - x2;
   x1 < screen_cx; x1 += 2, x2++, x3 -= 2, x4--) {
      // Scale top left quadrant
      lines[y1][x1] = lines[y2][x2];
      lines[y1][x1+1] = lines[y2][x2];
      lines[y1+1][x1] = lines[y2][x2];
      lines[y1+1][x1+1] = lines[y2][x2];

      // Scale top right quadrant
      lines[y1][x3] = lines[y2][x4];
      lines[y1][x3-1] = lines[y2][x4];
      lines[y1+1][x3] = lines[y2][x4];
      lines[y1+1][x3-1] = lines[y2][x4];

      // Scale bottom left quadrant
      lines[y3][x1] = lines[y4][x2];
      lines[y3][x1+1] = lines[y4][x2];
      lines[y3-1][x1] = lines[y4][x2];
      lines[y3-1][x1+1] = lines[y4][x2];

      // Scale bottom right quadrant
      lines[y3][x3] = lines[y4][x4];
      lines[y3][x3-1] = lines[y4][x4];
      lines[y3-1][x3] = lines[y4][x4];
      lines[y3-1][x3-1] = lines[y4][x4];
   }
}

//***************************************************************************
// update_fading
// Causes the fading to animate. Call this once per logical frame. This is
// done by the main loop.
//***************************************************************************

void update_fading(void) {
   // Determine what to do based on the current fading mode
   switch (fade.mode) {
      // Idling, do nothing
      case FADE_IDLE:
         break;

      // Fading on
      case FADE_ON:
         fade.pos += FADING_SPEED;
         if (fade.pos > 0xFF) {
            fade.pos = 0xFF;
            fade.mode = FADE_IDLE;
         }
         break;

      // Fading off
      case FADE_OFF:
         fade.pos -= FADING_SPEED;
         if (fade.pos < 0) {
            fade.pos = 0;
            fade.mode = FADE_IDLE;
            if (fade.switch_to != NUM_GAMEMODES)
               switch_game_mode(fade.switch_to);
         }
         break;
   }
}

//***************************************************************************
// draw_fading
// Modifies the framebuffer to apply the current fading. This is called by
// the main loop.
//***************************************************************************

void draw_fading(void) {
   dim_screen(fade.pos);
}

//***************************************************************************
// fade_on
// Starts fading from black.
//***************************************************************************

void fade_on(void) {
   fade.mode = FADE_ON;
   fade.switch_to = NUM_GAMEMODES;
}

//***************************************************************************
// fade_off
// Starts fading towards black.
//***************************************************************************

void fade_off(void) {
   fade.mode = FADE_OFF;
   fade.switch_to = NUM_GAMEMODES;
}

//***************************************************************************
// fade_off
// Same as fade_off, but switches to a new game mode when the fading is done.
//***************************************************************************

void fade_off_and_switch(GameMode mode) {
   fade.mode = FADE_OFF;
   fade.switch_to = mode;
}

//***************************************************************************
// is_fading
// Checks if a fading is going on currently.
//---------------------------------------------------------------------------
// return: non-zero if fading, zero otherwise
//***************************************************************************

int is_fading(void) {
   return fade.mode != FADE_IDLE;
}

//***************************************************************************
// take_screenshot
// Takes a screenshot and saves it to a file.
//***************************************************************************

void take_screenshot(void) {
   // Get current date and time (we'll use them to generate the filename)
   time_t nowstamp = time(NULL);
   struct tm *now = localtime(&nowstamp);
   if (now == NULL)
      abort_program(ERR_UNKNOWN, NULL);

   // Generate filename
   // The modulo operations in the time data is just me being paranoid, if
   // somebody can cheat these values then you probably have much bigger
   // issues than a potential buffer overflow here...
   char filename[0x40];
   snprintf(filename, sizeof(filename),
      "screenshots/soledad_%04d%02d%02d%02d%02d%02d.png",
      (now->tm_year + 1900) % 10000, (now->tm_mon + 1) % 100,
      now->tm_mday % 100, now->tm_hour % 100, now->tm_min % 100,
      now->tm_sec % 100);
   filename[sizeof(filename)-1] = '\0';

   // Now generate the PNG file
   save_png(filename);
}

//***************************************************************************
// save_png
// Saves the contents of the framebuffer to the specified file
//---------------------------------------------------------------------------
// param filename: name of file to write
//***************************************************************************

void save_png(const char *filename) {
   // Allocate memory for the buffer with the bitmap data
   // No, we can't use the framebuffer as-is since libpng expects something
   // different... (also we want to get rid of the alpha channel)
   size_t size = screen_w * screen_h;
   uint8_t *buffer = (uint8_t *) malloc(size * 3);
   if (buffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Fill temporary buffer with our data
   uint32_t *src = framebuffer;
   uint8_t *dest = buffer;
   for (unsigned i = 0; i < size; i++) {
      uint32_t color = *src++;
      *dest++ = color >> 16;
      *dest++ = color >> 8;
      *dest++ = color;
   }

   // And now allocate memory for a row table...
   uint8_t **rows = (uint8_t **) malloc(sizeof(uint8_t *) * screen_h);
   if (rows == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Fill in row table
   for (int i = 0; i < screen_h ; i++)
      rows[i] = &buffer[i * screen_w * 3];

   // Open file
   File *file = open_save_file(filename, FILE_WRITE);
   if (file == NULL)
      abort_program(ERR_UNKNOWN, NULL);

   // Create PNG writing structure
   png_structp png_ptr = png_create_write_struct
      (PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
   if (png_ptr == NULL) {
      close_file(file);
      free(rows);
      free(buffer);
      abort_program(ERR_UNKNOWN, NULL);
   }

   // Create PNG information structure
   png_infop info_ptr = png_create_info_struct(png_ptr);
   if (info_ptr == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Since libpng uses setjmp for error handling, set up the error point
   // here. To-do: find out exactly what happened so we can just use
   // abort_program with the appropriate error code instead.
   if (setjmp(png_jmpbuf(png_ptr))) {
      png_destroy_write_struct(&png_ptr, &info_ptr);
      close_file(file);
      free(rows);
      free(buffer);
      abort_program(ERR_UNKNOWN, NULL);
   }

   // Set up I/O functions
   png_set_write_fn(png_ptr, file, write_png_callback, flush_png_callback);

   // Set up PNG properties
   png_set_IHDR(png_ptr, info_ptr, screen_w, screen_h, 8,
      PNG_COLOR_TYPE_RGB, PNG_INTERLACE_NONE,
      PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

   // Tell libpng where to find the bitmap data
   png_set_rows(png_ptr, info_ptr, rows);

   // Write it!
   png_write_png(png_ptr, info_ptr, PNG_TRANSFORM_IDENTITY, NULL);

   // Done!
   png_destroy_write_struct(&png_ptr, &info_ptr);
   close_file(file);
   free(rows);
   free(buffer);
}

//***************************************************************************
// read_png_callback [internal]
// Callback used for reading from a PNG file
//---------------------------------------------------------------------------
// param png_ptr: pointer to PNG object
// param buffer: where to store the data
// param size: amount of bytes to read
//***************************************************************************

static void read_png_callback(png_structp png_ptr, png_bytep buffer,
png_size_t size) {
   // Attempt to read from the file
   if (read_file(png_get_io_ptr(png_ptr), buffer, size))
      png_error(png_ptr, "");
}

//***************************************************************************
// write_png_callback [internal]
// Callback used for writing into a PNG file
//---------------------------------------------------------------------------
// param png_ptr: pointer to PNG object
// param buffer: data to be written
// param size: amount of bytes to write
//***************************************************************************

static void write_png_callback(png_structp png_ptr, png_bytep buffer,
png_size_t size) {
   // Attempt to write into the file
   if (write_file(png_get_io_ptr(png_ptr), buffer, size))
      png_error(png_ptr, "");
}

//***************************************************************************
// flush_png_callback [internal]
// Placeholder callback that libpng wants for flushing files...
//---------------------------------------------------------------------------
// param png_ptr: pointer to PNG object
//***************************************************************************

static void flush_png_callback(png_structp png_ptr) {
   // Shut up, compiler
   (void) png_ptr;
}

//***************************************************************************
// get_cga_color
// Converts a full 32-bit ARGB color into something to be used for the CGA
// mode. Takes checkerboard patterns into account.
//---------------------------------------------------------------------------
// param original: original color
// param id: which of the two checkerboard colors to use (0 or 1)
// return: CGA-like color
//***************************************************************************

uint32_t get_cga_color(uint32_t original, unsigned id) {
   unsigned index = (original & 0xC00000) >> 18 |
                    (original & 0x00C000) >> 12 |
                    (original & 0x0000C0) >> 6;
   return (original & 0x80000000) | cga_table[id][index];
}

//***************************************************************************
// set_anim
// Changes an animation to the specified one.
//---------------------------------------------------------------------------
// param state: pointer to animation state
// param frame: pointer to frame to play
//***************************************************************************

void set_anim(AnimState *state, const AnimFrame *frame) {
   // Set the new frame
   state->frame = frame;
   if (frame == NULL)
      return;

   // Start delay for this frame
   state->frame_time = frame->duration;
}

//***************************************************************************
// update_anim
// Updates an animation. Call once per frame to play.
//---------------------------------------------------------------------------
// param state: pointer to animation state
//***************************************************************************

void update_anim(AnimState *state) {
   // Not showing anything?
   if (state->frame == NULL)
      return;

   // Delay until next frame
   state->frame_time--;
   if (state->frame_time)
      return;

   // Change to next frame
   set_anim(state, state->frame->next);
}
