//***************************************************************************
// "video.h"
// Video definitions.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

#ifndef VIDEO_H
#define VIDEO_H

// Required headers
#include <stdint.h>
#include "main.h"

// Framebuffer dimensions
// Note that these are the dimensions of the virtual resolution (i.e.
// 320×240, 384×240 and 384×216), not the real screen resolution. All drawing
// is performed on the virtual resolution anyways, so that's the only thing
// the code should care about...
extern int screen_w;        // Width
extern int screen_h;        // Height
extern int screen_cx;       // Horizontal center
extern int screen_cy;       // Vertical center

// Where we keep a list of valid resolutions
// To-do: come up with something smarter?
#define NUM_RESOLUTIONS (3*10)
typedef struct {
   int width;
   int height;
} Resolution;
extern Resolution resolutions[NUM_RESOLUTIONS];

// Definition of a sprite
// This is a game, do I need to explain what's a sprite...?
typedef struct Sprite {
   // Sprite properties
   uint32_t *data;               // Pointer to bitmap contents
   uint16_t width;               // Width in pixels
   uint16_t height;              // Height in pixels
   unsigned opaque: 1;           // Set if sprite is fully opaque
   unsigned translucent: 1;      // Set if sprite has partial translucency

   // Graphics set info
   struct Sprite *link;          // Next sprite in the list
   char *name;                   // Name of sprite
   uint8_t qhash;                // Quickhash of name
} Sprite;

// Definition of an animation frame
typedef struct AnimFrame {
   // Frame properties
   Sprite *sprite;               // Sprite to show
   int32_t x_offset;             // Horizontal offset
   int32_t y_offset;             // Vertical offset
   int32_t x_attachment;         // Attachment horizontal offset
   int32_t y_attachment;         // Attachment vertical offset
   unsigned duration;            // How long to show
   unsigned flags;               // Sprite flags
   struct AnimFrame *next;       // Next frame to show

   // Sound stuff
   int sfx;                      // Sound effect to play
   int32_t sfx_x;                // Sound effect X offset
   int32_t sfx_y;                // Sound effect Y offset
   unsigned clue: 1;             // Set if SFX is for audiovideo only

   // Graphics set info
   char *name;                   // Name of the frame (if any)
   struct AnimFrame *link;       // Next sprite in the list
} AnimFrame;

// To hold the current animation state
// Used when we have animated things
typedef struct {
   const AnimFrame *frame;    // Current animation frame
   unsigned frame_time;       // Time remaining until next frame
} AnimState;

// Definition of a graphics set
// A graphics set contains all the sprites and animations of a given type
// (e.g. player graphics, enemy graphics, level graphics, etc.)
typedef struct {
   Sprite *sprites;              // Pointer to sprites
   AnimFrame *frames;            // Pointer to animation frames
   char **spr_names;             // List of all sprite names
} GraphicsSet;

// Sprite flipping flags
#define SPR_NOFLIP      0x00     // Draw normally
#define SPR_HFLIP       0x01     // Flip horizontally
#define SPR_VFLIP       0x02     // Flip vertically
#define SPR_HVFLIP      0x03     // Flip both axes

// Function prototypes
void init_video(void);
void update_video(void);
void deinit_video(void);
void handle_gain_focus(void);
void handle_lose_focus(void);
void virtual_to_real_coord(int *, int *);
int is_fullscreen(void);
void toggle_fullscreen(void);
const char *get_window_title(void);
void set_window_title(const char *);
void clear_screen(uint32_t);
void draw_pixel(int, int, uint32_t);
void draw_hline(int, int, int, uint32_t);
void draw_vline(int, int, int, uint32_t);
void draw_rectangle(int, int, int, int, uint32_t);
void fill_rectangle(int, int, int, int, uint32_t);
void draw_sprite(const Sprite *, int, int, unsigned);
Sprite *create_sprite(uint16_t, uint16_t);
Sprite *load_sprite(const char *);
void destroy_sprite(Sprite *);
GraphicsSet *load_graphics_set(const char *);
Sprite *get_sprite(const GraphicsSet *, const char *);
AnimFrame *get_anim(const GraphicsSet *, const char *);
void destroy_graphics_set(GraphicsSet *);
void dim_screen(uint8_t);
void grayscale_filter(void);
void zoom_screen(void);
void update_fading(void);
void draw_fading(void);
void fade_on(void);
void fade_off(void);
void fade_off_and_switch(GameMode);
int is_fading(void);
void take_screenshot(void);
void save_png(const char *);
uint32_t get_cga_color(uint32_t, unsigned);
void set_anim(AnimState *, const AnimFrame *);
void update_anim(AnimState *);

#endif
