

#include "Brickmap.h"
#include "ShaderCore.h"

layout(lines)                           in;                                                     
layout(line_strip, max_vertices = 24)   out;                                                    
layout(std430) buffer                   BrickBuffer       { uint        brickBuffer[];       }; 
layout(std430) buffer                   BrickCountBuffer  { uint        brickCountBuffer[];  }; 
layout(std430) buffer                   VoxelBrickBuffer  { uvec2       voxelBrickBuffer[];  }; 
layout(std430) buffer                   VoxelCountBuffer  { uint        voxelCountBuffer[];  }; 
layout(std430) buffer                   VoxelConfigBuffer { VoxelConfig voxelConfigBuffer[]; }; 
flat in uint                            vInstanceId[];                                          


void DrawCell(in vec3 cellCoordinates, in float cellSize);







vec4 GridInverseTransform(in vec4 position)
{
    return vec4((position.xyz - position.w * voxelConfigBuffer[0].m_gridTransform.xyz) / voxelConfigBuffer[0].m_gridTransform.w, position.w);
}

_kernel


void DrawBricks()
{
    const uint brickIndex = uint(vInstanceId[0]),
               brickCount = voxelConfigBuffer[0].m_brickCount;
    if(brickCountBuffer[brickIndex] == 0)
        return; 
    const uvec3 brickCoordinates = To3D(brickIndex, brickCount);
    DrawCell(vec3(brickCoordinates), voxelConfigBuffer[0].m_brickSize);
}

_kernel


void DrawVoxels()
{
    uint voxelResolution, voxelOffset;
    const uvec2 voxelBrick = voxelBrickBuffer[vInstanceId[0]];
    const uint brickIndex = voxelBrick.x,
               indexInVoxel = voxelBrick.y,
               brickCount = voxelConfigBuffer[0].m_brickCount;
    DecodeBrick(
        brickBuffer[brickIndex],
        voxelResolution,
        voxelOffset);
    if(voxelCountBuffer[voxelOffset + indexInVoxel] == 0)
        return; 
    const uint voxelCount = (1u << voxelResolution);
    const uvec3 brickCoordinates = To3D(brickIndex, brickCount) << voxelResolution,
                voxelCoordinates = brickCoordinates + To3D(indexInVoxel, voxelCount);
    DrawCell(vec3(voxelCoordinates), 1.0f / float(brickCount << voxelResolution));
}






void DrawCell(in vec3 cellCoordinates, in float cellSize)
{
    
    vec4 xOffset = ViewProjection * GridInverseTransform(vec4(cellSize, 0.0f, 0.0f, 0.0f)),
         yOffset = ViewProjection * GridInverseTransform(vec4(0.0f, cellSize, 0.0f, 0.0f)),
         zOffset = ViewProjection * GridInverseTransform(vec4(0.0f, 0.0f, cellSize, 0.0f)),
         cellPos = ViewProjection * GridInverseTransform(vec4(cellCoordinates * cellSize, 1.0f));

    
    {
        
        gl_Position = cellPos;
        EmitVertex();
        gl_Position += xOffset;
        EmitVertex();
        EndPrimitive();

        EmitVertex();
        gl_Position += zOffset;
        EmitVertex();
        EndPrimitive();

        EmitVertex();
        gl_Position -= xOffset;
        EmitVertex();
        EndPrimitive();

        EmitVertex();
        gl_Position -= zOffset;
        EmitVertex();
        EndPrimitive();

        
        gl_Position = cellPos + yOffset;
        EmitVertex();
        gl_Position += xOffset;
        EmitVertex();
        EndPrimitive();

        EmitVertex();
        gl_Position += zOffset;
        EmitVertex();
        EndPrimitive();

        EmitVertex();
        gl_Position -= xOffset;
        EmitVertex();
        EndPrimitive();

        EmitVertex();
        gl_Position -= zOffset;
        EmitVertex();
        EndPrimitive();

        
        gl_Position = cellPos;
        EmitVertex();
        gl_Position += yOffset;
        EmitVertex();
        EndPrimitive();

        gl_Position = cellPos + xOffset;
        EmitVertex();
        gl_Position += yOffset;
        EmitVertex();
        EndPrimitive();

        gl_Position = cellPos + zOffset;
        EmitVertex();
        gl_Position += yOffset;
        EmitVertex();
        EndPrimitive();

        gl_Position = cellPos + xOffset + zOffset;
        EmitVertex();
        gl_Position += yOffset;
        EmitVertex();
        EndPrimitive();
    }
}
