

#include "Bxdf.h"
#include "Raytrace.h"



#define DEBUG_NORMAL_BUFFER         0



#define DEBUG_DETAILS_BUFFER        1



#define DEBUG_CURVATURE_BUFFER      2



#define DEBUG_POSITION_BUFFER       3



#define DEBUG_FRESNEL_BUFFER        4



#define DEBUG_VELOCITY_BUFFER       5



#define DEBUG_ROUGHNESS_BUFFER      6



#define DEBUG_AMBIENT_OCCLUSION     7



#define DEBUG_SPECULAR_OCCLUSION    8



#define DEBUG_GLOBAL_ILLUMINATION   9



#define DEBUG_GLOSSY_REFLECTIONS    10

flat                 in vec3    vColor;         
noperspective sample in vec2    vTexcoords;     
flat                 in vec3    vWorldPosition; 
uniform sampler2D               ColorBuffer;    
uniform sampler2D               NormalBuffer;   
uniform sampler2D               DepthBuffer;    
uniform uint                    DebugMode;      
out vec4                        FragColor;      







float ComputeGridAlpha(in vec3 worldPosition)
{
    vec2 gridCoordinates = abs(fract(worldPosition.xz - 0.5f) - 0.5f) / fwidth(worldPosition.xz);
    return 0.7f - min(min(gridCoordinates.x, gridCoordinates.y), 0.7f);
}

_kernel


void Debug()
{
    vec4 color = textureLod(ColorBuffer, vTexcoords, 0.0f);
    const float depth = textureLod(DepthBuffer, vTexcoords, 0.0f).x;
    if(depth >= 1.0f)
        color.xyz = GetPassthruColor(vec3(0.1f, 0.1f, 0.8f));
    else
        switch(DebugMode)
        {
        case DEBUG_NORMAL_BUFFER:   
        case DEBUG_DETAILS_BUFFER:
            color.xyz = GetPassthruColor(pow(color.xyz, vec3(1.3f)));
            break;
        case DEBUG_CURVATURE_BUFFER:
            {
                const float curvature = color.w - (color.w > 0.5f ? 128.0f / 255.0f : 0.0f);
                color.xyz = vec3(255.0f * curvature / 127.0f);
            }
            break;
        case DEBUG_POSITION_BUFFER:
            {
                const vec3 position = GetWorldPosition(GetScreenPosition(vTexcoords), depth);
                color.xyz = abs(vec3(uvec3(abs(position)) & 1u) - fract(abs(position)));
                color.xyz = GetPassthruColor(color.xyz);
            }
            break;
        case DEBUG_FRESNEL_BUFFER:
            color.xyz = pow(color.xyz, vec3(0.7f));
            break;
        case DEBUG_VELOCITY_BUFFER:
            {
                const vec2 velocity = color.xy;
                color.xyz = pow(vec3(
                    (velocity.y >= 0.0f ?  velocity.y : 0.0f) + (velocity.y < 0.0f ? -velocity.y : 0.0f),
                    (velocity.x <  0.0f ? -velocity.x : 0.0f) + (velocity.y < 0.0f ? -velocity.y : 0.0f),
                    (velocity.x >= 0.0f ?  velocity.x : 0.0f)), vec3(0.3f));
            }
            break;
        case DEBUG_ROUGHNESS_BUFFER:
            color.xyz = vec3(color.w);
            break;
        case DEBUG_AMBIENT_OCCLUSION:
            color.xyz = vec3(Square(1.0f - color.x));
            break;
        case DEBUG_SPECULAR_OCCLUSION:
            {
                const vec3 normal = normalize(2.0f * textureLod(NormalBuffer, vTexcoords, 0.0f).xyz - 1.0f);
                const vec3 worldPosition = GetWorldPosition(GetScreenPosition(vTexcoords), depth);
                const vec3 viewDirection = normalize(Eye - worldPosition);  
                color.xyz = vec3(SpecularOcclusion(max(dot(normal, viewDirection), 0.0f), 1.0f - color.x));
            }
            break;
        case DEBUG_GLOBAL_ILLUMINATION:
        case DEBUG_GLOSSY_REFLECTIONS:
            color.xyz /= FW_PI;
            break;
        default:
            break;
        }
    FragColor = vec4(color.xyz, 1.0f);
}

_kernel


void DrawGrid()
{
    vec3 worldPosition;

    
    float rayIntersectionDistance;
    vec2 normalizedDeviceCoordinates = 2.0f * vTexcoords - 1.0f;
    vec4 homogeneousCoordinates = ViewProjectionInverse * vec4(normalizedDeviceCoordinates, 0.0f, 1.0f);
    vec3 rayDestination = homogeneousCoordinates.xyz / homogeneousCoordinates.w,
         rayDirection   = normalize(rayDestination - Eye);
    if(!RayIntersectPlaneTest(vec3(0.0f, -1e-3f, 0.0f), vec3(0.0f, 1.0f, 0.0f), Eye, rayDirection, rayIntersectionDistance))
        discard;    
    worldPosition = Eye + rayIntersectionDistance * rayDirection;

    
    float normalizationFactor = mix(0.1f, 2.0f, min(rayIntersectionDistance * 0.01f, 1.0f)),
          normalizedRayIntersectionDistance = log(normalizationFactor * rayIntersectionDistance + 1.0f),
          normalizedRayIntersectionDistanceFract = fract(normalizedRayIntersectionDistance);
    uint hiGridRes = uint(normalizedRayIntersectionDistance + 0.0f) & (~1u),
         loGridRes = uint(normalizedRayIntersectionDistance + 1.0f) & (~1u);
    float hiGridAlpha = ComputeGridAlpha(3.0f * worldPosition / float(1u << hiGridRes)),
          loGridAlpha = ComputeGridAlpha(3.0f * worldPosition / float(1u << loGridRes));
    hiGridAlpha *= (uint(normalizedRayIntersectionDistance + 0.0f) == hiGridRes ? normalizedRayIntersectionDistanceFract : 1.0f - normalizedRayIntersectionDistanceFract);
    loGridAlpha *= (uint(normalizedRayIntersectionDistance + 1.0f) == loGridRes ? normalizedRayIntersectionDistanceFract : 1.0f - normalizedRayIntersectionDistanceFract);
    float gridAlpha = hiGridAlpha + loGridAlpha;    

    
    float distanceToCenter = distance(worldPosition, vWorldPosition),
          projectedDistanceToCenter = max(abs(Eye.y), 1.0f);
    gridAlpha /= 1.0f + 0.1f * max(distanceToCenter, 0.0f) / projectedDistanceToCenter;
    gridAlpha *= 1.0f - 0.001f * max(rayIntersectionDistance - 1000.0f, 0.0f);

    
    vec4 clipSpacePosition = ViewProjection * vec4(worldPosition, 1.0f);
    clipSpacePosition.xyz /= clipSpacePosition.w;   
    float depth = 0.5f * ((gl_DepthRange.far - gl_DepthRange.near) * clipSpacePosition.z + gl_DepthRange.near + gl_DepthRange.far);

    
    FragColor = vec4(GetPassthruColor(vec3(0.4f)), max(gridAlpha, 0.0f));
    gl_FragDepth = depth;   
}

_kernel


void DrawLines()
{
    FragColor = vec4(vec3(1.0f), 1.0f / 3.0f);
}

_kernel


void DrawGizmo()
{
    FragColor = vec4(vColor, 1.0f);
}
