

#include "Shader.h"
#include "Lighting.h"

layout(location = 0) out vec4   NormalOut;                                                  
layout(location = 1) out vec2   VelocityOut;                                                
uniform sampler2D               ColorBuffer;                                                
uniform sampler2D               DepthBuffer;                                                
uniform sampler2D               NormalBuffer;                                               
uniform sampler2D               DetailsBuffer;                                              
uniform sampler2D               SpecularBuffer;                                             
uniform sampler2D               VelocityBuffer;                                             
uniform sampler2D               ReflectionsBuffer;                                          
layout(std430) buffer           DepthBoundsBuffer      { vec2 depthBoundsBuffer[];      };  
layout(std430) buffer           DissolveConstantBuffer { vec4 dissolveConstantBuffer[]; };  
uniform samplerCube             LightProbeBuffer;                                           
uniform float                   LightProbeRoughness;                                        
uniform uint                    ShadowCasterCount;                                          
uniform mat4                    ViewMatrix;                                                 
uniform mat4                    ViewProjectionMatrix;                                       
in vec3                         vNormal;                                                    
flat in vec3                    vMeshData;                                                  
in vec2                         vTexcoords;                                                 
in vec3                         vObjectPosition;                                            
in vec3                         vWorldPosition;                                             
in vec4                         vCurrentPosition;                                           
in vec4                         vPreviousPosition;                                          
out vec4                        FragColor;                                                  
#ifdef USE_DISSOLVE_SHADER
uniform shader                  DissolveShader;                                             
#endif 

#include "StochasticDissolve.h"








vec2 CalculatePixelVelocity(in vec4 currentPosition, in vec4 previousPosition)
{
    vec4 currentUv = Reprojection * vec4(currentPosition.xyz / currentPosition.w, 1.0f),
         previousUv = 0.5f * previousPosition / previousPosition.w + 0.5f;
    currentUv = 0.5f * currentUv / currentUv.w + 0.5f;
    return currentUv.xy - previousUv.xy;
}





float CalculateCurvature()
{
    const vec3 normalDeriv = fwidth(vNormal);
    const float curvature = sqrt(length(normalDeriv));
    return (clamp(curvature, 0.0f, 127.0f / 255.0f) * 255.0f + (vMeshData.z != 0.0f ? 0.0f : 128.0f)) / 255.0f;
}

_kernel


void DrawDepth()
{
    StochasticAlphaTesting(false);
}

_kernel


void DrawDepthNormal()
{
    const float curvature = CalculateCurvature();
    StochasticAlphaTesting(true);   
    FragColor = vec4(0.5f * normalize(vNormal * (gl_FrontFacing ? 1.0f : -1.0f)) + 0.5f, curvature);
}

_kernel


void DrawDepthNormalVelocity()
{
    const float curvature = CalculateCurvature();
    StochasticAlphaTesting(true);   
    NormalOut = vec4(0.5f * normalize(vNormal * (gl_FrontFacing ? 1.0f : -1.0f)) + 0.5f, curvature);
    VelocityOut = CalculatePixelVelocity(vCurrentPosition, vPreviousPosition);
}

_kernel


void DrawHistory()
{
    vec3 color = textureLod(ColorBuffer, vTexcoords, 0.0f).xyz;
    const float depth = textureLod(DepthBuffer, vTexcoords, 0.0f).x;
    if(depth < 1.0f)
    {
        const vec3 kS = textureLod(SpecularBuffer, vTexcoords, 0.0f).xyz,
                   diffuse = textureLod(DiffuseBuffer, vTexcoords, 0.0f).xyz;
        color += kS * diffuse;  
    }
    FragColor = vec4(color, 1.0f);
}

_kernel


void DrawReflections()
{
    float ao = 1.0f;
    vec3 diffuse = vec3(0.0f), specular = vec3(0.0f);
    const float depth = textureLod(DepthBuffer, vTexcoords, 0.0f).x;
    if(depth >= 1.0f)
        discard;    
    const vec4 kS = textureLod(SpecularBuffer, vTexcoords, 0.0f);
    const vec4 reflections = textureLod(ReflectionsBuffer, vTexcoords, 0.0f);
    if(reflections.w <= 0.0f)
    {
        FragColor = vec4(kS.xyz * reflections.xyz, 1.0f);
        return; 
    }
    const vec3 position = GetWorldPosition(GetScreenPosition(vTexcoords), depth),
               normal = normalize(2.0f * textureLod(DetailsBuffer, vTexcoords, 0.0f).xyz - 1.0f),
               viewDirection = normalize(Eye - position);   
    const vec3 reflectionDirection = reflect(-viewDirection, normal);
#ifdef USE_IBL
    const float roughness = Square(kS.w),   
                mipLevel = roughness * float(FW_ENVIRONMENT_MIPS - 1u);
    diffuse += SkylightIntensity * textureLod(IrradianceBuffer, normal, 0.0f).xyz;
    specular += SkylightIntensity * textureLod(EnvironmentBuffer, reflectionDirection, mipLevel).xyz;
#endif 
#ifdef HAS_DIFFUSE_BUFFER
    diffuse += textureLod(DiffuseBuffer, vTexcoords, 0.0f).xyz;
#endif 
#ifdef HAS_OCCLUSION_BUFFER
    ao = min(ao, 1.0f - textureLod(OcclusionBuffer, vTexcoords, 0.0f).x);
#endif 
    const float nDotV = max(dot(normal, viewDirection), 0.0f);
    const float so = SpecularOcclusion(nDotV, ao);  
    FragColor = vec4(kS.xyz * mix(ao * kS.xyz * diffuse, specular, so), 1.0f);
}

_kernel


void DrawBrdf()
{
    const uint sampleCount = 1024u;

    
    const float nDotV = vTexcoords.x;
    const float roughness = vTexcoords.y;
    const vec3 N = vec3(0.0f, 0.0f, 1.0f);
    const vec3 V = vec3(
        sqrt(1.0f - nDotV * nDotV),
        0.0f,
        nDotV);

    
    float A = 0.0f;
    float B = 0.0f;

    
    for(uint i = 0u; i < sampleCount; ++i)
    {
        const vec2 Xi = Hammersley(i, sampleCount);
        const vec3 H  = ImportanceSampleGGX(Xi, N, roughness);
        const vec3 L  = normalize(2.0f * dot(V, H) * H - V);

        const float nDotL = max(L.z, 0.0f);
        const float nDotH = max(H.z, 0.0f);
        const float vDotH = max(dot(V, H), 0.0f);

        if(nDotL > 0.0f)
        {
            const float G = GeometrySmithIBL(nDotV, nDotL, roughness);
            const float G_Vis = (G * vDotH) / (nDotH * nDotV);
            const float Fc = pow(1.0f - vDotH, 5.0f);

            A += (1.0f - Fc) * G_Vis;
            B += Fc * G_Vis;
        }
    }

    
    FragColor = vec4(vec2(A, B) / float(sampleCount), 0.0f, 1.0f);
}

_kernel


void DrawEnvironment()
{
    float totalWeight = 0.0f;
    const uint sampleCount = 4096u;
    vec3 prefilteredColor = vec3(0.0f);

    
    const vec2 ndc = 2.0f * vTexcoords - 1.0f;
    vec4 worldPosition = ViewProjectionInverse * vec4(ndc, 0.0f, 1.0f);
    worldPosition.xyz /= worldPosition.w;   

    
    const vec3 N = normalize(worldPosition.xyz - Eye);
    const vec3 R = N;
    const vec3 V = R;

    
    for(uint i = 0u; i < sampleCount; ++i)
    {
        const vec2 Xi = Hammersley(i, sampleCount);
        const vec3 H  = ImportanceSampleGGX(Xi, N, LightProbeRoughness);
        const vec3 L  = normalize(2.0f * dot(V, H) * H - V);

        const float nDotL = max(dot(N, L), 0.0f);
        if(nDotL > 0.0f)
        {
            
            const float nDotH = max(dot(N, H), 0.0f);
            const float hDotV = max(dot(H, V), 0.0f);
            const float D     = DistributionGGX(nDotH, LightProbeRoughness);
            const float pdf   = D * nDotH / (4.0f * hDotV) + 0.0001f;

            
            const float resolution = ViewportWidthHeight.x;
            const float saTexel    = 4.0f * FW_PI / (6.0f * resolution * resolution);
            const float saSample   = 1.0f / (float(sampleCount) * pdf + 0.0001f);
            const float mipLevel   = (LightProbeRoughness == 0.0f ? 0.0f : 0.5f * log2(saSample / saTexel));

            
            prefilteredColor += textureLod(LightProbeBuffer, L, mipLevel).rgb * nDotL;
            totalWeight += nDotL;
        }
    }

    
    FragColor = vec4(prefilteredColor / totalWeight, 1.0f);
}

_kernel


void DrawIrradiance()
{
    vec3 irradiance = vec3(0.0f);

    
    const vec2 ndc = 2.0f * vTexcoords - 1.0f;
    vec4 worldPosition = ViewProjectionInverse * vec4(ndc, 0.0f, 1.0f);
    worldPosition.xyz /= worldPosition.w;   
    const vec3 forward = normalize(worldPosition.xyz - Eye);

    
    vec3 up = vec3(0.0f, 1.0f, 0.0f);
    vec3 right = cross(up, forward);
    up = cross(forward, right);

    
    float sampleCount = 0.0f, sampleDelta = 0.025f;
    for(float phi = 0.0f; phi < 2.0f * FW_PI; phi += sampleDelta)
        for(float theta = 0.0f; theta < 0.5f * FW_PI; theta += sampleDelta)
        {
            
            vec3 tangentSample = vec3(sin(theta) * cos(phi),  sin(theta) * sin(phi), cos(theta));
            
            vec3 sampleVector = tangentSample.x * right + tangentSample.y * up + tangentSample.z * forward;
            
            irradiance += textureLod(LightProbeBuffer, sampleVector, 0.0f).rgb * cos(theta) * sin(theta);
            
            ++sampleCount;
        }

    
    FragColor = vec4(FW_PI * irradiance / sampleCount, 1.0f);
}

_kernel


void DrawGBuffers()
{
    if(vTexcoords.y > 0.5f)
    {
        if(vTexcoords.x < 0.5f)
        {
            
            const vec2 uv = 2.0f * vec2(vTexcoords.x, vTexcoords.y - 0.5f);
            FragColor = sqrt(texture(SpecularBuffer, uv));
        }
        else
        {
            
            const vec2 uv = 2.0f * vTexcoords - 1.0f;
            FragColor = texture(NormalBuffer, uv);
        }
    }
    else
    {
        if(vTexcoords.x < 0.5f)
        {
            
#ifndef HAS_VELOCITY_BUFFER
            FragColor = vec4(0.0f, 0.0f, 0.0f, 1.0f);
#else 
            const vec2 uv = 2.0f * vTexcoords;
            const vec2 velocity = texture(VelocityBuffer, uv).xy;
            
            
            
            
            FragColor = vec4(pow(vec3(
                (velocity.y >= 0.0f ?  velocity.y : 0.0f) + (velocity.y < 0.0f ? -velocity.y : 0.0f),
                (velocity.x <  0.0f ? -velocity.x : 0.0f) + (velocity.y < 0.0f ? -velocity.y : 0.0f),
                (velocity.x >= 0.0f ?  velocity.x : 0.0f)), vec3(0.3f)),
                1.0f);
#endif 
        }
        else
        {
            
            const vec2 uv = 2.0f * vec2(vTexcoords.x - 0.5f, vTexcoords.y);
            const float depth = texture(DepthBuffer, uv).x; 
            const vec3 position = GetWorldPosition(GetScreenPosition(uv), depth);
            const vec3 fragColor = abs(vec3(uvec3(abs(position)) & 1u) - fract(abs(position)));
            FragColor = vec4(depth == 1.0f ? texture(SpecularBuffer, uv).xyz : pow(fragColor, vec3(1.0f / 1.5f)), 1.0f);
        }
    }
    FragColor = vec4(GetPassthruColor(FragColor.xyz), 1.0f);
}

_kernel


void DrawLightCount()
{
    uint lightCount = 0;
    const float depth = texture(DepthBuffer, vTexcoords).x;
    if(depth >= 1.0f)
    {
        FragColor = vec4(vec3(0.0f), 1.0f);
        return; 
    }
    const vec3 position = vec3(gl_FragCoord.xy * ViewportWidthHeightInverse, depth);
    for(LightIterator lightIt = LightBegin(position); !LightEnd(lightIt); LightNext(lightIt))
        ++lightCount;   
    FragColor = vec4(GetPassthruColor(ViridisQuintic(1e-2f * lightCount)), 1.0f);
}

_kernel


void DrawDepthBounds()
{
#ifndef HAS_DEPTH_BOUNDS
    
    FragColor = vec4(GetPassthruColor(vec3(1.0f, 0.0f, 0.0f)), 1.0f);
#else 
    uvec2 tileCoordinates = uvec2(vTexcoords * ViewportWidthHeight / float(FW_TILE_SIZE));
    vec2 depthBounds = depthBoundsBuffer[tileCoordinates.x + tileCoordinates.y * TileCount.x];
    FragColor = vec4(GetPassthruColor(sqrt(vec3(depthBounds.x / 100.0f))), 1.0f);
#endif 
}

_kernel


void DrawViewFrustum()
{
    FragColor = vec4(GetPassthruColor(vec3(1.0f)), 1.0f);
}

_kernel


void DrawShadowBuffer()
{
    vec3 shadows = 1.0f - texture(ShadowBuffer, vec3(vTexcoords, 0.0f)).xyz;
    if(ShadowCasterCount < 3)
        shadows.z = 0.0f;
    if(ShadowCasterCount < 2)
        shadows.y = 0.0f;
    if(ShadowCasterCount < 1)
        shadows.x = 0.0f;
    FragColor = vec4(GetPassthruColor(shadows), 1.0f);
}
