

#include "ShaderCore.h"



#define GHOST_TINT_PER_SAMPLE           1



#define DISABLE_HALO_ASPECT_RATIO       0



#define DISABLE_CHROMATIC_ABERRATION    0

noperspective sample in vec2    vTexcoords;             
uniform sampler2D               ColorBuffer;            
uniform sampler2D               LensDirtBuffer;         
uniform sampler2D               StarburstBuffer;        
uniform sampler2D               GhostColorBuffer;       
uniform float                   Intensity;              
uniform float                   HaloAspect;             
uniform float                   HaloRadius;             
uniform float                   HaloThickness;          
uniform float                   HaloThreshold;          
uniform uint                    GhostCount;             
uniform float                   GhostSpacing;           
uniform float                   GhostThreshold;         
uniform float                   LensDirtAspect;         
uniform float                   ChromaticAberration;    
uniform uint                    FeaturesMipLevel;       
out vec4                        FragColor;              








vec3 ApplyThreshold(in vec3 color, in float threshold)
{
    return max(color - threshold, 0.0f);
}







vec3 SampleSceneColor(in vec2 uv)
{
#if DISABLE_CHROMATIC_ABERRATION
    return textureLod(ColorBuffer, uv, int(FeaturesMipLevel)).xyz;
#else 
    const vec2 offset = normalize(vec2(0.5f) - uv) * ChromaticAberration;
    return vec3(
        textureLod(ColorBuffer, uv + offset, int(FeaturesMipLevel)).x,
        textureLod(ColorBuffer, uv,          int(FeaturesMipLevel)).y,
        textureLod(ColorBuffer, uv - offset, int(FeaturesMipLevel)).z);
#endif 
}








vec3 SampleGhosts(in vec2 uv, in float threshold)
{
    vec3 result = vec3(0.0f);

    
    const vec2 ghostVec = (0.5f - uv) * GhostSpacing;
    for(uint i = 0; i < GhostCount; ++i)
    {
        
        const vec2 suv = fract(uv + ghostVec * i);
        vec3 s = SampleSceneColor(suv);
        s = ApplyThreshold(s, threshold);

        
        const float distanceToCenter = distance(suv, vec2(0.5f));
#if GHOST_TINT_PER_SAMPLE
        s *= textureLod(GhostColorBuffer, vec2(distanceToCenter, 0.5f), 0.0f).xyz;  
#else 
        s *= 1.0f - smoothstep(0.0f, 0.75f, distanceToCenter);  
#endif 

        
        result += s;
    }
#if !GHOST_TINT_PER_SAMPLE
    result *= textureLod(GhostColorBuffer, vec2(distance(uv, vec2(0.5f)), 0.5f), 0.0f).xyz;
#endif 

    return result;
}









float Cubic(in float x, in float center, in float radius)
{
    x = min(abs(x - center) / radius, 1.0f);
    return 1.0f - x * x * (3.0f - 2.0f * x);
}










vec3 SampleHalo(in vec2 uv, in float aspect, in float radius, in float threshold)
{
    vec2 haloVec = vec2(0.5f) - uv;
#if DISABLE_HALO_ASPECT_RATIO
    haloVec = normalize(haloVec);
    float haloWeight = distance(uv, vec2(0.5f));
#else 
    haloVec.x /= aspect;
    haloVec = normalize(haloVec);
    haloVec.x *= aspect;
    const vec2 wuv = (uv - vec2(0.5f, 0.0f)) / vec2(aspect, 1.0f) + vec2(0.5f, 0.0f);
    float haloWeight = distance(wuv, vec2(0.5f));
#endif 
    haloVec *= radius;
    haloWeight = Cubic(haloWeight, radius, HaloThickness);
    return ApplyThreshold(SampleSceneColor(uv + haloVec), threshold) * haloWeight;
}

_kernel


void LensFlareFeatures()
{
    vec3 result = vec3(0.0f);
    result += SampleGhosts(vTexcoords, GhostThreshold);
    result += SampleHalo(vTexcoords, HaloAspect, HaloRadius, HaloThreshold);
    FragColor = vec4(result, 1.0f);
}

_kernel


void LensFlareComposite()
{
    
    const vec3 viewVec = GetForward();
    const float starburstOffset = (viewVec.x + viewVec.y + viewVec.z);

    
    const vec2 centerVec = vTexcoords - 0.5f;
    const float d = length(centerVec);
    const float radial = acos(centerVec.x / d);
    float mask =
        texture(StarburstBuffer, vec2(radial + starburstOffset * 1.0f, 0.0f)).x *
        texture(StarburstBuffer, vec2(radial - starburstOffset * 0.5f, 0.0f)).x;
    mask = clamp(mask + (1.0f - smoothstep(0.0f, 0.3f, d)), 0.0f, 1.0f);

    
    vec2 ndc = 2.0f * vTexcoords - 1.0f;
    const float aspect = Aspect / LensDirtAspect;
    if(aspect > 1.0f) ndc.y /= aspect;
                 else ndc.x *= aspect;
    mask *= texture(LensDirtBuffer, 0.5f * ndc + 0.5f).x;

    
    FragColor = vec4(textureLod(ColorBuffer, vTexcoords, 0.0f).xyz * mask * Intensity, 1.0f);
}
