

#ifndef LIGHTING_H
#define LIGHTING_H

#include "Bxdf.h"
#include "LinearlyTransformedCosines.h"


vec3 EvaluateLocalIllumination(in LightData light, in Material material, in vec3 normal, in vec3 viewDirection, in vec3 worldPosition);
vec3 EvaluateDirectIllumination(in LightData light, in Material material, in vec3 normal, in vec3 viewDirection, in vec3 worldPosition);
vec3 EvaluateIndirectIllumination(in vec2 uv, in Material material, in vec3 normal, in vec3 viewDirection, out vec4 specularTerm);
vec3 EvaluateIndirectIllumination(in vec2 uv, in Material material, in vec3 normal, in vec3 viewDirection);















float EvaluateAttenuation(in vec4 lightData, in float lightDistance)
{
    float lightAttenuationDenom = max(lightDistance - lightData.x, 0.0f) / lightData.x + 1.0f,
          lightAttenuation = 1.0f / (lightAttenuationDenom * lightAttenuationDenom);
    lightAttenuation = (lightAttenuation - lightData.y) / (1.0f - lightData.y);
    return max(lightAttenuation, 0.0f); 
}













vec3 EvaluateIllumination(in vec3 lightDirection, in Material material, in vec3 normal, in vec3 viewDirection)
{
    vec3 halfwayVector = normalize(viewDirection + lightDirection);

    
    float nDotH = max(dot(normal, halfwayVector), 0.0f),
          nDotL = max(dot(normal, lightDirection), 0.0f),
          nDotV = max(dot(normal, viewDirection), 0.0f),
          NDF   = DistributionGGX(nDotH, material.m_roughness),
          G     = GeometrySmith(nDotV, nDotL, material.m_roughness);
    vec3  F     = FresnelSchlick(max(dot(halfwayVector, viewDirection), 0.0f), material.m_fresnelBase);

    
    vec3 nominator = NDF * G * F;
    float denominator = 4.0f * nDotV * nDotL + 0.001f;
    vec3 specular = nominator / denominator;

    
    vec3 kD = 1.0f - F; 
    kD *= 1.0f - material.m_metallicity;
    vec3 diffuse = kD * material.m_albedo;

#ifndef USE_WATER_SHADER
    return (diffuse / FW_PI + specular) * nDotL;
#else 
    return diffuse / FW_PI * pow(0.4f * dot(normal, lightDirection) + 0.6f, 80.0f) + specular * nDotL;
#endif 
}















vec3 EvaluateIllumination(in vec3 points[4], in Material material, in vec3 normal, in vec3 position, in vec3 viewDirection, in bool twoSided)
{
    
    const float nDotV = max(dot(normal, viewDirection), 0.0f);
    const vec3 F = FresnelSchlickRoughness(nDotV, material.m_fresnelBase, material.m_roughness);
    const vec2 brdfSample = texture(BrdfBuffer, vec2(max(dot(normal, viewDirection), 0.0f), material.m_roughness)).rg;
    vec3 kS = F * brdfSample.x + brdfSample.y, kD = 1.0f - F;

    
    vec2 uv = vec2(material.m_roughness, sqrt(1.0f - nDotV));
    uv = uv * FW_LTC_LUT_SCALE + FW_LTC_LUT_BIAS;
    vec4 t1 = textureLod(LtcMatBuffer, uv, 0.0f);
    vec4 t2 = textureLod(LtcAmpBuffer, uv, 0.0f);

    
    vec3 T1, T2;
    T1 = normalize(viewDirection - normal * dot(viewDirection, normal));
    T2 = cross(normal, T1);

    mat3 tbn = transpose(mat3(T1, T2, normal));

    
    mat3 Minv = mat3(vec3(t1.x, 0.0f, t1.y),
                     vec3(0.0f, 1.0f, 0.0f),
                     vec3(t1.z, 0.0f, t1.w));
    vec3 specular = LTC_Evaluate(position, Minv * tbn, points, twoSided);
    specular *= kS * t2.x + (1.0f - kS) * t2.y;

    
    vec3 diffuse = LTC_Evaluate(position, tbn, points, twoSided);
    diffuse *= kD * (1.0f - material.m_metallicity);

#ifndef USE_WATER_SHADER
    return diffuse * material.m_albedo + specular;
#else 
    return specular;    
#endif 
}















vec3 EvaluateLocalIllumination(in LightData light, in Material material, in vec3 normal, in vec3 viewDirection, in vec3 worldPosition)
{
    vec3 lighting = light.m_radiance.xyz;

    
    const uint lightType = floatBitsToUint(light.m_radiance.w);
    switch(lightType)
    {
    case lightPoint:
#ifndef USE_LTC
    case lightArea: 
#endif 
        {
            
            vec3 lightDirection = light.m_position.xyz - worldPosition;
            float lightDistance = length(lightDirection);
            lightDirection /= lightDistance;    

            
            lighting *= EvaluateIllumination(
                lightDirection,
                material,
                normal,
                viewDirection);
            lighting *= EvaluateAttenuation(light.m_lightData, lightDistance);
        }
        break;
    case lightSpot:
        {
            
            vec3 lightDirection = light.m_position.xyz - worldPosition;
            float lightDistance = length(lightDirection);
            lightDirection /= lightDistance;    

            
            float spotFactor = dot(-lightDirection, light.m_direction.xyz);
            if(spotFactor < light.m_lightData.w)
                return vec3(0.0f);  

            
            lighting *= EvaluateIllumination(
                lightDirection,
                material,
                normal,
                viewDirection);
            lighting *= EvaluateAttenuation(light.m_lightData, lightDistance);

            
            float spotAttenuation = (light.m_lightData.w - spotFactor) / (light.m_lightData.w - light.m_direction.w);
            spotAttenuation = clamp(spotAttenuation, 0.0f, 1.0f);
            lighting *= spotAttenuation * spotAttenuation;
        }
        break;
    case lightDirectional:
        {
            
            lighting *= EvaluateIllumination(
                -light.m_direction.xyz,
                material,
                normal,
                viewDirection);
        }
        break;
#ifdef USE_LTC
    case lightArea:
        {
            
            vec3 points[4];
            int areaLightIndex = int(floatBitsToUint(light.m_direction.w) << 2);
            vec4 areaLight = texelFetch(AreaLightBuffer, areaLightIndex + 0);
            points[0] = areaLight.xyz;  
            points[1] = texelFetch(AreaLightBuffer, areaLightIndex + 1).xyz;
            points[2] = texelFetch(AreaLightBuffer, areaLightIndex + 2).xyz;
            points[3] = texelFetch(AreaLightBuffer, areaLightIndex + 3).xyz;

            
            lighting *= EvaluateIllumination(
                points,
                material,
                normal,
                worldPosition,
                viewDirection,
                areaLight.w != 0.0f);

            
            vec3 lightDirection = light.m_position.xyz - worldPosition;
            float lightDistance = length(lightDirection);
            lighting *= EvaluateAttenuation(light.m_lightData, lightDistance);
        }
        break;
#endif 
    default:
        return vec3(0.0f);  
    }

    return lighting;
}











vec3 EvaluateDirectIllumination(in LightData light, in Material material, in vec3 normal, in vec3 viewDirection, in vec3 worldPosition)
{
    vec3 lighting = EvaluateLocalIllumination(light, material, normal, viewDirection, worldPosition);
#ifdef HAS_SHADOWS
#ifdef FW_FRAGMENT_STAGE
    if(floatBitsToUint(light.m_lightData.z) != uint(-1) && dot(lighting, lighting) > 0.0f)
    {
        const uint shadowBufferIndex      = (floatBitsToUint(light.m_lightData.z) & 3),
                   shadowBufferSliceIndex = (floatBitsToUint(light.m_lightData.z) >> 2);
        lighting *= texture(ShadowBuffer, vec3(gl_FragCoord.xy * ViewportWidthHeightInverse, float(shadowBufferSliceIndex)))[shadowBufferIndex];
    }
#endif 
#endif 
    return lighting;
}











vec3 EvaluateIndirectIllumination(in vec2 uv, in Material material, in vec3 normal, in vec3 viewDirection, out vec4 specularTerm)
{
    
    float ao = material.m_ao;
    vec3 diffuse = vec3(0.0f);
    vec3 specular = vec3(0.0f);

    
    const float nDotV = max(dot(normal, viewDirection), 0.0f);
    const vec3 F = FresnelSchlickRoughness(nDotV, material.m_fresnelBase, material.m_roughness);
    const vec3 kD = (1.0f - F) * (1.0f - material.m_metallicity * (1.0f - material.m_roughness));
    const vec2 brdfSample = textureLod(BrdfBuffer, vec2(nDotV, material.m_roughness), 0.0f).xy;
    const vec3 kS = F * brdfSample.x + brdfSample.y;    

    
#ifdef USE_IBL
#ifdef USE_DIFFUSE_IBL
    diffuse += SkylightIntensity * texture(IrradianceBuffer, normal).xyz;
#endif 
#ifdef USE_SPECULAR_IBL
    const vec3 reflectionDirection = reflect(-viewDirection, normal);
    const float mipLevel = material.m_roughness * float(FW_ENVIRONMENT_MIPS - 1u);
    specular += SkylightIntensity * textureLod(EnvironmentBuffer, reflectionDirection, mipLevel).xyz;
#endif 
#endif 
#ifdef HAS_DIFFUSE_BUFFER
    diffuse += textureLod(DiffuseBuffer, uv, 0.0f).xyz;
#endif 

    
#ifdef HAS_OCCLUSION_BUFFER
    ao = min(ao, 1.0f - textureLod(OcclusionBuffer, uv, 0.0f).x);
#endif 
#ifdef USE_SPECULAR_IBL
    const float so = SpecularOcclusion(nDotV, ao);
#else 
    const float so = 1.0f;  
#endif 
#ifndef USE_WATER_SHADER
    diffuse *= ao;  
#else 
    const float nDotU = max(dot(normal, vec3(0.0f, 1.0f, 0.0f)), 0.0f);
    diffuse *= material.m_albedo * mix(vec3(1.0f), material.m_albedo, Square(Square(nDotU)));
#endif 

    
    specularTerm = vec4(kS, sqrt(material.m_roughness));

    return kD * diffuse * material.m_albedo + kS * mix(kS * diffuse, specular, so);
}










vec3 EvaluateIndirectIllumination(in vec2 uv, in Material material, in vec3 normal, in vec3 viewDirection)
{
    vec4 specularTerm;  
    return EvaluateIndirectIllumination(uv, material, normal, viewDirection, specularTerm);
}

#endif 
