

#include "Sampling.h"



struct LineConnection
{
    float   m_age;              
    float   m_life;             
    float   m_alpha;            
    uint    m_particleIndex;    
};

layout(std430) buffer   AgeBuffer                 { float          ageBuffer[];                 };  
layout(std430) buffer   LifeBuffer                { float          lifeBuffer[];                };  
layout(std430) buffer   ColorBuffer               { uvec2          colorBuffer[];               };  
layout(std430) buffer   LightingBuffer            { uvec2          lightingBuffer[];            };  
layout(std430) buffer   InitialColorBuffer        { uvec2          initialColorBuffer[];        };  
layout(std430) buffer   EmitterBuffer             { uint           emitterBuffer[];             };  
layout(std430) buffer   PositionBuffer            { vec4           positionBuffer[];            };  
layout(std430) buffer   AliveListBuffer           { uint           aliveListBuffer[];           };  
layout(std430) buffer   PerEmitterCountBuffer     { uint           perEmitterCountBuffer[];     };  
layout(std430) buffer   PerEmitterIndexBuffer     { uint           perEmitterIndexBuffer[];     };  
layout(std430) buffer   PerEmitterFlagsBuffer     { uint           perEmitterFlagsBuffer[];     };  
layout(std430) buffer   LineConnectionBuffer      { LineConnection lineConnectionBuffer[];      };  
layout(std430) buffer   LineConnectionIndexBuffer { uvec2          lineConnectionIndexBuffer[]; };  
uniform vec4            Params;                                                                     
uniform uint            EmitterMask;                                                                
flat out float          vParticleAge;                                                               
out vec4                vParticleColor;                                                             
flat out vec3           vParticleCenter;                                                            








uint GetParticleIndex(in uint globalId, out uint emitterIndex)
{
    uint particleIndex = globalId;
#ifndef SORT_PARTICLES
    if(EmitterMask == uint(-1))
#endif 
    {
        particleIndex = aliveListBuffer[particleIndex];
        emitterIndex = emitterBuffer[particleIndex];
        return particleIndex;
    }
#ifndef SORT_PARTICLES
    uint particleCount = 0;
    uint emitterMask = EmitterMask;
    for(;;)
    {
        emitterIndex = findLSB(emitterMask);
        particleCount += perEmitterCountBuffer[emitterIndex];
        if(particleIndex < particleCount)
        {
            particleIndex += perEmitterIndexBuffer[emitterIndex];
            break;  
        }
        emitterMask &= ~(1u << emitterIndex);
    }
    return aliveListBuffer[particleIndex];
#endif 
}







vec4 UnpackColor(in uvec2 packedColor)
{
    return vec4(unpackHalf2x16(packedColor.x), unpackHalf2x16(packedColor.y));
}







vec3 UnpackLighting(in uvec2 packedLighting)
{
    return vec3(unpackHalf2x16(packedLighting.x), unpackHalf2x16(packedLighting.y).x);
}








vec4 ReadColor(in uint particleIndex, in uint emitterIndex)
{
    vec4 color; 
    if((perEmitterFlagsBuffer[emitterIndex] & (1u << 1)) != 0)
        color = UnpackColor(colorBuffer[particleIndex]);
    else
        color = UnpackColor(initialColorBuffer[particleIndex]);
    if((perEmitterFlagsBuffer[emitterIndex] & (1u << 2)) != 0)
        color.xyz *= UnpackLighting(lightingBuffer[particleIndex]);
    return color;
}

_kernel


void RenderLine()
{
    const uvec2 lineConnectionIndex = lineConnectionIndexBuffer[gl_InstanceID];
    const LineConnection lineConnection = lineConnectionBuffer[lineConnectionIndex.y];
    const uint particleIndex = (Position.x < 0.0f ? lineConnectionIndex.x : lineConnection.m_particleIndex);
    gl_Position = ViewProjection * vec4(positionBuffer[particleIndex & 0x7FFFFFFu].xyz, 1.0f);
    vParticleColor = ReadColor(particleIndex & 0x7FFFFFFu, particleIndex >> 27);
    vParticleColor.w *= lineConnection.m_alpha; 
}

_kernel


void RenderPoint()
{
    uint emitterIndex;
    const uint particleIndex = GetParticleIndex(gl_InstanceID, emitterIndex);
    vec4 position = View * vec4(positionBuffer[particleIndex].xyz, 1.0f);
    vec4 particleCenter = Projection * position;    
    particleCenter = 0.5f * particleCenter / particleCenter.w + 0.5f;
    vParticleCenter = vec3(particleCenter.xy * vec2(Aspect, 1.0f), 0.0f);
    vParticleColor = ReadColor(particleIndex, emitterIndex);
    float particleSize = Params.x;  
    if(Params.y != 0.0f)
    {
        Sampler sampler = SamplerInit(particleIndex, 0, 0);
        particleSize += (2.0f * Sample1D(sampler) - 1.0f) * Params.y;
        particleSize = max(particleSize, 0.0f); 
    }
    if(vParticleColor.w <= 0.0f)
        particleSize = 0.0f;    
    position.xy += 1e-2f * particleSize * Position.xy;  
    gl_Position = Projection * position;    
    vParticleAge = clamp(ageBuffer[particleIndex] / lifeBuffer[particleIndex], 0.0f, 1.0f); 
    vParticleCenter.z = distance((0.5f * gl_Position.xy / gl_Position.w + 0.5f) * vec2(Aspect, 1.0f), vParticleCenter.xy);
    vParticleCenter.z = sqrt(0.5f * Square(vParticleCenter.z)); 
}
